# GNU Enterprise Forms - GTK UI Driver - base class for UI widgets
#
# Copyright 2001-2005 Free Software Foundation
#
# This file is part of GNU Enterprise
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: _base.py 7008 2005-02-11 17:03:06Z reinhard $

import gtk
import types

from gnue.common import events
from gnue.common.apps import i18n
from gnue.forms.GFForm import *
from gnue.forms.uidrivers._base.widgets._base import UIWidget
from gnue.forms.uidrivers.gtk2.common import _setDefaultEventHandlers


# =============================================================================
# This class implements the common behaviour of GTK2 widgets
# =============================================================================

class UIHelper (UIWidget):
  """
  Implements the common behaviour of GTK2 widgets
  """

  # ---------------------------------------------------------------------------
  # Create a new GTK2 widget
  # ---------------------------------------------------------------------------

  def __init__ (self, *args, **kwargs):

    UIWidget.__init__ (self, *args, **kwargs)
    self._useNewGTKAPI = gtk.pygtk_version >= (2, 4, 0)


  # ---------------------------------------------------------------------------
  # Create a new GTK widget
  # ---------------------------------------------------------------------------

  def createWidget (self, event, spacer):
    """
    This function creates a new GTK widget and adds it to the cross reference
    table.
    """
    gfObject           = event.object
    self._eventHandler = event.eventHandler

    if hasattr (gfObject, 'Char__y'):
      posY = gfObject.Char__y
      gap  = gfObject._gap + 1
      self.itemY = (posY + spacer * gap) * event.widgetHeight

    if hasattr (gfObject, 'Char__height'):
      self.itemHeight = gfObject.Char__height * event.widgetHeight

    newWidget = self._createWidget (event, spacer)
    if event.initialize:
      self._addToCrossRef (newWidget, gfObject, self)

    return newWidget


  # ---------------------------------------------------------------------------
  # Add a widget to the cross reference tables
  # ---------------------------------------------------------------------------

  def _addToCrossRef (self, widget, gfobject, uiobject):
    self._uiDriver._WidgetToGFObj [widget] = gfobject
    self._uiDriver._WidgetToUIObj [widget] = uiobject


  # ---------------------------------------------------------------------------
  # Remove a widget from the cross reference tables
  # ---------------------------------------------------------------------------

  def _deleteFromCrossRef (self, widget, object):
    try:
      del self._uiDriver._WidgetToGFObj [widget]
      del self._uiDriver._WidgetToUIObj [widget]

    except:
      pass


  # ---------------------------------------------------------------------------
  # Add all default event handler to the widget
  # ---------------------------------------------------------------------------

  def _addDefaultEventHandler (self, widget, lookup = None):
    _setDefaultEventHandlers (widget, self._eventHandler, True, self._uiDriver,
                              lookup)



  # ---------------------------------------------------------------------------
  # Add a focus-in handler to a widget
  # ---------------------------------------------------------------------------

  def _addFocusHandler (self, widget, lookupWidget = None, after = False):
    """
    This function connects a handler to the widget's focus-in-event signal
    passing an optional lookup widget to it. A lookup widget will be used for
    retrieving the GFObject from the lookup table. If no lookup widget is given
    the mandatory widget will be used for lookups.
    """

    if lookupWidget is None:
      lookupWidget = widget
    gDebug (6, "ADD FOCUSHANDLER %s (%s)" % (widget, lookupWidget))
    method = after and widget.connect_after or widget.connect
    widget._focusHandler = method ('focus-in-event', self.__focusInHandler,
                                   lookupWidget)


  # ---------------------------------------------------------------------------
  # Handle focus signals from widgets
  # ---------------------------------------------------------------------------

  def __focusInHandler (self, widget, event, lookupWidget):
    """
    This function handles 'focus-in-event' signals by creating an immediate
    call to 'requestFOCUS'. If the focused widget needs a jump to another
    record an additional 'requestJUMPRECORD' event will be fired.
    """
    gDebug (4, "FOCUS-IN-EVENT: %s for %s" % (widget, lookupWidget))

    gfObject = self._uiDriver._WidgetToGFObj [lookupWidget]
    _formRef = gfObject._form

    if _formRef._currentEntry == gfObject and \
       self.widgets.index (lookupWidget) == gfObject._visibleIndex:
      gDebug (6, "SKIPPING since we already have the focus!")
      return gtk.FALSE

    if _formRef._currentEntry != gfObject:
      action = events.Event ('requestFOCUS', gfObject, _form = _formRef)
      self._eventHandler (action)

    adjust = self.widgets.index (lookupWidget) - gfObject._visibleIndex
    if adjust:
      gDebug (6, "Adjusting record to %s" % adjust)
      action = events.Event ('requestJUMPRECORD', adjust, _form = _formRef)
      self._eventHandler (action)

    return gtk.FALSE


  # ---------------------------------------------------------------------------
  # Show all ui-widgets managed by this object
  # ---------------------------------------------------------------------------

  def show (self):
    """
    This function calls the show () on all gtk-widgets managed by this
    instance. This is needed if a row-count greater than 0 is given.
    """
    for widget in self.widgets:
      widget.show ()


  # ---------------------------------------------------------------------------
  # Hide all ui-widgets managed by this object
  # ---------------------------------------------------------------------------

  def hide (self):
    """
    This function calls the hide () on all gtk-widgets managed by this
    instance. This is needed if a row-count greater than 0 is given.
    """
    for widget in self.widgets:
      widget.hide ()


  def showModal (self):
    gDebug (1, "showModal not implemented for gtk2 driver")

  def destroy (self):
    gDebug (1, "destroy not implemented for gtk2 driver")


  # ---------------------------------------------------------------------------
  # Set the focus to a given widget
  # ---------------------------------------------------------------------------

  def indexedFocus (self, index):
    """
    This function set's the focus to the gtk-widget specified by index. If the
    widget has a focusHandler it will be blocked and unblocked to prevent a
    recursion. The focused widget will have a highlight color if it is set
    in gnue.conf.
    """
    widget = self.widgets [index]
    gDebug (6, "indexedFocus: %s [%s]" % (widget, index))

    isCombo = False
    if self._useNewGTKAPI and isinstance (widget, gtk.ComboBoxEntry):
      item = widget.child
      isCombo = True
    else:
      isCombo = isinstance (widget, gtk.Combo)
      item    = isCombo and widget.entry or widget

    self._blockHandler (item, '_focusHandler')

    gDebug (6, "Grab focus to %s for %s" % (item, widget))
    item.grab_focus ()


    # Change the background of a widget if requested via gnue.conf
    if gConfigForms ('focusColor'):
      focusColor = gtk.gdk.color_parse ('#' + gConfigForms ('focusColor'))
      mode   = ['base', 'bg'] [isinstance (item, gtk.Button)]
      setcol = getattr (item, 'modify_%s' % mode)

      if not hasattr (item, '_color'):
        item._color   = getattr (item.get_style (), mode) [gtk.STATE_NORMAL]
        item._setback = setcol

      setcol (gtk.STATE_NORMAL, focusColor)


    # To avoid the automatic select-all in a combo box, we explicitly set the
    # selection to the current position only
    if isCombo:
      pos = item.get_position ()
      item.select_region (pos, pos)

    self._blockHandler (item, '_focusHandler', True)


  # ---------------------------------------------------------------------------
  # On lose of the focus we un-select ComboBox-Entries
  # ---------------------------------------------------------------------------

  def loseFocus (self):
    for widget in self.widgets:
      gDebug (6, "Lose focus: %s" % widget)

      if self._useNewGTKAPI and isinstance (widget, gtk.ComboBoxEntry):
        item = widget.child
      else:
        item = isinstance (widget, gtk.Combo) and widget.entry or widget

      if hasattr (item, '_color'):
        item._setback (gtk.STATE_NORMAL, item._color)

      if isinstance (item, gtk.Entry):
        item.select_region (0,0)


  # ---------------------------------------------------------------------------
  # Set the value of a widget
  # ---------------------------------------------------------------------------

  def setValue (self, value, index = 0, enabled = True):
    """
    This function sets the value of a widget and optionally enables or disables
    the widget.
    """
    widget = self.widgets [index]
    gDebug (6, "setValue %s [%s] %s" % (repr (value), index, widget))

    if not isinstance (widget, gtk.CheckButton) and \
        type (value) != types.UnicodeType:
      gDebug (6, "converting %s to unicode using %s" % \
          (repr (value), gConfigForms ('textEncoding')))
      value = unicode (value, gConfigForms ('textEncoding'))


    # Once we change to PyGTK 2.4.* we can remove the gtk.Combo stuff
    if isinstance (widget, gtk.Combo):
      gfObject = self._uiDriver._WidgetToGFObj [widget]

      # Check if list of allowed value (~= foreign keys, ~= dropdown content)
      # changed
      if gfObject.style == "dropdown":
        self._updateChoices (widget, gfObject)

      self._blockHandler (widget.entry, '_insert_handler')
      self._blockHandler (widget.entry, '_delete_handler')

      widget.entry.set_text (value)

      self._blockHandler (widget.entry, '_insert_handler', True)
      self._blockHandler (widget.entry, '_delete_handler', True)

    elif self._useNewGTKAPI and isinstance (widget, gtk.ComboBoxEntry):
      gfObject = self._uiDriver._WidgetToGFObj [widget]

      # Check if list of allowed value (~= foreign keys, ~= dropdown content)
      # changed
      if gfObject.style == "dropdown":
        self._updateChoices (widget, gfObject)

      self._blockHandler (widget.child, '_insert_handler')
      self._blockHandler (widget.child, '_delete_handler')

      widget.child.set_text (value)

      self._blockHandler (widget.child, '_insert_handler', True)
      self._blockHandler (widget.child, '_delete_handler', True)


    elif isinstance (widget, gtk.CheckButton):
      self._blockHandler (widget, '_clickedHandler')
      widget.set_active (value)
      self._blockHandler (widget, '_clickedHandler', True)

    elif isinstance (widget, gtk.TreeView):
      gfObject = self._uiDriver._WidgetToGFObj [widget]
      if gfObject.style == "listbox":
        self._updateChoices (widget, gfObject)

      if value in self.items:
        path = "%s" % self.items.index (value)
        gDebug (6, "Changing path of treeView to %s" % path)
        self._blockHandler (self.selection, '_changedHandler')
        widget.set_cursor (path)
        self._blockHandler (self.selection, '_changedHandler', True)

    else:
      item = isinstance (widget, gtk.TextView) and widget.get_buffer () or \
          widget

      self._blockHandler (item, '_insert_handler')
      self._blockHandler (item, '_delete_handler')

      item.set_text (value)

      self._blockHandler (item, '_insert_handler', True)
      self._blockHandler (item, '_delete_handler', True)


    if enabled:
      widget.set_sensitive (1)
    else:
      widget.set_sensitive (0)


  # --------------------------------------------------------------------------
  # Set the cursor's location in a widget
  # --------------------------------------------------------------------------
  def setCursorPosition (self, position, index = 0):
    """
    Sets the cursor's position to the given location inside a capable widget.
    """
    if hasattr (self.widgets [index], 'set_position'):
      self.widgets [index].set_position (position)


  # --------------------------------------------------------------------------
  # Set the selection inside a widget
  # --------------------------------------------------------------------------
  def setSelectedArea (self, selection1, selection2, index=0):
    """
    Sets the selection start/end inside a capable widget.
    """
    try:
      self.widgets [index].select_region (selection1, selection2)

    except (AttributeError, TypeError):
      pass


  # ---------------------------------------------------------------------------
  # Block or unblock a given event handler of a widget
  # ---------------------------------------------------------------------------

  def _blockHandler (self, widget, handlerName, unblock = False):
    """
    This function blocks or unblocks an event handler of a widget it is
    assigned.
    """
    method = unblock and widget.handler_unblock or widget.handler_block
    if hasattr (widget, handlerName):
      handler = getattr (widget, handlerName)
      gDebug (6, "%s '%s' (%s) for %s" % \
          (['Block', 'Unblock'] [unblock], handlerName, handler, widget))
      method (handler)


  # ---------------------------------------------------------------------------
  # Make sure a string will be converted into unicode
  # ---------------------------------------------------------------------------

  def _makeSafe (self, value):
    if isinstance (value, types.StringType):
      value = unicode (value, i18n.encoding)
    return value
