/*
 * Copyright (C) 2002,2003 Pascal Haakmat.
 * Licensed under the GNU GPL.
 * Absolutely no warranty.
 */

#include <stdio.h>
#include <math.h>
#include <audiofile.h>
#include <gnome.h>
#include <config.h>
#include "grid.h"

void
grid_save(struct grid *g,
          const char *path) {
    char key[512];
    snprintf(key, 512, "=%s=/Grid/BPM", path);
    gnome_config_set_float(key, g->bpm);
    snprintf(key, 512, "=%s=/Grid/Units", path);
    gnome_config_set_float(key, g->units);
    snprintf(key, 512, "=%s=/Grid/Scale", path);
    gnome_config_set_int(key, g->measurement);
}

void
grid_load(struct grid *g,
          const char *path) {
    char key[512];
    snprintf(key, 512, "=%s=/Grid/BPM=120", path);
    g->bpm = gnome_config_get_float(key);
    snprintf(key, 512, "=%s=/Grid/Units=1", path);
    g->units = gnome_config_get_float(key);
    snprintf(key, 512, "=%s=/Grid/Scale=1", path);
    g->measurement = gnome_config_get_int(key);
    if(g->measurement != GRID_FRAMES && 
       g->measurement != GRID_SECONDS && 
       g->measurement != GRID_BEATS)
        g->measurement = GRID_SECONDS;
}

void
grid_configure(struct grid *g) {
    switch(g->measurement) {
    case GRID_FRAMES:
        g->gap = g->units;
        break;
    case GRID_SECONDS:
        g->gap = (g->units * g->rate);
        break;
    case GRID_BEATS:
        g->gap = g->units * ((60 / g->bpm) * g->rate);
        break;
    }
}

void
grid_measurement_set(struct grid *g,
                     enum grid_measurement m) {
    g->measurement = m;
    grid_units_set(g, g->units);
    grid_configure(g);
}

void
grid_units_set(struct grid *g,
               double u) {
    switch(g->measurement) {
    case GRID_FRAMES:
        if(u <= 1)
            u = 1;
        /* There is no such thing as a half frame. */
        u = floor(u);
        break;
    case GRID_BEATS:
        if(u <= 0.03125)
            u = 0.03125;
        break;
    case GRID_SECONDS:
        if(u <= 0.01)
            u = 0.01;
        break;
    }
    g->units = u;
    grid_configure(g);
}

void
grid_bpm_from_frames_set(struct grid *g,
                         AFframecount frame_count) {
    g->bpm = 60 / (frame_count / g->rate);
    grid_configure(g);
}

void
grid_bpm_set(struct grid *g,
             double b) {
    g->bpm = b;
    grid_configure(g);
}

void
grid_rate_set(struct grid *g,
              double r) {
    g->rate = r;
    grid_configure(g);
}

void
grid_format(struct grid *g,
            AFframecount frame_offset,
            char *str,
            int n) {
    int h, m, s, ms;
    double secs, beats;

    switch(g->measurement) {
    case GRID_FRAMES:
        snprintf(str, n, "%ld", frame_offset);
        break;
    case GRID_SECONDS:
        secs = (double)frame_offset / g->rate;
        h = floor(secs / 86400);
        secs -= h * 86400;
        m = floor(secs / 60);
        secs -= m * 60;
        s = floor(secs);
        ms = (secs - s) * 100;
        if(ms) 
            if(h) 
                snprintf(str, n, "%02d:%02d:%02d.%02d", h, m, s, ms);
            else
                snprintf(str, n, "%02d:%02d.%02d", m, s, ms);
        else 
            if(h) 
                snprintf(str, n, "%02d:%02d:%02d   ", h, m, s);
            else
                snprintf(str, n, "%02d:%02d   ", m, s);
        break;
    case GRID_BEATS:
        beats = (double)frame_offset / ((60 / g->bpm) * g->rate);
        if(beats == floor(beats))
            snprintf(str, n, "%d", (int)beats);
        else 
            snprintf(str, n, "%.2f", beats);
            
        break;
    }
}
