/* $Revision: 1.3 $ */
/*
 * "gpgp" Gnome/GTK Front for PGP
 * Copyright (C) 1998  Max Valianskiy
 * Copyright (c) 1999 tftp@netscape.net
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *
 * History:
 * 06-May-99 tftp       Replaced 'gpgm' with 'gpg' (no more gpgm!)
 */

#include "../config.h"
#include "../version.h"

#include <assert.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <ctype.h>
#include <errno.h>
#include <unistd.h>
#include <signal.h>
#include <gnome.h>
#include <stdio.h>
#include <string.h>
#include <sys/ipc.h>
#include <sys/shm.h>

#include "gpgp.h"
#include "pgman.h"

#define PGMAN_DEBUG     0

/*#define LogTextMessage(s) printf("%s\n",(s))*/

/* malloc blocks */
#define BLK_SIZE    500000
#define MAX_CMDLINE 1000
#define MY_COMMENT  GPGP_OFFICIAL_NAME " version " VERSION

struct show_error_pack
{
    FILE* fp;
    GtkWidget* dialog;
};

static const char unexp_EOF[] = N_("Unexpected end of gpg output");
static const char oper_complete[] = N_("Operation complete");
static const char shm_not_attached[] = N_("Shared memory not attached");
const char *pgman_error = NULL;

static const int opt_show_to_widget = 1;

static GList* read_signatures(char *ring, char *uid);

/*
 * show_error_cb()
 *
 * Procedure acts as an event handler bound to OK button in
 * the dialog box. File gets closed when user clicks OK button;
 * dialog then closes.
 *
 * History:
 * 24-Jun-99 Created.
 */
static void show_error_cb(GtkWidget* widget, struct show_error_pack* pack)
{
    gtk_widget_destroy(pack->dialog);
    fclose(pack->fp);
    g_free(pack);
}

/*
 * show_FILE_to_widget()
 *
 * Procedure reads file contents and displays it in a separate
 * dialog box. File gets closed when user clicks OK button;
 * dialog then closes.
 *
 * History:
 * 24-Jun-99 Created.
 */
static void show_FILE_to_widget(FILE *fp)
{
    GtkWidget *dialog;
    GtkWidget *less;
    struct show_error_pack *pack;

    assert(fp != NULL);

    dialog = gnome_dialog_new(
        _(GPGP_OFFICIAL_NAME),
        GNOME_STOCK_BUTTON_OK, NULL);
    gtk_widget_set_usize(dialog, 450, 250);

    pack = g_malloc(sizeof(struct show_error_pack));
    assert(pack != NULL);

    pack->dialog=dialog;
    pack->fp=fp;
    gnome_dialog_button_connect(
        GNOME_DIALOG (dialog), 0, GTK_SIGNAL_FUNC(show_error_cb), pack);
 
    less = gnome_less_new();
    gtk_box_pack_start(
        GTK_BOX(GNOME_DIALOG(dialog)->vbox), less,
        FALSE, TRUE, GNOME_PAD_SMALL);
    gtk_widget_show(less);
    gtk_widget_realize(GTK_WIDGET(GNOME_LESS(less)->text)); /* ??? */
    gnome_less_show_filestream(GNOME_LESS(less),fp);
    gtk_widget_show(dialog);
}

static void show_FILE_to_Log(FILE *fp)
{
    char buf[256];
    int len;

    while (fgets(buf, sizeof(buf)-1, fp) != NULL)
    {
        /* Remove trailing CR - only one! */
        len = strlen(buf);
        if (len > 0)
        {
            if (iscntrl(buf[len-1]))
                buf[--len] = '\0';
        }
        LogTextMessage(buf);
    }
    fclose(fp);
}

/*
 * show_error()
 *
 * Procedure reads from specified file descriptor and prints
 * everything into the output device.
 */
static void show_error(int fd)
{
    FILE *fp;

    fp = fdopen(fd, "r");
    if (fp != NULL)
    {
        int ch;

        /*
         * fdopen() does not set EOF flag, so feof(fp) always
         * returns FALSE here.
         */
        if ((ch = fgetc(fp)) != EOF)
        {
            ungetc(ch, fp); /* put it back! */
            if (opt_show_to_widget) /* FIXME: put to options */
                show_FILE_to_widget(fp);
            else
                show_FILE_to_Log(fp);
        }
    }
    else
        gnome_error_dialog(_("Internal Error: show_error can't open fd"));
}

/*
 * History:
 * 07-Jun-99 Changed back --list-secret-keys to --list-keys (gpg 0.9.7)
 *           Corrected memory leak (str was never disposed of on error).
 *           No more keyring selection - just use default ones.
 */
#pragma argsused
GList* get_keylist(char *ring, int is_secret)
{
    char buf[MAX_CMDLINE+1];
    GList* list;
    int pip[2];
    FILE* fp;

    pgman_error = NULL;
    pipe(pip);
    g_snprintf(
        buf, sizeof(buf),
#if GPGP_USES_DEFAULT_KEYRING
        "gpg --batch --list-%skeys %s 2>&%d",
#else
        "gpg --batch --keyring %s --no-default-keyring "
        "--list-%skeys %s 2>&%d", ring,
#endif
        is_secret ? "secret-" : "", gpgp_options.optstring, pip[1]);
#if PGMAN_DEBUG
    LogTextMessage(buf);
#endif
    fp = popen(buf,"r");
    if (fp == NULL)
    {
        /* Pipe failed to open */
        pgman_error = _("Pipe failed to open");
        return NULL;
    }
    close(pip[1]);
 
    if (!fgets(buf, sizeof(buf), fp))
    {
        pgman_error = unexp_EOF;
        pclose(fp);
        show_error(pip[0]);
        return NULL;
    }
 
    if (!fgets(buf, sizeof(buf), fp))
    {
        pgman_error = unexp_EOF;
        pclose(fp);
        show_error(pip[0]);
        return NULL;
    }

    list = g_list_alloc();
    while (fgets(buf, sizeof(buf), fp) != NULL)
    {
        char **str;
        int str_len = sizeof(char*) * 5;
        int n, valid_line;
        char tmp1[MAX_CMDLINE+1];
        char tmp2[MAX_CMDLINE+1];
        char tmp3[MAX_CMDLINE+1];
        char tmp4[MAX_CMDLINE+1];

        str = g_malloc(str_len);
        assert(str != NULL);
        memset(str, 0, str_len);
        valid_line = 0;
        n = sscanf(buf, "%s %s %s %[^\n]", tmp1, tmp2, tmp3, tmp4);
        if ((n == 4) && (strcmp(tmp1, "pub") == 0))
        {
            str[0] = _("public");
            valid_line++;
        }
        else if ((n == 4) && (strcmp(tmp1, "sec") == 0))
        {
            str[0] = _("secret");
            valid_line++;
        }
        if (valid_line)
        {
            str[1] = g_strdup(tmp3);
            str[2] = g_strdup(tmp4);
            g_list_append(list, str);
        }
        else
            g_free(str);
    }

    if (WEXITSTATUS(pclose(fp)) != 0)
    {
        pgman_error = _("get_keylist returned error");
        show_error(pip[0]);
        g_list_free(list);
        return NULL;
    }
    show_error(pip[0]);
    return list;
}

#pragma argsused
void pgp_add_key(char *file, char *ring)
{
    char buf[MAX_CMDLINE+1];
    int pip[2];
    int res;

    pipe(pip);
    g_snprintf(
        buf, sizeof(buf),
#if GPGP_USES_DEFAULT_KEYRING
        "gpg %s --batch --import %s 2>&%d",
#else
        "gpg --keyring %s %s --no-default-keyring "
        "--batch --import %s 2>&%d", ring,
#endif
        gpgp_options.optstring, file, pip[1]);
#if PGMAN_DEBUG
    LogTextMessage(buf);
#endif
    res=system(buf);
    close(pip[1]);
    show_error(pip[0]);
    pgman_error = (res==0) ? NULL : _("pgp_add_key returned error");
}

#pragma argsused
void pgp_del_key(char* ring, char* uid)
{
    char buf[MAX_CMDLINE+1];
    int pip[2];
    int res;

    pipe(pip);
    g_snprintf(
        buf, sizeof(buf),
#if GPGP_USES_DEFAULT_KEYRING
        "gpg --batch "
        "--yes %s --delete-key \"=%s\" 2>&%d",
#else
        "gpg --keyring %s --no-default-keyring --batch "
        "--yes %s --delete-key \"=%s\" 2>&%d", ring,
#endif
        gpgp_options.optstring, uid, pip[1]);

#if PGMAN_DEBUG
    LogTextMessage(buf);
#endif
    res=system(buf);

    close(pip[1]);
    show_error(pip[0]);
    pgman_error= (res==0) ? NULL : _("pgp_del_key returned error");
}

#pragma argsused
void pgp_export_key(char* ring, char* uid, char* file)
{
    char buf[MAX_CMDLINE+1];
    int pip[2];
    int res;

    pipe(pip);
    g_snprintf(
        buf, sizeof(buf),
#if GPGP_USES_DEFAULT_KEYRING
        "gpg --batch --armor %s --export \"=%s\" >> %s 2>&%d ",
#else
        "gpg --keyring %s --no-default-keyring --batch "
        "--armor %s  --export \"=%s\" >> %s 2>&%d ", ring,
#endif
        gpgp_options.optstring, uid, file, pip[1]);

#if PGMAN_DEBUG
    LogTextMessage(buf);
#endif
    res=system(buf);
    close(pip[1]);
    show_error(pip[0]);
    pgman_error = (res==0) ? NULL : _("pgp_export_key returned error");
}

#pragma argsused
void get_key_options(char* ring, char* uid, struct key_options* options)
{
    char buf[MAX_CMDLINE+1];
    char type[MAX_CMDLINE+1];
 /* FIXME: These aren't command line thingies! */
    char bits[MAX_CMDLINE+1];
    char keyid[MAX_CMDLINE+1];
    char date[MAX_CMDLINE+1];
    char finger[MAX_CMDLINE+1];
    int pip[2];
    int res;
    FILE* fp;

    options->signatures=NULL;
    pipe(pip);
    g_snprintf(
        buf, sizeof(buf),
#if GPGP_USES_DEFAULT_KEYRING
        "gpg --batch --with-colons %s --fingerprint \"=%s\" 2>&%d ",
#else
        "gpg --keyring %s --no-default-keyring --batch "
        "--with-colons %s --fingerprint \"=%s\" 2>&%d ",
        ring, gpgp_options.defpub, /* FIXME what ring? */
#endif
        gpgp_options.optstring, uid, pip[1]);

#if PGMAN_DEBUG
    LogTextMessage(buf);
#endif
    if (!(fp=popen(buf, "r")))
    {
        pgman_error = _("Failed to open pipe to process"); /*g_strerror(errno)*/
        close(pip[0]);
        close(pip[1]);
        return;
    }
  
    if (!fgets(buf, sizeof(buf), fp) )
    {
        pgman_error = unexp_EOF;
        close(pip[1]);
        show_error(pip[0]);
        pclose(fp);
        return;
    }

    sscanf(buf,
           "%[^:] %*c %*[^:] %*c %[^:] %*c %*[^:] %*c %[^:] %*c %[^:]",
           type, bits, keyid, date);

    options->bits=g_strdup(bits);
    options->keyid=g_strdup(keyid);
    options->date=g_strdup(date);

    if (strcmp(type,"pub")==0)
    {
        options->type=_("public");

        if (!fgets(buf, sizeof(buf), fp) )
        {
            pgman_error = unexp_EOF;
            close(pip[1]);
            show_error(pip[0]);
            pclose(fp);
            return;
        }
        sscanf(buf, "%*[^:] %*[^0-1A-Z] %[^:]", finger);
        options->fingerprint=g_strdup(finger);
    } 
    else
    {
        if (strcmp(type,"sec")==0)
        {
            options->type=_("secret");
            options->fingerprint="";
	}
	else
        {
            pgman_error = unexp_EOF;
            close(pip[1]);
            show_error(pip[0]);
            pclose(fp);
            return;
        }
    }
    res=WEXITSTATUS(pclose(fp));

    close(pip[1]);
    show_error(pip[0]);
    if (res==0)
    {
        pgman_error=NULL;
        options->signatures=read_signatures(ring, uid);
    }
    else
        pgman_error=_("get_key_options returned error");
}

#pragma argsused
static GList *read_signatures(char* ring, char* uid)
{
    char buf[MAX_CMDLINE+1]; 
    GList* list;
    int pip[2];
    FILE* fp;

    pgman_error=NULL;
    pipe(pip);
    g_snprintf(
        buf, sizeof(buf),
#if GPGP_USES_DEFAULT_KEYRING
        "gpg --batch --list-keys --verbose %s  \"=%s\" 2>&%d",
#else
        "gpg --batch --keyring %s --list-keys --verbose "
        "--no-default-keyring %s  \"=%s\" 2>&%d",
        ring, gpgp_options.defpub, /* FIXME what ring? */
#endif
        gpgp_options.optstring, uid, pip[1]);

#if PGMAN_DEBUG
    LogTextMessage(buf);
#endif
    fp = popen(buf,"r");
    if (!fp)
    {
        pgman_error=g_strerror(errno);
        return NULL;
    }
    close(pip[1]);
 
    list = g_list_alloc();
    while ( fgets(buf, sizeof(buf), fp) )
    {
        char** str = g_malloc ( sizeof(char*) * 6 );
        char tmp1[MAX_CMDLINE+1];
        char tmp2[MAX_CMDLINE+1];
        char tmp3[MAX_CMDLINE+1];
        char tmp4[MAX_CMDLINE+1];

        memset(str, 0, sizeof(char*)*6);
        sscanf(buf, "%s %s %s %[^\n]", tmp1, tmp2, tmp3, tmp4);

        if (strcmp(tmp1,"pub")==0)
            str[0]=_("public");
        else if (strcmp(tmp1,"sec")==0)
            str[0]=_("secret");
        else if (strcmp(tmp1,"sig")==0)
            str[0]=_("signature");
        else if (strcmp(tmp1, "sub")==0)
            str[0]=_("subkey");
        else 
            continue;
 
        str[1]=g_strdup(tmp3);
        str[2]=g_strdup(tmp2);
        str[3]=g_strdup(tmp4);

        g_list_append(list, str);
    }

    if (WEXITSTATUS(pclose(fp))!=0)
    { 
        pgman_error=_("read_signatures returned error");
        g_list_free(list);
        show_error(pip[0]);
        return NULL;
    }
    else
    {
        show_error(pip[0]);
        return list;
    }
}

/***************************/
/*   SIGN_ACTION           */
/***************************/

struct action_pack
{
    int inpip[2];
    int outpip[2];
    int errpip[2];
    char* data;
    int old_datalen;
    int datalen;
    char** new;
    int* newlen;
    int curlen;
    int read_key;
    int write_key;
    GnomeAppProgressKey key; 
    PgmanActionFunc func;
    gpointer data2;
};

static void action_finish(struct action_pack* pack, int cancel)
{
    int status;

    if (pack->datalen)
        gdk_input_remove(pack->write_key);
    gdk_input_remove(pack->read_key);

    if (pack->inpip[1])
        close(pack->inpip[1]);
    close(pack->outpip[0]);
    close(pack->inpip[0]);

    wait(&status);
 
    if (WEXITSTATUS(status)!=0 && !pgman_error)
        pgman_error=_("action_finish returned error");
       
    if (gpgp_options.use_status && !cancel)
        gnome_app_progress_done(pack->key);

    pack->func(pack->data2);
    show_error(pack->errpip[0]);
    g_free(pack);
    if (gpgp_options.use_status)
        gnome_app_flash(gpgp_options.app, oper_complete);
}

static void action_error(struct action_pack* pack)
{
    action_finish(pack, FALSE); /* Not cancelling */
}

static void cancel_cb(struct action_pack* pack)
{ 
    pgman_error=_("canceled by user");
    action_finish(pack,1);
}

static void action_read_cb(
    struct action_pack *pack,
    gint source,
    GdkInputCondition cond)
{
    if (!pack->curlen) /* realloc */
    {
        *pack->new = g_realloc(*pack->new,*pack->newlen+BLK_SIZE);
        pack->curlen = BLK_SIZE;
    }

    switch (read(pack->outpip[0], *pack->new+(*pack->newlen) , 1)) 
    {
        case 1:
            (*pack->newlen)++;
            pack->curlen--;
            break;

        case 0: /* EOF */
            action_finish(pack, FALSE);
            break;

        case -1: /* error */
            pgman_error=g_strerror(errno);
            action_error(pack);
            break;
    }
}

static void action_write_cb(
    struct action_pack *pack,
    gint source,
    GdkInputCondition cond)
{
    if (write (pack->inpip[1], pack->data, 1)!=1)
    {
        pgman_error=g_strerror(errno);
        if (gpgp_options.use_status)
            gnome_app_progress_done(pack->key);
        action_error(pack);
        return;
    }
		    
    pack->data++;
    pack->datalen--;

    if (!pack->datalen)
    {
        gdk_input_remove(pack->write_key);
        close(pack->inpip[1]);
        pack->inpip[1]=0;
        pack->write_key=0;
    }
}

static gdouble action_percent_cb(struct action_pack *pack)
{
    return ((float) pack->old_datalen-pack->datalen)/(float) pack->old_datalen;
}

void pgp_sign_action(
    char* options,
    char* data,
    int datalen,
    char** new,
    int* newlen,
    PgmanActionFunc func,
    gpointer data2)
{ 
    char buf[MAX_CMDLINE+1];
    struct action_pack* pack=g_malloc(sizeof(struct action_pack));

    pgman_error=NULL;

    pack->data=data;
    pack->datalen=datalen;
    pack->old_datalen=datalen;
    pack->new=new;
    pack->newlen=newlen;
    pack->func=func;
    pack->data2=data2;

    if (gpgp_options.use_status)
    {
        pack->key = gnome_app_progress_timeout(
            gpgp_options.app,
            _("Processing data"),
            100,
            (GnomeAppProgressFunc) action_percent_cb,
            (GnomeAppProgressCancelFunc) cancel_cb,
            pack);
    }
    g_snprintf(
        buf, sizeof(buf),
#if GPGP_USES_DEFAULT_KEYRING
        "gpg --batch %s %s",
#else
        "gpg --batch --no-default-keyring --keyring %s "
        "--secret-keyring %s %s %s",
        gpgp_options.defpub, gpgp_options.defsec,
#endif
        gpgp_options.optstring, options);

    if ((pipe(pack->inpip) == -1) ||
        (pipe(pack->outpip)==-1) ||
        (pipe(pack->errpip)==-1))
    {
        pgman_error=g_strerror(errno);
        return;
    }

    switch (fork())
    {
        case 0:{
            int res;
            /* child */
            dup2(pack->inpip[0], 0);
            dup2(pack->outpip[1], 1);
            dup2(pack->errpip[1], 2);
            close(pack->inpip[0]);
            close(pack->inpip[1]);
            close(pack->outpip[0]);
            close(pack->outpip[1]);
            close(pack->errpip[0]);
            close(pack->errpip[1]);

            /* I'm too lazy to do it right way. FIXME */
#if PGMAN_DEBUG
            LogTextMessage(buf);
#endif
            res=WEXITSTATUS(system(buf)); /* why WEXIT? */

            close(0);
            close(1);
            close(2);
            _exit(res); /* gnome is not fork safe??? */
        }
        case -1: /* error */
            pgman_error=g_strerror(errno);
            close(pack->inpip[0]);
            close(pack->inpip[1]);	     
            close(pack->outpip[0]);	     
            close(pack->outpip[1]);
            close(pack->errpip[0]);
            close(pack->errpip[1]);
            return;
        default: /* father */
        {
            *pack->new=g_malloc(BLK_SIZE);
            *pack->newlen=0;

            pack->curlen = BLK_SIZE;

            close(pack->outpip[1]);
            close(pack->errpip[1]);

            pack->read_key = gdk_input_add(
                pack->outpip[0],
                GDK_INPUT_READ,
                (GdkInputFunction) action_read_cb,
                pack);
            pack->write_key = gdk_input_add(
                pack->inpip[1],
                GDK_INPUT_WRITE,
                (GdkInputFunction) action_write_cb,
                pack);
        }
    }
}

/***************************/
/* UNSIGN_ACTION           */
/***************************/

struct action_pack2
{
    int inpip[2];
    int outpip[2];
    int errpip[2];
    int status[2];
    char* data;
    int old_datalen;
    int datalen;
    char** new;
    int* newlen;
    int curlen;
    int read_key;
    int write_key;
    int status_key;
    GnomeAppProgressKey key; 
    PgmanActionFunc func;
    gpointer data2;
    char *area;         /* shm area */
    int pid;            /* server pid */
    int id;             /* shm id */
};

static void action_finish2(struct action_pack2 *pack, int cancel)
{
    int status;

    if (pack->datalen)
        gdk_input_remove(pack->write_key);
    gdk_input_remove(pack->read_key);
    gdk_input_remove(pack->status_key);

    if (pack->inpip[1])
        close(pack->inpip[1]);
    close(pack->outpip[0]);
    close(pack->inpip[0]);
    close(pack->status[0]);

    wait(&status);
    if (WEXITSTATUS(status)!=0 && !pgman_error)
        pgman_error=_("action_finish2 returned error");

    if (gpgp_options.use_status && !cancel)
        gnome_app_progress_done(pack->key);

    pack->func(pack->data2);
    show_error(pack->errpip[0]);
    g_free(pack);

    if (gpgp_options.use_status)
        gnome_app_flash(gpgp_options.app, oper_complete);
}

static void action_error2(struct action_pack2 *pack)
{
    action_finish2(pack, FALSE);
}

static void cancel2_cb(struct action_pack2 *pack)
{ 
    pgman_error = _("Canceled by user");
    action_finish2(pack, TRUE);
}

static void action_read2_cb(
    struct action_pack2 *pack,
    gint source,
    GdkInputCondition cond)
{
    if (pack->curlen <= 0)
    {
        *pack->new = g_realloc(*pack->new, *pack->newlen+BLK_SIZE);
        pack->curlen = BLK_SIZE;
    }

    switch (read(pack->outpip[0], *pack->new+(*pack->newlen) , 1)) 
    {
        case 1:
            (*pack->newlen)++;
            pack->curlen--;
            break;

        case 0: /* EOF */
            action_finish2(pack,0);
            break;

        case -1: /* error */
            pgman_error = _("Read error"); /*g_strerror(errno)*/
            g_message("read error");
            action_error2(pack);
            return;
    }
}

static void do_get_string_cb(gchar* string, struct action_pack2* pack);
static void do_get_yes_cb(gint reply, struct action_pack2* pack);
static void do_action_finish(struct action_pack2* pack);

/*
 * do_get_string()
 *
 * This procedure initiates request for user-supplied data.
 * The actual transmission of response will occur from within
 * the callback/handler code bound to button(s).
 *
 * History:
 */
static void do_get_string(
    int mode,
    const char *keyword,
    struct action_pack2 *pack)
{
#if PGMAN_DEBUG
    LogTextMessage("do_get_string");
#endif

    if (strcmp(keyword, "passphrase.enter")==0)
        keyword = _("Enter passphrase");

    if( mode == 1 ) /* get string */
    {
        gnome_question_dialog(
            keyword,
            (GnomeReplyCallback) do_get_string_cb,
            pack);
    }
    else if( mode == 3 ) /* get password */
    {
        gnome_request_dialog(
            TRUE,               /* Password entry */
            keyword,            /* Prompt */
            NULL,               /* Default text */
            80,                 /* FIXME!!! Don't know max. size */
            (GnomeStringCallback) do_get_string_cb,
            pack,               /* Data */
            NULL);              /* Parent window */
    }
    else /* get yes/no */
    {
        gnome_request_string_dialog(
            keyword,
            (GnomeStringCallback) do_get_yes_cb,
            pack);
    }
}

static void do_get_string_cb(gchar *string, struct action_pack2 *pack)
{
    int n, len;

#if PGMAN_DEBUG
    LogTextMessage("do_get_string_cb");
#endif

    assert(pack != NULL);

    if (string != NULL)
        len = strlen(string);
    else
    {
        pgman_error = _("canceled by user");
        len = 0;
    }

    if (pack->area != NULL)
    {
        n = (pack->area[0] << 8) | (pack->area[1]);
        if (len > 0)
        {
            memcpy(pack->area+n+2, string, len);
        }
        pack->area[n] = len >> 8;
        pack->area[n+1] = len;
    }
    else
    {
        LogTextMessage(shm_not_attached);
    }
    do_action_finish(pack);
}

static void do_get_yes_cb(gint reply, struct action_pack2 *pack)
{
#if PGMAN_DEBUG
    LogTextMessage("do_get_yes_cb");
#endif

    if (pack->area != NULL)
    {
        int n = (pack->area[0] << 8) | (pack->area[1]);
        pack->area[n] = 0;
        pack->area[n+1] = 1;
        pack->area[n+2] = reply;
    }
    else
    {
        LogTextMessage(shm_not_attached);
    }
    do_action_finish(pack);
}

static void do_action_finish(struct action_pack2 *pack)
{
#if PGMAN_DEBUG
    LogTextMessage("do_action_finish");
#endif

    if (pack->area != NULL)
        pack->area[3] = 1; 
    else
        LogTextMessage(shm_not_attached);
    if (pgman_error)
    {
        kill(pack->pid, SIGTERM); /* Terminate GnuPG process */
        action_finish2(pack, 0);  /* Get rid of GUI windows */
    }
    else
        kill(pack->pid, SIGUSR1); /* Push coprocess */
}

static void ProcessStatusString(char *buf, struct action_pack2 *pack)
{
    int word = 0, is_info = 0,is_get = 0;
    char *p, *p2;
    static const char token[] = " \n";
    static const char gnupg_leader[] = "[GNUPG:] ";
    static const int gnupg_leader_len = sizeof(gnupg_leader) - 1;

#if PGMAN_DEBUG
    LogTextMessage("ProcessStatusString():");
    LogTextMessage(buf);
#endif

    if(memcmp(buf, gnupg_leader, gnupg_leader_len) != 0)
        return;

    for(p = strtok(buf+gnupg_leader_len, token); p; p=strtok(NULL, token))
    {
        word++;
        if((word == 1) && (strcmp(p,"SHM_INFO") == 0)) 
        {
            if(pack->area == NULL)
                is_info++;
            else
                g_print(_("duplicate SHM_INFO ignored\n"));
        }
        else 
        {
            if(is_info&&(p2=strchr(p,'='))) 
            {
                int val;
                *p2++ = 0;
                val = atoi(p2);
                if( !strcmp(p, "pv" ) ) 
                {
                    if( atoi(p2) != 1 )
                    {
                        pgman_error = N_("invalid protocol");
                        action_error2(pack);
                        return;
                    }
                    is_info = 2;
                }
                else if( !strcmp(p, "pid" ) )
                    pack->pid = val;
                else if( !strcmp(p, "shmid" ) )
                    pack->id = val;
            }
            else if( word == 1 && !strcmp(p,"SHM_GET") )
                is_get = 1;
            else if( word == 1 && !strcmp(p,"SHM_GET_BOOL") )
                is_get = 2;
            else if( word == 1 && !strcmp(p,"SHM_GET_HIDDEN") )
                is_get = 3;
            else if( word == 2 && is_get )	
            {
                do_get_string(is_get,p,pack);
                return;
            }
            else if( word == 1 )
            {
#if PGMAN_DEBUG
                char buf[80];
                strcpy(buf, _("Status: "));
                strcat(buf, p);
                LogTextMessage(buf);
#endif
            }
        }
    }
    if (is_info)
    {
        if( is_info < 2 )
            pgman_error=_("SHM info without protocol version");
        if( pack->pid == -1 )
            pgman_error=_("SHM info without server's pid");
        if( pack->id == -1 )
            pgman_error=_("SHM info without id");

        if (pgman_error) 
        {
            action_error2(pack);
            return;
        }

        /*
         * The function shmat attaches the shared memory segment
         * identified by shmid to the data segment of the calling
         * process.
         */
        pack->area = shmat( pack->id, NULL, 0 );
#if PGMAN_DEBUG
        {
            char tmp[80];
            g_snprintf(tmp,sizeof(tmp),"shmat: pack->area=%p",pack->area);
            LogTextMessage(tmp);
        }
#endif
        if( pack->area == ((void*) -1) )
        {
            pgman_error = _("SHM allocation failed");
            action_error2(pack);
            return;
        }
    }
}

/*
 * action_status2_cb()
 *
 * This procedure is attached as a callback to the status fd/pipe.
 * It gets called to fetch data from the pipe and do something.
 * The call occurs when "some" data is available - maybe one line,
 * maybe several. We read the whole block of data, split it into
 * separate strings and feed into processing code. This procedure
 * may be called to read just a small fragment of data, so we have
 * to accumulate the data in a buffer (mbuf) and check this buffer
 * for completed lines each time we add to it.
 *
 * History:
 * 12-Jun-99 tftp       Rewrote to use 'read' and perform block reads, splits.
 * 22-Jun-99 tftp       Added buffer reassembly.
 */
static void action_status2_cb(
    struct action_pack2 *pack,
    gint source,
    GdkInputCondition cond)
{
    int i, k, n;
    static char mbuf[1024];
    static int mlen = 0;

    if (pack == NULL) /* Magic for resetting the buffer */
    {
#if PGMAN_DEBUG
        LogTextMessage("Buffer reset.");
#endif
        mlen = 0;
        return;
    }

#if PGMAN_DEBUG
    LogTextMessage("Reading status stream...");
#endif

    if (mlen >= sizeof(mbuf))
    {
        LogTextMessage(_("Buffer overflow"));
        return;
    }
    n = read(pack->status[0], mbuf+mlen, sizeof(mbuf)-mlen);

#if PGMAN_DEBUG
    /*
     * The sole purpose of this block is to print the read data
     * in nice format - printables "as is", controls as \n etc.
     */
    {
        auto char tmp[1024];
        int i, j;

        g_snprintf(
            tmp, sizeof(tmp),
            "Status stream at %d: Read %d. \"", mlen, n);
        j = strlen(tmp);
        for (i=0; (i < n) && (j < (sizeof(tmp)-2)); i++, j++)
        {
            const char ch = mbuf[mlen+i];
            if (isprint(ch))
            {
                tmp[j] = ch;
            }
            else if (ch == '\n')
            {
                tmp[j++] = '\\';
                tmp[j] = 'n';
            }
            else if (ch == '\r')
            {
                tmp[j++] = '\\';
                tmp[j] = 'r';
            }
            else /* Print as hex */
            {
                g_snprintf(tmp+j, sizeof(tmp)-2-j, "\\x%02x", ch);
                j += 4; /* \xNN */
            }
        }
        tmp[j++] = '"';
        tmp[j] = '\0';
        LogTextMessage(tmp);
    }
#endif

    if (n <= 0) /* EOF */
    {
        pgman_error = _("No data read via pipe"); /*g_strerror(errno)*/
        action_error2(pack);
        return;
    }

    mlen += n;

    /*
     * Check the newly read data, look for '\n' - if found then
     * split buffer into ASCIIZ string(s), feed into processing code.
     *
     * TODO: It is not efficient to re-scan entire buffer; all we
     * need is to check the added portion.
     */
    for (i=0; i < mlen;)
    {
        if (mbuf[i] == '\n')
        {
            mbuf[i] = '\0';
            ProcessStatusString(mbuf, pack);

            /*
             * Now that we processed the string we have to move
             * remaining data in the buffer.
             */
            k = i+1;
            mlen -= k; /* We get rid of '\0' too */
            if (mlen > 0)
            {
                memmove(mbuf, mbuf+k, mlen);
            }

            /*
             * We finished with one string that got completed by
             * newly read block of data. However there may be more
             * strings... re-evaluate the whole block.
             */
            i=0;
        }
        else
            i++;
    }
}

static void action_write2_cb(
    struct action_pack2 *pack,
    gint source,
    GdkInputCondition cond)
{
    static const char msg[] = N_("action_write2_cb: Pipe write error");

    if (write(pack->inpip[1], pack->data, 1) != 1)
    {
        g_message(msg);
        pgman_error = msg; /*g_strerror(errno)*/
        if (gpgp_options.use_status)
            gnome_app_progress_done(pack->key);
        action_error2(pack);
        return;
    }
		    
    pack->data++;
    pack->datalen--;

    if (pack->datalen <= 0)
    {
        gdk_input_remove(pack->write_key);
        close(pack->inpip[1]);
        pack->inpip[1]=0;
        pack->write_key=0;
    }
}

static gdouble action_percent2_cb(struct action_pack2 *pack)
{
    gdouble res;
    int boundError = 0;

    /* Prevent divide by zero if somehow data block is empty */
    if (pack->old_datalen <= 0)
        return 0;

    res  = (float) (pack->old_datalen - pack->datalen);
    res /= (float) pack->old_datalen;

    /* Sanity check */
    if (res < 0)
    {
        res = 0;
        ++boundError;
    }
    else if (res > 1.0)
    {
        res = 1.0;
        ++boundError;
    }
    /* TODO: Scream as loud as needed if boundError happens! */
    return res;
}

void pgp_unsign_action(
    char *data,
    int datalen,
    char **new,
    int *newlen,
    PgmanActionFunc func,
    gpointer data2)
{ 
    char buf[MAX_CMDLINE+1];
    struct action_pack2 *pack;

    pgman_error = NULL;

    pack = g_malloc(sizeof(struct action_pack2));
    assert(pack != NULL);

    memset(pack, 0, sizeof(struct action_pack2));
    pack->data = data;
    pack->datalen = datalen;
    pack->old_datalen = datalen;
    pack->new = new;
    pack->newlen = newlen;
    pack->func = func;
    pack->data2 = data2;
    pack->area = NULL;

    if (gpgp_options.use_status)
    {
        pack->key = gnome_app_progress_timeout(
            gpgp_options.app,
            _("Processing data"),
            100,
            (GnomeAppProgressFunc) action_percent2_cb,
            (GnomeAppProgressCancelFunc) cancel2_cb,
            pack);
    }

    /*
     * pipe  creates  a  pair  of file descriptors, pointing to a
     * pipe inode, and places them in the  array  pointed  to  by
     * filedes. filedes[0]  is  for  reading, filedes[1] is for
     * writing. 
     */
    if ((pipe(pack->inpip) == -1) ||    /* stdin */
        (pipe(pack->outpip)== -1) ||    /* stdout */
        (pipe(pack->errpip)== -1) ||    /* stderr */
        (pipe(pack->status)== -1))      /* status fd */
    {
        pgman_error = _("Failed to open pipes"); /*g_strerror(errno)*/
        return;
    }

    g_snprintf(
        buf, sizeof(buf),
#if GPGP_USES_DEFAULT_KEYRING
        "gpg --run-as-shm-coprocess 0 --no-greeting "
        "--decrypt %s "
        "--status-fd %d ",
#else
        "gpg --run-as-shm-coprocess 0 --no-greeting "
        "--no-default-keyring --keyring %s --secret-keyring %s "
        "--decrypt %s "
        "--status-fd %d ",
        gpgp_options.defpub, gpgp_options.defsec,
#endif
        gpgp_options.optstring, /* options */
        pack->status[1]);       /* status filedesc */

#if PGMAN_DEBUG
    LogTextMessage(buf);
#endif

    switch (fork())
    {
        case 0: /* child */
        {
            int res;

            /*
             * Here we reopen (dup) standard file descriptors
             * (stdin, stdout, stderr). They will be closed
             * first and then duplicated as our own fd's
             * (created by pipe() call before).
             */
            dup2(pack->inpip[0], 0);    /* stdin, read */
            dup2(pack->outpip[1], 1);   /* stdout, write */
            dup2(pack->errpip[1], 2);   /* stderr, write */

            close(pack->inpip[0]);
            close(pack->inpip[1]);

            close(pack->outpip[0]);
            close(pack->outpip[1]);

            close(pack->errpip[0]);
            close(pack->errpip[1]);

            close(pack->status[0]);

            /* I'm too lazy to do it right way. FIXME */
            res=WEXITSTATUS(system(buf)); /* why WEXIT? */

            close(0);
            close(1);
            close(2);
            _exit(res); /* gnome is not fork safe??? */
        }
        case -1: /* error */
            pgman_error = "fork() error"; /*g_strerror(errno)*/
            close(pack->inpip[0]);
            close(pack->inpip[1]);	     
            close(pack->outpip[0]);	     
            close(pack->outpip[1]);
            close(pack->errpip[0]);
            close(pack->errpip[1]);
            return;
        default: /* father */
        {
            *pack->new=g_malloc(BLK_SIZE);
            *pack->newlen=0;

            pack->curlen = BLK_SIZE;

            close(pack->outpip[1]);
            close(pack->errpip[1]);
            close(pack->status[1]);

            pack->read_key = gdk_input_add(
                pack->outpip[0],
                GDK_INPUT_READ,
                (GdkInputFunction) action_read2_cb,
                pack);

            action_status2_cb(NULL, 0, 0); /* Reset buffer */
            pack->status_key = gdk_input_add(
                pack->status[0],
                GDK_INPUT_READ,
                (GdkInputFunction) action_status2_cb,
                pack);

            pack->write_key = gdk_input_add(
                pack->inpip[1],
                GDK_INPUT_WRITE,
                (GdkInputFunction) action_write2_cb,
                pack);
        } 
    }
}	     

/*
 * update_gpgp_options()
 *
 * Procedure replaces existing option string with either
 * "--comment ..." or "--no-comment" depending on state of
 * structure gpgp_options.
 *
 * History:
 * 07-May-99 tftp       Simplified the code.
 */
void update_gpgp_options()
{
    gpgp_options.optstring = g_strdup(
        gpgp_options.comment ?
        " --comment \"" MY_COMMENT "\"" : " --no-comment");
}
