/***************************************************************************
                          gpc_image.c  -  description
                             -------------------
    begin                : Wed May 17 2000
    copyright            : (C) 2000 by Thierry Florac
    email                : tflorac@free.fr
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/stat.h>
#include <dirent.h>
#include <gnome.h>
#include <gdk-pixbuf/gdk-pixbuf.h>

#include "gpc_app.h"
#include "gpc_image.h"
#include "gpc_utils.h"

#include "pg_connection.h"
#include "pg_query.h"

#include "support.h"


/** Data structure to store current image informations */
struct _GpcImageInfos {
  gint      ID;
  gint      film_ID;
  gchar    *filename;
  gint      width;
  gint      height;
  gchar    *mime_type;
  gchar    *number;
  gchar    *lens_maker;
  gchar    *lens_model;
  gshort    focal;
  gfloat    aperture;
  gchar    *speed;
  gfloat    latitude;
  gfloat    longitude;
  gfloat    elevation;
  gfloat    direction;
  gfloat    pitch;
  gfloat    roll;
  gshort    interest_note;
  gshort    quality_note;
  gchar    *description;
  GList    *keywords;
};


/** Private function to get physical informations from a given ABSOLUTE image file name */
gboolean gpc_image_get_infos_from_file (gchar *filename, GpcImageInfos *infos);

gboolean gpc_image_get_infos_from_file (gchar *filename, GpcImageInfos *infos)
{
  const gchar   *mime_type;
  GdkPixbuf     *pixbuf = NULL;
  
  if ((filename == NULL) || 
      (strlen (filename) == 0) ||
      (infos == NULL))
    return FALSE;
  
  pixbuf = gdk_pixbuf_new_from_file (filename);
  if (pixbuf) {
    mime_type = gnome_mime_type_of_file (filename);
    gpc_image_set_mime_type (infos, (gchar *) mime_type);
    gpc_image_set_width (infos, gdk_pixbuf_get_width (pixbuf));
    gpc_image_set_height (infos, gdk_pixbuf_get_height (pixbuf));
    gdk_pixbuf_unref (pixbuf);
    return TRUE;
  }
  else
    return FALSE;
}



/** Create a new GpcImageInfos structure */
GpcImageInfos *gpc_image_infos_new (void) {

  GpcImageInfos *infos;

  infos = g_new0(GpcImageInfos, 1);
  infos->ID = -1;

  return infos;
}


/** Get informations corresponding to a given image ID */
GpcImageInfos *gpc_image_get_infos (pgConnection *connection, gint ID)
{
  GpcImageInfos *infos;
  pgQuery       *query;
  gchar         *sql;

  gchar         *old_locale;
  gchar         *saved_locale;

  if (!connection || !pg_connection_is_active (connection))
    return NULL;
  sql = g_strdup_printf ("select * from GPC_IMAGE "
                         "where (ID = %d)", ID);
  query = pg_query_new_with_sql (connection, sql);
  g_free (sql);
  if (!query)
    return NULL;
  if (pg_query_open (query) > 0) {
    old_locale = setlocale (LC_NUMERIC, NULL);
    saved_locale = g_strdup (old_locale);
    setlocale (LC_NUMERIC, "C");
    infos = gpc_image_infos_new ();
    infos->ID = atoi (pg_query_get_fieldvalue_byname (query, "ID"));
    infos->film_ID = atoi (pg_query_get_fieldvalue_byname (query, "FILM_ID"));
    infos->filename = g_strdup (pg_query_get_fieldvalue_byname (query, "FILENAME"));
    infos->width = atoi (pg_query_get_fieldvalue_byname (query, "WIDTH"));
    infos->height = atoi (pg_query_get_fieldvalue_byname (query, "HEIGHT"));
    infos->mime_type = g_strdup (pg_query_get_fieldvalue_byname (query, "MIME_TYPE"));
    infos->number = g_strdup (pg_query_get_fieldvalue_byname (query, "NUMBER"));
    infos->lens_maker = g_strdup (pg_query_get_fieldvalue_byname (query, "LENS_MAKER"));
    infos->lens_model = g_strdup (pg_query_get_fieldvalue_byname (query, "LENS_MODEL"));
    infos->focal = atoi (pg_query_get_fieldvalue_byname (query, "FOCAL"));
    infos->aperture = atof (pg_query_get_fieldvalue_byname (query, "APERTURE"));
    infos->speed = g_strdup (pg_query_get_fieldvalue_byname (query, "SPEED"));
    infos->latitude = atof (pg_query_get_fieldvalue_byname (query, "LATITUDE"));
    infos->longitude = atof (pg_query_get_fieldvalue_byname (query, "LONGITUDE"));
    infos->elevation = atof (pg_query_get_fieldvalue_byname (query, "ELEVATION"));
    infos->direction = atof (pg_query_get_fieldvalue_byname (query, "DIRECTION"));
    infos->pitch = atof (pg_query_get_fieldvalue_byname (query, "PITCH"));
    infos->roll = atof (pg_query_get_fieldvalue_byname (query, "ROLL"));
    infos->interest_note = atoi (pg_query_get_fieldvalue_byname (query, "INTEREST_NOTE"));
    infos->quality_note = atoi (pg_query_get_fieldvalue_byname (query, "QUALITY_NOTE"));
    infos->description = g_strdup (pg_query_get_fieldvalue_byname (query, "DESCRIPTION"));
    pg_query_free (query);
    sql = g_strdup_printf ("select * from GPC_KEYWORD "
                           "where (IMAGE_ID = %d)", ID);
    query = pg_query_new_with_sql (connection, sql);
    g_free (sql);
    if (query && (pg_query_open (query) > 0)) {
      pg_query_move_first (query);
      while (! pg_query_at_EOF (query)) {
        infos->keywords = g_list_insert_sorted (infos->keywords, g_strdup (pg_query_get_fieldvalue_byname (query, "CODE")), (GCompareFunc) g_strcasecmp);
        pg_query_move_next (query);
      }
      pg_query_free (query);
    }
    setlocale (LC_NUMERIC, saved_locale);
    g_free (saved_locale);
    return infos;
  }
  else {
    pg_query_free (query);
    return NULL;
  }

}


/** Check to see if an image exists into database with a given RELATIVE directory and filename */
gboolean gpc_image_exists (pgConnection *connection, gchar *dirname, gchar *filename)
{
  pgQuery  *query;
  gchar    *where;
  gchar    *sql_filename;
  gchar    *sql_dirname;
  gboolean result;

  if (!connection)
    return FALSE;
  sql_filename = get_sql_string (filename);
  sql_dirname = get_sql_string (dirname);
  where = g_strdup_printf ("(FILENAME = %s) and (FILM_ID in (select ID from GPC_FILM where PATHNAME = %s))", sql_filename, sql_dirname);
  query = gpc_image_get_list_where (connection, where);
  result = query && (pg_query_get_recordcount (query) > 0);
  pg_query_free (query);
  g_free (where);
  g_free (sql_filename);
  g_free (sql_dirname);
  return result;
}


/** Check to see if an image exists into database with a given ID */
gboolean gpc_image_exists_with_id (pgConnection *connection, gint ID)
{
  pgQuery  *query;
  gchar    *where;
  gboolean result;

  if (!connection)
    return FALSE;
  where = g_strdup_printf ("(ID = %d)", ID);
  query = gpc_image_get_list_where (connection, where);
  result = query && (pg_query_get_recordcount (query) > 0);
  pg_query_free (query);
  g_free (where);
  return result;
}


/** Check to see if the image file matching a given image ID exists */
gboolean gpc_image_file_exists (pgConnection *connection, gint ID)
{
  gchar    *pathname;
  gchar    *filename;
  gchar    *sql;
  pgQuery  *query;
  gboolean  result = FALSE;
  
  if ((connection == NULL) || 
      (!pg_connection_is_active (connection)))
    return FALSE;
  
  sql = g_strdup_printf ("select FILM.PATHNAME, IMAGE.FILENAME "
                         "from GPC_FILM FILM, GPC_IMAGE IMAGE "
                         "where FILM.ID = IMAGE.FILM_ID "
                         "  and IMAGE.ID = %d", ID);
  query = pg_query_new_with_sql (connection, sql);
  if (!pg_query_open (query) > 0) {
    pg_query_free (query);
    g_free (sql);
    return FALSE;
  }
  pathname = pg_query_get_fieldvalue_byname (query, "PATHNAME");
  if (pathname[0] == '.') 
    filename = g_strconcat (gpc_repository, pathname, pg_query_get_fieldvalue_byname (query, "FILENAME"), NULL);
  else
    filename = g_strconcat (pathname, pg_query_get_fieldvalue_byname (query, "FILENAME"), NULL);
  result = g_file_test (filename, G_FILE_TEST_ISFILE);
  g_free (filename);
  g_free (sql);
  pg_query_free (query);
  return result;
}


/** Create thumbnail for specified image, using command defined in preferences */
gboolean gpc_image_create_thumbnail (pgConnection *connection, GpcImageInfos *infos)
{
  gint           film_ID;
  GpcFilmInfos  *film_infos;
  gchar         *imagepath;
  gchar         *imagename;
  gchar         *source;
  gchar         *dest;
  gchar         *thumbpath;
  gchar         *command;
  gchar         *image_command;
  gint           result = 0;

  if (!infos)
    return FALSE;
  film_ID = gpc_image_get_film_id (infos);
  film_infos = gpc_film_get_infos (connection, film_ID);
  imagepath = gpc_film_get_full_pathname (film_infos);
  imagename = gpc_image_get_filename (infos);
  thumbpath = g_strconcat (imagepath, "/.gpc", NULL);
  if (!g_file_test (thumbpath, G_FILE_TEST_ISDIR))    /* Thumbnails directory not found */
    result = mkdir (thumbpath, S_IRWXU | S_IRGRP | S_IXGRP);
  g_free (thumbpath);
  if (!result) {
    source = g_strdup_printf ("%s%s", imagepath, imagename);
    dest = g_strdup_printf ("%s.gpc/%s", imagepath, imagename);
    command = gnome_config_get_string ("/gphotocoll/thumbnails/convert=convert -geometry \"150x150>\" -enhance \"%s\" \"%s\"");
    image_command = g_strdup_printf (command, source, dest);
    system (image_command);
    g_free (image_command);
    g_free (command);
    g_free (dest);
    g_free (source);
    g_free (imagepath);
    gpc_film_infos_free (film_infos);
    return TRUE;
  }
  else {
    g_free (imagepath);
    return FALSE;
  }
}


/** Append an image into database, for which ABSOLUTE filename is given.
    The corresponding film is created if needed, but NOT updated with other images */
gboolean gpc_image_append (pgConnection *connection, gchar *filename)
{
  GpcImageInfos *infos;
  gchar         *filmpath;
  gchar         *abs_path;
  gchar         *gpc_path;
  gchar         *basename;
  gint           ID = 0;
  gboolean       result = FALSE;
  
  filmpath = g_dirname (filename);
  if (filmpath[strlen (filmpath) - 1] != '/')
    abs_path = g_strconcat (filmpath, "/", NULL);
  else
    abs_path = g_strdup (filmpath);
  gpc_path = gpc_path_relname (abs_path);
  basename = g_strdup (g_basename (filename));
  if (!gpc_image_exists (connection, gpc_path, basename)) {
    infos = gpc_image_infos_new ();
    gpc_image_set_filename (infos, basename);
    gpc_image_set_description (infos, basename);
    result = gpc_image_get_infos_from_file (filename, infos);
    if (result && (ID = gpc_film_append_without_update (connection, filmpath))) {
      infos->film_ID = ID;
      gpc_image_modified = TRUE;
      result = gpc_image_infos_store (connection, infos);
      if (gpc_store_thumbnails)
        result = result && gpc_image_create_thumbnail (connection, infos);
      if (result && films_clist)
        on_films_clist_select_row (films_clist, gpc_film_selection, 0, NULL, NULL);
    }
    gpc_image_infos_free (infos);
  }
  g_free (basename);
  g_free (gpc_path);
  g_free (filmpath);
  return result;
}


/** Insert an image into database, with the given RELATIVE pathname and a filename (without path) */
gboolean gpc_image_insert (pgConnection *connection, gint film_id, gchar *dirname, gchar *filename)
{
  GpcImageInfos   *infos;
  gboolean         result = FALSE;
  
  if ((connection == NULL) || (!pg_connection_is_active (connection)) ||
      (dirname == NULL) || (strlen (dirname) == 0) ||
      (filename == NULL) || (strlen (filename) == 0))
    return FALSE;
  
  infos = gpc_image_infos_new ();
  infos->film_ID = film_id;
  gpc_image_set_filename (infos, filename);
  gpc_image_set_description (infos, filename);
  if (dirname[0] == '.')
    result = gpc_image_get_infos_from_file (g_strconcat (gpc_repository, dirname, filename, NULL), infos);
  else
    result = gpc_image_get_infos_from_file (g_strconcat (dirname, filename, NULL), infos);
  gpc_image_modified = result;
  result = result && gpc_image_infos_store (connection, infos);
  if (gpc_store_thumbnails)
    result = result && gpc_image_create_thumbnail (connection, infos);
  gpc_image_infos_free (infos);
  return result;
}


/** Update an image stored into database, with the given RELATIVE pathname and a filename (without path) */
gboolean gpc_image_update (pgConnection *connection, gint film_id, gchar *dirname, gchar *filename)
{
  GpcImageInfos  *infos;
  pgQuery        *query;
  gchar          *sql;
  gchar          *sql_filename;
  gchar          *image_id;
  gboolean        result = FALSE;
  
  if ((connection == NULL) || (!pg_connection_is_active (connection)) ||
      (dirname == NULL) || (strlen (dirname) == 0) ||
      (filename == NULL) || (strlen (filename) == 0))
    return FALSE;

  sql_filename = get_sql_string (filename);
  sql = g_strdup_printf ("select ID "
                         "from GPC_IMAGE "
                         "where (FILM_ID = %d) "
                         "  and (FILENAME = %s)", film_id, sql_filename);
  query = pg_query_new_with_sql (connection, sql);
  g_free (sql);
  g_free (sql_filename);
  if (!query)
    return FALSE;
  if (pg_query_open (query) != 1) {
    pg_connection_show_error (connection, _("An error occured :"), _("Retrieving multiple records from GPC_IMAGE table...!"));
    pg_query_free (query);
    return FALSE;
  }
  image_id = pg_query_get_fieldvalue_byname (query, "ID");
  infos = gpc_image_get_infos (connection, atoi (image_id));
  pg_query_free (query);
  if (dirname[0] == '.') 
    result = gpc_image_get_infos_from_file (g_strconcat (gpc_repository, dirname, filename, NULL), infos);
  else
    result = gpc_image_get_infos_from_file (g_strconcat (dirname, filename, NULL), infos);
  gpc_image_modified = result;
  result = result && gpc_image_infos_store (connection, infos);
  if (gpc_store_thumbnails)
    result = result && gpc_image_create_thumbnail (connection, infos);
  gpc_image_infos_free (infos);
  return result;
}


gboolean gpc_image_delete (pgConnection *connection, gint ID)
{
  gchar     *sql;
  pgQuery   *query;
  gboolean   result = FALSE;
  
  if ((connection ==  NULL) ||
      (!pg_connection_is_active (connection)))
    return result;
  
  sql = g_strdup_printf ("delete from GPC_IMAGE "
                         "where ID = %d", ID);
  query = pg_query_new_with_sql (connection, sql);
  result = pg_query_execute (query);
  pg_query_free (query);
  g_free (sql);
  return result;
}


/** Execute a query extracting every image in a given film */
pgQuery* gpc_image_get_list_from_film (pgConnection *connection, gint film_ID)
{
  pgQuery *query;
  gchar   *sql;

  if (!connection || !pg_connection_is_active (connection))
    return NULL;
  sql = g_strdup_printf ("select ID, NUMBER, DESCRIPTION "
                         "from GPC_IMAGE "
                         "where (FILM_ID = %d) "
                         "order by NUMBER", film_ID);
  query = pg_query_new_with_sql (connection, sql);
  g_free (sql);
  if (!query)
    return NULL;
  if (pg_query_open (query) < 0) {
    pg_connection_show_error (connection, _("An error occured :"), _("Can't open query on GPC_IMAGE table...!"));
    pg_query_free (query);
    return NULL;
  }
  return query;
}


/** Execute a query extracting images from database.
    Condition about GPC_IMAGE is given in SQL language without the "WHERE" keyword */
pgQuery* gpc_image_get_list_where (pgConnection *connection, gchar *where)
{
  gchar   *sql;
  pgQuery *query;

  if (!connection || !pg_connection_is_active (connection))
    return NULL;
  sql = g_strdup_printf ("select ID, NUMBER, DESCRIPTION "
                         "from GPC_IMAGE "
                         "where %s "
                         "order by NUMBER", where);
  query = pg_query_new_with_sql (connection, sql);
  g_free (sql);
  if (!query)
    return NULL;
  if (pg_query_open (query) < 0) {
    pg_connection_show_error (connection, _("An error occured :"), _("Can't open query on GPC_IMAGE table...!"));
    pg_query_free (query);
    return NULL;
  }
  return query;
}


gint gpc_image_get_id (GpcImageInfos *infos)
{
  if (!infos)
    return -1;
  return infos->ID;
}


gint gpc_image_get_film_id (GpcImageInfos *infos)
{
  if (!infos)
    return -1;
  return infos->film_ID;
}


gint gpc_image_get_width (GpcImageInfos *infos)
{
  if (!infos)
    return -1;
  return infos->width;
}


gint gpc_image_get_height (GpcImageInfos *infos)
{
  if (!infos)
    return -1;
  return infos->height;
}


gchar* gpc_image_get_mime_type (GpcImageInfos *infos)
{
  if (!infos)
    return NULL;
  return infos->mime_type;
}


gchar* gpc_image_get_filename (GpcImageInfos *infos)
{
  if (!infos)
    return NULL;
  return infos->filename;
}


gchar* gpc_image_get_number (GpcImageInfos *infos)
{
  if (!infos)
    return NULL;
  return infos->number;
}


gchar* gpc_image_get_lens_maker (GpcImageInfos *infos)
{
  if (!infos)
    return NULL;
  return infos->lens_maker;
}


gchar* gpc_image_get_lens_model (GpcImageInfos *infos)
{
  if (!infos)
    return NULL;
  return infos->lens_model;
}


gshort gpc_image_get_focal (GpcImageInfos *infos)
{
  if (!infos)
    return -1;
  return infos->focal;
}


gfloat gpc_image_get_aperture (GpcImageInfos *infos)
{
  if (!infos)
    return -1;
  return infos->aperture;
}


gchar* gpc_image_get_speed (GpcImageInfos *infos)
{
  if (!infos)
    return NULL;
  return infos->speed;
}


gfloat gpc_image_get_latitude (GpcImageInfos *infos)
{
  if (!infos)
    return -1;
  return infos->latitude;
}


gfloat gpc_image_get_longitude (GpcImageInfos *infos)
{
  if (!infos)
    return -1;
  return infos->longitude;
}


gfloat gpc_image_get_elevation (GpcImageInfos *infos)
{
  if (!infos)
    return -1;
  return infos->elevation;
}


gfloat gpc_image_get_direction (GpcImageInfos *infos)
{
  if (!infos)
    return -1;
  return infos->direction;
}


gfloat gpc_image_get_pitch (GpcImageInfos *infos)
{
  if (!infos)
    return -1;
  return infos->pitch;
}


gfloat gpc_image_get_roll (GpcImageInfos *infos)
{
  if (!infos)
    return -1;
  return infos->roll;
}


gshort gpc_image_get_interest_note (GpcImageInfos *infos)
{
  if (!infos)
    return -1;
  return infos->interest_note;
}


gshort gpc_image_get_quality_note (GpcImageInfos *infos)
{
  if (!infos)
    return -1;
  return infos->quality_note;
}


gchar* gpc_image_get_description (GpcImageInfos *infos)
{
  if (!infos)
    return NULL;
  return infos->description;
}


GList* gpc_image_get_keywords (GpcImageInfos *infos)
{
  if (!infos)
    return NULL;
  return infos->keywords;
}


void gpc_image_set_filename (GpcImageInfos *infos, gchar *value)
{
  if (!infos)
    return;
  if (infos->filename)
    g_free (infos->filename);
  infos->filename = g_strdup (value);
}


void gpc_image_set_width (GpcImageInfos *infos, gint value)
{
  if (!infos)
    return;
  infos->width = value;
}


void gpc_image_set_height (GpcImageInfos *infos, gint value)
{
  if (!infos)
    return;
  infos->height = value;
}


void gpc_image_set_mime_type (GpcImageInfos *infos, gchar *value)
{
  if (!infos)
    return;
  if (infos->mime_type)
    g_free (infos->mime_type);
  infos->mime_type = g_strdup (value);
}


void gpc_image_set_number (GpcImageInfos *infos, gchar *value)
{
  if (!infos)
    return;
  if (infos->number)
    g_free (infos->number);
  infos->number = g_strdup (value);
}


void gpc_image_set_lens_maker (GpcImageInfos *infos, gchar *value)
{
  if (!infos)
    return;
  if (infos->lens_maker)
    g_free (infos->lens_maker);
  infos->lens_maker = g_strdup (value);
}


void gpc_image_set_lens_model (GpcImageInfos *infos, gchar *value)
{
  if (!infos)
    return;
  if (infos->lens_model)
    g_free (infos->lens_model);
  infos->lens_model = g_strdup (value);
}


void gpc_image_set_focal (GpcImageInfos *infos, gshort value)
{
  if (!infos)
    return;
  infos->focal = value;
}


void gpc_image_set_aperture (GpcImageInfos *infos, gfloat value)
{
  if (!infos)
    return;
  infos->aperture = value;
}


void gpc_image_set_speed (GpcImageInfos *infos, gchar *value)
{
  if (!infos)
    return;
  if (infos->speed)
    g_free (infos->speed);
   infos->speed = g_strdup (value);
}


void gpc_image_set_latitude (GpcImageInfos *infos, gfloat value)
{
  if (!infos)
    return;
  infos->latitude = value;
}


void gpc_image_set_longitude (GpcImageInfos *infos, gfloat value)
{
  if (!infos)
    return;
  infos->longitude = value;
}


void gpc_image_set_elevation (GpcImageInfos *infos, gfloat value)
{
  if (!infos)
    return;
  infos->elevation = value;
}


void gpc_image_set_direction (GpcImageInfos *infos, gfloat value)
{
  if (!infos)
    return;
  infos->direction = value;
}


void gpc_image_set_pitch (GpcImageInfos *infos, gfloat value)
{
  if (!infos)
    return;
  infos->pitch = value;
}


void gpc_image_set_roll (GpcImageInfos *infos, gfloat value)
{
  if (!infos)
    return;
  infos->roll = value;
}


void gpc_image_set_interest_note (GpcImageInfos *infos, gshort value)
{
  if (!infos)
    return;
  infos->interest_note = value;
}


void gpc_image_set_quality_note (GpcImageInfos *infos, gshort value)
{
  if (!infos)
    return;
  infos->quality_note = value;
}


void gpc_image_set_description (GpcImageInfos *infos, gchar *value)
{
  if (!infos)
    return;
  if (infos->description)
    g_free (infos->description);
  infos->description = g_strdup (value);
}


void gpc_image_set_keywords (GpcImageInfos *infos, GList *value, gboolean free_it)
{
  if (!infos)
    return;
  if (infos->keywords && free_it) {
    g_list_foreach (infos->keywords, (GFunc) g_free, NULL);
    g_list_free (infos->keywords);
  }
  infos->keywords = value;
}


gboolean gpc_image_infos_store (pgConnection *connection, GpcImageInfos *infos)
{
  gchar   *sql;
  gchar   *filename;
  gchar   *mime_type;
  gchar   *number;
  gchar   *lens_maker;
  gchar   *lens_model;
  gchar   *speed;
  gchar   *description;

  gchar   *old_locale;
  gchar	  *saved_locale;

  pgQuery *query;
  gboolean result;

  if (!connection || !infos || !pg_connection_is_active (connection))
    return FALSE;
  if (!gpc_image_modified)
    return TRUE;
  if (!pg_connection_start (connection))
    return FALSE;
  old_locale = setlocale (LC_NUMERIC, NULL);
  saved_locale = g_strdup (old_locale);
  setlocale (LC_NUMERIC, "C");
  filename = get_sql_string (gpc_image_get_filename (infos));
  mime_type = get_sql_string (gpc_image_get_mime_type (infos));
  number = get_sql_string (gpc_image_get_number (infos));
  lens_maker = get_sql_string (gpc_image_get_lens_maker (infos));
  lens_model = get_sql_string (gpc_image_get_lens_model (infos));
  speed = get_sql_string (gpc_image_get_speed (infos));
  description = get_sql_string (gpc_image_get_description (infos));
  if (infos->ID == -1) 
    sql = g_strdup_printf ("insert into GPC_IMAGE "
                           "(FILM_ID,"
                           " FILENAME,"
                           " WIDTH,"
                           " HEIGHT,"
                           " MIME_TYPE,"
                           " NUMBER,"
                           " LENS_MAKER,"
                           " LENS_MODEL,"
                           " FOCAL,"
                           " APERTURE,"
                           " SPEED,"
                           " LATITUDE,"
                           " LONGITUDE,"
                           " ELEVATION,"
                           " DIRECTION,"
                           " PITCH,"
                           " ROLL,"
                           " INTEREST_NOTE,"
                           " QUALITY_NOTE,"
                           " DESCRIPTION) "
                           "values "
                           "(%d,"
                           " %s,"
                           " %d,"
                           " %d,"
                           " %s,"
                           " %s,"
                           " %s,"
                           " %s,"
                           " %d,"
                           " %f,"
                           " %s,"
                           " %f,"
                           " %f,"
                           " %f,"
                           " %f,"
                           " %f,"
                           " %f,"
                           " %d,"
                           " %d,"
                           " %s)",
                           gpc_image_get_film_id (infos),
                           filename,
                           gpc_image_get_width (infos),
                           gpc_image_get_height (infos),
                           mime_type,
                           number,
                           lens_maker,
                           lens_model,
                           gpc_image_get_focal (infos),
                           gpc_image_get_aperture (infos),
                           speed,
                           gpc_image_get_latitude (infos),
                           gpc_image_get_longitude (infos),
                           gpc_image_get_elevation (infos),
                           gpc_image_get_direction (infos),
                           gpc_image_get_pitch (infos),
                           gpc_image_get_roll (infos),
                           gpc_image_get_interest_note (infos),
                           gpc_image_get_quality_note (infos),
                           description);
  else
    sql = g_strdup_printf ("update GPC_IMAGE set "
                           " FILENAME = %s,"
                           " WIDTH = %d,"
                           " HEIGHT = %d,"
                           " MIME_TYPE = %s,"
                           " NUMBER = %s,"
                           " LENS_MAKER = %s,"
                           " LENS_MODEL = %s,"
                           " FOCAL = %d,"
                           " APERTURE = %f,"
                           " SPEED = %s,"
                           " LATITUDE = %f,"
                           " LONGITUDE = %f,"
                           " ELEVATION = %f,"
                           " DIRECTION = %f,"
                           " PITCH = %f,"
                           " ROLL = %f,"
                           " INTEREST_NOTE = %d,"
                           " QUALITY_NOTE = %d,"
                           " DESCRIPTION = %s "
                           "where ID = %d",
                           filename,
                           gpc_image_get_width (infos),
                           gpc_image_get_height (infos),
                           mime_type,
                           number,
                           lens_maker,
                           lens_model,
                           gpc_image_get_focal (infos),
                           gpc_image_get_aperture (infos),
                           speed,
                           gpc_image_get_latitude (infos),
                           gpc_image_get_longitude (infos),
                           gpc_image_get_elevation (infos),
                           gpc_image_get_direction (infos),
                           gpc_image_get_pitch (infos),
                           gpc_image_get_roll (infos),
                           gpc_image_get_interest_note (infos),
                           gpc_image_get_quality_note (infos),
                           description,
                           gpc_image_get_id (infos));
  query = pg_query_new_with_sql (connection, sql);
  result = pg_query_execute (query);
  if (!result)
    pg_connection_show_error (connection, _("PostgreSQL error"), _("Can't store image informations..."));
  pg_query_free (query);
  g_free (sql);
  g_free (filename);
  g_free (mime_type);
  g_free (number);
  g_free (lens_maker);
  g_free (lens_model);
  g_free (speed);
  g_free (description);
  result = result && gpc_image_infos_store_keywords (connection, infos);
  result = result && gpc_image_store_references (connection, infos);
  result = result && pg_connection_commit (connection);
  if (!result)
    pg_connection_rollback (connection);
  setlocale (LC_NUMERIC, saved_locale);
  g_free (saved_locale);
  gpc_image_modified = !result;
  return result;
}


gboolean gpc_image_infos_store_keywords (pgConnection *connection, GpcImageInfos *infos)
{
  guint   elements;
  guint   index;
  gchar   *sql;
  gchar   *sql_code;
  GList   *keywords;

  pgQuery *query;
  gboolean result;

  if (!connection || !infos || !pg_connection_is_active(connection))
    return FALSE;
  if (!gpc_image_modified)
    return TRUE;
  sql = g_strdup_printf ("delete from GPC_KEYWORD "
                         "where (IMAGE_ID = %d)",
                         gpc_image_get_id (infos));
  query = pg_query_new_with_sql (connection, sql);
  result = pg_query_execute (query);
  pg_query_free (query);
  g_free (sql);
  keywords = gpc_image_get_keywords (infos);
  elements = g_list_length (keywords);
  for (index = 0; index < elements; index++) {
    sql_code = get_sql_string ((gchar *) g_list_nth_data (keywords, index));
    sql = g_strdup_printf ("insert into GPC_KEYWORD (CODE, FILM_ID, IMAGE_ID) "
                           "values (%s, 0, %d)",
                           sql_code,
                           gpc_image_get_id (infos));
    query = pg_query_new_with_sql (connection, sql);
    result = result && pg_query_execute (query);
    pg_query_free (query);
    g_free (sql);
    g_free (sql_code);
  }
  return result;
}


gboolean gpc_image_store_references (pgConnection *connection, GpcImageInfos *infos)
{
  gboolean result;
  
  if (!connection || !infos || !pg_connection_is_active (connection))
    return FALSE;
  result = TRUE;
  if (gpc_added_lens_makers) {
    result = result && gpc_app_store_keywords (connection, "GPC_LENS_MAKER", gpc_added_lens_makers);
    g_list_foreach (gpc_added_lens_makers, (GFunc) g_free, NULL);
    g_list_free (gpc_added_lens_makers);
    gpc_added_lens_makers = NULL;
  }
  if (gpc_added_lens_maker_codes) {
    result = result && gpc_app_store_maker_codes (connection, "GPC_LENS_MODEL", gpc_image_get_lens_maker (infos), gpc_added_lens_maker_codes);
    g_list_foreach (gpc_added_lens_maker_codes, (GFunc) g_free, NULL);
    g_list_free (gpc_added_lens_maker_codes);
    gpc_added_lens_maker_codes = NULL;
  }
  if (gpc_added_keywords) {
    result = result && gpc_app_store_keywords (connection, "GPC_KEY", gpc_added_keywords);
    g_list_foreach (gpc_added_keywords, (GFunc) g_free, NULL);
    g_list_free (gpc_added_keywords);
    gpc_added_keywords = NULL;
  }
  return result;
}


void gpc_image_infos_free(GpcImageInfos *infos)
{
  if (!infos)
    return;
  if (infos->filename)
    g_free (infos->filename);
  if (infos->mime_type)
    g_free (infos->mime_type);
  if (infos->number)
    g_free (infos->number);
  if (infos->lens_maker)
    g_free (infos->lens_maker);
  if (infos->lens_model)
    g_free (infos->lens_model);
  if (infos->speed)
    g_free (infos->speed);
  if (infos->description)
    g_free (infos->description);
  if (infos->keywords) {
    g_list_foreach (infos->keywords, (GFunc) g_free, NULL);
    g_list_free (infos->keywords);
  }
  g_free(infos);
}
