/* gtd_filter - Read & access user filter preferences.
 *
 * Copyright (C) 1997, 1998 Free Software Foundation
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, you can either send email to this
 * program's author (see below) or write to:
 * 
 *              The Free Software Foundation, Inc.
 *              675 Mass Ave.
 *              Cambridge, MA 02139, USA. 
 * 
 * Please send bug reports, etc. to zappo@gnu.org
 * 
 * 
 * Description:
 *   This file knows how to read user filter preferences, and provides
 * accessors for getting at the data.
 *
 * $Log: gtd_filter.c,v $
 * Revision 1.4  1998/02/15 13:58:29  zappo
 * Use extended name buffers which prevent buffer overflows.
 *
 * Revision 1.3  1998/01/04 13:33:03  zappo
 * Fixed warnings
 *
 * Revision 1.2  1997/12/14 19:19:04  zappo
 * Renamed package to gtalk, renamed symbols and files apropriately
 * Fixed copyright and email address.
 *
 * Revision 1.1  1997/12/02 22:53:51  zappo
 * Initial revision
 *
 *
 * Tokens: ::Header:: gtalkd.h
 */

#include "gtalklib.h"
#include "gtalkd.h"

struct UserFilter {
  struct GenericListNode  link; /* List part             */  
  char                   *name;	/* User pattern affected */
  char                   *host;	/* Host pattern affected */
  int                     deny;	/* How to deny the user  */
};

struct NameDeny {
  char *string;
  int   method;
};

/* Permissable ways to deny service to someone. */
static struct NameDeny deny_methods[] = {
  { "ALLOW", 0 },
  { "NOT_HERE", NOT_HERE },
  { "FAILED", FAILED },
  { "PERMISSION_DENIED", PERMISSION_DENIED },
  { "SELECTIVE_REFUSAL", SELECTIVE_REFUSAL }
};


/*
 * Function: FILT_ReadFile
 *
 *   Reads the specified users' blacklist file.  This should only be
 * called ONCE per user by the user allocator in gtd_fusr.
 *
 * Returns:     Nothing.
 * Parameters:  uo - User whose blacklist we are going to read.
 *
 * History:
 * zappo 11/26/97 Created */
void FILT_ReadFile(uo)
     struct UserObject *uo;
{
  FILE *f;
  char buffer[100];
  
  /* Start the filter list initialization (not handled by macros) */
  uo->filters.first = NULL;
  uo->filters.last = NULL;
  uo->filters.next = NULL;

  /* Try to get at the file. */
  f = fopen(uo->filterf, "r");
  if(f == NULL)
    {
#ifdef TEST
      printf("Failed to open blacklist file %s!\n", uo->filterf);
#endif
      return;
    }

  while(fgets(buffer, sizeof(buffer), f)) 
    {
      char *s = buffer;
      char *u, *h, *t;

      /* Jump whitespace */
      while(*s && ((*s == ' ') || (*s == '\t'))) s++;

      /* Skip comments and blank lines */
      if((*s == '#') || (*s == '\n') || (*s == '\0'))
	{
#ifdef TEST
	  printf("Junk: %s", s);
#endif
	  continue;
	}

      /* If we get here then there is a proposed new filter */
      if(*s != '@')
	{
	  u = s;

	  while(*s && (*s != '@') && (*s != ' ')) s++;
	  
	  if(*s == '@') {
	    *s = '\0';
	    h = ++s;
	  } else {
	    if(*s) {
	      *s = '\0';
	      s++;
	    }
	    h = NULL;
	  }

      } else {
	
	u = NULL;
	s++;
	h = s;

      }
	
      if(h)
	{
	  /* Find the rest of the hostname. */
	  while(*s && (*s != ' ')) s++;
      
	  if(*s) {
	    *s = '\0';
	    s++;
	  }
	}

      /* At last, read in the method */
      t = s;
      while(*s && (*s != '\n') && (*s != ' ') && (*s != '#')) s++;
      *s = '\0';

      if(*t) {
	/* Here we have parsed all the way, so create a new node */
	struct UserFilter *new;
	int i;

	new = (struct UserFilter *)LIST_alloc(&uo->filters,
					      sizeof(struct UserFilter));
	
	new->name = u?strdup(u):u;
	new->host = h?strdup(h):h;

	for(i = 0; i < (sizeof(deny_methods)/sizeof(struct NameDeny)); i++)
	  if(strcmp(deny_methods[i].string, t) == 0)
	    {
	      new->deny = deny_methods[i].method;
	      break;
	    }

	if(i == (sizeof(deny_methods)/sizeof(struct NameDeny))) {
#ifdef TEST
	  printf("Failed to find type [%s]\n", t);
	  i = 3;
#endif
	  /* Default unknown type to PERMISSION_DENIED */
	  new->deny = PERMISSION_DENIED;
	}
#ifdef TEST
	printf("Filter Read: %s@%s %s\n",
	       new->name?new->name:"<none>",
	       new->host?new->host:"<none>",
	       deny_methods[i].string);
      }
      else {
	printf("Failed to read line!\n");
#endif
      }
    }
}


/*
 * Function: FILT_CheckFilterSimple
 *
 *   Simplified parameter checker.  See FILT_CheckFilters.
 *
 * Returns:     int - 
 * Parameters:  uo     - User object with filters
 *              host   - Host caller is on
 *              caller - String Pointer of caller's name
 *              Ctxt   - Context
 * History:
 * zappo   11/26/97   Created
 */
int FILT_CheckFilterSimple(uo, host, caller, Ctxt)
     struct UserObject    *uo;
     struct HostObject    *host;
     char                 *caller;
     struct DaemonContext *Ctxt;
{
  int rval = 0;
  int confidence = 10;
  struct UserFilter *loop;

  loop = FirstElement(uo->filters);
  while(loop) {

    int local_confidence = 0;

    if(loop->name) {
      if(strcmp(loop->name, caller) != 0) {
	local_confidence = -1;
      }
    } else {
      local_confidence += 2;
    }

    if(local_confidence != -1) {
      if(loop->host) {
	if(strstr(host->name, loop->host) == NULL) {
	  local_confidence = -1;
	}
      } else {
	local_confidence++;
      }
    }

    if(local_confidence != -1) {
      if(local_confidence < confidence)
	{
	  confidence = local_confidence;
	  rval = loop->deny;
	}
    }

#ifdef TEST
    printf("Scan: %s==%s %s in %s: Confidence %d\n",
	   loop->name?loop->name:"<none>",
	   caller,
	   loop->host?loop->host:"<none>",
	   host->name,
	   local_confidence);
#endif
    
    loop = NextElement(loop);
  }

  if((Ctxt->type < GTALKD) && (rval == SELECTIVE_REFUSAL))
    rval = PERMISSION_DENIED;

  return rval;
}


/*
 * Function: FILT_CheckFilters
 *
 *   Checks the list of filters for the specified user.  Returns the
 * filter code or 0 (ALLOW) based on the loaded filters.
 *
 * Returns:     int - Response code, or 0.
 * Parameters:  uo   - User Object we are checking against
 *              ro   - Request object with caller in it.
 *              Ctxt - Context
 * History:
 * zappo   11/26/97   Created
 */
int FILT_CheckFilters(uo, ro, Ctxt)
     struct UserObject    *uo;
     struct RequestObject *ro;
     struct DaemonContext *Ctxt;
{
  struct HostObject *host;
  int ans;

  /* Get the hostname for the message we are examining. */
  host = HOST_gen_host_by_addr((struct sockaddr *)&ro->request.addr,
			       sizeof(ro->request.addr));

  ans = FILT_CheckFilterSimple(uo, host, ro->l_name, Ctxt);

  if((ans == SELECTIVE_REFUSAL) && (ro->request.vers < TALK_VERSION_GNU))
    ans = PERMISSION_DENIED;

  return ans;
}

