// -*- C++ -*-

/*

  Heap Layers: An Extensible Memory Allocation Infrastructure
  
  Copyright (C) 2000-2003 by Emery Berger
  http://www.cs.umass.edu/~emery
  emery@cs.umass.edu
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#ifndef _HOARD_H_
#define _HOARD_H_

#include "hldefines.h"

#define SUPERBLOCK_SIZE 8192 // 4096
#define EMPTINESS_CLASSES 8


// Hoard-specific Heap Layers

#include "check.h"
#include "superblock.h"
#include "fixedrequestheap.h"
#include "ownerheader.h"
#include "hoardmanager.h"
#include "segregatedheap.h"
#include "addheaderheap.h"
#include "semilockedheap.h"
#include "alignedheap.h"
#include "threadpoolheap.h"
#include "noheader.h"
#include "addlock.h"
#include "redirectfree.h"
#include "ignoreinvalidfree.h"
#include "addlocksuperblockheader.h"
#include "conformantheap.h"
#include "hoardsuperblock.h"
#include "lockmallocheap.h"
#include "alignedsuperblockheap.h"

// Generic Heap Layers

#include "ansiwrapper.h"
#include "debugheap.h"
#include "spinlock.h"
#include "lockedheap.h"
#include "spinlock.h"
#include "winlock.h"
#include "bins4k.h"
#include "bins8k.h"
#include "bins16k.h"
#include "oneheap.h"
#include "freelistheap.h"
#include "nullheap.h"
#include "threadheap.h"
#include "hybridheap.h"
#include "posixlock.h"
#include "spinlock.h"


using namespace HL;

#if defined(_WIN32)
typedef HL::WinLockType TheLockType;
#else
typedef HL::SpinLockType TheLockType;
#endif



class bogusThresholdFunctionClass {
public:
  static inline int function (int, int, size_t) {
    // We *never* cross the threshold for a process heap.
    return 0;
  }
};


class ProcessHeap1 :
  public ConformantHeap<
                        HoardManager<AlignedSuperblockHeap<TheLockType, SUPERBLOCK_SIZE>,
  			EmptyHoardManager<HoardSuperblock<TheLockType, SUPERBLOCK_SIZE, ProcessHeap1>, SUPERBLOCK_SIZE>,
  			HoardSuperblock<TheLockType, SUPERBLOCK_SIZE, ProcessHeap1>,
  			SUPERBLOCK_SIZE,
		   	EMPTINESS_CLASSES,
			TheLockType,
  			bogusThresholdFunctionClass,
                        ProcessHeap1> > {
public:

  ProcessHeap1 (void) {}

  // Disable allocation from this heap.
  inline void * malloc (size_t);


private:
  ProcessHeap1 (const ProcessHeap1&);
  ProcessHeap1& operator=(const ProcessHeap1&);

};

// Just one process (global) heap.
class ProcessHeap {
public:

  ProcessHeap (void) 
    : theHeap (getHeap())
    {
      //      printf ("instantiating processheap %x\n", this);
    }
  
  inline size_t getSize (void * ptr) {
    return theHeap->getSize (ptr);
  }
  typedef ProcessHeap1 SuperHeap;
  typedef HoardSuperblock<TheLockType, SUPERBLOCK_SIZE, ProcessHeap1> SuperblockType;

  enum { Alignment = SuperHeap::Alignment };


  void put (SuperblockType * s, size_t sz) {
    assert (s);
    assert (s->isValidSuperblock());
    theHeap->put (s, sz);
  }

  SuperblockType * get (size_t sz, ProcessHeap * dest) {
    SuperblockType * s = theHeap->get (sz, reinterpret_cast<ProcessHeap1 *>(dest));
    if (s) {
      assert (s->isValidSuperblock());
    }
    return s;
  }

private:

  SuperHeap * theHeap;

  inline static SuperHeap * getHeap (void) {
    static double theHeapBuf[sizeof(SuperHeap) / sizeof(double) + 1];
    static SuperHeap * theHeap = new (&theHeapBuf[0]) SuperHeap;
    return theHeap;
  }

  ProcessHeap (const ProcessHeap&);
  ProcessHeap& operator=(const ProcessHeap&);

};


class hoardThresholdFunctionClass {
public:
  inline static int function (int u, int a, size_t objSize) {
    /*
      Returns 1 iff we've crossed the emptiness threshold:

      U < A - 2S   &&   U < 7/8 A

     */
    int r = ((8 * u) < (7 * a)) && ((u < a - (2 * SUPERBLOCK_SIZE) / (int) objSize));
    return r;
  }
};


// The heap that manages small objects.
class SmallHeap : 
  public HoardManager<AlignedSuperblockHeap<TheLockType, SUPERBLOCK_SIZE>,
                      ProcessHeap,
                      HoardSuperblock<TheLockType, SUPERBLOCK_SIZE, SmallHeap>,
                      SUPERBLOCK_SIZE,
                      EMPTINESS_CLASSES,
                      TheLockType,
                      hoardThresholdFunctionClass,
                      SmallHeap> {};

#if defined(_WIN32)

#include "winmmapheap.h"

// The heap that manages large objects.
class BigHeapSuper :
  public AddHeaderHeap<HoardSuperblock<TheLockType, SUPERBLOCK_SIZE, BigHeapSuper>,
		       SUPERBLOCK_SIZE,
		       WinMmapHeap<SUPERBLOCK_SIZE, 256 * 1024> >
{
  HL::sassert<(((int) Alignment % SUPERBLOCK_SIZE) == 0)> VerifyAlignment;
};

#else

#include "alignedmmap.h"

// The heap that manages large objects.
class BigHeapSuper :
  public AddHeaderHeap<HoardSuperblock<TheLockType, SUPERBLOCK_SIZE, BigHeapSuper>,
  SUPERBLOCK_SIZE,
  AlignedMmap<SUPERBLOCK_SIZE, HL::MmapHeap> >
{
  HL::sassert<(((int) Alignment % SUPERBLOCK_SIZE) == 0)> VerifyAlignment;
};

#endif


// BigHeap is a locked variant of the above that we actually use for
// allocation, since AlignedMmap is not thread-safe.

class BigHeap : public LockedHeap<TheLockType, BigHeapSuper> {
  HL::sassert<(((int) Alignment % SUPERBLOCK_SIZE) == 0)> VerifyAlignment;
public:
  inline void * malloc (size_t sz) {
    void * ptr = LockedHeap<TheLockType, BigHeapSuper>::malloc (sz);
    assert (((size_t) ptr - sizeof(HoardSuperblock<TheLockType, SUPERBLOCK_SIZE, BigHeapSuper>::Header)) % SUPERBLOCK_SIZE == 0);
    return ptr;
  }
};


class PerThreadHoardHeap :
  public RedirectFree<LockMallocHeap<SmallHeap>,
                      HoardSuperblock<TheLockType, SUPERBLOCK_SIZE, SmallHeap>,
                      SUPERBLOCK_SIZE> {};


template <int N, int NH>
class HoardHeap :
  public IgnoreInvalidFree<
         HL::ANSIWrapper<
         HL::HybridHeap<bins<HoardSuperblock<TheLockType, SUPERBLOCK_SIZE, SmallHeap>::Header,
                          SUPERBLOCK_SIZE>::BIG_OBJECT,
			ThreadPoolHeap<N, NH, PerThreadHoardHeap>,
			BigHeap> > >
{
public:
  enum { BIG_OBJECT = bins<HoardSuperblock<TheLockType, SUPERBLOCK_SIZE, SmallHeap>::Header, SUPERBLOCK_SIZE>::BIG_OBJECT };
};


#endif
