#include <stdio.h>
#include <stdlib.h>
#include <dlfcn.h>
#include <sys/mman.h>
#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>

#include <map>


// Control output of debugging messages.

#if 0
#define IF_DEBUG(x) x
#else
#define IF_DEBUG(x) 
#endif


volatile int anyThreadCreated = 1;

const size_t minimumPageSize = 4096;

#include <heaplayers.h>

using namespace std;
using namespace HL;

struct MyLess {
  bool operator()(const void * a, const void * b) const {
    return (size_t) a < (size_t) b;
  }
};

class MyHeap : public FreelistHeap<StaticHeap<1048576> > {};

template <class TYPE>
class MyAllocator : public STLAllocator<TYPE, MyHeap> {};

typedef map<const void *, size_t, MyLess, MyAllocator<pair<const void *, size_t> > > mapType;

mapType mappedRegions;

#if !defined(RTLD_NEXT)
#define RTLD_NEXT ((void *) -1)
#endif

// Keep track of all heap memory allocated via sbrk and mmap.

extern "C" {

  typedef void * mmapFunctionType (void * addr, size_t len, int prot, int flags, int fildes, off_t off);
  typedef int munmapFunctionType (void * addr, size_t len);
  typedef void exitFunctionType (int);
  typedef void * sbrkFunctionType (intptr_t);
  typedef int brkFunctionType (void * endds);

  static mmapFunctionType * mmapFn = NULL;
  static munmapFunctionType * munmapFn = NULL;
  static sbrkFunctionType * sbrkFn = NULL;
  static brkFunctionType  * brkFn  = NULL;
  static int devZeroFd = 0;
  static size_t currentAllocatedMmap = 0;
  static size_t currentAllocatedSbrk = 0;
  static size_t maxAllocated = 0;
  static size_t initialBreak = 0;
  static void * currentBreak = 0;
  bool instrument = true;

  void * mmap (void * addr, size_t len, int prot, int flags, int fildes, off_t off) {
    if (!mmapFn) {
      devZeroFd = open ("/dev/zero", O_RDWR);
      mmapFn = (mmapFunctionType *) dlsym (RTLD_NEXT, "mmap");
    }
    // Check to see if these are heap pages (anonymous or /dev/zero).
    if (((flags & MAP_ANON) && (fildes == -1)) || (fildes == devZeroFd)) {
      if (instrument) {
	void * ptr = (*mmapFn)(addr, len, prot, flags, fildes, off);
	fprintf (stderr, "original mmap call: %x, %d\n", (void *) ptr, len);
	size_t mapped = 0;
	while (mapped < len) {
	  mappedRegions[((char *) ptr + mapped)] = minimumPageSize;
	  IF_DEBUG(fprintf (stderr, "mmap %x, %d\n", (void *) ((char *) ptr + mapped), minimumPageSize));
	  mapped += minimumPageSize;
	  currentAllocatedMmap += minimumPageSize;
	}
	if (currentAllocatedMmap + currentAllocatedSbrk > maxAllocated) {
	  maxAllocated = currentAllocatedMmap + currentAllocatedSbrk;
	}
	return (void *) ptr;
      }
    }
    return (*mmapFn)(addr, len, prot, flags, fildes, off);
  }

  int munmap (void * addr, size_t len) {
    if (!munmapFn) {
      munmapFn = (munmapFunctionType *) dlsym (RTLD_NEXT, "munmap");
    }
    if (instrument) {
      size_t unmapped = 0;
      while (unmapped < len) {
	mapType::iterator i = mappedRegions.find ((char *) addr + unmapped);
	mappedRegions.erase (i);
	unmapped += minimumPageSize;
	currentAllocatedMmap -= minimumPageSize;
      }
      fprintf (stderr, "munmap %x, %d\n", (void *) addr, len);
    }
    return (*munmapFn) (addr, len);
  }

  void * sbrk (intptr_t incr) {
    if (!sbrkFn) {
      sbrkFn = (sbrkFunctionType *) dlsym (RTLD_NEXT, "sbrk");
      initialBreak = (size_t) (*sbrkFn)(0);
    }
    void * p = (*sbrkFn)(incr);
    if (instrument) {
      currentAllocatedSbrk = (size_t) ((char *) ((*sbrkFn)(0)) - (char *) initialBreak);
      if (currentAllocatedMmap + currentAllocatedSbrk > maxAllocated) {
	maxAllocated = currentAllocatedMmap + currentAllocatedSbrk;
      }
      IF_DEBUG(fprintf (stderr, "sbrk = %d\n", (size_t) incr));
    }
    return p;
  }

  int brk (void * endds) {
    IF_DEBUG(fprintf (stderr, "brk!\n"));

    if (!brkFn) {
      brkFn = (brkFunctionType *) dlsym (RTLD_NEXT, "brk");
      initialBreak = (size_t) (*sbrkFn)(0);
    }
    if (instrument) {
      IF_DEBUG(fprintf (stderr, "brk = %d\n", (size_t) endds));
      currentAllocatedSbrk = (size_t) endds;
      if (currentAllocatedMmap + currentAllocatedSbrk > maxAllocated) {
	maxAllocated = currentAllocatedMmap + currentAllocatedSbrk;
      }
    }
    return (*brkFn)(endds);
  }

  void exit (int status) {
    instrument = false;
    exitFunctionType * fn = (exitFunctionType *) dlsym (RTLD_NEXT, "exit");
    fprintf (stderr, "Current allocated with mmap = %d\n", currentAllocatedMmap);
    fprintf (stderr, "Current allocated with sbrk = %d\n", currentAllocatedSbrk);
    fprintf (stderr, "Max allocated overall = %d\n", maxAllocated);
    if (fn) {
      (*fn)(status);
    }
  }
}
