#include "hitopcommands.h"
#ifndef __htmlh__
  #include "html.h"
#endif
#ifndef __parammaph__
  #include "parammap.h"
#endif
#ifndef __errorh__
  #include "error.h"
#endif
#ifndef __imgstoreh__
  #include "imgstore.h"
#endif
#ifndef __defstoreh__
  #include "defstore.h"
#endif
#ifndef __navh__
  #include "nav.h"
#endif
#ifndef __mathsh__
  #include "maths.h"
#endif
#ifndef __pluginmgrh__
  #include "pluginmgr.h"
#endif
#ifndef __polytypeh__
  #include "polytype.h"
#endif

#include <fstream>
#include <map>
#include <ctime>

HTMLStream::iterator HitopCommands::IMPOSSIBLE(HTMLStream& stream,
  HTMLStream::iterator cur,const ParamMap& paramMap,const string& tag){
  Error(*cur,tag+" resulted from variable expansion");
  return cur;
}

HTMLStream::iterator HitopCommands::IF(HTMLStream& stream,
  HTMLStream::iterator cur,const ParamMap& paramMap,const string& tag){
  if(cur->m_myPair==NULL){
    if(cur->Token()==TokenMap::ELSEIF){
      Error(*cur,"@ELSEIF resulted from variable expansion");
    }else{
      Error(*cur,"@IF resulted from variable expansion");
    }
  }
  string name,value,compare="EQ";
  HTMLStream::iterator next;
  bool eval=false;
  if(paramMap.Retrieve("EXPR",name)){
    eval=PolyType(evaluate(name)).AsBool();
  }else{
    if(!paramMap.Retrieve("NAME",name)) Error(*cur,e_NoName);
    if(paramMap.Retrieve("COMPARE",compare)&&!g_isXML) transform(compare.begin(),compare.end(),compare.begin(),ToUpper());
    if(paramMap.Retrieve("VALUE",value)){
      string var;
      if(!stream.m_curVars.Get(*cur,name,var)){
        Error(*cur,e_UndefVar,name);
      }else{
        if(compare=="EQ"){
          eval=(var==value);
        }else if(compare=="NE"){
          eval=(var!=value);
        }else if(compare=="LT"){
          eval=(var<value);
        }else if(compare=="GT"){
          eval=(var>value);
        }else if(compare=="LE"){
          eval=(var<=value);
        }else if(compare=="GE"){
          eval=(var>=value);
        }else if(compare=="IN"){
          eval=value.find(var)!=string::npos;
        }else if(compare=="NI"){
          eval=value.find(var)==string::npos;
        }else if(compare=="EQINT"){
          eval=(atoi(var.c_str())==atoi(value.c_str()));
        }else if(compare=="NEINT"){
          eval=(atoi(var.c_str())!=atoi(value.c_str()));
        }else if(compare=="LTINT"){
          eval=(atoi(var.c_str())<atoi(value.c_str()));
        }else if(compare=="GTINT"){
          eval=(atoi(var.c_str())>atoi(value.c_str()));
        }else if(compare=="LEINT"){
          eval=(atoi(var.c_str())<=atoi(value.c_str()));
        }else if(compare=="GEINT"){
          eval=(atoi(var.c_str())>=atoi(value.c_str()));
        }else{
          Error(*cur,e_BadCompare);
        }
      }
    }else{
      eval=stream.m_curVars.IsDefName(name);
    }
  }

  if(paramMap.Defined("NOT")) eval=!eval;
  if(cur->m_myElse==NULL){
    if(eval){
      next=cur;
      ++next;
      if(next==cur->m_myPair) ++next;
      stream.m_stream.erase(cur->m_myPair);
      stream.m_stream.erase(cur);
    }else{
      next=cur->m_myPair;
      ++next;
      stream.m_stream.erase(cur,next);
    }
  }else{
    if(eval){
      next=cur;
      ++next;
      if(next==cur->m_myElse){
        next=cur->m_myPair;
        ++next;
      }
      ++cur->m_myPair;
      stream.m_stream.erase(cur->m_myElse,cur->m_myPair);
      stream.m_stream.erase(cur);
    }else{
      next=cur->m_myElse;
      if(next->Token()==TokenMap::ELSEIF){
        stream.m_stream.erase(cur,cur->m_myElse);
      }else{
        ++next;
        if(next==cur->m_myPair) ++next;
        stream.m_stream.erase(cur->m_myPair);
        ++cur->m_myElse;
        stream.m_stream.erase(cur,cur->m_myElse);
      }
    }
  }
  return next;
}

static bool FileToString(const string& fileName,string& content){
  ifstream inputFile;
  inputFile.open(fileName.c_str(),ios::in);
  if(!inputFile.good()) return false;
  char ch='\0';
  while(!inputFile.eof()){
    inputFile.get(ch);
    if(!inputFile.eof()) content+=ch;
  }
  if(ch=='\n') content=content.substr(0,content.length()-1);
  inputFile.close();
  return true;
}

HTMLStream::iterator HitopCommands::SET(HTMLStream& stream,
  HTMLStream::iterator cur,const ParamMap& paramMap,const string& tag){
  Vars::Scope scope=Vars::Local;
  string scopeText;
  if(paramMap.Retrieve("SCOPE",scopeText)){
    transform(scopeText.begin(),scopeText.end(),scopeText.begin(),ToUpper());
    if(scopeText=="LOCAL"){
      scope=Vars::Local;
    }else if(scopeText=="PARENT"){
      scope=Vars::Parent;
    }else if(scopeText=="GLOBAL"){
      scope=Vars::Global;
    }else{
      Error(*cur,e_BadScope);
    }
  }
  string name,value,def;
  if(paramMap.Retrieve("NAME",name)){
    if(!(paramMap.Retrieve("DEFAULT",def)&&stream.m_curVars.IsDefName(name))){
      if(paramMap.Retrieve("VALUE",value)){
        if(!stream.m_curVars.Set(name,value,scope)){
          Error(*cur,e_BadVarName);
        }
      }else if(paramMap.Retrieve("EXPR",value)){
        value=evaluate(value);
        if(!stream.m_curVars.Set(name,value,scope)){
          Error(*cur,e_BadVarName);
        }
      }else if(paramMap.Retrieve("SRC",value)){
        string text;
        value=stream.m_ThisPath+value;
        if(!FileToString(value,text)) Error(*cur,e_FileNotOpen,value);
        if(!stream.m_curVars.Set(name,text,scope)){
          Error(*cur,e_BadVarName);
        }
      }else{
        if(!stream.m_curVars.Set(name,"",scope)){
          Error(*cur,e_BadVarName);
        }
      }
    }
  }else{
    Error(*cur,e_NoName);
  }
  HTMLStream::iterator Next=cur;
  ++Next;
  stream.m_stream.erase(cur);
  return Next;
}

HTMLStream::iterator HitopCommands::UNSET(HTMLStream& stream,
  HTMLStream::iterator cur,const ParamMap& paramMap,const string& tag){
  Vars::Scope scope=Vars::Local;
  string scopeText;
  if(paramMap.Retrieve("SCOPE",scopeText)){
    transform(scopeText.begin(),scopeText.end(),scopeText.begin(),ToUpper());
    if(scopeText=="LOCAL"){
      scope=Vars::Local;
    }else if(scopeText=="PARENT"){
      scope=Vars::Parent;
    }else if(scopeText=="GLOBAL"){
      scope=Vars::Global;
    }else{
      Error(*cur,e_BadScope);
    }
  }
  string name;
  if(paramMap.Retrieve("NAME",name)){
    if(!stream.m_curVars.Unset(name,scope)){
      Error(*cur,e_BadVarName);
    }
  }else{
    Error(*cur,e_NoName);
  }
  HTMLStream::iterator Next=cur;
  ++Next;
  stream.m_stream.erase(cur);
  return Next;
}

HTMLStream::iterator HitopCommands::GET(HTMLStream& stream,
  HTMLStream::iterator cur,const ParamMap& paramMap,const string& tag){
  string var,value,format;
  if(paramMap.Retrieve("EXPR",value)){
    value=evaluate(value);
  }else{
    if(!paramMap.Retrieve("NAME",var)) Error(*cur,e_NoName);
    if(paramMap.Retrieve("FORMAT",format)) var+=":"+format;
    if(!stream.m_curVars.Get(*cur,var,value)) Error(*cur,e_UndefVar,var);
  }
  cur->ImposeText(value);
  ++cur;
  return cur;
}

HTMLStream::iterator HitopCommands::FILE(HTMLStream& stream,
  HTMLStream::iterator cur,const ParamMap& paramMap,const string& tag){
  HTMLStream::iterator next=cur;
  ++next;
  string filename,formatName;
  if(!paramMap.Retrieve("SRC",filename)) Error(*cur,e_NoSrc);
  enum FileFormat{Hitop,Verbatim,Safe};
  FileFormat fileFormat=Hitop;
  if(paramMap.Retrieve("FORMAT",formatName)){
    if(!g_isXML) transform(formatName.begin(),formatName.end(),formatName.begin(),ToUpper());
    if(formatName=="HITOP"){
      fileFormat=Hitop;
    }else if(formatName=="VERBATIM"){
      fileFormat=Verbatim;
    }else if(formatName=="SAFE"){
      fileFormat=Safe;
    }else{
      Error(*cur,e_BadFormat2);
    }
  }
  if(fileFormat==Hitop || fileFormat==Safe){
    HTMLStream insert(&stream);
    insert.m_debugLevel=stream.m_debugLevel;
    insert.m_FName=filename;
    ifstream inputFile;
    insert.m_ThisPath=PathPart(stream.m_ThisPath+filename);
    inputFile.open((stream.m_ThisPath+filename).c_str(),ios::in);
    if(inputFile.good()){
      inputFile>>insert;
      inputFile.close();
    }else{
      Error(*cur,e_FileNotOpen,stream.m_ThisPath+filename);
    }
    if(fileFormat==Hitop) insert.Process();
    stream.m_stream.splice(cur,insert.m_stream);
    stream.m_stream.erase(cur);
  }else{
    string text;
    filename=stream.m_ThisPath+filename;
    if(!FileToString(filename,text)) Error(*cur,e_FileNotOpen,filename);
    cur->ImposeText(text);
    cur->SetPreformat(true);
  }
  return next;
}

HTMLStream::iterator HitopCommands::UNDEF(HTMLStream& stream,
  HTMLStream::iterator cur,const ParamMap& paramMap,const string& tag){
  HTMLStream::iterator Next=cur;
  ++Next;
  string Name;
  if(paramMap.Retrieve("NAME",Name)){
    TokenMap::Token token=TokenMap::NameToTokenNum(Name);
    if(!DEFStore::IsValidName(Name)){
      Error(*cur,e_BadProcName);
    }
    if(!DEFStore::Exists(token)){
      Error(*cur,e_UndefProc);
    }
    DEFStore::Delete(token);
    stream.m_funcMap.erase(token); // TODO: funcmap isn't exported. write an access fn.
  }else{
    Error(*cur,e_NoName);
  }
  stream.m_stream.erase(cur);
  return Next;
}

HTMLStream::iterator HitopCommands::HITOP(HTMLStream& stream,
  HTMLStream::iterator cur,const ParamMap& paramMap,const string& tag){
  HTMLStream::iterator Next=cur;
  ++Next;
  paramMap.RetrieveNum("DEBUG",stream.m_debugLevel);
  paramMap.Retrieve("LIBS",PluginManager::s_loadpath);
  paramMap.Retrieve("CONTENTTYPE",g_contentType);
  stream.m_stream.erase(cur);
  return Next;
}

HTMLStream::iterator HitopCommands::DEBUG(HTMLStream& stream,
  HTMLStream::iterator cur,const ParamMap& paramMap,const string& tag){
  HTMLStream::iterator Next=cur;
  ++Next;
  string message;
  if((stream.m_debugLevel>=1)&&paramMap.Retrieve("MESSAGE",message)) Debug(*cur,message);
  stream.m_stream.erase(cur);
  return Next;
}

HTMLStream::iterator HitopCommands::DEF(HTMLStream& stream,
  HTMLStream::iterator cur,const ParamMap& paramMap,const string& tag){
  HTMLStream::iterator Next=cur->m_myPair;
  if(Next==NULL) Error(*cur,"@DEF resulted from variable expansion");
  ++Next;
  string Name;
  if(paramMap.Retrieve("NAME",Name)){
    TokenMap::Token token=TokenMap::NameToTokenNum(Name);
    if(!DEFStore::IsValidName(Name)){
      Error(*cur,e_BadProcName);
    }
    HTMLStream::iterator firstLine=cur;
    ++firstLine;
    DEFStore::Set(token,firstLine,cur->m_myPair);
    HTMLStream::HTMLStream::RegisterCommand(Name,&USER);
  }else{
    Error(*cur,e_NoName);
  }
  ++cur->m_myPair;
  stream.m_stream.erase(cur,cur->m_myPair);
  return Next;
}

HTMLStream::iterator HitopCommands::COMMENT(HTMLStream& stream,
  HTMLStream::iterator cur,const ParamMap& paramMap,const string& tag){
  HTMLStream::iterator Next=cur->m_myPair;
  if(Next==NULL) Error(*cur,"@COMMENT resulted from variable expansion");
  ++Next;
  stream.m_stream.erase(cur,Next);
  return Next;
}

HTMLStream::iterator HitopCommands::IMG(HTMLStream& stream,
  HTMLStream::iterator cur,const ParamMap& paramMap,const string& tag){
  string curPath,FName;
  stream.m_curVars.Get(*cur,"FILE",curPath);
  curPath=PathPart(curPath);
  if(paramMap.Retrieve("SRC",FName)){
    FName=unescape(FName);
    if((!paramMap.Defined("WIDTH"))&&(!paramMap.Defined("HEIGHT"))){
      if(!isAbsolute(FName)){
        int width,height;
        if((!isAbsolute(FName))&&(IMGStore::GetSize(curPath+FName,width,height))){
          if(g_isXML){
            cur->AppendToContent(" width="+IToS(width)+" height="+IToS(height));
          }else{
            cur->AppendToContent(" WIDTH="+IToS(width)+" HEIGHT="+IToS(height));
          }
        }else{
          Warn(*cur,"WARNING: Cannot open image file");
        }
      }
    }
  }
  ++cur;
  return cur;
}

HTMLStream::iterator HitopCommands::ForEngine(HTMLStream& stream,
  HTMLStream::iterator cur,const ParamMap& paramMap,TokenMap::Token
  entryToken,int step){
  int from=0,to=0;
  if(!paramMap.RetrieveNum("FROM",from)) Error(*cur,e_NoFrom);
  if(!paramMap.RetrieveNum("TO",to)) Error(*cur,e_NoTo);
  if(step==0) Error(*cur,e_StepIsZero);
  if(step<0) step=-step;
  ParamMap newParam(paramMap);
  if(from<to){
    for(int i=from;i<=to;i+=step){
      newParam["COUNT"]=IToS(i);
      ProcCallByNum(stream,cur,newParam,entryToken);
    }
  }else{
    for(int i=from;i>=to;i-=step){
      newParam["COUNT"]=IToS(i);
      ProcCallByNum(stream,cur,newParam,entryToken);
    }
  }
  HTMLStream::iterator next=cur;
  ++next;
  stream.m_stream.erase(cur);
  return next;
}

HTMLStream::iterator HitopCommands::FOR(HTMLStream& stream,
  HTMLStream::iterator cur,const ParamMap& paramMap,const string& tag){
  TokenMap::Token entryToken=0;
  int step=1;
  if(!paramMap.RetrieveToken("ENTRY",entryToken)) Error(*cur,e_NoEntry);
  if(!DEFStore::Exists(entryToken)) Error(*cur,e_UndefEntry);
  paramMap.RetrieveNum("STEP",step);
  return ForEngine(stream,cur,paramMap,entryToken,step);
}

HTMLStream::iterator HitopCommands::FOREACH(HTMLStream& stream,
  HTMLStream::iterator cur,const ParamMap& paramMap,const string& tag){
  TokenMap::Token entryToken=0;
  if(!paramMap.RetrieveToken("ENTRY",entryToken)) Error(*cur,e_NoEntry);
  if(!DEFStore::Exists(entryToken)) Error(*cur,e_UndefEntry);
  string src,data,prefix;
  bool hasPREFIX=paramMap.Retrieve("PREFIX",prefix);
  bool hasSRC=paramMap.Retrieve("SRC",src);
  bool hasCONTENT=paramMap.Retrieve("CONTENT",data);
  bool hasDATA=paramMap.Retrieve("DATA",data);
  if(hasPREFIX){
    map<string,string> results;
    stream.m_curVars.Prefix(results,prefix);
    ParamMap newParam(paramMap);
    for(map<string,string>::const_iterator i=results.begin();i!=results.end();++i){
      newParam["NAME"]=i->first;
      newParam["VALUE"]=i->second;
      ProcCallByNum(stream,cur,newParam,entryToken);
    }
  }else if(hasSRC){
    ifstream inFile;
    inFile.open(src.c_str(),ios::in);
    if(!inFile.good()) Error(*cur,e_FileNotOpen,src);
    int count=1;
    ParamMap newParam(paramMap);
    string line;
    while(!inFile.eof()){
      getline(inFile,line);
      if(!inFile.eof()){
        newParam["CONTENT"]=EscapeSignificant(line);
        newParam["COUNT"]=IToS(count);
        ProcCallByNum(stream,cur,newParam,entryToken);
        ++count;
      }
    }
    inFile.close();
  }else if(hasCONTENT||hasDATA){
    if(hasCONTENT) Warn(*cur,"CONTENT is deprecated in favour of DATA");
    string sep;
    paramMap.Retrieve("SEP",sep);
    int from=1,to=65535;
    paramMap.RetrieveNum("FROM",from);
    paramMap.RetrieveNum("TO",to);
    data+=sep;
    ParamMap newParam(paramMap);
    string current;
    int sepLength=sep.length();
    if(sepLength==0){
      if(from<1) from=1;
      if(to>(int)data.length()) to=data.length();
      for(int count=from;count<=to;++count){
        newParam["CONTENT"]=data[count-1];
        newParam["COUNT"]=IToS(count);
        ProcCallByNum(stream,cur,newParam,entryToken);
      }
    }else{
      for(int count=1;(count<=to)&&(!data.empty());++count){
        int pos=data.find(sep);
        current=data.substr(0,pos);
        data=data.substr(pos+sepLength,string::npos);
        if(count>=from){
          newParam["CONTENT"]=current;
          newParam["COUNT"]=IToS(count);
          ProcCallByNum(stream,cur,newParam,entryToken);
        }
      }
    }
  }else{
    return ForEngine(stream,cur,paramMap,entryToken,1);
  }
  HTMLStream::iterator next=cur;
  ++next;
  stream.m_stream.erase(cur);
  return next;
}

extern multimap<string,string> g_httpHeader;
extern string g_statusHeader;

HTMLStream::iterator HitopCommands::RAWHEADER(HTMLStream& stream,
  HTMLStream::iterator cur,const ParamMap& paramMap,const string& tag){
  HTMLStream::iterator next=cur;
  ++next;
  string name,value;
  if(!paramMap.Retrieve("NAME",name)) Error(*cur,e_NoName);
  if(!paramMap.Retrieve("VALUE",value)) Error(*cur,e_NoValue);
  string temp;
  if(!g_isXML) transform(name.begin(),name.end(),temp.begin(),ToUpper());
  if(temp=="STATUS"){
    g_statusHeader=value;
  }else{
    g_httpHeader.insert(make_pair(name,value));
  }
  stream.m_stream.erase(cur);
  return next;
}

HTMLStream::iterator HitopCommands::COOKIE(HTMLStream& stream,
  HTMLStream::iterator cur,const ParamMap& paramMap,const string& tag){
  HTMLStream::iterator next=cur;
  ++next;
  string name,value,path,domain,expires;
  if(!paramMap.Retrieve("NAME",name)) Error(*cur,e_NoName);
  if(!paramMap.Retrieve("VALUE",value)) Error(*cur,e_NoValue);
  paramMap.Retrieve("PATH",path);
  paramMap.Retrieve("DOMAIN",domain);
  paramMap.Retrieve("EXPIRES",expires);
  if (expires.empty()) {
    string maxage;
    paramMap.Retrieve("MAXAGE",maxage);
    if (!maxage.empty()) {
      time_t ttl = time(NULL)+atoi(maxage.c_str());
      struct tm *expiry= gmtime(&ttl);
      char buffer[80];
      strftime(buffer,80,"%A, %d-%b-%Y %H:%M:%S GMT",expiry);    
      expires=buffer;
    }
  }
  if (value.end()!=find(value.begin(),value.end(),' ')) {
    value="\""+value+"\"";
  }
  string cookie=name+"="+value;
  if (!path.empty()) cookie+=string("; Path=")+path;
  if (!domain.empty()) cookie+=string("; Domain=")+domain;
  if (!expires.empty()) cookie+=string("; Expires=")+expires;
  if (paramMap.Retrieve("SECURE",path)) cookie+="; Secure";
  g_httpHeader.insert(make_pair(string("Set-Cookie"),cookie));
  stream.m_stream.erase(cur);
  return next;
}

void HitopCommands::ProcCallByNum(HTMLStream& stream,
  HTMLStream::iterator cur,const ParamMap& paramMap,TokenMap::Token token){
  HTMLStream* call=DEFStore::Find(token);
  HTMLStream doThis(&stream);
  doThis.m_debugLevel=stream.m_debugLevel;
  doThis=*call;
  doThis.m_ThisPath=stream.m_ThisPath;
  doThis.Recurse=stream.Recurse+1;
  doThis.SomeVars(*cur,paramMap);
  doThis.Process();
  stream.m_stream.splice(cur,doThis.m_stream);
  //stream.m_stream.insert(cur,doThis.m_stream.begin(),doThis.m_stream.end());
}

void HitopCommands::UserEngine(HTMLStream& stream,
  HTMLStream::iterator cur,const ParamMap& paramMap,const string& tag){
  string tempTag(tag);
  TokenMap::Token token=TokenMap::NameToTokenNum(tempTag);
  ProcCallByNum(stream,cur,paramMap,token);
}

void HitopCommands::ProcCallInline(HTMLStream& stream,
  HTMLStream::iterator cur,const ParamMap& paramMap,const string& tag){
  ProcCallByNum(stream,cur,paramMap,cur->Token());
}

HTMLStream::iterator HitopCommands::USER(HTMLStream& stream,
  HTMLStream::iterator cur,const ParamMap& paramMap,const string& tag){
  HTMLStream::iterator Next=cur;
  ++Next;
  ProcCallInline(stream,cur,paramMap,tag);
  stream.m_stream.erase(cur);
  return Next;
}

HTMLStream::iterator HitopCommands::NAV(HTMLStream& stream,
  HTMLStream::iterator cur,const ParamMap& paramMap,const string& tag){
  HTMLStream::iterator Next=cur;
  ++Next;

  string FormatStr;
  if (paramMap.Retrieve("FORMAT",FormatStr)&&!g_isXML) transform(FormatStr.begin(),FormatStr.end(),FormatStr.begin(),ToUpper());
  NavStore::OutputFormat format =NavStore::of_open;

  if (FormatStr == "FLAT") {
    format=NavStore::of_flat;
  } else if (FormatStr == "OPEN") {
    format=NavStore::of_open;
  } else if ((FormatStr.empty())||(FormatStr == "FOLDED")) {
    format=NavStore::of_folded;
  } else {
    Error(*cur,"FORMAT must be one of FLAT,OPEN or FOLDED");
  }

  string pre,post,entry;
  TokenMap::Token preToken=-1,postToken=-1,entryToken=-1;
  if (paramMap.Retrieve("PREGROUP",pre)) {
    preToken=TokenMap::NameToTokenNum(pre);
    if (!DEFStore::Exists(preToken)) Error(*cur,"PREGROUP Function doesn't exist");
  }
  if (paramMap.Retrieve("ENTRY",entry)) {
    entryToken=TokenMap::NameToTokenNum(entry);
    if (!DEFStore::Exists(entryToken)) Error(*cur,"ENTRY Function doesn't exist");
  }
  if (paramMap.Retrieve("POSTGROUP",post)) {
    postToken=TokenMap::NameToTokenNum(post);
    if (!DEFStore::Exists(postToken)) Error(*cur,"POSTGROUP Function doesn't exist");
  }

  int from=0,to=-1;
  paramMap.RetrieveNum("TO",to);
  paramMap.RetrieveNum("FROM",from);

  string curPath=PathPart(stream.m_ThisPath),navsource,curFile;
  if (navsource[0]!='/') navsource = curPath+navsource;

  Nav navbar;

  stream.m_curVars.Get(*cur,"FILE",curFile);
  if (paramMap.Retrieve("SRC",navsource)){
    navbar.Load(stream,cur,navsource,curFile,format);
  }else if (paramMap.Retrieve("DATA",navsource)){
    navbar.Set(stream,cur,navsource,curFile,format);
  }else{
    Error(*cur,e_NoDataSrc);
  }

  navbar.Execute(stream,cur,paramMap,format,preToken,entryToken,postToken,from,to);
  stream.m_stream.erase(cur);
  return Next;
}

HTMLStream::iterator HitopCommands::REQUIRES(HTMLStream& stream,HTMLStream::iterator cur,
  const ParamMap& paramMap,const string& tag) {
  string ns,src;
  if (paramMap.Retrieve("NAMESPACE",ns)&&!g_isXML) transform(ns.begin(),ns.end(),ns.begin(),ToUpper());
  if (!paramMap.Retrieve("PLUGIN",src)) Error(*cur,"PLUGIN must be defined for a plugin to be loaded");
  int minversion=0;
  paramMap.RetrieveNum("VERSION",minversion);
  PluginManager::Load(cur,src,ns,minversion);
  HTMLStream::iterator Next=cur;
  ++Next;
  stream.m_stream.erase(cur);
  return Next;
}

void HitopCommands::Init(){
  if(g_isHTML){
    if(g_isXML){
      HTMLStream::RegisterCommand("img",&IMG);
    }else{
      HTMLStream::RegisterCommand("IMG",&IMG);
    }
  }
  if(!g_isXML){
    HTMLStream::RegisterCommand("DEF",&DEF);
    HTMLStream::RegisterCommand("/DEF",&IMPOSSIBLE);
    HTMLStream::RegisterCommand("ELSE",&IMPOSSIBLE);
    HTMLStream::RegisterCommand("FILE",&FILE);
    HTMLStream::RegisterCommand("FOREACH",&FOREACH);
    HTMLStream::RegisterCommand("GET",&GET);
    HTMLStream::RegisterCommand("IF",&IF);
    HTMLStream::RegisterCommand("/IF",&IMPOSSIBLE);
    HTMLStream::RegisterCommand("NAV",&NAV);
    HTMLStream::RegisterCommand("SET",&SET);
  }
  HTMLStream::RegisterCommand("@COMMENT",&COMMENT);
  HTMLStream::RegisterCommand("/@COMMENT",&IMPOSSIBLE);
  HTMLStream::RegisterCommand("@DEBUG",&DEBUG);
  HTMLStream::RegisterCommand("@DEF",&DEF);
  HTMLStream::RegisterCommand("/@DEF",&IMPOSSIBLE);
  HTMLStream::RegisterCommand("@ELSE",&IMPOSSIBLE);
  HTMLStream::RegisterCommand("@ELSEIF",&IF);
  HTMLStream::RegisterCommand("@FILE",&FILE);
  HTMLStream::RegisterCommand("@FOR",&FOR);
  HTMLStream::RegisterCommand("@FOREACH",&FOREACH);
  HTMLStream::RegisterCommand("@GET",&GET);
  HTMLStream::RegisterCommand("@HITOP",&HITOP);
  HTMLStream::RegisterCommand("@IF",&IF);
  HTMLStream::RegisterCommand("/@IF",&IMPOSSIBLE);
  HTMLStream::RegisterCommand("@NAV",&NAV);
  HTMLStream::RegisterCommand("@RAWHEADER",&RAWHEADER);
  HTMLStream::RegisterCommand("@REQUIRES",&REQUIRES);
  HTMLStream::RegisterCommand("@SET",&SET);
  HTMLStream::RegisterCommand("@SETCOOKIE",&COOKIE);
  HTMLStream::RegisterCommand("@UNDEF",&UNDEF);
  HTMLStream::RegisterCommand("@UNSET",&UNSET);
  PluginManager::Init();
}
