/*
 *  JLib - Jacob's Library.
 *  Copyright (C) 2003, 2004  Juan Carlos Seijo Prez
 * 
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 * 
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 * 
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 *  Juan Carlos Seijo Prez
 *  jacob@mainreactor.net
 */

/** Sistema de ficheros virtual.
 * @file    JFS.h
 * @author  Juan Carlos Seijo Prez
 * @date    23/12/2003
 * @version 0.0.1 - 23/12/2003 - Primera versin.
 */
#ifndef _JFS_INCLUDED
#define _JFS_INCLUDED

#include <JLib/Util/JTypes.h>
#include <list>
#include <JLib/Util/JObject.h>
#include <JLib/Util/JString.h>
#include <JLib/Util/JFile.h>
#include <JLib/Util/JLoadSave.h>
#include <JLib/Graphics/JImage.h>
#include <JLib/Graphics/JImageSprite.h>

// Formatos de recurso estndar de JLib

/** Identificador de recurso de bloque */
#define JRES_RESOURCEBLOCK 0xFFFFFFFF
/** Imagen bsica */
#define JRES_IMAGE         0x00000001
/** Sprite de imgenes */
#define JRES_IMAGESPRITE   0x00000010
/** Objeto de JScened */
#define JRES_JSCENEDOBJECT 0x00000100
/** Escena de JScened */
#define JRES_JSCENEDSCENE  0x00000200

/** Crea y devuelve un nuevo dato de recurso adecuado a partir del tipo pasado.
 * En el caso de un bloque de recursos, lo crea con nombre vaco y padre = 0.
 * @param type Identificador de recurso a crear.
 * @return Puntero al nuevo recurso  0 (cero) en caso de no existir un recurso de ese tipo.
 */
void * JFSNewFromType(u32 type);

/** Cabecera de recurso
 */
class JResourceHeader : public JLoadSave
{
public:
    u32 dataOffset;         /**< Offset de los datos de este recurso desde el comienzo del fichero */
    u32 nextOffset;         /**< Offset del siguiente recurso desde el comienzo del fichero */
    u32 type;               /**< Tipo de recurso */
    JString name;           /**< Nombre del recurso */

    /** Crea una nueva cabecera de recurso.
     * @param  _name Nombre del recurso
     * @param  _type Tipo del recurso
     * @param  _dataOffset Offset de los datos de este recurso desde el comienzo del fichero.
     * @param  _nextOffset Offset del siguiente recurso, desde el comienzo del fichero.
     */
    JResourceHeader(const JString &_name = "", 
                    u32 _type = 0, 
                    u32 _dataOffset = 0, 
                    u32 _nextOffset = 0) 
      : name(_name), type(_type), dataOffset(_dataOffset), nextOffset(_nextOffset)
    {}

    /** Carga la cabecera.
     * @param  f Fichero abierto y correctamente posicionado desde el que cargar los datos.
     * @return 0 si todo va bien, 1 en caso de error de E/S de fichero, 2 en caso de error
     * de integridad de datos.
     */
    virtual u32 Load(JFile &f);

    /** Salva la cebecera.
     * @param  f Fichero abierto y correctamente posicionado desde el que cargar los datos.
     * @return 0 si todo va bien, 1 en caso de error de E/S de fichero, 2 en caso de error
     * de integridad de datos.
     */
    virtual u32 Save(JFile &f);
};

/** Recurso bsico.
 */
class JResource : public JLoadSave
{
private:
  static u32 idCount;       /**< Contador de instancias pedidas de este recurso */
  u32 id;                   /**< Identificador de recurso nico */

protected:
  JResource *parent;                  /**< Recurso contenedor */
  JResourceHeader header;             /**< Cabecera del recurso */
  
  JLoadSave *data;                    /**< Datos especficos del recurso */
  bool loaded;                        /**< Indica si el recurso ya se ha cargado */

  /** Comienza la grabacin del recurso. Escribe la cabecera.
   * @param  f Fichero abierto y correctamente posicionado desde el que cargar los datos.
   * @return 0 si todo va bien, 1 en caso de error de E/S de fichero, 2 en caso de error
   * de integridad de datos.
   */
  u32 BeginSave(JFile &f);

  /** Finaliza la grabacin del recurso. Actualiza los offsets.
   * @param  f Fichero abierto y correctamente posicionado desde el que cargar los datos.
   * @return 0 si todo va bien, 1 en caso de error de E/S de fichero, 2 en caso de error
   * de integridad de datos.
   */
  u32 EndSave(JFile &f);

  /** Comienza la carga del recurso. Carga la cabecera.
   * @param  f Fichero abierto y correctamente posicionado desde el que cargar los datos.
   * @return 0 si todo va bien, 1 en caso de error de E/S de fichero, 2 en caso de error
   * de integridad de datos.
   */
  u32 BeginLoad(JFile &f) {return header.Load(f);}

  /** Finaliza la carga del recurso. Marca el recurso como cargado.
   */
  void EndLoad() {loaded = true;}

public:
  /** Constructor para cargar.
   */
  JResource() : data(0), parent(0), loaded(false)
  {
  }

  /** Constructor para salvar
   * @param  name Nombre del recurso
   * @param  type Tipo de recurso.
   * @param  _data Datos asociados al recurso
   * @param  _parent Padre de este recurso. Debe ser un recurso de bloque.
   */
  JResource(const JString &name, s32 type, JLoadSave *_data = 0, JResource *_parent = 0) 
    : header(name, type, 0, 0), data(_data), parent(_parent), loaded(false)
  {
    id = idCount++;
  }

  /** Devuelve el offset absoluto del siguiente recurso en el fichero
   * @return Offset desde el comienzo del fichero para el siguiente recurso.
   */
  u32 NextOffset() {return header.nextOffset;}

  /** Devuelve el offset desde el comienzo del fichero de los datos de este recurso.
   * @return Posicin dentro del fichero desde la que se debe cargar el recurso con Load()
   */
  u32 DataOffset() {return header.dataOffset;}

  /** Devuelve el tipo de recurso.
   * @return Tipo de recurso tal y como aparece en la cabecera.
   */
  u32 Type() const {return header.type;}

  /** Devuelve el identificador de recurso
   * @return Identificador nico en tiempo de ejecucin del recurso.
   */
  u32 Id() const {return id;}

  /** Devuelve el nombre del recurso
   * @return Nombre de este recurso tal y como aparece en la cabecera.
   */
  const JString & Name() const {return header.name;}

  /** Establece el nombre del recurso
   * @param  name Nuevo nombre para este recurso.
   */
  void Name(const JString &name) {header.name = name;}

  /** Devuelve los datos del recurso.
   * @return Datos asociados a este recurso.
   */
  JLoadSave* Data() const {return data;}

  /** Determina si ya se carg el recurso
   * @return <b>true</b> si ya se carg, <b>false</b> en caso contrario.
   */
  bool Loaded() {return loaded;}

  /** Devuelve el recurso padre.
   * @return Recurso contenedor de este recurso.
   */
  JResource * Parent() {return parent;}

  /** Destruye el recurso, lo descarga y libera su memoria asociada.
   */
  virtual ~JResource() {}

  /** Carga slo la cabecera de este recurso. Para cargar los datos del recurso actual 
   * hay que usar Load(JFile, JLoadSave*)
   * @param  f Fichero abierto y correctamente posicionado desde el que cargar los datos.
   * @return 0 si todo va bien, 1 en caso de error de E/S de fichero, 2 en caso de error
   * de integridad de datos.
   */
  virtual u32 Load(JFile &f);

  /** Carga el recurso.
   * @param  f Fichero abierto y correctamente posicionado desde el que cargar los datos.
   * @param  where Puntero al objeto donde cargar el recurso.
   * @return 0 si todo va bien, 1 en caso de error de E/S de fichero, 2 en caso de error
   * de integridad de datos.
   */
  virtual u32 Load(JFile &f, JLoadSave *where);

  /** Salva el recurso.
   * @param  f Fichero abierto y correctamente posicionado desde el que cargar los datos.
   * @return 0 si todo va bien, 1 en caso de error de E/S de fichero, 2 en caso de error
   * de integridad de datos.
   */
  virtual u32 Save(JFile &f);
};

/** Iterador de lista de recursos 
  */
typedef std::list<JResource*>::iterator JResIterator;

/** Bloque de recursos.
 * Se puede entender un bloque de recursos como una carpeta del
 * sistema de ficheros que contiene archivos (los recursos que contiene).
 */
class JResourceBlock : public JResource
{
protected:
  JResourceBlock *parent;               /**< Bloque padre */
  std::list<JResource*> block;          /**< Lista de recursos */

public:
  /** Crea un nuevo recurso de bloque.
   * @param  _name Nombre del recurso.
   * @param  _parent Nombre del recurso contenedor.
   */
  JResourceBlock(const JString& _name, 
                 JResourceBlock *_parent = 0) 
	: JResource(_name, JRES_RESOURCEBLOCK), parent(_parent)
  {}

  /** Destruye el recurso y libera su memoria asociada.
   */
  virtual ~JResourceBlock();

  /** Devuelve la lista de recursos. En una implementacin futura se sustituir
   * ste mtodo por los correspondientes de acceso al bloque de recursos (aadir,
   * borrar, etc.).
   */
  std::list<JResource*> & List() {return block;}

  /** Carga el recurso.
   * @param  f Fichero abierto y correctamente posicionado desde el que cargar los datos.
   * @return 0 si todo va bien, 1 en caso de error de E/S de fichero, 2 en caso de error
   * de integridad de datos.
   */
  virtual u32 Load(JFile &f);

  /** Carga el recurso con el nombre dado.
   * Busca el recurso con el numbre dado dentro del fichero y lo carga.
   * @param  name Nombre del recurso a buscar.
   * @param  f Fichero abierto y correctamente posicionado desde el que cargar los datos.
   * @param  where Objeto donde cargar el recurso.
   * @return 0 si todo va bien, 1 en caso de error de E/S de fichero, 2 en caso 
   * de que no se encuentre el recurso con el nombre dado.
   */
  virtual u32 Load(const JString &name, JFile &f, JLoadSave *where);

  /** Carga el recurso dado.
   * @param  res Objeto recurso a cargar.
   * @param  f Fichero abierto y correctamente posicionado desde el que cargar los datos.
   * @param  where Objeto donde cargar el recurso.
   * @return 0 si todo va bien, 1 en caso de error de E/S de fichero, 2 en caso 
   * de que no se encuentre el recurso con el nombre dado.
   */
  virtual u32 Load(JResource *res, JFile &f, JLoadSave *where);

  /** Salva el bloque.
   * @param  f Fichero abierto y correctamente posicionado donde salvar los datos.
   * @return 0 si todo va bien, 1 en caso de error de E/S de fichero, 2 en caso de error
   * de integridad de datos.
   */
  virtual u32 Save(JFile &f);
};

/** Sistema de ficheros virtual.
 * Consiste en un archivo organizado de forma jerrquica en bloques.
 * Siempre existe al menos un bloque, el bloque raz del que 'cuelgan' el
 * resto de recursos en el archivo.
 */
class JFS
{
protected:
  JString name;                     /**< Nombre (nombre del fichero) */
  JFile f;                          /**< Fichero de recursos */
  JResourceBlock root;              /**< Bloque raz */
  JResourceBlock *cur;              /**< Bloque actual */
  JResIterator it;                  /**< Iterador */

  /** Carga el recurso dado.
    * @param res Puntero al recurso a cargar.
    * @return 0 si todo va bien, 1 en caso de error de E/S y 2 en caso de error
    * de integridad de datos.
    */
  u32 Load(JResource *res) {return res->Load(f);}

public:
  /** Abre un fichero de recursos o lo crea si no existiera.
   * @param  _name Nombre del archivo de recursos.
   */
  JFS(const JString &_name) : root("/"), cur(&root), name(_name)
  {}

  /** Carga el fichero de recursos
    * @return 0 si todo va bien, 1 en caso de error de E/S y 2 en caso de error
    * de integridad de datos.
    */
  u32 Load();

  /** Salva el fichero de recursos
    * @return 0 si todo va bien, 1 en caso de error de E/S y 2 en caso de error
    * de integridad de datos.
    */
  u32 Save();

  /** Devuelve el bloque actual.
    * @return Bloque de recursos en el que se encuentra el sistema de ficheros.
    */
  JResourceBlock * Current() {return cur;}

  /** Establece el bloque raz como bloque actual.
   */
  void CDRoot() {cur = &root;}

  /** Cambia al bloque con el nombre dado.
   * @param  _name Nombre del bloque a establecer como bloque actual.
   * @return <b>true</b> si se encontr <b>false</b> si no.
   */
  bool CD(JString &_name);

  /** Va al bloque padre
   * @return <b>true</b> si el bloque actual tiene bloque padre <b>false</b> si es el bloque raz.
   */
  bool CDParent() {if (cur->Parent()) {cur = (JResourceBlock *)cur->Parent(); return true;} return false;}

  /** Carga el recurso dado.
    * @param  _name Nombre del recurso a cargar.
    * @param  where Puntero al objeto donde cargar el recurso.
    * @return 0 si todo va bien, 1 en caso de error de E/S y 2 en caso de error
    * de integridad de datos.
    */
  u32 Load(JString &_name, JLoadSave *where);

  /** Devuelve una instancia del recurso con el nombre dado. Si no est cargado
   * lo carga.
   * @param  _name Nombre del recurso.
   * @return Recurso pedido  0 (cero) si no se encontr o hubo un error al cargar
   * el recurso.
   */
  JResource * Get(JString &_name);
  
  /** Borra el recurso dado.
   * @param  _name Nombre del recurso.
   * @return <b>true</b> si se encontr y se pudo borrar <b>false</b> si no se encontr o
   * no se pudo borrar.
   */
  bool Delete(JString &_name);

  /** Devuelve el nombre del fichero de recursos.
   * @return Nombre del fichero de recursos asociado a este sistema de ficheros.
   */
  const JString & Name() {return name;}
};

#endif  // _JFS_INCLUDED
