/* ptal-printd -- PTAL print daemon */

/* Copyright (C) 2000-2002 Hewlett-Packard Company
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 *
 * In addition, as a special exception, Hewlett-Packard Company
 * gives permission to link the code of this program with any
 * version of the OpenSSL library which is distributed under a
 * license identical to that listed in the included LICENSE.OpenSSL
 * file, and distribute linked combinations including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * this file, you may extend this exception to your version of the
 * file, but you are not obligated to do so.  If you do not wish to
 * do so, delete this exception statement from your version.
 */

/* Original author: David Paschal */

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <pwd.h>	/* TODO: getpwuid() */
#include <grp.h>	/* TODO: getgrgid() */
#include <errno.h>
#include <syslog.h>
#include <time.h>
#include <ptal.h>

#define PTAL_PRINTD_DEV_DIR VAR_RUN_PREFIX "/ptal-printd"

#define LEN_BUFFER 4096
static char uel1[]="\033%-12345X";
static char uel2[]="\033E\033%-12345X";

#define _ptalName (ptalName?ptalName:"???")

#define _PTAL_LOG_ERROR(args...) \
	do { \
		PTAL_LOG_ERROR(args); \
		syslog(LOG_LPR|LOG_ERR,args); \
	} while(0)

#define ASSERT(condition,string,recovery) \
	if (!(condition)) { \
		_PTAL_LOG_ERROR("%s(%s): %s failed, errno=%d!\n", \
			argv0,_ptalName,string,errno); \
		recovery; \
	}
#define ASSERTA(condition,string) ASSERT(condition,string,goto abort)

int main(int argc,char **argv) {
	char *argv0=0,*ptalName=0;
	int morePipes=0;
	char *pipeBaseName=0,*pipePath=PTAL_PRINTD_DEV_DIR;
	char colonReplacement='_';
	mode_t pipeMode=S_IRUSR|S_IWUSR|S_IWGRP|S_IWOTH;
	uid_t pipeOwner=getuid();
	gid_t pipeGroup=getgid();
	int nofork=0;
	int retryDelay=30;
	int timeout=-1;
	int sendUel=0;

	struct stat statbuf;
	ptalDevice_t dev;
	ptalChannel_t chan;
	int i,len,r,w,wsize,alreadyOpen,bitBucket,offset;
	char buffer[LEN_BUFFER+1],*oldPipeName;
	pid_t pid;
	int numPipes=0;
	struct PipeInfo {
		char *name;
		int fd;
	} *pipeList=0;
	int currentChannel=0,emptyJobCount=0;
	fd_set rset;
	int maxFdPlus1;

	argv0=*argv;

	/* Standard I/O file descriptors may be missing when invoked from
	 * a Linux USB hotplug script.  Let /dev/null take their place. */
	while (42) {
		int fd=open("/dev/null",O_RDWR);
		if (fd<0) break;
		if (fd>2) {	/* 2=standard error */
			close(fd);
			break;
		}
	}
	/* Set up syslog. */
	openlog("ptal-printd",LOG_NDELAY,LOG_LPR);

	ptalInit();

	while (42) {
		argc--; argv++; if (argc<=0) break;
		PTAL_LOG_DEBUG("%s(%s): Token=<%s>\n",argv0,_ptalName,*argv);

		if (**argv!='-') {
			if (ptalName) goto syntaxError;
			ptalName=*argv;

		} else if (!strcmp(*argv,"-morepipes")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			morePipes=atoi(*argv);
			if (morePipes<0) goto syntaxError;

		} else if (!strcmp(*argv,"-pipename")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			pipeBaseName=*argv;

		} else if (!strcmp(*argv,"-pipepath")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			pipePath=*argv;

		} else if (!strcmp(*argv,"-colon")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			colonReplacement=**argv;

		} else if (!strcmp(*argv,"-mode")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			pipeMode=strtol(*argv,0,0);

		} else if (!strcmp(*argv,"-owner")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			pipeOwner=atoi(*argv);

		} else if (!strcmp(*argv,"-group")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			pipeGroup=atoi(*argv);

		} else if (!strcmp(*argv,"-like")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			if (stat(*argv,&statbuf)<0) {
				_PTAL_LOG_ERROR("%s(%s): Unable to read "
					"file permissions from \"%s\"!\n",
					argv0,_ptalName,*argv);
				goto abort;
			}
			pipeMode=statbuf.st_mode;
			pipeOwner=statbuf.st_uid;
			pipeGroup=statbuf.st_gid;

		} else if (!strcmp(*argv,"-nofork")) {
			nofork=1;

		} else if (!strcmp(*argv,"-retrydelay")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			retryDelay=atoi(*argv);

		} else if (!strcmp(*argv,"-timeout")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			timeout=atoi(*argv);

		} else if (!strcmp(*argv,"-uel")) {
			sendUel=1;

		} else if (!strcmp(*argv,"-nouel")) {
			sendUel=0;

		} else {
syntaxError:
			PTAL_LOG_ERROR(
"Syntax: %s <devname> [<options>...]\n"
"Where <devname> may be one of:\n"
				,argv0);
			ptalDeviceEnumerate(0,
				ptalDeviceEnumeratePrintCallback,0);
			PTAL_LOG_ERROR(
"Supported <options>:\n"
"  -morepipes <n>        -- creates <n> additional pipes for this device\n"
"  -pipename <filename>  -- sets name of pipe (default is based on <devname>)\n"
"  -pipepath <path>      -- sets path of pipe if name contains no slashes\n"
"  -colon <c>            -- sets character to replace colon in pipename\n"
"  -mode 0<octal>        -- sets pipe permissions (use leading 0 for octal)\n"
"  -owner <uid>          -- sets pipe owner (numeric uid only)\n"
"  -group <gid>          -- sets pipe group (numeric gid only)\n"
"  -like </dev/lp0>      -- copies mode, owner and group from another device\n"
"  -nofork               -- runs in foreground (default is background daemon)\n"
"  -retrydelay <seconds> -- sets retry delay if device open failed\n"
"  -timeout <seconds>    -- sets idle timeout (default is no timeout)\n"
"  -uel, -nouel          -- enables/disables wrapping job with UEL escapes\n"
				);
			goto abort;
		}
	}
	PTAL_LOG_DEBUG("%s(%s): No more tokens.\n",argv0,_ptalName);

	if (!ptalName) goto syntaxError;
	/* Create default pipeBaseName suffix based on ptalName. */
	if (!pipeBaseName) {
		len=strlen(ptalName);
		pipeBaseName=malloc(len+1);
		ASSERTA(pipeBaseName,"malloc(pipeBaseName #1)");
		strcpy(pipeBaseName,ptalName);
		for (i=0;i<len;i++) {
			if (pipeBaseName[i]==':') {
				pipeBaseName[i]=colonReplacement;
			}
		}
	}
	PTAL_LOG_DEBUG("%s(%s): pipeBaseName=<%s>\n",
		argv0,ptalName,pipeBaseName);
	/* If pipeBaseName didn't include a path, then prepend default path. */
	if (!strchr(pipeBaseName,'/')) {
		oldPipeName=pipeBaseName;
		pipeBaseName=malloc(strlen(pipePath)+strlen(pipeBaseName)+2);
		ASSERTA(pipeBaseName,"malloc(pipeBaseName #2)");
		strcpy(pipeBaseName,pipePath);
		strcat(pipeBaseName,"/");
		strcat(pipeBaseName,oldPipeName);
		/* We should deallocate the old pipeBaseName if it was 
		 * previously malloced, as opposed to referenced from argv. */
	}

	pipeMode&=0777;
	PTAL_LOG_DEBUG("%s(%s): pipeBaseName=<%s>, "
		"timeout=%d, mode=0%3.3o, owner=%d, group=%d.\n",
		argv0,ptalName,pipeBaseName,timeout,
		pipeMode,pipeOwner,pipeGroup);

	dev=ptalDeviceOpen(ptalName);
	ASSERTA(dev,"ptalDeviceOpen");
	chan=ptalChannelAllocate(dev);
	ASSERTA(chan,"ptalChannelAllocate");
	ptalChannelSetRemoteService(chan,PTAL_STYPE_PRINT,0,0);

	numPipes=1+morePipes;
	pipeList=malloc(numPipes*sizeof(struct PipeInfo));
	ASSERTA(pipeList,"malloc(pipeList)");
	for (i=0;i<numPipes;i++) {
		if (!i) {
			pipeList[i].name=pipeBaseName;
		} else {
			pipeList[i].name=malloc(strlen(pipeBaseName)+100);
			ASSERTA(pipeList[i].name,"malloc(pipeList.name)");
			sprintf(pipeList[i].name,"%s%c%c%d",pipeBaseName,
				colonReplacement,colonReplacement,i);
		}
		unlink(pipeList[i].name);
		ASSERTA(mkfifo(pipeList[i].name,pipeMode)>=0,"mkfifo");
		ASSERTA(chmod(pipeList[i].name,pipeMode)>=0,"chmod");
		ASSERTA(chown(pipeList[i].name,pipeOwner,pipeGroup)>=0,"chown");
		pipeList[i].fd=open(pipeList[i].name,O_RDONLY|O_NONBLOCK);
		ASSERTA(pipeList[i].fd>=0,"open #1");
	}

	if (!nofork) {
		pid=fork();
		ASSERTA(pid>=0,"fork");
		if (pid) {
			return 0;
		}
	}

	syslog(LOG_LPR|LOG_NOTICE,
		"%s(%s) successfully initialized using %s*.\n",
		argv0,ptalName,pipeBaseName);

 while (42) {
	FD_ZERO(&rset);
	maxFdPlus1=0;
	for (i=0;i<numPipes;i++) {
		if (pipeList[i].fd>=0) {
			FD_SET(pipeList[i].fd,&rset);
			if (pipeList[i].fd>=maxFdPlus1) {
				maxFdPlus1=pipeList[i].fd+1;
			}
		}
	}
	r=select(maxFdPlus1,&rset,0,0,0);
	ASSERTA(r>=0,"select");

   if (pipeList[currentChannel].fd>=0 &&
       FD_ISSET(pipeList[currentChannel].fd,&rset)) {

	PTAL_LOG_DEBUG("\n\n%s(%s): Taking input from channel %d.\n",
		argv0,ptalName,currentChannel);

	alreadyOpen=0;
	bitBucket=0;
	offset=0;
	wsize=LEN_BUFFER;
     while (42) {
	struct timeval tvTimeout={timeout,0},*pTimeout=&tvTimeout;
	FD_ZERO(&rset);
	FD_SET(pipeList[currentChannel].fd,&rset);
	if (timeout<0) {
		pTimeout=0;
	}

	if (select(pipeList[currentChannel].fd+1,&rset,0,0,
	     pTimeout)<=0) {
		PTAL_LOG_DEBUG("%s(%s,%d): Timed out at offset=%d, "
			"assuming end of job.\n",
			argv0,ptalName,currentChannel,offset);
		break;
	}

	r=read(pipeList[currentChannel].fd,buffer,wsize);
	PTAL_LOG_DEBUG("%s(%s,%d): read() returns %d at offset=%d.\n",
		argv0,ptalName,currentChannel,r,offset);
	if (r<=0) {
		if (!offset) {
			/* Needed because select() on a FIFO never seems
			 * to block on FreeBSD: */
			emptyJobCount=(emptyJobCount+1)%numPipes;
			if (!emptyJobCount) {
				PTAL_LOG_DEBUG("%s(%s): too many empty "
					"jobs make me sleepy.\n",
					argv0,ptalName);
				sleep(2);
			}
		}
		break;
	}

       if (!alreadyOpen) {
	while (42) {
		time_t t1=time(0),t2;
		int d;
		if (ptalChannelOpen(chan)!=PTAL_ERROR) break;
		t2=time(0);
		d=retryDelay-(time(0)-t1);
		if (d<=0) d=1;
		_PTAL_LOG_ERROR("%s(%s,%d): Unable to open print "
			"channel!  Will delay %d second(s) and retry.\n",
			argv0,ptalName,currentChannel,d);
		sleep(d);
	}

	ptalChannelAdjustPacketSizes(chan,&wsize,0);

	if (sendUel) {
		len=strlen(uel1);
		w=ptalChannelWrite(chan,uel1,len);
		if (w!=len) {
			_PTAL_LOG_ERROR("%s(%s,%d): Error sending job header "
				"string (wrote %d bytes instead of %d)!  "
				"Discarding rest of print job.\n",
				argv0,ptalName,currentChannel,w,len);
			bitBucket=1;
		}
	}

	PTAL_LOG_DEBUG("\n%s(%s,%d): Beginning of job.\n\n",
		argv0,ptalName,currentChannel);
	alreadyOpen=1;
       }

	if (!bitBucket) {
		w=ptalChannelWrite(chan,buffer,r);
		if (w!=r) {
			_PTAL_LOG_ERROR("%s(%s,%d): Error sending print data "
				"(wrote %d bytes instead of %d at offset=%d)!  "
				"Discarding rest of print job.\n",
				argv0,ptalName,currentChannel,w,r,offset);
			bitBucket=1;
		}
	}

	offset+=r;
     }
	PTAL_LOG_DEBUG("\n\n%s(%s,%d): End of job (%d bytes).\n\n",
		argv0,ptalName,currentChannel,offset);

     if (alreadyOpen) {
	if (sendUel && !bitBucket) {
		len=strlen(uel2);
		w=ptalChannelWrite(chan,uel2,len);
		if (w!=len) {
			_PTAL_LOG_ERROR("%s(%s,%d): Error sending job trailer "
				"string (wrote %d bytes instead of %d)!\n",
				argv0,ptalName,currentChannel,w,len);
		}
	}

	ptalChannelClose(chan);
     }

	close(pipeList[currentChannel].fd);
	pipeList[currentChannel].fd=
		open(pipeList[currentChannel].name,O_RDONLY|O_NONBLOCK);
	ASSERTA(pipeList[currentChannel].fd>=0,"open #2");
   }
	currentChannel=(currentChannel+1)%numPipes;
 }

abort:
	ptalDone();
	if (pipeList) for (i=0;i<numPipes;i++) {
		unlink(pipeList[i].name);
	}
	return 1;
}
