/* ptal -- Peripheral Transport Abstraction Library */

/* Copyright (C) 2000-2003 Hewlett-Packard Company
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 *
 * In addition, as a special exception, Hewlett-Packard Company
 * gives permission to link the code of this program with any
 * version of the OpenSSL library which is distributed under a
 * license identical to that listed in the included LICENSE.OpenSSL
 * file, and distribute linked combinations including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * this file, you may extend this exception to your version of the
 * file, but you are not obligated to do so.  If you do not wish to
 * do so, delete this exception statement from your version.
 */

/* Original author: David Paschal */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdarg.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <dirent.h>
#include <signal.h>
#include "ptal-internal.h"

#define PTAL_FLUSH_LEN_BUFFER		4096
#define PTAL_FLUSH_TIMEOUT_SEC		4
#define PTAL_FLUSH_TIMEOUT_USEC		0

const char *ptalEtcPrefix="/etc/ptal";

static int ptalDebugFlag=0;
static ptalFile_t ptalGlobalConfigFile=0;
static ptalDevice_t ptalFirstDevice=0;
static ptalDevice_t ptalLastDevice=0;

int ptalLogMsg(int level,char *format,...) {
	int r;
	va_list ap;

	if (level>ptalDebugFlag || !format) return PTAL_OK;

	va_start(ap,format);
	r=vfprintf(stderr,format,ap);
	va_end(ap);
	return r;
}

void ptalDeviceDump(ptalDevice_t dev,int level) {
	ptalChannel_t chan=dev->firstChannel;

	fprintf(stderr,"dev=0x%8.8X, prev=0x%8.8X, next=0x%8.8X\n",
		(int)dev,(int)dev->prev,(int)dev->next);
	fprintf(stderr,"provider=0x%8.8X <%s>\n",
		(int)dev->provider,dev->provider->name);
	fprintf(stderr,"devname=<%s>\n",
		dev->devname);
	fprintf(stderr,"lenPrefix=%d\n",
		dev->lenPrefix);
	fprintf(stderr,"appInfo=0x%8.8X\n",
		(int)dev->appInfo);
	fprintf(stderr,"firstChannel=0x%8.8X\n",
		(int)dev->firstChannel);
	fprintf(stderr,"lastChannel=0x%8.8X\n",
		(int)dev->lastChannel);
	fprintf(stderr,"pmlChannel=0x%8.8X\n",
		(int)dev->pmlChannel);
	fprintf(stderr,"firstPmlObject=0x%8.8X\n",
		(int)dev->firstPmlObject);
	fprintf(stderr,"lastPmlObject=0x%8.8X\n",
		(int)dev->lastPmlObject);
	if (dev->provider->deviceDump)
		dev->provider->deviceDump(dev,level);
	fprintf(stderr,"\n");

	while (chan) {
		ptalChannelDump(chan,level);
		chan=chan->next;
	}
}

void ptalDump(int level) {
	ptalDevice_t dev=ptalFirstDevice;

	fprintf(stderr,"ptalDebugFlag=%d\n",ptalDebugFlag);
	fprintf(stderr,"ptalFirstDevice=0x%8.8X\n",(int)ptalFirstDevice);

	while (dev) {
		ptalDeviceDump(dev,level);
		dev=dev->next;
	}

	fprintf(stderr,"ptalLastDevice=0x%8.8X\n",(int)ptalLastDevice);
}

char *ptalFilenameMalloc(const char *prefix,const char *suffix) {
	int len1,len2,size;
	char *filename;

	if (!prefix) prefix=ptalEtcPrefix;
	len1=strlen(prefix);
	len2=strlen(suffix);
	size=len1+1+len2+1;
	filename=malloc(size);
	if (!filename) {
		PTAL_LOG_WARN("ptalFileNameMalloc(%s/%s): malloc(%d) failed!\n",
			prefix,suffix,size);
	} else {
		memcpy(filename+0,prefix,len1);
		filename[len1]='/';
		memcpy(filename+len1+1,suffix,len2);
		filename[len1+1+len2]=0;
	}

	return filename;
}

void ptalFilenameFree(char *filename) {
	free(filename);
}

void ptalFileFree(ptalFile_t file) {
	free(file);
}

ptalFile_t ptalFileRead(const char *prefix,const char *suffix) {
	char *filename;
	int fd,len;
	ptalFile_t file=0;

	filename=ptalFilenameMalloc(prefix,suffix);
	if (filename) {
		fd=open(filename,O_RDONLY);
		if (fd>=0) {
			len=lseek(fd,0,SEEK_END);
			if (len>0) {
				/* VERY important to leave 1 extra byte: */
				int size=sizeof(struct ptalFile_s)+len;
				lseek(fd,0,SEEK_SET);
				file=malloc(size);
				if (!file) {
					PTAL_LOG_WARN("ptalFileRead(%s/%s): "
						"malloc(%d) failed!\n",
						prefix,suffix,size);
				} else {
					int r=read(fd,file->data,len);
					if (r<=0) {
						ptalFileFree(file);
						file=0;
					} else {
						file->len=r;
					}
				}
			}
			close(fd);
		}
		ptalFilenameFree(filename);
	}

	return file;
}

ptalDevice_t ptalDeviceAdd(ptalProvider_t provider,
    char *devname,int lenPrefix,void *cbd) {
	ptalDevice_t dev=0;
	int size;

	/* Sanity-check sizeofDevice. */
	size=provider->sizeofDevice;
	if (size<sizeof(struct ptalDevice_s)) {
		PTAL_LOG_WARN("ptalDeviceAdd(%s): invalid sizeofDevice=%d!\n",
			devname,size);
		goto abort;
	}

	/* Malloc and zero device. */
	dev=malloc(size);
	if (!dev) {
		PTAL_LOG_WARN("ptalDeviceAdd(%s): malloc(dev=%d) failed!\n",
			devname,size);
		goto abort;
	}
	memset(dev,0,size);

	/* Initialize fields. */
	dev->provider=provider;
	dev->lenPrefix=lenPrefix;
	dev->configFile=ptalFileRead(0,devname);

	/* Malloc and copy devname. */
	size=strlen(devname);
	dev->devname=malloc(size+1);
	if (!dev->devname) {
		PTAL_LOG_WARN("ptalDeviceAdd(%s): malloc(devname=%d) failed!\n",
			devname,size+1);
		goto abort;
	}
	memcpy(dev->devname,devname,size);
	dev->devname[size]=0;

	/* Call provider's constructor if defined. */
	if (provider->deviceConstructor) {
		provider->deviceConstructor(dev,cbd);
	}

	/* Append to linked list of devices. */
	if (!ptalFirstDevice) ptalFirstDevice=dev;
	dev->prev=ptalLastDevice;
	dev->next=0;
	if (ptalLastDevice) ptalLastDevice->next=dev;
	ptalLastDevice=dev;

	PTAL_LOG_DEBUG("ptalDeviceAdd(%s): dev=0x%8.8X.\n",devname,dev);
	return dev;

abort:
	if (dev) {
		if (dev->devname) free(dev->devname);
		if (dev->configFile) free(dev->configFile);
		free(dev);
	}
	return 0;
}

int ptalProviderEnumerate(ptalProviderEnumerate_f callback,void *cbd) {
	ptalProvider_t *pp=ptalProviders;
	int r=0;

	while (*pp) {
		if (callback) {
			r+=callback((*pp)->name,cbd);
		}
		pp++;
	}

	return r;
}

ptalDevice_t ptalDeviceOpen(char *name) {
	int lenPrefix=0;
	char c;
	ptalProvider_t *pp=ptalProviders,provider;

	if (!name || !*name) {
		return ptalFirstDevice;
	}

	while ((c=name[lenPrefix])!=':') {
		if (!c) {
			PTAL_LOG_WARN("ptalDeviceOpen(%s): missing colon!\n",
				name);
			return 0;
		}
		lenPrefix++;
	}

	while (42) {
		provider=*pp;
		if (!provider) {
			PTAL_LOG_WARN("ptalProviderLookup(name=<%s>): "
				"provider not found!\n",name);
			return 0;
		}
		if (strlen(provider->name)==lenPrefix &&
		    !memcmp(provider->name,name,lenPrefix)) {
			break;
		}
		pp++;
	}

	while (name[lenPrefix]==':') lenPrefix++;

	if (!provider->deviceOpen) {
		PTAL_LOG_WARN("ptalDeviceOpen: no deviceOpen method for "
			"provider=<%s>!\n",provider->name);
		return 0;
	}

	return provider->deviceOpen(name,lenPrefix);
}

void ptalDeviceReadDefaultDeviceFile(const char *prefix,const char *suffix) {
	ptalFile_t file=ptalFileRead(prefix,suffix);
	if (file) {
		int i=0;
		while (i<file->len) {
			char c=file->data[i];
			if (c>' ' && c<127) {
				int j=i;
				while (j<file->len) {
					c=file->data[j];
					if (c<=' ' || c>=127) {
						break;
					}
					j++;
				}
				file->data[j]=0;
				ptalDeviceOpen(file->data+i);
				break;
			}
			i++;
		}
		free(file);
	}
}

int ptalDeviceProbe(void) {
	char *devname=getenv("PTAL_DEFAULT_DEVICE");
	char *homedir=getenv("HOME");
	DIR *dir;

	if (devname) {
		PTAL_LOG_DEBUG("ptalDeviceProbe: env default dev=<%s>.\n",
			devname);
		ptalDeviceOpen(devname);
	}
	if (homedir) {
		ptalDeviceReadDefaultDeviceFile(
			homedir,".ptal-default-device");
	}
	ptalDeviceReadDefaultDeviceFile(0,"default-device");

	dir=opendir(ptalEtcPrefix);
	if (dir) {
		while (42) {
			struct dirent *dirent=readdir(dir);
			if (!dirent) break;
			devname=dirent->d_name;
			PTAL_LOG_DEBUG("ptalDeviceProbe: dev=<%s>.\n",devname);
			ptalDeviceOpen(devname);
		}
		closedir(dir);
	}

	return PTAL_OK;
}

#if 0
void ptalSignalHandler(int signum) {
	signal(signum,ptalSignalHandler);
	PTAL_LOG_DEBUG("ptalSignalHandler: Caught signal %d!\n",signum);
}
#endif

int ptalInit(void) {
	char *debugString;

	if ((debugString=getenv("PTAL_DEBUG"))) {
		ptalDebugFlag=atoi(debugString);
		PTAL_LOG_ERROR("ptalInit(): debug level set to %d.\n",
			ptalDebugFlag);
	}

	PTAL_LOG_DEBUG("ptalInit()\n");

	signal(SIGHUP,SIG_IGN);
	signal(SIGPIPE,SIG_IGN);
#if 0
	signal(SIGHUP,ptalSignalHandler);
	signal(SIGPIPE,ptalSignalHandler);
#endif

	ptalGlobalConfigFile=ptalFileRead(0,"defaults");

	if (ptalDeviceProbe()<PTAL_OK) return PTAL_ERROR;

	return PTAL_OK;
}

int ptalDeviceClose(ptalDevice_t dev) {
	PTAL_LOG_DEBUG("ptalDeviceClose(dev=0x%8.8X)\n",dev);

	ptalPmlDeallocateAll(dev);
	ptalChannelEnumerate(dev,ptalChannelDeallocateCbd,0);

	return PTAL_OK;
}

int ptalDeviceDelete(ptalDevice_t dev) {
	PTAL_LOG_DEBUG("ptalDeviceDelete(dev=0x%8.8X)\n",dev);

	ptalDeviceClose(dev);

	if (dev->prev) {
		dev->prev->next=dev->next;
	} else {
		ptalFirstDevice=dev->next;
	}
	if (dev->next) {
		dev->next->prev=dev->prev;
	} else {
		ptalLastDevice=dev->prev;
	}

	if (dev->provider->deviceDestructor)
		dev->provider->deviceDestructor(dev);

	free(dev);

	return PTAL_OK;
}
int ptalDeviceDeleteCbd(ptalDevice_t dev,void *cbd) {
	return ptalDeviceDelete(dev);
}

int ptalDeviceEnumerate(ptalProvider_t provider,
    ptalDeviceEnumerate_f callback,void *cbd) {
	int count=0;
	ptalDevice_t current,next;

	current=ptalFirstDevice;
	while (current) {
		next=current->next;
		if ((!provider || provider==current->provider) &&
		    callback) {
			count+=callback(current,cbd);
		}
		current=next;
	}

	return count;
}

int ptalDeviceEnumeratePrintCallback(ptalDevice_t dev,void *cbd) {
	PTAL_LOG_ERROR("\t%s\n",ptalDeviceGetName(dev));
	return 1;
}

char *ptalDeviceGetName(ptalDevice_t dev) {
	return dev->devname;
}

int ptalDone(void) {
	PTAL_LOG_DEBUG("ptalDone()\n");

	return ptalDeviceEnumerate(0,ptalDeviceDeleteCbd,0);
}

void *ptalDeviceGetAppInfo(ptalDevice_t dev) {
	return dev->appInfo;
}

void ptalDeviceSetAppInfo(ptalDevice_t dev,void *appInfo) {
	dev->appInfo=appInfo;
}

int ptalDeviceGetDeviceIDString(ptalDevice_t dev,char *buffer,int maxlen) {
	if (!dev->provider->deviceGetDeviceIDString) {
		PTAL_LOG_WARN("ptalDeviceGetDeviceIDString(dev=0x%8.8X): "
			"no deviceGetDeviceIDString method!\n",dev);
		return PTAL_ERROR;
	}
	return dev->provider->deviceGetDeviceIDString(dev,buffer,maxlen);
}

int ptalDeviceGetPreviousDeviceIDString(ptalDevice_t dev,
    char *buffer,int maxlen) {
	if (!dev->provider->deviceGetPreviousDeviceIDString) {
		PTAL_LOG_WARN("ptalDeviceGetDeviceIDString(dev=0x%8.8X): "
			"no deviceGetPreviousDeviceIDString method!\n",dev);
		return PTAL_ERROR;
	}
	return dev->provider->deviceGetPreviousDeviceIDString(dev,
		buffer,maxlen);
}

int ptalDeviceIDGetField(char *_devID,char *field,
    char **pValue,int *pLenValue) {
	int r=PTAL_ERROR,len=strlen(_devID)+1;
	char *devID=malloc(len);
	char *string=devID,*ppLast=0,*token,*colon,*value;
	memcpy(devID,_devID,len);

	while (42) {
		token=strtok_r(string,";",&ppLast); string=0;
		if (!token) break;

		/* Strip leading spaces before field name. */
		while (isspace(*token)) token++;

		/* Split field at colon or ignore if none.
		 * Strip trailing spaces after field name. */
		colon=strchr(token,':');
		if (!colon) continue;
		value=colon+1;
		while (colon>token && isspace(*(colon-1))) colon--;
		*colon=0;

		if (!strcmp(token,field)) {
			char *uvalue=_devID+(token-devID);
			if (pValue) *pValue=uvalue;
			if (pLenValue) {
				*pLenValue=strlen(token)+strlen(value)+1;
				memcpy(devID,_devID,len);
				if (token[*pLenValue]==';') {
					(*pLenValue)++;
				}
			}
			r=PTAL_OK;
			break;
		}
	}

	memset(devID,0,len);
	free(devID);
	return r;
}

void ptalDeviceIDPruneField(char **pValue,int *pLenValue) {
	/* Remove the trailing semicolon. */
	while (*pLenValue>0 && (*pValue)[*pLenValue-1]==';') (*pLenValue)--;

	/* Remove the leading field name and colon. */
	while (*pLenValue && **pValue!=':') {
		(*pLenValue)--; (*pValue)++;
	}
	while (*pLenValue && **pValue==':') {
		(*pLenValue)--; (*pValue)++;
	}
}

int ptalDeviceIDGetEitherField(char *devID,char *field1,char *field2,
    char **pValue,int *pLenValue) {
	if (ptalDeviceIDGetField(devID,field1,pValue,pLenValue)==PTAL_ERROR &&
	    ptalDeviceIDGetField(devID,field2,pValue,pLenValue)==PTAL_ERROR) {
		return PTAL_ERROR;
	}
	return PTAL_OK;
}

int ptalDeviceIDGetManufacturer(char *devID,char **pValue,int *pLenValue) {
	return ptalDeviceIDGetEitherField(devID,"MFG","MANUFACTURER",
		pValue,pLenValue);
}

int ptalDeviceIDGetModel(char *devID,char **pValue,int *pLenValue) {
	return ptalDeviceIDGetEitherField(devID,"MDL","MODEL",
		pValue,pLenValue);
}

int ptalDeviceIDGetCommandSet(char *devID,char **pValue,int *pLenValue) {
	return ptalDeviceIDGetEitherField(devID,"CMD","COMMAND SET",
		pValue,pLenValue);
}

int ptalDeviceIDGetSerialNumber(char *devID,char **pValue,int *pLenValue) {
	return ptalDeviceIDGetEitherField(devID,"SERN","SN",
		pValue,pLenValue);
}

/* TODO: Functions to convert between service names and socket IDs. */


void ptalChannelDump(ptalChannel_t chan,int level) {
	fprintf(stderr,"chan=0x%8.8X, prev=0x%8.8X, next=0x%8.8X\n",
		(int)chan,(int)chan->prev,(int)chan->next);
	fprintf(stderr,"dev=0x%8.8X, provider=0x%8.8X <%s>\n",
		(int)chan->dev,(int)chan->provider,chan->provider->name);
	fprintf(stderr,"serviceType=%d\n",
		chan->serviceType);
	fprintf(stderr,"socketID=%d\n",
		chan->socketID);
	fprintf(stderr,"serviceName=<%s>\n",
		chan->serviceName);
	fprintf(stderr,"desiredHPSize=%d\n",
		chan->desiredHPSize);
	fprintf(stderr,"desiredPHSize=%d\n",
		chan->desiredPHSize);
	fprintf(stderr,"actualHPSize=%d\n",
		chan->actualHPSize);
	fprintf(stderr,"actualPHSize=%d\n",
		chan->actualPHSize);
	fprintf(stderr,"retryCount=%d\n",
		chan->retryCount);
	fprintf(stderr,"retryDelay=%d\n",
		chan->retryDelay);
	fprintf(stderr,"fd=%d\n",
		chan->fd);
	if (chan->provider->channelDump)
		chan->provider->channelDump(chan,level);
	fprintf(stderr,"\n");
}

ptalChannel_t ptalChannelAllocate(ptalDevice_t dev) {
	int size;
	ptalChannel_t chan;

	/* Sanity-check sizeofChannel. */
	size=dev->provider->sizeofChannel;
	if (size<sizeof(struct ptalChannel_s)) {
		PTAL_LOG_WARN("ptalChannelAllocate(dev=0x%8.8X): "
			"invalid sizeofChannel=%d!\n",dev,size);
		return 0;
	}

	/* Malloc and zero channel. */
	chan=malloc(size);
	if (!chan) {
		PTAL_LOG_WARN("ptalChannelAllocate(dev=0x%8.8X): "
			"malloc(chan=%d) failed!\n",dev,size);
		return 0;
	}
	memset(chan,0,size);

	/* Initialize fields. */
	chan->dev=dev;
	chan->provider=dev->provider;
	chan->retryCount=PTAL_DEFAULT_RETRY_COUNT;
	chan->retryDelay=PTAL_DEFAULT_RETRY_DELAY;
	chan->fd=PTAL_NO_FD;

	/* Call provider's constructor if defined. */
	if (chan->provider->channelConstructor) {
		chan->provider->channelConstructor(chan);
	}

	/* Insert into linked list of channels. */
	if (!dev->firstChannel) dev->firstChannel=chan;
	chan->prev=dev->lastChannel;
	chan->next=0;
	if (dev->lastChannel) dev->lastChannel->next=chan;
	dev->lastChannel=chan;

	PTAL_LOG_DEBUG("ptalChannelAllocate(dev=0x%8.8X): "
		"chan=0x%8.8X.\n",dev,chan);
	return chan;
}

int ptalChannelEnumerate(ptalDevice_t dev,
    ptalChannelEnumerate_f callback,void *cbd) {
	int count=0;
	ptalChannel_t current,next;

	current=dev->firstChannel;
	while (current) {
		next=current->next;
		if (callback) {
			count+=callback(current,cbd);
		}
		current=next;
	}

	return count;
}

int ptalChannelDeallocate(ptalChannel_t chan) {
	PTAL_LOG_DEBUG("ptalChannelDeallocate(chan=0x%8.8X)\n",chan);

	ptalChannelClose(chan);
	if (chan->provider->channelDestructor)
		chan->provider->channelDestructor(chan);

	if (chan->prev) {
		chan->prev->next=chan->next;
	} else {
		chan->dev->firstChannel=chan->next;
	}
	if (chan->next) {
		chan->next->prev=chan->prev;
	} else {
		chan->dev->lastChannel=chan->prev;
	}

	free(chan);

	return PTAL_OK;
}
int ptalChannelDeallocateCbd(ptalChannel_t chan,void *cbd) {
	return ptalChannelDeallocate(chan);
}

int ptalChannelGetRemoteService(ptalChannel_t chan,
    int *pServiceType,int *pSocketID,char **pServiceName) {
	if (pServiceType) *pServiceType=chan->serviceType;
	if (pSocketID) *pSocketID=chan->socketID;
	if (pServiceName) *pServiceName=chan->serviceName;

	return PTAL_OK;
}

int ptalChannelSetRemoteService(ptalChannel_t chan,
    int serviceType,int socketID,char *serviceName) {
	PTAL_LOG_DEBUG("ptalChannelSetRemoteService(chan=0x%8.8X,"
		"serviceType=%d,socketID=%d,serviceName=<%s>)\n",
		chan,serviceType,socketID,serviceName?serviceName:"");

	chan->serviceType=serviceType;
	chan->socketID=socketID;
	if (!serviceName) serviceName="";
	strncpy(chan->serviceName,serviceName,PTAL_MAX_SERVICE_NAME_LEN);
	chan->serviceName[PTAL_MAX_SERVICE_NAME_LEN]=0;

	return PTAL_OK;
}

typedef struct ptalChannelFindOrAllocate_s {
	int serviceType;
	int socketID;
	char *serviceName;

	ptalChannel_t chan;
} *ptalChannelFindOrAllocate_t;

int ptalChannelFindOrAllocateCallback(ptalChannel_t chan,void *_cbd) {
	ptalChannelFindOrAllocate_t cbd=(ptalChannelFindOrAllocate_t)_cbd;
	int serviceType,socketID;
	char *serviceName;

	ptalChannelGetRemoteService(chan,&serviceType,&socketID,&serviceName);

	if (cbd->serviceType!=serviceType) return 0;
	if (serviceType==PTAL_STYPE_GENERIC) {
		if (socketID!=cbd->socketID) return 0;
		if (!cbd->serviceName && strlen(serviceName)) return 0;
		if (strcmp(serviceName,cbd->serviceName)) return 0;
	}

	if (cbd->chan) {

		return 0;
	}
	cbd->chan=chan;
	return 1;
}

ptalChannel_t ptalChannelFindOrAllocate(ptalDevice_t dev,
    int serviceType,int socketID,char *serviceName) {
	struct ptalChannelFindOrAllocate_s cbd;

	cbd.serviceType=serviceType;
	cbd.socketID=socketID;
	cbd.serviceName=serviceName;
	cbd.chan=0;

	ptalChannelEnumerate(dev,ptalChannelFindOrAllocateCallback,&cbd);

	if (!cbd.chan) {
		cbd.chan=ptalChannelAllocate(dev);
		if (cbd.chan) {
			ptalChannelSetRemoteService(cbd.chan,
				serviceType,socketID,serviceName);
		}
	}

	return cbd.chan;
}

int ptalChannelSetPacketSizes(ptalChannel_t chan,
    int desiredHPSize,int desiredPHSize) {
	PTAL_LOG_DEBUG("ptalChannelSetPacketSizes(chan=0x%8.8X,"
		"desiredHPSize=%d,desiredPHSize=%d)\n",
		chan,desiredHPSize,desiredPHSize);

	chan->desiredHPSize=desiredHPSize;
	chan->desiredPHSize=desiredPHSize;

	return PTAL_OK;
}

void ptalChannelAdjustOnePacketSize(int *pDesired,int actual) {
	if (pDesired && actual && actual<*pDesired) {
		*pDesired=actual;
	}
}

int ptalChannelAdjustPacketSizes(ptalChannel_t chan,
    int *pDesiredHPSize,int *pDesiredPHSize) {
	ptalChannelAdjustOnePacketSize(pDesiredHPSize,chan->actualHPSize);
	ptalChannelAdjustOnePacketSize(pDesiredPHSize,chan->actualPHSize);

	return PTAL_OK;
}

int ptalChannelSetErrorHandling(ptalChannel_t chan,
    int retryCount,int retryDelay) {
	PTAL_LOG_DEBUG("ptalChannelSetErrorHandling(chan=0x%8.8X,"
		"retryCount=%d,retryDelay=%d)\n",
		chan,retryCount,retryDelay);

	chan->retryCount=retryCount;
	chan->retryDelay=retryDelay;

	return PTAL_OK;
}

int ptalChannelIsOpen(ptalChannel_t chan) {
	if (chan->fd!=PTAL_NO_FD) return 1;
	if (!chan->provider->channelIsOpen) return 0;
	return chan->provider->channelIsOpen(chan);
}

int ptalChannelOpen(ptalChannel_t chan) {
	if (ptalChannelIsOpen(chan)) {
		PTAL_LOG_WARN("ptalChannelOpen(chan=0x%8.8X): "
			"already open (fd=%d).\n",chan,chan->fd);
		return PTAL_OK;
	}
	chan->actualHPSize=chan->actualPHSize=0;

	if (!chan->provider->channelOpen) {
		PTAL_LOG_WARN("ptalChannelOpen(chan=0x%8.8X): "
			"no open method!\n",chan);
		return PTAL_ERROR;
	}

	if (chan->provider->channelOpen(chan)==PTAL_ERROR) {
		PTAL_LOG_WARN("ptalChannelOpen(chan=0x%8.8X): "
			"provider failed open!\n",chan);
		return PTAL_ERROR;
	}

	PTAL_LOG_DEBUG("ptalChannelOpen(chan=0x%8.8X): fd=%d.\n",
		chan,chan->fd);
	return PTAL_OK;
}

int ptalChannelOpenOrReopen(ptalChannel_t chan) {
	if (ptalChannelIsStale(chan)) {
		ptalChannelClose(chan);
	}

	return ptalChannelOpen(chan);
}

int ptalChannelClose(ptalChannel_t chan) {
	int r;

	PTAL_LOG_DEBUG("ptalChannelClose(chan=0x%8.8X)\n",chan);

	if (chan->provider->channelClose) {
		return chan->provider->channelClose(chan);
	}

	if (chan->fd==PTAL_NO_FD) {
		PTAL_LOG_WARN("ptalChannelClose(chan=0x%8.8X): "
			"not open!\n",chan);
		return PTAL_ERROR;
	}

	r=close(chan->fd);
	if (r<0) {
		PTAL_LOG_WARN("ptalChannelClose(chan=0x%8.8X): "
			"close(fd=%d) returns %d!\n",chan,chan->fd,r);
		return PTAL_ERROR;
	}
	chan->fd=PTAL_NO_FD;

	return PTAL_OK;
}

int ptalFdPrepareForSelect(int fd,int *pn,
    fd_set *prset,fd_set *pwset,fd_set *pxset) {
	if (pn && *pn<=fd) *pn=fd+1;
	if (prset) FD_SET(fd,prset);
	if (pwset) FD_SET(fd,pwset);
	if (pxset) FD_SET(fd,pxset);

	return PTAL_OK;
}

int ptalChannelPrepareForSelect(ptalChannel_t chan,
    int *pfd,int *pn,fd_set *prset,fd_set *pwset,fd_set *pxset) {
	if (chan->fd==PTAL_NO_FD) {
		PTAL_LOG_WARN("ptalChannelPrepareForSelect(chan=0x%8.8X): "
			"not open!\n",chan);
		return PTAL_ERROR;
	}

	if (pfd) *pfd=chan->fd;
	return ptalFdPrepareForSelect(chan->fd,pn,prset,pwset,pxset);
}

int _ptalChannelSelect(ptalChannel_t chan,int *pr,int *pw,int *px,
    struct timeval *timeout) {
	int fd,n=0,r;
	fd_set rset,wset,xset;
	fd_set *prset=0,*pwset=0,*pxset=0;
	struct timeval timeoutCopy;

	if (chan->provider->channelSelect) {
		return chan->provider->channelSelect(chan,pr,pw,px,timeout);
	}

	if (pr && *pr) { prset=&rset; FD_ZERO(prset); }
	if (pw && *pw) { pwset=&wset; FD_ZERO(pwset); }
	if (px && *px) { pxset=&xset; FD_ZERO(pxset); }

	if (ptalChannelPrepareForSelect(chan,&fd,&n,prset,pwset,pxset)==
	    PTAL_ERROR) {
		return PTAL_ERROR;
	}

	if (timeout) {
		timeoutCopy.tv_sec=timeout->tv_sec;
		timeoutCopy.tv_usec=timeout->tv_usec;
		timeout=&timeoutCopy;
	}
	r=select(n,prset,pwset,pxset,timeout);
	if (prset) *pr=FD_ISSET(fd,prset);
	if (pwset) *pw=FD_ISSET(fd,pwset);
	if (pxset) *px=FD_ISSET(fd,pxset);

	PTAL_LOG_DEBUG("_ptalChannelSelect(chan=0x%8.8X) returns %d, "
		"errno=%d, n=%d, timeout=0x%8.8X, tv_sec=%d, tv_usec=%d.\n",
		chan,r,errno,n,timeout,
		timeout?timeout->tv_sec:0,timeout?timeout->tv_usec:0);
	return r;
}

int ptalChannelSetSelectPollTimeout(ptalChannel_t chan,
    struct timeval *pTimeout) {
	chan->selectPollTimeout=pTimeout?pTimeout->tv_sec:0;
	return PTAL_OK;
}

int ptalChannelSetSelectPollCallback(ptalChannel_t chan,
    ptalChannelSelectPoll_f callback,void *cbd) {
	chan->selectPollCallback=callback;
	chan->selectPollCallbackData=cbd;
	return PTAL_OK;
}

int ptalChannelSelect(ptalChannel_t chan,int *pr,int *pw,int *px,
    struct timeval *pTimeout) {
	struct timeval shortTimeout={chan->selectPollTimeout,0};
	struct timeval remainingTimeout={0,0},*pSelectTimeout;
	int r,oldr=0,oldw=0,oldx=0;

	if (!chan->selectPollCallback || chan->selectPollTimeout<=0) {
		return _ptalChannelSelect(chan,pr,pw,px,pTimeout);
	}

	if (pr) oldr=*pr;
	if (pw) oldw=*pw;
	if (px) oldx=*px;
	if (pTimeout) {
		remainingTimeout.tv_sec=pTimeout->tv_sec;
		remainingTimeout.tv_usec=pTimeout->tv_usec;
	}

	while (42) {
		if (pTimeout && remainingTimeout.tv_sec<shortTimeout.tv_sec) {
			pSelectTimeout=&remainingTimeout;
		} else {
			pSelectTimeout=&shortTimeout;
		}
		if (pr) *pr=oldr;
		if (pw) *pw=oldw;
		if (px) *px=oldx;
		r=_ptalChannelSelect(chan,pr,pw,px,pSelectTimeout);
		if (r) break;

		if (chan->selectPollCallback(chan,
		     chan->selectPollCallbackData)==PTAL_ERROR) {
			break;
		}

		if (pTimeout) {
			if (remainingTimeout.tv_sec<shortTimeout.tv_sec) {
				break;
			}
			remainingTimeout.tv_sec-=shortTimeout.tv_sec;
		}
	}

	return r;
}

int ptalChannelRead(ptalChannel_t chan,char *buffer,int count) {
	int r;

	PTAL_LOG_DEBUG("ptalChannelRead(chan=0x%8.8X,"
		"buffer=0x%8.8X,count=%d)\n",
		chan,buffer,count);

	if (chan->provider->channelRead) {
		r=chan->provider->channelRead(chan,buffer,count);

	} else if (chan->fd==PTAL_NO_FD) {
		PTAL_LOG_WARN("ptalChannelRead(chan=0x%8.8X): "
			"not open!\n",chan);
		r=PTAL_ERROR;

	} else {
		r=read(chan->fd,buffer,count);
	}

	PTAL_LOG_DEBUG("ptalChannelRead(chan=0x%8.8X,"
		"buffer=0x%8.8X,count=%d) returns %d.\n",
		chan,buffer,count,r);
	return r;
}

int ptalChannelIsStale(ptalChannel_t chan) {
	char c;
	int r=1,x=1,s;
	struct timeval timeout={0,0};

	if (!ptalChannelIsOpen(chan)) return 0;

	s=ptalChannelSelect(chan,&r,0,&x,&timeout);

	if (!s) return 0;
	if (s<0 || x) return 1;

	if (r && ptalChannelRead(chan,&c,1)<=0) return 1;

	return 0;
}

int ptalChannelReadTimeout(ptalChannel_t chan,char *buffer,int countdown,
    struct timeval *startTimeout,struct timeval *continueTimeout) {
	struct timeval *timeout=startTimeout;
	int r,count,countup=0;

	while (42) {
		r=1;
		count=ptalChannelSelect(chan,&r,0,0,timeout);
		if (count<=0) break;

		count=ptalChannelRead(chan,buffer,countdown);
		if (count<=0) break;

		countup+=count;
		buffer+=count;
		countdown-=count;
		if (countdown<=0) break;

		timeout=continueTimeout;
	}

	if (countup) count=countup;
	return count;
}

/* It's ugly to put this here, but I didn't want to maintain it in
 * multiple places for SANE. */
/* TODO: Find a more appropriate place for this? */
static int ptalSclBufferIsPartialReply(unsigned char *data,int datalen) {
	int i=0,value=0;
	unsigned char d;  

	if (i>=datalen) return 0;
	if (data[i++]!=27) return 0;
	if (i>=datalen) return 0;
	if (data[i++]!='*') return 0;
	if (i>=datalen) return 0;
	if (data[i++]!='s') return 0;
	while (42) {
		if (i>=datalen) return 0;
		d=data[i]-'0';
		if (d>9) break;
		i++;
	}
	d=data[i++];
	if (d<'a' || d>'z') return 0;
	while (42) {
		if (i>=datalen) return 0;
		d=data[i]-'0';
		if (d>9) break;
		i++;
		value=(value*10)+d;
	}
	if (i>=datalen) return 0;
	if (data[i++]!='W') return 0;
	value=i+value-datalen;
	if (value<0) value=0;
	return value;
}

/* It's ugly to put this here, but I didn't want to maintain it in
 * multiple places for SANE. */
/* TODO: Find a more appropriate place for this? */
int ptalSclChannelRead(ptalChannel_t chan,char *buffer,int countdown,
    struct timeval *startTimeout,struct timeval *continueTimeout,
    int isSclResponse) {
	char *bufferStart=buffer;
	int bufferLen=countdown,countup=0,r;
	struct timeval myContinueTimeout={0,0};

	if (!isSclResponse) {
		return ptalChannelReadTimeout(chan,buffer,countdown,
			startTimeout,continueTimeout);
	}

	while (42) {
		r=ptalChannelReadTimeout(chan,buffer,countdown,
			startTimeout,&myContinueTimeout);
		PTAL_LOG_DEBUG("ptalSclChannelRead(chan=0x%8.8X): "
			"ptalChannelReadTimeout(buffer=0x%8.8X,count=%d) "
			"returns %d, errno=%d.\n",
			chan,buffer,countdown,r,errno);
		if (r<=0) break;
		countup+=r;

		countdown=ptalSclBufferIsPartialReply(bufferStart,countup);
		if (countup+countdown>bufferLen) {
			countdown=bufferLen-countup;
		}
		if (countdown<=0) break;

		PTAL_LOG_DEBUG("ptalSclChannelRead(chan=0x%8.8X): "
			"read %d of %d bytes, %d remaining.\n",
			chan,countup,bufferLen,countdown);

		buffer+=r;
		startTimeout=continueTimeout;
	}

	if (!countup) return r;
	return countup;
}

int ptalChannelFlush(ptalChannel_t chan,
    struct timeval *startTimeout,struct timeval *continueTimeout) {
	struct timeval defaultStartTimeout,defaultContinueTimeout;
	char buffer[PTAL_FLUSH_LEN_BUFFER];
	int r,countup=0;

	if (!startTimeout) {
		startTimeout=&defaultStartTimeout;
		startTimeout->tv_sec=0;
		startTimeout->tv_usec=0;
	}

	if (!continueTimeout) {
		continueTimeout=&defaultContinueTimeout;
		continueTimeout->tv_sec=PTAL_FLUSH_TIMEOUT_SEC;
		continueTimeout->tv_usec=PTAL_FLUSH_TIMEOUT_USEC;
	}

	while (42) {
		r=ptalChannelReadTimeout(chan,buffer,PTAL_FLUSH_LEN_BUFFER,
			startTimeout,continueTimeout);
		if (r<=0) break;
		countup+=r;
		startTimeout=continueTimeout;
	}

	return countup;
}

int ptalChannelWrite(ptalChannel_t chan,char *buffer,int count) {
	int r;

	PTAL_LOG_DEBUG("ptalChannelWrite(chan=0x%8.8X,"
		"buffer=0x%8.8X,count=%d)\n",chan,buffer,count);

	if (chan->provider->channelWrite) {
		r=chan->provider->channelWrite(chan,buffer,count);
	
	} else if (chan->fd==PTAL_NO_FD) {
		PTAL_LOG_WARN("ptalChannelWrite(chan=0x%8.8X): "
			"not open!\n",chan);
		r=PTAL_ERROR;

	} else {
		r=write(chan->fd,buffer,count);
	}

	PTAL_LOG_DEBUG("ptalChannelWrite(chan=0x%8.8X,"
		"buffer=0x%8.8X,count=%d) returns %d.\n",
		chan,buffer,count,r);
	return r;
}








int ptalPmlOpen(ptalDevice_t dev) {
	PTAL_LOG_DEBUG("ptalPmlOpen(dev=0x%8.8X)\n",dev);

	if (dev->provider->pmlOpen) {
		return dev->provider->pmlOpen(dev);
	}

	if (!dev->pmlChannel) {
		dev->pmlChannel=ptalChannelAllocate(dev);
		if (!dev->pmlChannel) {
			return PTAL_ERROR;
		}
		ptalChannelSetRemoteService(dev->pmlChannel,PTAL_STYPE_PML,0,0);
	}

	return ptalChannelOpenOrReopen(dev->pmlChannel);
}

int ptalPmlClose(ptalDevice_t dev) {
	PTAL_LOG_DEBUG("ptalPmlClose(dev=0x%8.8X)\n",dev);

	if (dev->provider->pmlClose) {
		return dev->provider->pmlClose(dev);
	}

	if (!dev->pmlChannel) {
		return PTAL_ERROR;
	}

	return ptalChannelClose(dev->pmlChannel);
}

ptalPmlObject_t ptalPmlAllocate(ptalDevice_t dev) {
	int size=sizeof(struct ptalPmlObject_s);
	ptalPmlObject_t obj;

	PTAL_LOG_DEBUG("ptalPmlAllocate(dev=0x%8.8X)\n",dev);

	/* Malloc and zero object. */
	obj=malloc(size);
	if (!obj) {
		PTAL_LOG_WARN("ptalPmlAllocate(dev=0x%8.8X): "
			"malloc(obj=%d) failed!\n",dev,size);
		return 0;
	}
	memset(obj,0,size);

	/* Initialize fields. */
	obj->dev=dev;

	/* Insert into linked list of PML objects for this device. */
	if (!dev->firstPmlObject) dev->firstPmlObject=obj;
	obj->prev=dev->lastPmlObject;
	obj->next=0;
	if (dev->lastPmlObject) dev->lastPmlObject->next=obj;
	dev->lastPmlObject=obj;

	PTAL_LOG_DEBUG("ptalPmlAllocate(dev=0x%8.8X) returns obj=0x%8.8X.\n",
		dev,obj);
	return obj;
}

int ptalPmlEnumerate(ptalDevice_t dev,
    ptalPmlObjectEnumerate_f callback,void *cbd) {
	int count=0;
	ptalPmlObject_t current,next;

	current=dev->firstPmlObject;
	while (current) {
		next=current->next;
		if (callback) {
			count+=callback(current,cbd);
		}
		current=next;
	}

	return count;
}

int ptalPmlDeallocate(ptalPmlObject_t obj) {
	PTAL_LOG_DEBUG("ptalPmlDeallocate(obj=0x%8.8X)\n",obj);

	if (obj->prev) {
		obj->prev->next=obj->next;
	} else {
		obj->dev->firstPmlObject=obj->next;
	}
	if (obj->next) {
		obj->next->prev=obj->prev;
	} else {
		obj->dev->lastPmlObject=obj->prev;
	}

	free(obj);

	return PTAL_OK;
}
int ptalPmlDeallocateCallback(ptalPmlObject_t obj,void *_cbd) {
	return ptalPmlDeallocate(obj);
}

int ptalPmlDeallocateAll(ptalDevice_t dev) {
	return ptalPmlEnumerate(dev,ptalPmlDeallocateCallback,0);
}

int ptalPmlSetID(ptalPmlObject_t obj,char *oid) {
	int len=0;	/* TODO: Do we need this parameter? */

	PTAL_LOG_DEBUG("ptalPmlSetID(obj=0x%8.8X)\n",obj);

	if (!len) {
		len=strlen(oid);
		if (!len) len++;
	}
	if (len>PTAL_PML_MAX_OID_LEN) return PTAL_ERROR;

	/* TODO: Disable trap (if enabled) on old OID. */

	memcpy(obj->oid,oid,len);
	obj->oid[len]=0;

	obj->numberOfValidValues=0;

	/* TODO: Clear out other trap-related fields. */

	PTAL_LOG_DEBUG("ptalPmlSetID(obj=0x%8.8X) returns OK.\n",obj);
	return PTAL_OK;
}

ptalPmlObject_t ptalPmlAllocateID(ptalDevice_t dev,char *oid) {
	ptalPmlObject_t obj=ptalPmlAllocate(dev);
	if (!obj) {
		PTAL_LOG_WARN("ptalPmlAllocateID: out of memory!\n");
	} else if (ptalPmlSetID(obj,oid)==PTAL_ERROR) {
		PTAL_LOG_WARN("ptalPmlAllocateID: error setting OID!\n");
 		ptalPmlDeallocate(obj);
		obj=0;
	}
	return obj;
}

int ptalPmlSetAsciiID(ptalPmlObject_t obj,char *s) {
	char oid[PTAL_PML_MAX_OID_LEN+1];
	int len=0,c;

	while (42) {
		while (*s=='.') s++;
		if (!*s) break;
		if (*s<'0' || *s>'9') return PTAL_ERROR;
		c=atoi(s);
		if (c<0 || c>255) return PTAL_ERROR;
		if (len>=PTAL_PML_MAX_OID_LEN) return PTAL_ERROR;
		oid[len++]=c;
		while (*s>='0' && *s<='9') s++;
	}
	oid[len]=0;

	return ptalPmlSetID(obj,oid);
}

int ptalPmlGetID(ptalPmlObject_t obj,char *buffer,int maxlen) {
	if (maxlen<=1) return PTAL_ERROR;
	buffer[maxlen-1]=0;
	strncpy(buffer,obj->oid,maxlen);
	if (buffer[maxlen-1]) return PTAL_ERROR;

	return PTAL_OK;
}

ptalPmlValue_t ptalPmlGetLastValue(ptalPmlObject_t obj) {
	if (obj->numberOfValidValues<=0) return 0;
	return &obj->value[obj->indexOfLastValue];
}

ptalPmlValue_t ptalPmlGetPreviousLastValue(ptalPmlObject_t obj) {
	if (obj->numberOfValidValues<=1) return 0;

	return &obj->value[(PTAL_PML_MAX_OID_VALUES+obj->indexOfLastValue-1)%
		PTAL_PML_MAX_OID_VALUES];
}

ptalPmlValue_t ptalPmlPrepareNextValue(ptalPmlObject_t obj) {
	obj->indexOfLastValue=(obj->indexOfLastValue+1)%PTAL_PML_MAX_OID_VALUES;
	if (obj->numberOfValidValues<PTAL_PML_MAX_OID_VALUES) {
		obj->numberOfValidValues++;
	}
	return &obj->value[obj->indexOfLastValue];
}

void ptalPmlClearOldValues(ptalPmlObject_t obj) {
	if (obj->numberOfValidValues) {
		obj->numberOfValidValues=1;
	}
}

int ptalPmlSetPrefixValue(ptalPmlObject_t obj,int type,
    char *prefix,int lenPrefix,char *value,int lenValue) {
	ptalPmlValue_t v=ptalPmlPrepareNextValue(obj);
	int r=PTAL_ERROR;

	PTAL_LOG_DEBUG("ptalPmlSetPrefixValue(obj=0x%8.8X,type=0x%4.4X,"
		"lenPrefix=%d,lenValue=%d)\n",
		obj,type,lenPrefix,lenValue);

	if (lenPrefix<0 || lenValue<0 ||
	    (lenPrefix+lenValue)>PTAL_PML_MAX_VALUE_LEN) {
		PTAL_LOG_WARN("ptalPmlSetPrefixValue(obj=0x%8.8X): "
			"invalid lenPrefix=%d and/or lenValue=%d!\n",
			obj,lenPrefix,lenValue);
		goto abort;
	}

	v->type=type;
	v->len=lenPrefix+lenValue;
	if (lenPrefix) memcpy(v->value,prefix,lenPrefix);
	if (lenValue) memcpy(v->value+lenPrefix,value,lenValue);
	v->value[lenPrefix+lenValue]=0;

	r=PTAL_OK;
abort:
	PTAL_LOG_DEBUG("ptalPmlSetPrefixValue(obj=0x%8.8X) returns %d.\n",
		obj,r);
	return r;
}

int ptalPmlSetValue(ptalPmlObject_t obj,int type,char *value,int len) {
	return ptalPmlSetPrefixValue(obj,type,0,0,value,len);
}

int ptalPmlSetStringValue(ptalPmlObject_t obj,int symbolSet,
    char *value,int len) {
	char prefix[2];
	prefix[0]=(symbolSet>>8)&0xFF;
	prefix[1]=(symbolSet)&0xFF;

	if (!len) len=strlen(value);
	return ptalPmlSetPrefixValue(obj,PTAL_PML_TYPE_STRING,
		prefix,2,value,len);
}

int ptalPmlSetIntegerValue(ptalPmlObject_t obj,int type,int value) {
	char buffer[sizeof(int)];
	int len=sizeof(int),i=len-1;

	while (42) {
		buffer[i]=value&0xFF;
		value>>=8;
		if (!i) break;
		i--;
	}
	for (;!buffer[i] && i<(len);i++);

	return ptalPmlSetPrefixValue(obj,type,buffer+i,len-i,0,0);
}

int ptalPmlGetType(ptalPmlObject_t obj) {
	ptalPmlValue_t v=ptalPmlGetLastValue(obj);
	if (!v) {
		return PTAL_ERROR;
	}
	return v->type;
}

int ptalPmlGetPrefixValue(ptalPmlObject_t obj,int *pType,
    char *prefix,int lenPrefix,char *buffer,int maxlen) {
	int len;
	ptalPmlValue_t v=ptalPmlGetLastValue(obj);

	if (!v) {

		return PTAL_ERROR;
	}
	if (pType) *pType=v->type;
	if (!prefix && !buffer) return PTAL_OK;

	if (lenPrefix<0 || maxlen<0) {

		return PTAL_ERROR;
	}

	if (v->len>lenPrefix+maxlen) {

		return PTAL_ERROR;
	}
	if (v->len<lenPrefix) {

		return PTAL_ERROR;
	}

	if (lenPrefix) memcpy(prefix,v->value,lenPrefix);
	len=v->len-lenPrefix;
	if (len) memcpy(buffer,v->value+lenPrefix,len);
	if (len<maxlen) buffer[len]=0;

	return len;
}

int ptalPmlGetValue(ptalPmlObject_t obj,int *pType,char *buffer,int maxlen) {
	return ptalPmlGetPrefixValue(obj,pType,0,0,buffer,maxlen);
}

int ptalPmlGetStringValue(ptalPmlObject_t obj,int *pSymbolSet,
    char *buffer,int maxlen) {
	int type,len;
	unsigned char prefix[2];

	if (ptalPmlGetPrefixValue(obj,&type,0,0,0,0)==PTAL_ERROR) {

		return PTAL_ERROR;
	}

	len=ptalPmlGetPrefixValue(obj,&type,prefix,2,buffer,maxlen);
	if (len==PTAL_ERROR) {

		return PTAL_ERROR;
	}
	if (pSymbolSet) *pSymbolSet=((prefix[0]<<8)|prefix[1]);

	return len;
}

int ptalPmlGetIntegerValue(ptalPmlObject_t obj,int *pType,int *pValue) {
	int type;
	unsigned char svalue[sizeof(int)];
	int accum=0,i,len;

	if (!pType) pType=&type;

	len=ptalPmlGetPrefixValue(obj,pType,0,0,svalue,sizeof(int));
	if (len==PTAL_ERROR) {

		return PTAL_ERROR;
	}

	for (i=0;i<len;i++) {
		accum=((accum<<8)|(svalue[i]&0xFF));
	}
	if (pValue) *pValue=accum;

	return PTAL_OK;
}

int ptalPmlDoLastValuesDiffer(ptalPmlObject_t obj) {
	ptalPmlValue_t vNew=ptalPmlGetLastValue(obj);
	ptalPmlValue_t vOld=ptalPmlGetPreviousLastValue(obj);

	return (vNew && vOld &&
		(vOld->type!=vNew->type ||
		 vOld->len!=vNew->len ||
		 memcmp(vOld->value,vNew->value,vOld->len)));
}

int ptalPmlSetStatus(ptalPmlObject_t obj,int status) {
	obj->status=status;

	return status;
}

int ptalPmlGetStatus(ptalPmlObject_t obj) {
	return obj->status;
}

int ptalPmlReadReply(ptalDevice_t dev,unsigned char *data,int maxDatalen,
    int request) {

	return ptalChannelRead(dev->pmlChannel,data,maxDatalen);

	/* TODO: Check for and handle traps. */

}

int ptalPmlRequestSet(ptalPmlObject_t obj) {
	unsigned char data[PTAL_PML_MAX_DATALEN];
	int i,datalen=0,request=PTAL_PML_REQUEST_SET,r;

	PTAL_LOG_DEBUG("ptalPmlRequestSet(obj=0x%8.8X)\n",obj);
	ptalPmlSetStatus(obj,PTAL_PML_OK);

	if (obj->dev->provider->pmlSet) {
		return obj->dev->provider->pmlSet(obj);
	}
	if (!obj->dev->pmlChannel) {

		return PTAL_ERROR;
	}

	/* Default SET implementation: */
	data[datalen++]=request;
	data[datalen++]=PTAL_PML_TYPE_OBJECT_IDENTIFIER;
	r=strlen(obj->oid);
	data[datalen++]=r;
	memcpy(data+datalen,obj->oid,r);
	datalen+=r;

	r=ptalPmlGetValue(obj,&i,data+datalen+2,PTAL_PML_MAX_DATALEN-datalen-2);
	if (r==PTAL_ERROR) {

		return PTAL_ERROR;
	}
	data[datalen++]=i|(r>>8);
	data[datalen++]=r;
	datalen+=r;

	r=ptalChannelWrite(obj->dev->pmlChannel,data,datalen);
	if (r!=datalen) {

		return PTAL_ERROR;
	}

	datalen=ptalPmlReadReply(obj->dev,data,PTAL_PML_MAX_DATALEN,request);
	if (datalen==PTAL_ERROR) {

		return PTAL_ERROR;
	}

	i=0;
	r=data[i++];	/* Read command. */
	if (r!=(request|PTAL_PML_COMMAND_REPLY)) {

		return PTAL_ERROR;
	}

	r=data[i++];	/* Read execution outcome. */
	if (ptalPmlSetStatus(obj,r)&PTAL_PML_ERROR) {

		return PTAL_ERROR;
	}

	r=data[i++];	/* Read data type. */
	if (r==PTAL_PML_TYPE_ERROR_CODE) {
		r=data[i++];	/* Read length (should be 1). */
		r=data[i++];	/* Read error code. */
		ptalPmlSetStatus(obj,r);
		if (r&PTAL_PML_ERROR || i>=datalen) {

			return PTAL_ERROR;
		}
		r=data[i++];	/* Read data type. */
	}

	/* TODO: Validate OID, copy value back in. */

	return PTAL_OK;
}

int ptalPmlRequestSetRetry(ptalPmlObject_t obj,int count,int delay) {
	int r=PTAL_ERROR;

	if (count<=0) count=20;
	if (delay<=0) delay=2;
	while (42) {
		r=ptalPmlRequestSet(obj);
		if (r!=PTAL_ERROR || count<=0 || (ptalPmlGetStatus(obj)!=
		     PTAL_PML_ERROR_ACTION_CAN_NOT_BE_PERFORMED_NOW)) {
			break;
		}
		sleep(delay);
		count--;
	}

	return r;
}

int ptalPmlRequestGet(ptalPmlObject_t obj,ptalPmlObject_t next) {
	unsigned char data[PTAL_PML_MAX_DATALEN];
	int i,datalen=0,request=PTAL_PML_REQUEST_GET,r;

	PTAL_LOG_DEBUG("ptalPmlRequestGet(obj=0x%8.8X,next=0x%8.8X)\n",
		obj,next);
	ptalPmlSetStatus(obj,PTAL_PML_OK);

	if (obj->dev->provider->pmlGet) {
		return obj->dev->provider->pmlGet(obj,next);
	}
	if (!obj->dev->pmlChannel) {

		return PTAL_ERROR;
	}

	/* Default GET/GETNEXT implementation: */
	if (next) request=PTAL_PML_REQUEST_GETNEXT;
	data[datalen++]=request;
	data[datalen++]=PTAL_PML_TYPE_OBJECT_IDENTIFIER;
	r=strlen(obj->oid);
	data[datalen++]=r;
	memcpy(data+datalen,obj->oid,r);
	datalen+=r;

	r=ptalChannelWrite(obj->dev->pmlChannel,data,datalen);
	if (r!=datalen) {

		return PTAL_ERROR;
	}

	datalen=ptalPmlReadReply(obj->dev,data,PTAL_PML_MAX_DATALEN,request);
	if (datalen==PTAL_ERROR) {

		return PTAL_ERROR;
	}

	i=0;
	r=data[i++];	/* Read command. */
	if (r!=(request|PTAL_PML_COMMAND_REPLY)) {

		return PTAL_ERROR;
	}

	r=data[i++];	/* Read execution outcome. */
	if (ptalPmlSetStatus(obj,r)&PTAL_PML_ERROR) {

		return PTAL_ERROR;
	}

	r=data[i++];	/* Read data type. */
	if (r==PTAL_PML_TYPE_ERROR_CODE) {
		r=data[i++];	/* Read length (should be 1). */
		r=data[i++];	/* Read error code. */
		ptalPmlSetStatus(obj,r);
		if (r&PTAL_PML_ERROR || i>=datalen) {

			return PTAL_ERROR;
		}
		r=data[i++];	/* Read data type. */
	}

	if (r!=PTAL_PML_TYPE_OBJECT_IDENTIFIER) {

		return PTAL_ERROR;
	}
	r=data[i++];	/* Read length. */
	/* For GET, make sure it's the same OID. */
	if (!next) {
		/* TODO */

	/* For GETNEXT, copy the OID. */
	} else {
		obj=next;
		if (ptalPmlSetID(obj,data+i)==PTAL_ERROR) {

			return PTAL_ERROR;
		}
	}
	i+=r;

	r=data[i]&PTAL_PML_TYPE_MASK;	/* Read data type. */
	datalen=((data[i]&(~PTAL_PML_TYPE_MASK))<<8)|data[i+1];
	if (ptalPmlSetValue(obj,r,data+i+2,datalen)==PTAL_ERROR) {
		return PTAL_ERROR;
	}

	return PTAL_OK;
}

int ptalPmlRequestSetTrap(ptalPmlObject_t obj,int enable) {
	/* TODO: Implement! */

	return PTAL_ERROR;
}

int ptalPmlIsTrapEnabled(ptalPmlObject_t obj) {
	return obj->trapEnabled;
}

/* Returns old count. */
int ptalPmlGetAndClearTrapCount(ptalPmlObject_t obj) {
	int count=obj->trapCount;
	obj->trapCount-=count;
	return count;
}

int ptalPmlServiceTraps(ptalDevice_t dev) {
	/* TODO: Implement! */

	return 0;
}
