import sys
from gtk import *
from constants import *
from Preferences import *
from MessageBox import *
import GtkExtra


class KeyEditor(GtkDialog):

    def __init__(self, preferences):
        GtkDialog.__init__(self)
        self.set_title("IceMe: shortcut editor")
        self.set_default_size(550, 250)
        self.connect("delete_event", self.close)

        self.preferences = preferences

        self.clist = GtkCList(2, ("Shortcut", "Command"))
        self.clist.set_selection_mode(SELECTION_BROWSE)
        self.clist.set_column_width(0, 100)
        self.clist.set_column_width(1, 200)
        self.clist.connect("select-row", self.on_select_row)
        self.clist.connect("click-column", self.on_click_column)
        self.clist.show()
        self.clist_sort_col  = -1
        self.clist_sort_type = SORT_ASCENDING

        scr = GtkScrolledWindow()
        scr.set_policy(POLICY_AUTOMATIC, POLICY_AUTOMATIC)
        scr.add(self.clist)
        scr.show()

        cmd_label = GtkLabel("Command:")
        cmd_label.set_alignment(1.0, 0.5)
        cmd_label.show()

        self.cmd = GtkEntry()
        self.cmd.connect("changed", self.on_cmd_changed)
        self.cmd.show()

        cmd_iconname = os.path.join(SCRIPT_DIR, "pixmaps", "open.xpm")
        cmd_icon = GtkPixmap(self, cmd_iconname, None)
        cmd_icon.show()
        cmd_button = GtkButton()
        cmd_button.add(cmd_icon)
        cmd_button.connect("clicked", self.on_cmd_button_clicked)
        cmd_button.show()

        cmd_hbox = GtkHBox(FALSE, 5)
        cmd_hbox.pack_start(cmd_label, expand=FALSE, fill=FALSE)
        cmd_hbox.pack_start(self.cmd)
        cmd_hbox.pack_start(cmd_button, expand=FALSE, fill=FALSE)
        cmd_hbox.show()

        self.flag_alt   = GtkCheckButton("Alt")
        self.flag_ctrl  = GtkCheckButton("Ctrl")
        self.flag_shift = GtkCheckButton("Shift")
        self.flag_meta  = GtkCheckButton("Meta")
        self.flag_super = GtkCheckButton("Super")
        self.flag_hyper = GtkCheckButton("Hyper")
        self.flag_alt.connect("toggled", self.on_flag_toggled)
        self.flag_ctrl.connect("toggled", self.on_flag_toggled)
        self.flag_shift.connect("toggled", self.on_flag_toggled)
        self.flag_meta.connect("toggled", self.on_flag_toggled)
        self.flag_super.connect("toggled", self.on_flag_toggled)
        self.flag_hyper.connect("toggled", self.on_flag_toggled)
        self.flag_alt.show()
        self.flag_ctrl.show()
        self.flag_shift.show()
        self.flag_meta.show()
        self.flag_super.show()
        self.flag_hyper.show()

        self.key = GtkCombo()
        self.key.set_usize(100, -1)
        self.key.set_popdown_strings( ("Esc", "Enter", "Space", "BackSp",
            "Del", "F1", "F2", "F3", "F4", "F5", "F6", "F7", "F8", "F9",
            "F10", "F11", "F12",
            "A", "B", "C", "D", "E", "F", "G", "H", "I", "J", "K", "L", "M",
            "N", "O", "P", "Q", "R", "S", "T", "U", "V", "W", "X", "Y", "Z",
            "0", "1", "2", "3", "4", "5", "6", "7", "8", "9") )
        self.key.entry.connect("changed", self.on_key_changed)
        self.key.show()

        table = GtkTable(4, 3, FALSE)
        table.attach(cmd_hbox,        0, 4, 0, 1, yoptions=0)
        table.attach(self.flag_alt,   0, 1, 1, 2, yoptions=0)
        table.attach(self.flag_ctrl,  1, 2, 1, 2, yoptions=0)
        table.attach(self.flag_shift, 2, 3, 1, 2, yoptions=0)
        table.attach(self.flag_meta,  0, 1, 2, 3, yoptions=0)
        table.attach(self.flag_super, 1, 2, 2, 3, yoptions=0)
        table.attach(self.flag_hyper, 2, 3, 2, 3, yoptions=0)
        table.attach(self.key,        3, 4, 1, 3, yoptions=0)
        table.set_border_width(10)
        table.set_row_spacings(5)
        table.set_col_spacings(5)
        table.show()

        self.rightframe = GtkFrame("Command/Shortcut")
        self.rightframe.set_border_width(10)
        self.rightframe.add(table)
        self.rightframe.show()

        hpaned = GtkHPaned()
        hpaned.pack1(scr, TRUE, TRUE)
        hpaned.pack2(self.rightframe, FALSE, FALSE)
        hpaned.show()

        add = GtkButton("Add")
        add.connect("clicked", self.on_add_clicked)
        add.show()

        delete = GtkButton("Delete")
        delete.connect("clicked", self.on_delete_clicked)
        delete.show()

        close = GtkButton("Close")
        close.connect("clicked", self.close)
        close.show()

        self.vbox.add(hpaned)
        self.action_area.pack_start(add)
        self.action_area.pack_start(delete)
        self.action_area.pack_start(close)

        self.show()
        self.load()
        self.update_widgets()


    ########################################################################
    # misc. functions
    ########################################################################

    def load(self):
        self.clist.clear()
        keysfile = self.preferences.getIceWMFile("keys")
        if keysfile == None:
            return
        file = open(keysfile)
        lines = file.readlines()
        file.close()
        i = 0
        for line in lines:
            line = string.strip(line)
            if line[0] == "#":
                continue
            s = string.split(line)
            if s[0] != "key":
                sys.stderr.write('*** wrong tag in keys file "%s"\n'
                                 '*** line %d: %s\n' % (keysfile, i, line))
                continue
            keystr = s[1]
            if keystr[0]  == '"':
                keystr = keystr[1:]
            if keystr[-1] == '"':
                keystr = keystr[:-1]
            cmd_pos = string.find(line, s[1]) + len(s[1]) + 1
            cmd = string.strip(line[cmd_pos:])
            self.clist.append((keystr, cmd))
            i = i + 1
        if i:
            self.clist.columns_autosize()

    def save(self):
        f = None
        full_filename = self.preferences.getIceWMFile("keys", FALSE)
        home_filename = os.path.join(HOME_ICEWM, "keys")
        errmsgs = []
        if self.preferences.getIgnoreHomeOption():
            # try to save to the original location (should only work as root):
            try:
                f = open(full_filename, "w")
            except IOError, msg:
                errmsgs.append(
                    "You specified '--ignore-home', but I couldn't\n"
                    "write the keys file, probably because you\n"
                    "don't have write access. The error message was:\n"
                    "%s\n"
                    "I'll try writing to your $HOME.\n" % msg)
        if not f:
            # try to save in the home directory:
            try:
                if not os.path.isdir(HOME_ICEWM): os.mkdir(HOME_ICEWM)
                f = open(home_filename, "w")
            except IOError, msg:
                errmsgs.append(
                    "I couldn't write the keys file to your home dir.\n"
                    "The error message was:\n"
                    "%s\n"
                    "The file will not be saved!\n" % msg)
        if f:
            for i in range(0, self.clist.rows):
                keystr = self.clist.get_text(i, 0)
                cmd = self.clist.get_text(i, 1)
                f.write('key "%s" %s\n' % (keystr, cmd))
            f.close()
        if errmsgs:
            message("Error saving shortcuts", errmsgs, pixmap=ICON_ALERT)

    def close(self, x=None, y=None):
        self.hide()

    def update_widgets(self):
        if self.clist.rows == 0:
            self.rightframe.set_sensitive(FALSE)
        else:
            self.rightframe.set_sensitive(TRUE)
            self.cmd.set_text(self.clist.get_text(self.cur_row, 1))
            keystr = self.clist.get_text(self.cur_row, 0)
            self.flag_alt.set_active(string.find(keystr, "Alt+") > -1)
            self.flag_ctrl.set_active(string.find(keystr, "Ctrl+") > -1)
            self.flag_shift.set_active(string.find(keystr, "Shift+") > -1)
            self.flag_meta.set_active(string.find(keystr, "Meta+") > -1)
            self.flag_super.set_active(string.find(keystr, "Super+") > -1)
            self.flag_hyper.set_active(string.find(keystr, "Hyper+") > -1)
            pluspos = string.rfind(keystr, "+")
            key = keystr[pluspos+1:]
            self.key.entry.set_text(key)

    def update_clist_key_column(self):
        keystr = ""
        if self.flag_alt.get_active():   keystr = keystr + "Alt+"
        if self.flag_ctrl.get_active():  keystr = keystr + "Ctrl+"
        if self.flag_shift.get_active(): keystr = keystr + "Shift+"
        if self.flag_meta.get_active():  keystr = keystr + "Meta+"
        if self.flag_super.get_active(): keystr = keystr + "Super+"
        if self.flag_hyper.get_active(): keystr = keystr + "Hyper+"
        keystr = keystr + self.key.entry.get_text()
        self.clist.set_text(self.cur_row, 0, keystr)

    def add_shortcut(self, keystr, command):
        self.cur_row = self.clist.append((keystr, command))
        self.clist.select_row(self.cur_row, 0)
        self.clist.moveto(self.cur_row, 0, 1.0, 0.0)

    def remove_shortcut(self, row):
        self.clist.remove(row)
        self.update_widgets()


    ########################################################################
    # callbacks
    ########################################################################

    def on_select_row(self, clist, row, col, a):
        self.cur_row  = row
        self.update_widgets()

    def on_cmd_changed(self, cmdfield):
        self.clist.set_text(self.cur_row, 1, cmdfield.get_text())

    def on_key_changed(self, keyfield):
        self.update_clist_key_column()

    def on_flag_toggled(self, flag):
        self.update_clist_key_column()

    def on_cmd_button_clicked(self, button):
        command = GtkExtra.file_open_box()
        if command:
            self.cmd.set_text(command)

    def on_add_clicked(self, button):
        self.add_shortcut("None", "")

    def on_delete_clicked(self, button):
        self.remove_shortcut(self.cur_row)

    def on_click_column(self, clist, col):
        if col == 0:
            if self.clist_sort_col == 0:
                if self.clist_sort_type == SORT_ASCENDING:
                    self.clist_sort_type = SORT_DESCENDING
                else:
                    self.clist_sort_type = SORT_ASCENDING
            else:
                self.clist_sort_type = SORT_ASCENDING
            self.clist_sort_col = 0
        else:
            if self.clist_sort_col == 1:
                if self.clist_sort_type == SORT_ASCENDING:
                    self.clist_sort_type = SORT_DESCENDING
                else:
                    self.clist_sort_type = SORT_ASCENDING
            else:
                self.clist_sort_type = SORT_ASCENDING
            self.clist_sort_col = 1
        self.clist.set_sort_column(self.clist_sort_col)
        self.clist.set_sort_type(self.clist_sort_type)
        self.clist.sort()




def test():
    prefs = Preferences(sys.argv)
    k = KeyEditor(prefs)
    mainloop()


if __name__ == "__main__":
    test()

