/****************************************************************************
 *                                DFIO.h
 *
 * Author: Matthew Ballance
 * Desc:   
 *
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 *
 ****************************************************************************/
#ifndef INCLUDED_DFIO_H
#define INCLUDED_DFIO_H
#include <tcl.h>
#include "types.h"
#include "ivi_String.h"
#include "DFIOTrace.h"
#include "vector.h"
#include "HashTable.h"
#include <stdio.h>

class DFIOTrace;
    
/********************************************************************/
/** \class DFIO
 *  \brief Describes the public interface of a Dump-File IO object
 *
 *  DFIOs provide an interface to signal-data storage implementations
 ********************************************************************/
class DFIO {
   
    /****************************************************************
     * Public Methods
     ****************************************************************/
    public:

        /************************************************************/
        /** DFIO()
         *
         * \param interp Interpretter to install commands in
         * \param argc   count of arguments to constructor
         * \param argv   arguments to constructor. First two are
         *               mandatory:
         *               - argv[0] = "dfio"
         *               - argv[1] = <inst_name>
         ************************************************************/
        DFIO(
                Tcl_Interp           *interp,
                Uint32                argc,
                Char                **argv);
    
        /************************************************************/
        /** ~DFIO()
         ************************************************************/
        virtual ~DFIO();
    
        /************************************************************/
        /** setTime()
         *
         * Sets the time for all traces managed by this DFIO. This 
         * method is generally called via a method on the DFIOTrace
         * object
         ************************************************************/
        virtual void setTime(Uint32 time);
    
        /************************************************************/
        /** InstCmd()
         *
         * Implements the command interface to the DFIO object
         ************************************************************/
        virtual int InstCmd(int argc, const char **argv);

        /************************************************************/
        /** setTimeUnit()
         *
         * Sets the time unit (or resolution) of this DFIO.
         *
         * \param unit Time unit expressed as a power of 10
         * -  0 is seconds
         * - -3 is ms
         * - -6 is us
         * - -9 is ns
         * - -12 is ps
         * - etc ...
         ************************************************************/
        virtual void setTimeUnit(Int32 unit);

        /************************************************************/
        /** getTimeUnit()
         *
         * Retrieves the time unit for this DFIO
         ************************************************************/
        virtual Int32 getTimeUnit();

        /************************************************************/
        /** setUnitMultiplier()
         *
         ************************************************************/
        virtual void setUnitMultiplier(Int32 multiplier);

        /************************************************************/
        /** getUnitMultiplier()
         ************************************************************/
        virtual Int32 getUnitMultiplier();

        /************************************************************/
        /** getMaxTime()
         *
         * Returns the time of the max-time data-point
         ************************************************************/
        virtual Uint32 getMaxTime();

        /************************************************************/
        /** setMaxTime()
         *
         * Sets the maximum time. This is only called by DFIOTrace
         * object.
         ************************************************************/
        virtual void setMaxTime(Uint32 max);

        /************************************************************/
        /** setSeparateTraceMax()
         *
         * Configures whether the max-time for each trace is based
         * on the max-time of the DFIO or on the max time of the
         * trace itself. 
         *
         * Independent max-time calculation is used when the DFIO 
         * is used for the wave editor
         *
         * \param en Set to
         * - true if traces should determine their own max time
         * - false if traces should use the DFIO max time 
         *
         * DFIO defaults to 'false' - trace gives DFIO max-time 
         * as its own max time
         ************************************************************/
        virtual void setSeparateTraceMax(bool en) { d_separateTraceMax = en; }
    
        virtual Int32  addTrace(DFIOTrace  *newTrace) { 
            base_addTrace(newTrace); 
        }
    
        Int32          base_addTrace(DFIOTrace  *newTrace);
    
        virtual DFIOTrace *findTrace(Char *traceName) { 
            return base_findTrace(traceName); 
        }
    
        DFIOTrace *base_findTrace(Char *traceName);

        virtual DFIOTrace *newTrace(
                Char        *name,
                Uint32       rows,
                Uint32       msb,
                Uint32       lsb,
                Uint32       traceType) {
            return base_newTrace(name, rows, msb, lsb, traceType);
        }

        DFIOTrace *base_newTrace(
                Char        *name,
                Uint32       rows,
                Uint32       msb,
                Uint32       lsb,
                Uint32       traceType);
    
        virtual Int32  openFile(Char *fileName, Char *mode) { 
            return base_openFile(fileName, mode);
        }
    
        Int32          base_openFile(Char *fileName, Char *mode);
    
        enum {
            OM_Read       = 1,
            OM_Write      = 2, 
            OM_ReadWrite  = 3
        };
    	
    /****************************************************************
     * Protected Methods
     ****************************************************************/
    protected:

        /************************************************************/
        /** base_setTime()
         *
         * Performs the base functionality of setTime(). Any class
         * that extends setTime() should call this function from
         * within the setTime() method.
         ************************************************************/
        void base_setTime(Uint32  time);

        /************************************************************/
        /** base_setTimeUnit()
         *
         * Performs the base functionality of setTimeUnit. Classes
         * that override setTimeUnit() must call this method from
         * within their implementation of setTimeUnit().
         ************************************************************/
        void base_setTimeUnit(Int32 unit);

        /************************************************************/
        /** Delete()
         *
         * Requests that this DFIO be deleted. Since the DFIO will
         * usually be deleted from within the command function, 
         * Delete() waits until execution isn't within the DFIO 
         * object before it is deleted.
         ************************************************************/
        void Delete();


    /****************************************************************
     * Private Methods
     ****************************************************************/
    private:

        /************************************************************
         * Delete()
         *
         * Actually performs the work of deleting this object...
         ************************************************************/
        static void Delete(voidPtr dfio) {
            delete (DFIO *)dfio;
        }

        /************************************************************
         * InstCmd()
         ************************************************************/
        static int InstCmd(ClientData ud, Tcl_Interp *interp,
                int argc, const char **argv) {
            return ((DFIO *)ud)->InstCmd(argc, argv);
        }


    /****************************************************************
     * Public Members
     ****************************************************************/
    public:

       /****************************************************
        * Member Variables
        ****************************************************/
        Vector<DFIOTrace>        *traces;
        HashTable<DFIOTrace>     *signalHash;
        FILE                     *rfp; 
        FILE                     *wfp; 
        Uint32                    readPos;
        Uint32                    writePos;
        Uint32                    openMode;
        Uint32                    currentTime;
        Uint64                    currentTime64;
        Uint32                    maxTime;
        Int32                     timeUnit;
        Int32                     timeMult;
        bool                      d_separateTraceMax;
        String                    d_instName;
        Tcl_Interp               *d_interp;
        bool                      d_ok;
};
    
#endif /* INCLUDED_DFIO_H */
    
