/*****************************************************************************
 *                               String.cc
 * Author: Matthew Ballance
 * Desc:   Simple string class..
 *
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 *
 *****************************************************************************/
#include "ivi_String.h"
#include <stdio.h>
#include <string.h>

#define ilstrcpy(dp, sp) { \
    Uint32     i = 0; \
    do { \
        dp[i] = sp[i]; \
    } while(sp[i++]); \
}

/***********************************************************
 * String(char)
 ***********************************************************/
String::String(const char *str)
{
    if (str) {
        str_len = str_max = strlen(str);
        this->str = new char[str_len+1];
        ilstrcpy(this->str, str);
    } else {
        str_len = str_max = 0;
        this->str = 0;
    }
}

/***********************************************************
 * String(char, len)
 ***********************************************************/
String::String(Char *str, Uint32 len)
{
    str_len = str_max = len;
    this->str = new char[len+1];
    memcpy(this->str, str, len);
    this->str[len] = 0;
}

/***********************************************************
 * String(void)
 ***********************************************************/
String::String(void)
{
    str_len = str_max = 0;
    str = 0;
}

/*********************************************************** 
 * String(String *)
 ***********************************************************/
String::String(String *tmpl)
{
    char *tmp_str = tmpl->value();

    this->str = new char[tmpl->length()+1];
    ilstrcpy(this->str, tmp_str);
    this->str_len = this->str_max = tmpl->length();
}

/*********************************************************** 
 * String(String &)
 ***********************************************************/
String::String(const String &cpy)
{
    char *tmp_str = cpy.value();

    this->str = new char[cpy.length()+1];
    if (cpy.length()) {
        ilstrcpy(this->str, tmp_str);
    } else {
        this->str[0] = 0;
    }
    this->str_len = this->str_max = cpy.length();
}
/*********************************************************** 
 * String(length)
 ***********************************************************/
String::String(Uint32 len)
{
    str_len = len;
    str_max = len;

    if (len) {
        this->str = new char[len+1];
        this->str[0] = 0;
    }
}

/***********************************************************
 * ~String()
 ***********************************************************/
String::~String(void)
{
    if (str) {
        delete[] str;
    }
}

/***********************************************************
 * String =
 ***********************************************************/
void String::operator = (const String &rhs)
{
    /**** Reallocate if necessary ****/
    if (rhs.str_len > str_max) {
        if (str) {
            delete[] str;
        }
        str = new Char[rhs.str_len+1];
        str_max = rhs.str_len;
    }

    str_len = rhs.str_len;
    ilstrcpy(str, rhs.str);
}

/***********************************************************
 * String =
 ***********************************************************/
void String::operator = (const Char *str_in)
{
    Uint32    in_len = strlen(str_in);

    /**** Reallocate if necessary ****/
    if (in_len > str_max) {
        if (str) {
            delete[] str;
        }
        str = new Char[in_len+1];
        str_max = in_len;
    }

    str_len = in_len;
    ilstrcpy(str, str_in);
}

/***********************************************************
 * operator == 
 ***********************************************************/
int String::operator == (const String &rhs) const
{
    return String::equal(str, rhs.str);
}

/***********************************************************
 * operator ==
 ***********************************************************/
int String::operator == (const char *str) const
{
    return String::equal(str);
}

/***********************************************************
 * operator +=
 ***********************************************************/
void String::operator += (const Char *add)
{
    Uint32  len = strlen(add);
    Char   *tmp;

    if ((len+str_len) > str_max) {
        tmp = new Char[(len+str_len)*2];
        if (str) {
            ilstrcpy(tmp, str);
            delete[] str;
        }
        str = tmp;
    }

    memcpy(&str[str_len], add, len+1);
    str_len += len;
}

/***********************************************************
 * operator +=
 ***********************************************************/
void String::operator += (char ch)
{
    if ((str_len+1) > str_max) {
        char *tmp = new Char[(str_len+1)*2];
        ilstrcpy(tmp, str);
        delete [] str;
        str = tmp;
    }

    str[str_len++] = ch;
    str[str_len] = 0;
}

/***********************************************************
 * equal(String)
 ***********************************************************/
int String::equal(String *str) const
{
  return str->equal(this->str);    
}

/***********************************************************
 * equal(char, strlen)
 ***********************************************************/
int String::equal(const char *str, Uint32 len) const
{
    Uint32 i;
    if (str_len != len) return 0;
 
    for (i=0; i<str_len; i++) {
        if (str[i] != this->str[i]) return 0;
    }

    return 1; 
}

/***********************************************************
 * equal(char)
 ***********************************************************/
int String::equal(const char *str) const
{
    Uint32 o_len = strlen(str);
    return equal(str, o_len);
}

/***********************************************************
 * equal(char *, char *)
 ***********************************************************/
int String::equal(const char *str1, const char *str2)
{
    Uint32 i = 0, ret = 0;

    if ((str1 == 0) || (str2 == 0)) {
        return 0;
    }

    while ((str1[i] != 0) && (str2[i] != 0) &&
           (str1[i] == str2[i])) {
        i++;
    }

    if (!str1[i] && !str2[i]) {
        ret = 1;
    }

    return ret;
}

/***********************************************************
 * strchr()
 ***********************************************************/
Int32 String::strchr(Char ch, Uint32 start_idx)
{
    Uint32 idx = start_idx;

    while (str[idx] && (idx < str_len)) {
        if (str[idx] == ch) {
            return idx;
        } else {
            idx++;
        }
    }

    return -1;
}

/***********************************************************
 * strchr()
 ***********************************************************/
Int32 String::strchr(Char ch)
{
    return strchr(ch, 0);
}

/***********************************************************
 * strrchr()
 ***********************************************************/
Int32 String::strrchr(Char ch)
{
    Uint32 idx = 0;
    Int32  ret_idx = -1;

    while (str[idx] && (idx <= str_len)) {
        if (str[idx] == ch) {
            ret_idx = idx;
        } 
        idx++;
    }
    return ret_idx;
}

/***********************************************************
 * chkSize()
 ***********************************************************/
void String::chkSize(Uint32 new_max)
{
    Char *tmp;

    if (new_max > str_max) {
        tmp = str;
        str = new Char[new_max+1];
        if (tmp) {
            memcpy(str, tmp, str_len+1);
            delete[] tmp;
        } else {
            str[0] = 0;
        }
        str_max = new_max;
    }
}

/***********************************************************
 * setValue
 ***********************************************************/
void String::setValue(char *str, Int32 n_len)
{
    if (n_len < 0) {
        n_len = strlen(str);
    }

    if (n_len >  str_max) {
        delete[] this->str;
        this->str = new char[n_len+1];
        str_max   = n_len+1;
    }

   str_len = n_len;
 
   ilstrcpy(this->str, str);
}

/***********************************************************
 * setLen()
 ***********************************************************/
void String::setLen(Int32 len)
{
    Char    *tmp;

    if (len < 0) {
        this->str_len = strlen(str);
    } else {
        if (str_max < len) {
            tmp = str;
            str = new char[len+1];
            memcpy(str, tmp, str_len+1);
        } 

        this->str_len = len;
    }
}

/***********************************************************
 * match()
 ***********************************************************/
Int32 String::match(char *pattern)
{
    return match(this->str, pattern);
}


#define	CHAR		0177
#define	QUOTE 		0200	/* Eighth char bit used for 'ing */
#define	TRIM		0177	/* Mask to strip quote bit */
/***********************************************************
 * match()
 * Adapted from csh's pattern-matching code. License follows
 * function
 ***********************************************************/
Int32 String::match(char *string, char *pattern)
{
    register char stringc, patternc;
    int     match;
    char    rangec;

    for (;; ++string) {
	stringc = *string & TRIM;
	patternc = *pattern++;
	switch (patternc) {
	case 0:
	    return (stringc == 0);
	case '?':
	    if (stringc == 0)
		return (0);
	    break;
	case '*':
	    if (!*pattern)
		return (1);
	    while (*string)
		if (String::match(string++, pattern))
		    return (1);
	    return (0);
	case '[':
	    match = 0;
	    while (rangec = *pattern++) {
		if (rangec == ']')
		    if (match)
			break;
		    else
			return (0);
		if (match)
		    continue;
		if (rangec == '-' && *(pattern - 2) != '[' && *pattern != ']') {
		    match = (stringc <= (*pattern & TRIM) &&
			     (*(pattern - 2) & TRIM) <= stringc);
		    pattern++;
		}
		else
		    match = (stringc == rangec);
	    }
	    if (rangec == 0) {
                return -1;
                /*
		stderror(ERR_NAME | ERR_MISSING, ']');
                 */
            }
	    break;
	default:
	    if ((patternc & TRIM) != stringc)
		return (0);
	    break;

	}
    }
}
/*-
 * Copyright (c) 1980, 1991 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */


