/*****************************************************************************
 *                               WaveWidget.h
 *
 * Author: Matthew Ballance
 * Desc:   Implements a widget for displaying waveforms. The WaveWidget
 *         utilizes the SigDB to store the signal data...
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 *
 *****************************************************************************/
#ifndef INCLUDED_WAVE_WIDGET_H
#define INCLUDED_WAVE_WIDGET_H
#include "types.h"
#include "XObjWidgetBase.h"
#include "CmdSwitcher.h"
#include "ivi_String.h"
#include <tk.h>
#include <stdlib.h>
#include <string.h>
#include "CursorMgr.h"
#include "LogMgr.h"
#include "LogInstance.h"
#include <math.h>
#include <unistd.h>

/*****************************************************************
 * Defines
 *****************************************************************/
#define SET_FLAG(flg) d_RedrawFlags |= (flg)
#define CLR_FLAG(flg) d_RedrawFlags &= ~(flg)
#define IS_FLAG_SET(flg) (d_RedrawFlags & (flg))

typedef enum {
   WWC_Redraw = 1,
   WWC_YScrollCmd,
   WWC_XScrollCmd,
   WWC_Xview,
   WWC_Yview,
   WWC_TimeBounds,
   WWC_SetLimits,
   WWC_GetLimits,
   WWC_PixToTime,
   WWC_TimeToPix,
   WWC_Cget,
   WWC_Print,
   WWC_PageLayout,
   WWC_TimePerPage,
   WWC_PixPerTime,
   WWC_UpdateSigVals,
   WWC_NumCmds
} WW_Cmds;

static ObjWidgetBase::CmdStruct ww_cmds[] = {
   {"redraw",             WWC_Redraw      },
   {"-yscrollcommand",    WWC_YScrollCmd  },
   {"-xscrollcommand",    WWC_XScrollCmd  },
   {"xview",              WWC_Xview       },
   {"yview",              WWC_Yview       },
   {"time_bounds",        WWC_TimeBounds  },
   {"set_limits",         WWC_SetLimits   },
   {"get_limits",         WWC_GetLimits   },
   {"pixels_to_time",     WWC_PixToTime   },
   {"time_to_pixels",     WWC_TimeToPix   },
   {"cget",               WWC_Cget        },
   {"print",              WWC_Print       },
   {"page_layout",        WWC_PageLayout  },
   {"time_per_page",      WWC_TimePerPage },
   {"pix_per_time",       WWC_PixPerTime  },
   {"update_sigvals",     WWC_UpdateSigVals },
   {"",                   0               }
};

class TreeWidget;
class SdbReader;
class DFIOValChg;
class SdbSignal;
class PsPixmapObj;

class LogInstance;
class LogInstLevel;

class WaveWidget : public XObjWidgetBase {

    /****************************************************************
     * Public Methods
     ****************************************************************/
    public:

        /************************************************************
         * WaveWidget()
         ************************************************************/
        WaveWidget(
                Tcl_Interp  *interp,
                Uint32       argc,
                Tcl_Obj     *const objv[]);

        /************************************************************
         * ~WaveWidget()
         ************************************************************/
        virtual ~WaveWidget();

        /************************************************************
         * CreateInst()
         ************************************************************/
        static int CreateInst(
                ClientData        clientData,
                Tcl_Interp       *interp,
                int               objc,
                Tcl_Obj          *const objv[]);

        /************************************************************
         * getOptionSpec()
         ***********************************************************/
        static Tk_OptionSpec *getOptionSpec();

        /************************************************************
         * getGCs()
         ************************************************************/
        GCObj *getGCs() { return (d_useXGC)?d_gcObjs:d_psGcObjs; }

        void ComputeGeometry(void);
        virtual void IdleProc(void);

        int  Configure(int argc, char **argv, int flags);
        int  Configure(Uint32 objc, Tcl_Obj *const objv[]);

        virtual void EventProc(XEvent *eventPtr);
        void UpdateScrollbars(void);
        Int32 Xview(Uint32 objc, Tcl_Obj *const objv[]);
        Int32 Yview(Uint32 objc, Tcl_Obj *const objv[]);
 
        virtual int InstCmd(Uint32 argc, Tcl_Obj *const objv[]);

        static int cursorEventCB(
                ClientData        clientData,
                Tcl_Interp       *interp,
                int               ojbc,
                Tcl_Obj          *const objv[]);

    typedef Uint64    TraceSpec;

    static Tk_OptionTable d_OptTab;

    /****************************************************************
     * PaintFunc()
     *
     * This is the function-type for pre- and post-paint callouts
     ****************************************************************/
    typedef void (*PaintFunc)(
            voidPtr        user_data,
            PixmapObj     &pixmap,
            TraceSpec     &startTrace,
            TraceSpec     &endTrace);

    /******************************************************************
     * DrawTimeScale
     ******************************************************************/
    void DrawTimeScale(
           PixmapObj    &pixmap,
           Uint32        startTime,
           Uint32        endTime,
           Uint32        pmStartOffset,
           Uint32        pixHeight,
           Uint32        drawTags);

    virtual int set_PreOverlayFunc(WaveWidget::PaintFunc pf, voidPtr ud) {
        return d_PreOverlay(pf, ud);
    }

    virtual int set_PostOverlayFunc(WaveWidget::PaintFunc pf, voidPtr ud) {
        return d_PostOverlay(pf, ud);
    }

    /****************************************************************
     * Private Methods
     ****************************************************************/
    private:

        /************************************************************
         * SetupGraphics()
         */
        /**
         * Configures the graphics objects based on the color-scheme
         * option
         ************************************************************/
        void SetupGraphics();

        /************************************************************
         * getSchemeValue()
         */
        /** Gets the specified value from the current scheme. If 
         *  the value isn't set in the scheme, a default value is 
         *  returned
         ************************************************************/
        const char *getSchemeValue(const char *val_name);

        void RepaintBackingPixmaps(void);
        void UpdatePixmaps(void);
        void DrawSelectionBox(PixmapObj &pixmap);

        /************************************************************
         * FormatTime()
         ************************************************************/
        void FormatTime(char *buf, Uint32 time);

        /************************************************************
         * SetupBindings()
         ************************************************************/
        void SetupBindings();

        /************************************************************
         * SetupPsGraphics()
         ************************************************************/
        void SetupPsGraphics();

        /************************************************************
         * GetPageAvailableWidth()
         ************************************************************/
        Uint32 GetPageAvailableWidth(PsPixmapObj *pixmap);

        /************************************************************
         * GetPageAvailableHeight()
         ************************************************************/
        Uint32 GetPageAvailableHeight(PsPixmapObj *pixmap);

        /************************************************************
         * CalcPagePixPerTime()
         ************************************************************/
        int CalcPagePixPerTime(int argc, Tcl_Obj *const objv[]);

        /************************************************************
         * CalcTimePerPage()
         ************************************************************/
        int CalcTimePerPage(int argc, Tcl_Obj *const objv[]);

        /************************************************************
         * PrintPage()
         ************************************************************/
        void PrintPage(
                PsPixmapObj        *pm,
                Uint32              start_time,
                Uint32              end_time,
                Uint32              start_ypos,
                Uint32              ysize,
                Double              pixPerTime,
                Uint32              page_num,
                Uint32              page_total);

        /************************************************************
         * PrintPageBorder()
         ************************************************************/
        void PrintPageBorder(
                PsPixmapObj        *pm,
                Uint32              page_num,
                Uint32              page_total,
                Uint32             &head_border,
                Uint32             &bottom_border);

        /************************************************************
         * Print()
         ************************************************************/
        int Print(int objc, Tcl_Obj *const objv[]);

        /************************************************************
         * PageLayout()
         ************************************************************/
        int PageLayout(int objc, Tcl_Obj *const objv[]);

        /************************************************************
         * PageLayout()
         ************************************************************/
        void PageLayout(
                PsPixmapObj  *pm,
                Uint32        startTime,
                Uint32        endTime,
                Double        pixPerTime,
                Uint32       &pages_w,
                Uint32       &pages_h);

        /************************************************************
         * DrawWaveProper()
         ************************************************************/
        void DrawWaveProper(
                PixmapObj         &pixmap,
                Uint32             startPixY,
                Uint32             endPixY,
                Uint32             startTime,
                Uint32             endTime,
                Double             pixPerTime);

        /************************************************************
         * UpdateSigValTree()
         ************************************************************/
        void UpdateSigValTree(Uint64 &time);

        /************************************************************
         * getCursorPaneHeight()
         ************************************************************/
        Uint32 getCursorPaneHeight();

        /************************************************************
         * ButtonEventProc
         ************************************************************/
        static int ButtonEventProc(
                ClientData                clientData,
                Tcl_Interp               *interp,
                int                       objc,
                Tcl_Obj                  *const objv[]) {
            return ((WaveWidget *)clientData)->ButtonEventProc(objc, objv);
        }

        /************************************************************
         * ButtonEventProc()
         ************************************************************/
        int ButtonEventProc(
                int                objc,
                Tcl_Obj           *const objv[]);

        void UpdateXScroll(void);

        enum RepaintFlags {
            RepaintExpose    = RedrawNumFlags,
            RepaintB1Down    = (RedrawNumFlags << 1),
            RepaintB2Down    = (RedrawNumFlags << 2),
            RepaintB3Down    = (RedrawNumFlags << 3),
            RepaintMouseDrag = (RedrawNumFlags << 4),
            RepaintBRelease  = (RedrawNumFlags << 5),
            RepaintScrollX   = (RedrawNumFlags << 6),
            RepaintScrollY   = (RedrawNumFlags << 7),
            RepaintCursor    = (RedrawNumFlags << 8)
        };

#define REPAINT_FLAGS (RepaintScrollX|RepaintScrollY)
#define BUTTONS_DOWN  (RepaintB1Down|RepaintB2Down|RepaintB3Down)

    /****************************************************************
     * PaintFuncStruct
     ****************************************************************/
    struct PaintFuncStruct {
        PaintFuncStruct() : d_pf(0), d_user_data(0) { ; }

        PaintFuncStruct(WaveWidget::PaintFunc pf, voidPtr user_data) {
            d_pf        = pf;
            d_user_data = user_data;
        }

        int operator () (WaveWidget::PaintFunc pf, voidPtr user_data) {

            /**** If the cb is already set, it's an error... ****/
            if (d_pf && pf) {
                return -1;
            }

            d_pf        = pf;
            d_user_data = user_data;
            return 0;
        }

        void invoke(PixmapObj &pm, TraceSpec &start, TraceSpec &end) {
            if (d_pf) {
                d_pf(d_user_data, pm, start, end);
            }
        }

        void operator () (PixmapObj &pm, TraceSpec &start, TraceSpec &end) {
            if (d_pf) {
                d_pf(d_user_data, pm, start, end);
            }
        }

        private:
            WaveWidget::PaintFunc    d_pf;
            voidPtr                  d_user_data;
    };

    /****************************************************************
     * Private Data
     ****************************************************************/
    private:

        enum {
            TC_VectExp1Color,
            TC_VectExp0Color,
            TC_VectExpXColor,
            TC_VectExpZColor,
            TC_VectExpChgColor,

            TC_VectBinColor,
            TC_VectChgColor,
            TC_VectXColor,
            TC_VectZColor,

            TC_Scalar1Color,
            TC_Scalar0Color,
            TC_ScalarXColor,
            TC_ScalarZColor,
            TC_ScalarChgColor,

            TC_CursorUnselColor,
            TC_CursorSelColor,
            TC_CursorLockUnselColor,
            TC_CursorLockSelColor,

            TC_ScaleColor,
            TC_TimescaleTextColor,

            TC_SelBoxColor,
            TC_CursorMeasureColor,
            TC_SeparatorColor,

            TC_NumColors
        };

        struct WaveConfig {
            char            *colorScheme;
            Tk_3DBorder      background;
            Tk_Font          tkfont;
            XColor          *traceColors[TC_NumColors];
            XColor          *colors[TC_NumColors];
            Uint32           traceHeight; /*Total height of trace and padding.*/
            Uint32           cursorWidth;
            Int32            cursorSelWidth;
            Char            *sdbrName;
            Char            *sigTreeName;
            Char            *sigValTreeName;
            Char            *cursorMode;
            Char            *yscrollcommand;
            Char            *xscrollcommand;
            Char            *progressCommand;
            Char            *cursorMoveCmd;
            Char            *selectBoxCmd;
            Tk_Cursor        activeCursor;
            Tk_Cursor        busyCursor;
            Int32            trace_pady;
            Int32            cursorpane_height;
            Tk_Cursor        cursor;
        };

        WaveConfig       d_config;
        XPixmapObj       d_backingPixmap;
        XPixmapObj       d_displayPixmap;
        Uint32           d_backingPixmapValid;


        String           strVal;
        Tcl_Command      waveWidgetCmd;

        PaintFuncStruct  d_PreOverlay;
        PaintFuncStruct  d_PostOverlay;

        ClientData       SDBRSigUpdateCb;
        ClientData       SDBRSigListCb;
        ClientData       SDBRSigDataUpdateCb;
        ClientData       d_cursorCB;
        ClientData       d_cursorListCB;

    
   /**** Graphics Contexts ****/
    GCObj            d_gcObjs[TC_NumColors];
    GCObj            d_psGcObjs[TC_NumColors];
    bool             d_useXGC;

    /***** Line params... *****/
    Uint32           dy;

    Double           pixelsPerTimeUnit;      /*** Number of pixels to 
                                              *** display per unit of time
                                              ***/

    Uint32           widgetHeight;           /*** Height of the display
                                              *** pixmap
                                              ***/
    Uint32           widgetWidth;            /*** Width of the display
                                              *** pixmap 
                                              ***/

    Uint32           pixmapHeight;           /*** ***/
    Uint32           pixmapWidth;            /*** ***/

    Uint32           pixmapStartTime;        /*** First time in the backing
                                              *** pixmap
                                              ***/
    Uint32           pixmapEndTime;          /*** Last time in the backing
                                              *** pixmap
                                              ***/
    Uint32           pixmapStartPixPos;      /*** First virtual pixel in 
                                              *** pixmap
                                              ***/
    Uint32           pixmapEndPixPos;        /*** Last virtual pixel in 
                                              *** pixmap
                                              ***/

    Uint64           d_startTrace;           /*** First trace in the wave
					      *** window
					      ***/
    Uint64           d_endTrace;             /*** Last trace in the wave
					      *** window
					      ***/

    Uint32           pixmapXReallocThresh;   /***  ***/
    Uint32           pixmapYReallocThresh;   /***  ***/

    Uint32           displayStartTime;       /*** Current startTime     ***/
    Uint32           displayEndTime; 
    Uint32           newDisplayStartTime;    /*** Requested startTime   ***/
    Uint32           newDisplayEndTime;
   
    Uint32           displayStartPos;        /*** Current startPixelY   ***/
    Uint32           newDisplayStartPos;     /*** Requested startPixelY ***/

    Uint32           d_runningMax;

    SdbReader       *sdbr;
    TreeWidget      *d_sigTree;
    TreeWidget      *d_sigValTree;

    Uint32           button_down;

    Int32            sel_xi, sel_yi;
    Int32            sel_xf, sel_yf;

    Uint32           totalTime;             /*** Max time on the trace ***/



    Uint32          *yTmpStorage;
    Uint32           yTmpStorageSize;

    SdbrCursor      *grabbedCursor;

    Char            *scratchBuf;
    Uint32           scratchBufLen;

    Uint32           timeScaleValid;
    Uint32           timeScale;

    LogInstance     *log_inst;
    LogInstLevel     log_high;
    LogInstLevel     log_med;
    LogInstLevel     log_low;

    Uint32           xscroll_valid;

    Uint32           d_scaleTextHeight;

    void DrawScalarTrace(SdbSignal *signal, PixmapObj &pixmap, Uint32 x, 
            Uint32 y, Vector<DFIOValChg> *chgs, Uint32 scale,
            Uint32 startTime, Uint32 endTime);
    void DrawVectorTrace(
        SdbSignal *signal, PixmapObj &pixmap, Uint32 x, Uint32 y, 
        Vector<DFIOValChg> *chgs, Uint32 scale, 
        Uint32 startTime, Uint32 endTime);

    void DrawCursors(CursorMgr *mgr, PixmapObj &pixmap,
            Uint32     drawMain,
            Uint32     tagCenter,
            Uint32     drawTag);

    void CursorB1Action(Int32 x, Int32 y, Int32 mod);

    /****************************************************************
     * SDBRSigUpdateCB()
     ****************************************************************/
    static int SDBRSigUpdateCB(
        ClientData        clientData,
        Tcl_Interp       *interp,
        int               objc,
        Tcl_Obj          *const objv[]);

    /****************************************************************
     * SDBRSigDataUpdateCB()
     ****************************************************************/
    static int SDBRSigDataUpdateCB(
        ClientData        clientData,
        Tcl_Interp       *interp,
        int               objc,
        Tcl_Obj          *const objv[]);

    /******************************************************************
     * UpdateTimeScale()
     *
     * Based on the current timescale, find out the distance between
     * timescale bars...
     *
     * Inputs:
     * - pixmapStartTime
     *   Start time for the pixmap
     *
     * - pixmapEndTime
     *   End time for the pixmap
     *
     * - pixmapWidth
     *   Width in pixels of the pixmap
     *
     * Return:
     * - Returns the distance between bars
     ******************************************************************/
     Uint32 WaveWidget::UpdateTimeScale(
            Uint32        pixmapStartTime,
            Uint32        pixmapEndTime,
            Uint32        pixmapWidth);



   /*******************************************************************
    * PixelsToTime()
    *******************************************************************/
    inline Uint32 PixelsToTime(Uint32    pixels) {
        Double dpix = (Double)pixels;
        Double ret =  dpix / pixelsPerTimeUnit;
        Double c = ceil(ret);

        if ((c-ret) < 0.5) {
            return (Uint32)c;
        } else {
            return (Uint32)floor(c);
        }
    }

    inline Uint32 TimeToPixels(Uint32   time) {
        return (Uint32)(((Double)time)*pixelsPerTimeUnit);
    }

   /*******************************************************************
    * Center()
    *******************************************************************/
    void Center(
        Uint32        displayBegin,
        Uint32        displaySize,
        Uint32        pixmapSize,
        Uint32       *pixmapBegin,
        Uint32       *pixmapEnd);

   /******************************************************************
    * CalcStartEndPos()
    * 
    * Calculates the starting and ending trace/subtrace pairs. 
    * Routine also suggests a starting position.
    ******************************************************************/
    void CalcStartEndPos(
        Uint64       *startPos,
        Uint64       *endPos,
        Uint32       *startY);

   /******************************************************************
    * DrawVectorTraceMaster()
    ******************************************************************/
    Uint32 DrawVectorTraceMaster(
        SdbSignal             *signal,
        PixmapObj             &pixmap, 
        Uint32                 xOff,
        Uint32                 yOff,
        Uint64                 traceSpec,
        Vector<DFIOValChg>    *chgs,
        Uint32                 scale,
        Uint32                 startTime,
        Uint32                 endTime);


    /******************************************************************
     * DrawExpandedTrace()
     ******************************************************************/
     Uint32 DrawExpandedTrace(
        SdbSignal           *signal,
        PixmapObj           &pixmap,
        Uint32               xOff,
        Uint32               yOff,
        Uint64               traceSpec,
        Vector<DFIOValChg>  *chgs,
        Uint32               scale,
        Uint32               startTime,
        Uint32               endTime);

    /******************************************************************
     * SendSelBoxCmd()
     ******************************************************************/
     void SendSelBoxCmd(Uint32 xi, Uint32 yi, Uint32 xf, Uint32 yf);

    /******************************************************************
     * TimeBounds()
     * Returns the time bounding box (lb, ub)
     ******************************************************************/
     Int32 TimeBounds(void);

       /******************************************************************
        * SetLimits
        ******************************************************************/
        Int32 SetLimits(Uint32 objc, Tcl_Obj *const objv[]);
        
       /******************************************************************
        * getTraceSep()
        ******************************************************************/
        Uint32 getTraceSep() { 
            return d_config.traceHeight+(2*d_config.trace_pady); }

       /******************************************************************
        * setWindow()
        ******************************************************************/
        void setWindow();

        enum OptFlags {
            Opt_Graphics    = 0x0001,
            Opt_Sdbr        = 0x0002,
            Opt_SigTree     = 0x0004,
            Opt_SigValTree  = 0x0008,
            Opt_ColorScheme = 0x0010
        };

        Tcl_Obj            *d_cbArgs[16];
        CallbackList       *d_WinChangeCBList;

        Uint64              d_sigValTime;
        bool                d_sigValNeedUpdate;
};

#endif /* INCLUDED_WAVE_WIDGET_H */


