/***************************************************************************
                       IPTChain.cpp  -  description
                          -------------------
 begin                : Mon Jan 28 2002
 copyright            : (C) 2002 by Christian Hubinger
 email                : a9806056@unet.univie.ac.at
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "qptrlist.h"
#include "qstring.h"
#include "qmessagebox.h"

#include <stdlib.h>
#include <iostream.h>
// kde includes
#include <kdebug.h>
#include <klocale.h>
#include <kmessagebox.h>
//my includes
#include "iptchain.h"
#include "iptrule.h"
#include "iptable.h"
#include "kmferror.h"

IPTChain::IPTChain( const QString& name, IPTable* table, bool buildin ) {
	kdDebug() << "IPTChain::IPTChain(const QString& name,IPTable* table,bool buildin)" << endl;
	m_err = new KMFError();

	// Initialize members
	m_name = "UNDEFINED";
	setName( name );
	setTable( table );
	is_build_in_chain = false;
	has_default_target = false;
	setBuildIn( buildin );
	enable_log = false;
	m_ruleset.setAutoDelete( true );
	m_cmd_default_target = "UNDEFINED";
	m_cmd_chain_definition = "UNDEFINED";
	m_log_limit = "UNDEFINED";
	m_log_prefix = "UNDEFINED";
	m_log_burst = "UNDEFINED";
	setUsed( true );
}

IPTChain::~IPTChain() {
	kdDebug() << "\n\nIPChain::~IPTChain()" << endl;
	m_ruleset.clear();
	delete m_err;
}

void IPTChain::setName( const QString& name ) {
	m_name = name;
}
void IPTChain::setDefaultTarget( const QString& target ) {
	//	kdDebug() << "void IPTChain::setDefaultTarget(const QString& target) " << endl;
	m_default_target = target;
	has_default_target = true;
}

void IPTChain::setTable( IPTable* table ) {
	m_table = table;
}

IPTable* IPTChain::table() const {
	return m_table;
}

QPtrList<IPTRule>& IPTChain::chainRuleset() const {
	QPtrList<IPTRule>* ret_list = new QPtrList<IPTRule>;
	*ret_list = m_ruleset;
	return *ret_list;
}

void IPTChain::setBuildIn( bool build_in ) {
	is_build_in_chain = build_in;
	if ( is_build_in_chain ) {
		const QString target = "DROP";
		setDefaultTarget( target );
	}
}

void IPTChain::hasCustomDefaultTarget( bool has ) {
	if ( is_build_in_chain )
		has_default_target = true;
	else
		has_default_target = has;
}


void IPTChain::setUsed( bool valid ) {
	is_used = valid;
}

KMFError* IPTChain::addRule( IPTRule& rule ) {
	kdDebug() << "bool IPTChain::addRule(IPTRule& rule)" << endl;

	QString name = rule.name();
	for ( uint i = 0; i < m_ruleset.count(); i++ ) {
		QString tmp_name = m_ruleset.at( i ) ->name();
		if ( tmp_name == name ) {
			m_err->setErrType( "NORMAL" );
			const QString& msg = i18n( "<qt>Unable to add rule %1 into chain %2.<br>"
														" There is already a rule defined with that name. "
														"Please try again with another name (has to be unique in that chain).</qt>" ).arg( tmp_name ).arg( m_name );
			m_err->setErrMsg( msg );
			return m_err;
		}
	}
	m_ruleset.append( &rule );
	int num = m_ruleset.find( &rule );
	kdDebug() << "Added Rule Nr: " << num << endl;
	rule.setRuleNum( num );
	rule.setChain( this );
	regenerateRuleNumbers();
	m_err->setErrType( "OK" );
	const QString& msg = "";
	m_err->setErrMsg( msg );
	return m_err;
}

IPTRule* IPTChain::addRule( const QString& rule_name ) {
	kdDebug() << "IPTRule* IPTChain::addRule(QString& rule_name) " << endl;

	QString target = "DROP";
	const QString& new_name = rule_name;
	QString name = new_name;

	for ( uint i = 0; i < m_ruleset.count(); i++ ) {
		QString tmp_name = m_ruleset.at( i ) ->name();
		if ( tmp_name == name ) {
			KMessageBox::sorry( 0, i18n( "<qt>Unable to add rule %1 into chain %2.<br>"
																						" There is already a rule defined with that name. "
																						"Please try again with another name (must be unique in that chain)."
																						 ).arg( tmp_name ).arg( m_name ), i18n( "Warning" ) );
			return 0;
		}
	}

  IPTRule* new_rule = new IPTRule( new_name, this, target );
	if ( new_rule == 0 )
		return 0;

	m_ruleset.append( new_rule );
	regenerateRuleNumbers();
	return new_rule;
}

bool IPTChain::delRule( IPTRule& rule ) {
	if ( m_ruleset.count() < 1 )
		return false;

	QString name = rule.name();
	for ( uint i = 0; i < m_ruleset.count(); i++ ) {
		QString tmp_name = m_ruleset.at( i ) ->name();
		if ( tmp_name == name ) {
			kdDebug() << "Delete Rule: " << rule.name() << endl;
			m_ruleset.remove( i );
			regenerateRuleNumbers();
			return true;
		}
	}

	kdDebug() << "Sorry can't delete Rule: " << rule.name() << ".\nRule not found in Chain !!!" << endl;
	return false;
}

bool IPTChain::moveRule( IPTRule* rule, int how_much ) {
	kdDebug() << "void IPTChain::moveRule(IPTRule* rule,int how_much)" << endl;

	int num = m_ruleset.find( rule );
	kdDebug() << "Rule is at position: " << num << endl;
	int new_pos = num + how_much;
	if ( new_pos < 0 || new_pos > ( int ) m_ruleset.count() - 1 ) {
		kdDebug() << "New Position is not Valid: " << new_pos << endl;
		return false;
	} else {
		kdDebug() << "Moving rule to position: " << new_pos << endl;
		IPTRule *tmp_rule = m_ruleset.take( num );
		m_ruleset.insert( new_pos, tmp_rule );
		regenerateRuleNumbers();
		return true;
	}
}

void IPTChain::regenerateRuleNumbers() {
	kdDebug() << "void IPTChain::regenerateRuleNumbers()" << endl;
	IPTRule* tmp_rule;
	int i = 0;
	for ( tmp_rule = m_ruleset.first(); tmp_rule; tmp_rule = m_ruleset.next() ) {
		tmp_rule->setRuleNum( i );
		i++;
	}
	kdDebug() << "exit: void IPTChain::regenerateRuleNumbers()" << endl;
}

void IPTChain::setDropLogging( bool enabled, QString& limit, QString& burst, QString& prefix ) {
	enable_log = enabled;

	if ( !limit.isEmpty() )
		m_log_limit = limit;
	else
		m_log_limit = "";
	if ( !prefix.isEmpty() )
		m_log_prefix = prefix;
	else
		m_log_prefix = "";
	if ( !burst.isEmpty() )
		m_log_burst = burst;
	else
		m_log_burst = "";
}


QPtrList<IPTRule>& IPTChain::chainFwds() {
	QPtrList<IPTRule>* fwds = new QPtrList<IPTRule>;
	for ( uint i = 0;i < m_ruleset.count();i++ ) {
		IPTRule& cr = *m_ruleset.at( i );
		QString target = cr.target();
		// lots of targets missing !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
		if ( !target.isEmpty() && target != "ACCEPT" && target != "DROP" && target != "LOG"
		        && target != "REJECT" && target != "RETURN" && target != "DNAT" && target != "SNAT"
		        && target != "QUEUE" && target != "MIRROR" && target != "REDIRECT" && target != "MASQUERADE" ) {
			//kdDebug() << "Found Forward to chain: " << target << endl;
			IPTRule * curr_rule = new IPTRule( cr );
			fwds->append( curr_rule );
		}
	}
	return *fwds;
}

QPtrList<IPTRule>& IPTChain::chainFeeds() {
	kdDebug() << "QPtrList<IPTRule> IPTChain::chainFeeds()" << endl;
	QPtrList<IPTRule>* feeds = new QPtrList<IPTRule>;
	IPTChain* tmp_chain;
	QPtrList<IPTChain> table_chains = table() ->chains();
	for ( tmp_chain = table_chains.first(); tmp_chain; tmp_chain = table_chains.next() ) {
		//		kdDebug() << "Searching in chain: "  << tmp_chain->name() << endl;
		if ( !tmp_chain->chainRuleset().isEmpty() ) {
			IPTRule * tmp_rule;
			QPtrList<IPTRule> rules = tmp_chain->chainRuleset();
			for ( tmp_rule = rules.first(); tmp_rule; tmp_rule = rules.next() ) {
				QString target = tmp_rule->target();
				//kdDebug() << "Found Rule: " << tmp_rule->name() << " with target: " << target << endl;
				if ( target == m_name ) {
					feeds->append( tmp_rule );
					//kdDebug() << "Found feed: " << tmp_rule->name() << " from chain: " << tmp_chain->name() << endl;
				}
			}
		}
	}
	return *feeds;
}

QString IPTChain::createIPTablesChainDefinition() {
	QString chainDef = "";

	if ( is_build_in_chain )
		return chainDef;
	else {
		m_cmd_chain_definition = "$IPT -t ";
		m_cmd_chain_definition.append( m_table->name() );
		m_cmd_chain_definition.append( " -N " );
		m_cmd_chain_definition.append( m_name );
		//kdDebug() << "Set Chain Definition to: " << m_cmd_chain_definition << endl;
		return m_cmd_chain_definition;
	}
}

QString IPTChain::createIPTablesChainDefaultTarget() {
	kdDebug() << "IPTChain::createIPTablesChainDefaultTarget()" << endl;

	m_cmd_default_target = "$IPT -t ";
	m_cmd_default_target.append( m_table->name() );
	if ( is_build_in_chain ) {
		m_cmd_default_target.append( " -P " );
		m_cmd_default_target.append( m_name );
		m_cmd_default_target.append( " " );
	} else {
		m_cmd_default_target.append( " -A " );
		m_cmd_default_target.append( m_name );
		m_cmd_default_target.append( " -j " );
	}
	m_cmd_default_target.append( m_default_target );
	//kdDebug() << "Default Target Cmd: " << m_cmd_default_target << endl;

	return m_cmd_default_target;
}

QPtrList<QStringList>& IPTChain::createIPTablesChainRules() {
	kdDebug() << "QPtrList<QStringList>& IPTChain::createIPTablesChainRules()" << endl;
	QPtrList<QStringList>* all_rules = new QPtrList<QStringList>;
	if ( !m_ruleset.isEmpty() ) {
		//kdDebug() << "------- Chain Rules --------"<< endl;
		IPTRule * rule;
		for ( rule = m_ruleset.first(); rule; rule = m_ruleset.next() ) {
			QString rule_name = rule->name();
			QString rule_cmd = rule->cmdString();
			QStringList* chainDefs = new QStringList();
			chainDefs->append( rule_name );
			if ( rule->enabled() ) {
				chainDefs->append( rule_cmd );
			} else {
				QString warning = "echo -n \"\nSkipping Disabled Rule " + rule_name + "!!!\"";
				chainDefs->append( warning );
			}
			all_rules->append( chainDefs );
		}
	}
	if ( enable_log ) {
		//kdDebug() << "------- Chain Logging --------"<< endl;
		QString chain_log = "";
		chain_log.append( "$IPT -t " );
		chain_log.append( m_table->name() );
		chain_log.append( " -A " );
		chain_log.append( m_name );
		if ( !m_log_limit.isEmpty() ) {
			chain_log.append( " -m limit --limit " );
			chain_log.append( m_log_limit );
			if ( !m_log_burst.isEmpty() ) {
				chain_log.append( " --limit-burst " );
				chain_log.append( m_log_burst );
			}
		}
		chain_log.append( " -j LOG" );
		if ( !m_log_prefix.isEmpty() ) {
			chain_log.append( " --log-prefix \"" );
			chain_log.append( m_log_prefix );
			chain_log.append( "\"" );
		}
		kdDebug() << chain_log << endl;
		QStringList* chainDefs = new QStringList();
		QString rule_name = i18n( "Chain: %1 Drop Logging" ).arg( m_name );
		chainDefs->append( rule_name );
		chainDefs->append( chain_log );
		all_rules->append( chainDefs );
	}
	//kdDebug() << "------- Chain Default Target --------"<< endl;
	if ( has_default_target && !m_cmd_default_target.isEmpty() ) {
		QString deftg = createIPTablesChainDefaultTarget();
		QStringList* chainDefs = new QStringList();
		QString rule_name = i18n( "Chain: %1 Default Target" ).arg( m_name );
		chainDefs->append( rule_name );
		chainDefs->append( deftg );
		all_rules->append( chainDefs );
	}
	kdDebug() << "Leaving IPTChain::createIPTablesChainRules()" << endl;
	return *all_rules;
}

IPTChain* IPTChain::createChainClone() {
	kdDebug() << "\n\nIPTChain* createChainClone()\n\nCREATING CHAIN CLONE\n\n" << endl;
	IPTChain* copy = new IPTChain( name(), table(), isBuildIn() );
	copy->setUsed( isUsed() );
	QString limit = logLimit();
	QString burst = logBurst();
	QString prefix = logPrefix();

	copy->setDropLogging( logging(), limit, burst, prefix );
	if ( hasDefaultTarget() )
		copy->setDefaultTarget( defaultTarget() );

	if ( m_ruleset.count() > 0 )
		for ( uint i = 0; i < m_ruleset.count(); i++ ) {
			if ( m_ruleset.at( i ) ) {
				IPTRule & tmp = *m_ruleset.at( i ) ->createRuleClone();
				copy->addRule( tmp );
				kdDebug() << "Added Rule: " << tmp.name() << " to chain: " << name() << "\n" << endl;
			}
		}

	kdDebug() << "\n\nEXIT:createChainClone()\n\nCreated Chain: " << copy->name() << endl;
	return copy;
}
