// =============================================================================
//
//      --- kvi_irc_userchanlist.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviIrcUserChanList"

#include "kvi_debug.h"
#include "kvi_irc_user.h"
#include "kvi_irc_userchanlist.h"
#include "kvi_string.h"

// TODO: We want a hash table here for 2.1.0!!!

// ===========================================================================
// Client list
//
KviIrcUserChanList::KviIrcUserChanList(KviIrcUserList *pList)
{
	m_pGlobalList  = pList;
	m_pHead        = 0;
	m_pTail        = 0;
	m_pCur         = 0;
	m_iCount       = 0;
	m_iOwnerCount  = 0;
	m_iOpCount     = 0;
	m_iHalfOpCount = 0;
	m_iVoiceCount  = 0;
	m_iUserOpCount = 0;
}

KviIrcUserChanList::~KviIrcUserChanList()
{
	clearList();
}

KviIrcUser *KviIrcUserChanList::firstUser()
{
	m_pCur = m_pHead;
	return m_pCur ? m_pCur->pNode->pUser : 0;
}

KviIrcUser *KviIrcUserChanList::lastUser()
{
	m_pCur = m_pTail;
	return m_pCur ? m_pCur->pNode->pUser : 0;
}

KviIrcUser *KviIrcUserChanList::nextUser()
{
	if( m_pCur )
		m_pCur = m_pCur->next;
	return m_pCur ? m_pCur->pNode->pUser : 0;
}
KviIrcUser *KviIrcUserChanList::prevUser()
{
	if( m_pCur )
		m_pCur = m_pCur->prev;
	return m_pCur ? m_pCur->pNode->pUser : 0;
}

KviIrcUserChanData *KviIrcUserChanList::firstData()
{
	m_pCur = m_pHead;
	return m_pHead;
}

KviIrcUserChanData *KviIrcUserChanList::lastData()
{
	m_pCur = m_pTail;
	return m_pTail;
}

KviIrcUserChanData *KviIrcUserChanList::nextData()
{
	if( m_pCur )
		m_pCur = m_pCur->next;
	return m_pCur;
}

KviIrcUserChanData *KviIrcUserChanList::prevData()
{
	if( m_pCur )
		m_pCur = m_pCur->prev;
	return m_pCur;
}

void KviIrcUserChanList::clearList()
{
	__range_valid((m_pHead && m_pTail) || (!(m_pHead || m_pTail)));
	while( m_pHead ) {
		part(*(m_pHead->pNode->pUser));
	}
	__range_valid(m_pHead == m_pTail);
	__range_valid(m_pTail == 0);
}

KviIrcUserChanData *KviIrcUserChanList::join(const KviIrcUser &user, char bOp, char bVoice, char bHalfOp, char bUserOp, char bOwner)
{
	// Adds a user to this list,
	// adding it to the global one too
	KviIrcUserChanData *data = findData(user.nick());
	if( data ) {
		// Already in the list... double join?
		__debug_1arg("Double join for user %s", user.nick());
		// Update the fields only
		if( data->qFlag != 0 ) {
			if( bOwner == 0 ) {
				m_iOwnerCount--;
				data->qFlag = bOwner;
			}
		} else if( bOwner != 0 ) {
			m_iOwnerCount++;
			data->qFlag = bOwner;
		}
		if( data->oFlag != 0 ) {
			if( bOp == 0 ) {
				m_iOpCount--;
				data->oFlag = bOp;
			}
		} else if( bOp != 0 ) {
			m_iOpCount++;
			data->oFlag = bOp;
		}
		if( data->hFlag != 0 ) {
			if( bHalfOp == 0 ) {
				m_iHalfOpCount--;
				data->hFlag = bHalfOp;
			}
		} else if( bHalfOp != 0 ) {
			m_iOpCount++;
			data->hFlag = bHalfOp;
		}
		if( data->vFlag != 0 ) {
			if( bVoice == 0 ) {
				m_iVoiceCount--;
				data->vFlag = bVoice;
			}
		} else if( bVoice != 0 ) {
			m_iVoiceCount++;
			data->vFlag = bVoice;
		}
		if( data->uFlag != 0 ) {
			if( bUserOp == 0 ) {
				m_iUserOpCount--;
				data->uFlag = bUserOp;
			}
		} else if( bUserOp != 0 ) {
			m_iUserOpCount++;
			data->uFlag = bUserOp;
		}
		return data;
	}
	data = new KviIrcUserChanData;
	data->qFlag = bOwner;
	data->oFlag = bOp;
	data->hFlag = bHalfOp;
	data->vFlag = bVoice;
	data->uFlag = bUserOp;
	// Set extFlag to zero now...
	// This flag can be used by the external classes.
	// KviUserListBox uses it to keep track of the selected items
	data->extFlag = 0;
	data->pNode = m_pGlobalList->addUser(user);
	__range_valid(data->pNode);
	insertData(data);
	return data;
}

bool KviIrcUserChanList::part(const KviIrcUser &user)
{
	return part(user.nick());
}

bool KviIrcUserChanList::part(const char *nick)
{
	KviIrcUserChanData *data = findData(nick);
	__range_valid(data);
	if( !data ) return false;
	__range_valid(data->pNode == m_pGlobalList->findNode(nick));
	__range_valid(data->pNode);
	// It deletes the whole data->pNode structure
	m_pGlobalList->killUserByNode(data->pNode);
	// Remove data, deleting it
	removeData(data);
	return true;
}

bool KviIrcUserChanList::select(const KviIrcUser &user)
{
	return select(user.nick());
}

bool KviIrcUserChanList::select(const char *nick)
{
	KviIrcUserChanData *data = findData(nick);
	if( !data ) return false;
	data->extFlag = 1;
	return true;
}

bool KviIrcUserChanList::deselect(const KviIrcUser &user)
{
	return deselect(user.nick());
}

bool KviIrcUserChanList::deselect(const char *nick)
{
	KviIrcUserChanData *data = findData(nick);
	if( !data ) return false;
	data->extFlag = 0;
	return true;
}

bool KviIrcUserChanList::nickChange(const KviIrcUser &nicker, const char *newNick)
{
	KviIrcUserChanData *data = findData(nicker.nick());
	if( !data ) return false;
	__range_valid(data->pNode == m_pGlobalList->findNode(nicker.nick()));
	__range_valid(data->pNode);
	char qFlag = data->qFlag;
	char oFlag = data->oFlag;
	char hFlag = data->hFlag;
	char vFlag = data->vFlag;
	char uFlag = data->uFlag;
	// Now like part...
	m_pGlobalList->killUserByNode(data->pNode);
	removeData(data);
	KviIrcUser newNicker(nicker);
	newNicker.setNick(newNick);
	join(newNicker, oFlag, vFlag, hFlag, uFlag, qFlag);
	return true;
}

KviIrcUser *KviIrcUserChanList::findUser(const KviIrcUser &user)
{
	return findUser(user.nick());
}

KviIrcUser *KviIrcUserChanList::findUser(const char *nick)
{
	KviIrcUserChanData *data = findData(nick);
	if( !data ) return 0;
	__range_valid(data->pNode);
	__range_valid(data->pNode->pUser);
	return data->pNode->pUser;
}

int KviIrcUserChanList::findUserPosition(const char *nick)
{
	KviIrcUserChanData *cur = m_pHead;
	int item = 0;
	int result;
	while( cur ) {
		// Compare
		result = kvi_strcmpCI(nick, cur->pNode->pUser->m_nick_ptr);
		if( result == 0 ) return item;
		item++;
		cur = cur->next;
	}
	return item; // Not found
}

KviIrcUserChanData *KviIrcUserChanList::findData(const KviIrcUser &user)
{
	return findData(user.nick());
}

KviIrcUserChanData *KviIrcUserChanList::findData(const char *nick)
{
	// Find a data entry in the list
	KviIrcUserChanData *cur = m_pHead;
	int result;
	while( cur ) {
		// Compare
		result = kvi_strcmpCI(nick, cur->pNode->pUser->m_nick_ptr);
		if( result == 0 ) return cur; // Found (case insensitive)
		if(result > 0 ) {
			// The cur is greater...
			if( cur->qFlag != 0 ) {
				// We were looking in the ops block.
				// Skip all the ops and continue
				while( cur ) {
					if( cur->qFlag == 0 ) break;
					// Sanity: must not be here, if it is then the order is wrong!
					__range_invalid(kvi_strEqualCI(cur->pNode->pUser->m_nick_ptr, nick));
					cur = cur->next;
				}
			} else if( cur->oFlag != 0 ) {
				// We were looking in the ops block.
				// Skip all the ops and continue
				while( cur ) {
					if( cur->oFlag == 0 ) break;
					// Sanity: must not be here, if it is then the order is wrong!
					__range_invalid(kvi_strEqualCI(cur->pNode->pUser->m_nick_ptr, nick));
					cur = cur->next;
				}
			} else if( cur->hFlag != 0 ) {
				// We were looking in the halfop block.
				// Skip all the halfops and continue
				while( cur ) {
					if( cur->hFlag == 0 ) break;
					// Sanity: must not be here, if it is then the order is wrong!
					__range_invalid(kvi_strEqualCI(cur->pNode->pUser->m_nick_ptr, nick));
					cur = cur->next;
				}
			} else if( cur->vFlag != 0 ) {
				// We were looking in the voiced block.
				// Skip all the voiced and continue
				while( cur ) {
					if( cur->vFlag == 0 ) break;
					// Sanity: must not be here, if it is then the order is wrong!
					__range_invalid(kvi_strEqualCI(cur->pNode->pUser->m_nick_ptr, nick));
					cur = cur->next;
				}
			} else if( cur->uFlag != 0 ) {
				// We were looking in the userop block.
				// Skip all the userops and continue
				while( cur ) {
					if( cur->uFlag == 0 ) break;
					// Sanity: must not be here, if it is then the order is wrong!
					__range_invalid(kvi_strEqualCI(cur->pNode->pUser->m_nick_ptr, nick));
					cur = cur->next;
				}
			} else return 0; // (normal user) cannot be...
		} else cur = cur->next;
	}
	// Ran up to the end...
	return 0; // Not found
}

bool KviIrcUserChanList::owner(const char *nick, char bOwner)
{
	KviIrcUserChanData *data = findData(nick);
	__range_valid(data);
	if( !data ) return false;
	if( data->qFlag != bOwner ) {
		// Adjust the list order
		removeDataNoDelete(data);
		__range_invalid(findData(nick));
		data->qFlag = bOwner;
		insertData(data);
		__range_valid(findData(nick));
	}
	return true;
}

bool KviIrcUserChanList::owner(const KviIrcUser &user, char bOwner)
{
	return owner(user.nick(), bOwner);
}

bool KviIrcUserChanList::op(const char *nick, char bOp)
{
	KviIrcUserChanData *data = findData(nick);
	__range_valid(data);
	if( !data ) return false;
	if( data->oFlag != bOp ) {
		// Adjust the list order
		removeDataNoDelete(data);
		__range_invalid(findData(nick));
		data->oFlag = bOp;
		insertData(data);
		__range_valid(findData(nick));
	}
	return true;
}

bool KviIrcUserChanList::op(const KviIrcUser &user, char bOp)
{
	return op(user.nick(),bOp);
}

bool KviIrcUserChanList::halfop(const char *nick, char bHalfOp)
{
	KviIrcUserChanData *data = findData(nick);
	__range_valid(data);
	if( !data ) return false;
	if( data->hFlag != bHalfOp ) {
		// Adjust the list order
		removeDataNoDelete(data);
		__range_invalid(findData(nick));
		data->hFlag = bHalfOp;
		insertData(data);
		__range_valid(findData(nick));
	}
	return true;
}

bool KviIrcUserChanList::halfop(const KviIrcUser &user, char bHalfOp)
{
	return halfop(user.nick(), bHalfOp);
}

bool KviIrcUserChanList::voice(const char *nick, char bVoice)
{
	KviIrcUserChanData *data = findData(nick);
	__range_valid(data);
	if( !data ) return false;
	if( data->vFlag != bVoice ) {
		// Adjust the list order
		removeDataNoDelete(data);
		__range_invalid(findData(nick));
		data->vFlag = bVoice;
		insertData(data);
		__range_valid(findData(nick));
	}
	return true;
}

bool KviIrcUserChanList::voice(const KviIrcUser &user, char bVoice)
{
	return voice(user.nick(), bVoice);
}

bool KviIrcUserChanList::userop(const char *nick, char bUserOp)
{
	KviIrcUserChanData *data = findData(nick);
	__range_valid(data);
	if( !data ) return false;
	if( data->uFlag != bUserOp ) {
		// Adjust the list order
		removeDataNoDelete(data);
		__range_invalid(findData(nick));
		data->uFlag = bUserOp;
		insertData(data);
		__range_valid(findData(nick));
	}
	return true;
}

bool KviIrcUserChanList::userop(const KviIrcUser &user, char bUserOp)
{
	return userop(user.nick(), bUserOp);
}
bool KviIrcUserChanList::isOwner(const char *nick)
{
	KviIrcUserChanData *data = findData(nick);
	if( !data ) return false;
	return (data->qFlag != 0);
}

bool KviIrcUserChanList::isOwner(const KviIrcUser &user)
{
	return isOwner(user.nick());
}

bool KviIrcUserChanList::isOp(const char *nick)
{
	KviIrcUserChanData *data = findData(nick);
	if( !data ) return false;
	return (data->oFlag != 0);
}

bool KviIrcUserChanList::isOp(const KviIrcUser &user)
{
	return isOp(user.nick());
}
bool KviIrcUserChanList::isHalfOp(const char *nick)
{
	KviIrcUserChanData *data = findData(nick);
	if( !data )return false;
	return (data->hFlag != 0);
}

bool KviIrcUserChanList::isHalfOp(const KviIrcUser &user)
{
	return isHalfOp(user.nick());
}

bool KviIrcUserChanList::isVoice(const char *nick)
{
	KviIrcUserChanData *data = findData(nick);
	if( !data ) return false;
	return (data->vFlag != 0);
}

bool KviIrcUserChanList::isVoice(const KviIrcUser &user)
{
	return isVoice(user.nick());
}

bool KviIrcUserChanList::isUserOp(const char *nick)
{
	KviIrcUserChanData *data = findData(nick);
	if( !data ) return false;
	return (data->uFlag != 0);
}

bool KviIrcUserChanList::isUserOp(const KviIrcUser &user)
{
	return isUserOp(user.nick());
}

int KviIrcUserChanList::count()
{
	return m_iCount;
}

int KviIrcUserChanList::ownerCount()
{
	return m_iOwnerCount;
}

int KviIrcUserChanList::opCount()
{
	return m_iOpCount;
}

int KviIrcUserChanList::halfopCount()
{
	return m_iHalfOpCount;
}

int KviIrcUserChanList::voiceCount()
{
	return m_iVoiceCount;
}

int KviIrcUserChanList::useropCount()
{
	return m_iUserOpCount;
}

void KviIrcUserChanList::insertData(KviIrcUserChanData *data)
{
	// Inserts a data node in the local list
	__range_valid(data->pNode);
	__range_valid(data->pNode->pUser);
	__range_valid(data->pNode->pUser->hasNick());
	__range_invalid(findData(data->pNode->pUser->nick()));
	// Increase the count
	m_iCount++;
	if( m_pHead ) {
		// Have items inside
		if( data->qFlag != 0 ) {
			insertOwnerData(data);
			m_iOwnerCount++;
		} else if( data->oFlag != 0 ) {
			insertOpData(data); // isOp
			m_iOpCount++;
		} else if( data->hFlag != 0 ) {
			insertHalfOpData(data);
			m_iHalfOpCount++;
		} else if( data->vFlag != 0 ) {
			insertVoiceData(data);
			m_iVoiceCount++;
		} else if( data->uFlag != 0 ) {
			insertUserOpData(data);
			m_iUserOpCount++;
		} else insertNormalData(data);
	} else {
		// First in the list
		m_pHead = data;
		m_pTail = data;
		data->next = 0;
		data->prev = 0;
		if( data->qFlag != 0 ) m_iOwnerCount++;
		else if( data->oFlag != 0 ) m_iOpCount++;
		else if( data->hFlag != 0 ) m_iHalfOpCount++;
		else if( data->vFlag != 0 ) m_iVoiceCount++;
		else if( data->uFlag != 0 ) m_iUserOpCount++;
	}
}

void KviIrcUserChanList::insertOwnerData(KviIrcUserChanData *data)
{
	// Walk the op list...
	__range_valid(m_pHead);
	__range_valid(data->qFlag != 0);
	KviIrcUserChanData *cur = m_pHead;
	while( cur ) {
		if( (kvi_strcmpCI(data->pNode->pUser->m_nick_ptr, cur->pNode->pUser->m_nick_ptr) > 0) ||
			(cur->qFlag == 0) )
		{
			// Cur is greater or is non-op... insert before
			data->next = cur;
			data->prev = cur->prev;
			if( data->prev )
				data->prev->next = data;
			else
				m_pHead = data; // No prev item
			cur->prev = data;
			return;
		}
		cur = cur->next;
	}
	// Ran to the end of the list! (only ops inside)
	m_pTail->next = data;
	data->prev = m_pTail;
	data->next = 0;
	m_pTail = data;
}

void KviIrcUserChanList::insertOpData(KviIrcUserChanData *data)
{
	// Walk the op list...
	__range_valid(m_pHead);
	__range_valid(data->oFlag != 0);
	KviIrcUserChanData *cur = m_pHead;
	while( cur ) { // Skip the owners
		if( cur->qFlag == 0 ) break;
		cur = cur->next;
	}
	while( cur ) {
		if( (kvi_strcmpCI(data->pNode->pUser->m_nick_ptr, cur->pNode->pUser->m_nick_ptr) > 0) ||
			(cur->oFlag == 0) )
		{
			// Cur is greater or is non-op... insert before
			data->next = cur;
			data->prev = cur->prev;
			if( data->prev )
				data->prev->next = data;
			else
				m_pHead = data; // No prev item
			cur->prev = data;
			return;
		}
		cur = cur->next;
	}
	// Ran to the end of the list! (only ops inside)
	m_pTail->next = data;
	data->prev = m_pTail;
	data->next = 0;
	m_pTail = data;
}

void KviIrcUserChanList::insertHalfOpData(KviIrcUserChanData *data)
{
	// Walk the halfop list...
	__range_valid(m_pHead);
	__range_valid(data->hFlag != 0);
	KviIrcUserChanData *cur = m_pHead;
	while( cur ) { // Skip the owners
		if( cur->qFlag == 0 ) break;
		cur = cur->next;
	}
	while( cur ) { // Skip the ops
		if( cur->oFlag == 0 ) break;
		cur = cur->next;
	}
	while( cur ) {
		if( (kvi_strcmpCI(data->pNode->pUser->m_nick_ptr, cur->pNode->pUser->m_nick_ptr) > 0) ||
			(cur->hFlag == 0) )
		{
			// Cur is greater or is non-halfop... insert before
			data->next = cur;
			data->prev = cur->prev;
			if( data->prev )
				data->prev->next = data;
			else
				m_pHead = data; // No prev item
			cur->prev = data;
			return;
		}
		cur = cur->next;
	}
	// Ran to the end of the list! (only ops and halfops inside)
	m_pTail->next = data;
	data->prev = m_pTail;
	data->next = 0;
	m_pTail = data;
}

void KviIrcUserChanList::insertVoiceData(KviIrcUserChanData *data)
{
	// Walk the voice list...
	__range_valid(m_pHead);
	__range_valid(data->vFlag != 0);
	KviIrcUserChanData *cur = m_pHead;
	while( cur ) { // Skip the owners
		if( cur->qFlag == 0 ) break;
		cur = cur->next;
	}
	while( cur ) { // Skip the ops
		if( cur->oFlag == 0 ) break;
		cur = cur->next;
	}
	while( cur ) { // Skip the halfops
		if( cur->hFlag == 0 ) break;
		cur = cur->next;
	}
	// Now pointing to the first non-op or to the end of the list
	while( cur ) {
		// First non op
		if( (kvi_strcmpCI(data->pNode->pUser->m_nick_ptr, cur->pNode->pUser->m_nick_ptr) > 0) ||
			(cur->vFlag == 0) )
		{
			// cur is greater or is non-voice... insert before
			data->next = cur;
			data->prev = cur->prev;
			if( data->prev )
				data->prev->next = data;
			else
				m_pHead = data; // No prev item
			cur->prev = data;
			return;
		}
		cur = cur->next;
	}
	// Ran to the end of the list! (only ops, halfops, and voiced inside)
	m_pTail->next = data;
	data->prev = m_pTail;
	data->next = 0;
	m_pTail = data;
}

void KviIrcUserChanList::insertUserOpData(KviIrcUserChanData *data)
{
	// Walk the userop list...
	__range_valid(m_pHead);
	__range_valid(data->uFlag != 0);
	KviIrcUserChanData *cur = m_pHead;
	while( cur ) { // Skip the owners
		if( cur->qFlag == 0 ) break;
		cur = cur->next;
	}
	while( cur ) { // Skip the ops
		if( cur->oFlag == 0 ) break;
		cur = cur->next;
	}
	while( cur ) { // Skip the halfops
		if( cur->hFlag == 0 ) break;
		cur = cur->next;
	}
	while( cur ) { // Skip the voiced ones
		if( cur->vFlag == 0 ) break;
		cur = cur->next;
	}
	while( cur ) {
		if( (kvi_strcmpCI(data->pNode->pUser->m_nick_ptr, cur->pNode->pUser->m_nick_ptr) > 0) ||
			(cur->uFlag == 0) )
		{
			// Cur is greater or is non-userop... insert before
			data->next = cur;
			data->prev = cur->prev;
			if( data->prev )
				data->prev->next = data;
			else
				m_pHead = data; // No prev item
			cur->prev = data;
			return;
		}
		cur = cur->next;
	}
	// Ran to the end of the list! (only ops, halfops, voiced, and userops inside)
	m_pTail->next = data;
	data->prev = m_pTail;
	data->next = 0;
	m_pTail = data;
}

void KviIrcUserChanList::insertNormalData(KviIrcUserChanData *data)
{
	// Walk the voice list...
	__range_valid(m_pHead);
	__range_valid((data->uFlag == 0) && (data->vFlag == 0) && (data->hFlag == 0) && (data->oFlag == 0) && (data->qFlag == 0));
	KviIrcUserChanData *cur = m_pHead;
	while( cur ) { // Skip the owners
		if( cur->qFlag == 0 ) break;
		cur = cur->next;
	}
	while( cur ) { // Skip the ops
		if( cur->oFlag == 0 ) break;
		cur = cur->next;
	}
	while( cur ) { // Skip the halfops
		if( cur->hFlag == 0 ) break;
		cur = cur->next;
	}
	while( cur ) { // Skip the voiced ones
		if( cur->vFlag == 0 ) break;
		cur = cur->next;
	}
	while( cur ) { // Skip the userops
		if( cur->uFlag == 0 ) break;
		cur = cur->next;
	}
	// Now pointing to the first non-op/voice or to the end of the list
	while( cur ) {
		// First non-op/voice
		if( kvi_strcmpCI(data->pNode->pUser->m_nick_ptr, cur->pNode->pUser->m_nick_ptr) > 0 ) {
			// Cur is greater... insert before
			data->next = cur;
			data->prev = cur->prev;
			if( data->prev )
				data->prev->next = data;
			else
				m_pHead = data; // No prev item
			cur->prev = data;
			return;
		}
		cur = cur->next;
	}
	// Ran to the end of the list!
	m_pTail->next = data;
	data->prev = m_pTail;
	data->next = 0;
	m_pTail = data;
}

void KviIrcUserChanList::removeDataNoDelete(KviIrcUserChanData *data)
{
	// Removes just the data node from the list, nothing else
	__range_valid(data);
	__range_valid(data->pNode);

	m_iCount--; // One less

	if( data->qFlag != 0 ) m_iOwnerCount--;
	else if( data->oFlag != 0 ) m_iOpCount--;     // Was op?
	else if( data->hFlag != 0 ) m_iHalfOpCount--; // Was halfop?
	else if( data->vFlag != 0 ) m_iVoiceCount--;  // Was voice?
	else if( data->uFlag != 0 ) m_iUserOpCount--; // Was userop?

	if( data == m_pHead ) {
		// Was the first item in the list
		if( data->next ) {
			// And has a next one
			data->next->prev = 0;
			m_pHead = data->next;
		} else {
			// And was the only one
			__range_valid(m_pTail == data);
			m_pTail = 0;
			m_pHead = 0;
		}
	} else {
		// Somewhere in the middle or the last one
		__range_valid(data->prev);
		if( data->next ) {
			// In the middle
			data->next->prev = data->prev;
			data->prev->next = data->next;
		} else {
			// Was the last
			m_pTail = data->prev;
			data->prev->next = 0;
		}
	}
}

void KviIrcUserChanList::removeData(KviIrcUserChanData *data)
{
	removeDataNoDelete(data);
	if( data ) delete data;
}

