#ifndef _KVI_WINDOW_H_INCLUDED_
#define _KVI_WINDOW_H_INCLUDED_

// =============================================================================
//
//      --- kvi_window.h ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#include <qrect.h>
#include <qwidget.h>

#include "kvi_settings.h"
#include "kvi_string.h"

class QPixmap;
class QSplitter;

class KviAccel;
class KviFindDialog;
class KviFrame;
class KviInput;
class KviIrcView;
class KviMdiChild;
class KviTaskBarButton;
class KviUserListBox;
class KviWindowProperty;

/**
 * Base class for all interface windows.<br>
 * Defines some virtual functions for later common use.<br>
 * The derived windows 'live' attached to a KviMdiChild widget<br>
 * managed by KviMdiManager, or detached (managed by the window manager.)<br>
 * So remember that the parent() pointer may change, and may be 0 too.<br>
 *
 * @short Base class for all interface windows.
 */
class KviWindow : public QWidget
{
	friend class KviFrame;
	friend class KviUserParser;
	friend class KviInput;
#ifdef COMPILE_PLUGIN_SUPPORT
	friend class KviPluginManager;
#endif
	Q_OBJECT
public:
	KviWindow(const char *name, int type, KviFrame *lpFrm);
	~KviWindow();
	/**
	 * The frame that this window belongs to.
	 * It is always non-null and never changes.
	 */
	KviFrame *frame() const;
protected:
	KviFrame         *m_pFrm;
	QString           m_szCaption;
	int               m_type;
	bool              m_bJustGotFocused;
	KviTaskBarButton *m_pTaskBarButton;
	KviFindDialog    *m_pFindDialog;
	/** The focus handler widget. */
	QWidget          *m_pFocusHandler;
	QWidget          *m_pLastFocusedChild;
#ifdef COMPILE_PLUGIN_SUPPORT
	/** Plugin that this window belongs to (usually 0) */
	void             *m_pluginHandle;
#endif
public:
	KviAccel         *m_pAccelerators;
	/** The view splitter... */
	QSplitter        *m_pSplitter;
	/**
	 * This is the KviIrcView pointer.<br>
	 * It is set to 0 in the constructor.<br>
	 * If you do not use this pointer for output,
	 * you should override output().
	 * @see KviIrcView::output
	 */
	KviIrcView       *m_pView;
	/**
	 * KviUserListBox pointer; it is set to 0 in the constructor
	 * and is used by derived classes.
	 */
	KviUserListBox   *m_pListBox;
	/**
	 * KviInput pointer; it is set to 0 in the constructor
	 * and used by derived classes.
	 */
	KviInput         *m_pInput;
public:
	/**
	 * Retuns the window type.<br>
	 * Window types are declared in <a href="kvi_defines-h.html">kvi_defines.h</a><br>
	 * (inline)
	 */
	int type();
	/**
	 * Returns true if this window is attached to the MDI manager.<br>
	 * (inline)
	 */
	bool isAttached();
	/**
	 * Returns the caption text.<br>
	 * (inline)
	 */
	const char *caption();
	/**
	 * Returns window target (if any)
	 */
	const char *target();
	/**
	 * Sets the window caption string...<br>
	 * Calls updateButton on the taskbar button if it has been set.<br>
	 */
	void setWindowCaption(const char *szCaption);
	/**
	 * Highlights the related taskbar button.<br>
	 * Should be called from all do*Output overridden calls.<br>
	 * (inline)
	 */
	void highlight(bool bAltColor = false);
	/**
	 * Returns the KviMdiChild parent widget (or 0 if the window is not attached).
	 */
	KviMdiChild *mdiParent();
	/**
	 * Tells if the window is minimized when attached to the MDI manager,
	 * or if it is VISIBLE when 'floating'.
	 */
	bool isMinimized();// Useful only when attached (?)
	/**
	 * Tells if the window is minimized when attached to the MDI manager,
	 * otherwise returns false.
	 */
	bool isMaximized();// Useful only when attached (?)
	/**
	 * Returns the geometry of this window or that of the parent if there is any...
	 */
	QRect externalGeometry();

	// Methods to ABSOLUTELY override

	/**
	 * You SHOULD override this function in the derived class
	 * and return one of the g_pWindowIcon pointers managed by KviApp.
	 */
	virtual QPixmap *myIconPtr();
	/**
	 * Must be overridden in child classes.
	 */
	virtual void applyOptions();
	/**
	 * Sets the related taskbar button progress.
	 */
	void setProgress(int progress);
	/**
	 * Minimizes this window when it is attached to the MDI manager.<br>
	 * Otherwise has no effect.
	 */
	void minimize(bool bAnimate);   // Useful only when attached
	/**
	 * Maximizes this window when it is attached to the MDI manager.<br>
	 * Otherwise has no effect.
	*/
	void maximize(bool bAnimate);   // Useful only when attached

	/**
	 * output(message_type, format_string, arguments...);<br>
	 * This should be overridden in derived classes that do not use the m_pView pointer,
	 * and highlight() should be called in that function implementation.<br>
	 * The default implementation outputs to the (KviIrcView *) m_pView class,
	 * segfaulting if !m_pView :)
	 */
	virtual void output(int msg_type, const char *format, ...);
	virtual void outputNoFmt(int msg_type, const char *text);

	virtual void setProperties(KviWindowProperty *); // Do nothing here
	virtual void saveProperties();
	virtual void getDefaultLogName(KviStr &str);
	void delayedAutoRaise();
	void delayedClose();

public slots:
	/**
	 * Attaches this window to the MDI manager.<br>
	 * It calls the KviFrame attachWindow() function, so if you have a pointer
	 * to this KviFrame you will be faster calling that function.<br>
	 * Useful as slot.
	 */
	void attach();
	/**
	 * Detaches this window from the MDI manager.<br>
	 * It calls the KviFrame detachWindow() function, so if you have a pointer
	 * to this KviFrame you will be faster calling that function.<br>
	 * Useful as slot.
	 */
	void detach();

	void findText();
	void minimize(); // Overload and slot
	void maximize(); // Overload and slot
	/**
	 * Restores this window when it is attached to the MDI manager.
	 * Useful only when attached
	 */
	void restore();
	void autoRaise();
protected:
	void setFocusHandler(QWidget *);
	virtual void youAreAttached(KviMdiChild *);
	virtual void youAreDetached();

	void installAccelerators();
	void destroyAccelerators();

	// This one is called from KviFrame
	void setTaskBarButton(KviTaskBarButton *);
	/**
	 * Ignores the event and calls KviFrame::childWindowCloseRequest
	 *
	 * @see KviFrame::childWindowCloseRequest
	 */
	virtual void  closeEvent(QCloseEvent *);
	virtual bool eventFilter(QObject *, QEvent *);
	virtual void  wheelEvent(QWheelEvent *);

	void childInserted(QWidget *);
	void childRemoved(QWidget *);
	void childEvent(QChildEvent *);

	void activateSelf();
	void focusInEvent(QFocusEvent *);
};

inline int          KviWindow::type()        { return m_type; }
inline bool         KviWindow::isAttached()  { return (parent() != 0); }
inline const char  *KviWindow::caption()     { return m_szCaption.latin1(); }
inline const char  *KviWindow::target()      { return m_szCaption.latin1(); }
inline KviMdiChild *KviWindow::mdiParent()   { return (KviMdiChild *) parent(); }

#endif // _KVI_WINDOW_H_INCLUDED_
