/*
 *   LADCCA
 *    
 *   Copyright (C) 2002 Robert Ham <rah@bash.sh>
 *    
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdint.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include <ladcca/comm_event.h>

void
cca_comm_event_init (cca_comm_event_t * event)
{
  event->type = 0;
  event->event_data.number = 0;
}

void
cca_comm_event_free (cca_comm_event_t * event)
{
  switch (event->type)
  {
    case CCA_Comm_Event_Connect:
      if (event->event_data.connect)
        cca_connect_params_destroy (event->event_data.connect);
      break;
    case CCA_Comm_Event_Event:
      if (event->event_data.event)
        cca_event_destroy (event->event_data.event);
      break;
    case CCA_Comm_Event_Config:
      if (event->event_data.config)
        cca_config_destroy (event->event_data.config);
      break;
    case CCA_Comm_Event_Exec:
      if (event->event_data.exec)
        cca_exec_params_destroy (event->event_data.exec);
      break;
  }
}

cca_comm_event_t *
cca_comm_event_new ()
{
  cca_comm_event_t * event;
  event = cca_malloc (sizeof (cca_comm_event_t));
  cca_comm_event_init (event);
  return event;
}

void
cca_comm_event_destroy (cca_comm_event_t * event)
{
  cca_comm_event_free (event);
  free (event);
}

void
cca_comm_event_set_type (cca_comm_event_t * event, enum CCA_Comm_Event_Type type)
{
  event->type = type;
}

void
cca_comm_event_set_connect (cca_comm_event_t * event, cca_connect_params_t * params)
{
  cca_comm_event_set_type (event, CCA_Comm_Event_Connect);
  event->event_data.connect = params;
}

void
cca_comm_event_set_event (cca_comm_event_t * event, cca_event_t * cevent)
{
  cca_comm_event_set_type (event, CCA_Comm_Event_Event);
  event->event_data.event = cevent;
}

void
cca_comm_event_set_config (cca_comm_event_t * event, cca_config_t * config)
{
  cca_comm_event_set_type (event, CCA_Comm_Event_Config);
  event->event_data.config = config;
}

void
cca_comm_event_set_protocol_mismatch (cca_comm_event_t * event, cca_protocol_t protocol)
{
  cca_comm_event_set_type (event, CCA_Comm_Event_Protocol_Mismatch);
  event->event_data.number = protocol;
}

void
cca_comm_event_set_exec              (cca_comm_event_t * event, cca_exec_params_t * params)
{
  cca_comm_event_set_type (event, CCA_Comm_Event_Exec);
  event->event_data.exec = params;
}

enum CCA_Comm_Event_Type
cca_comm_event_get_type (cca_comm_event_t * event)
{
  return event->type;
}

cca_connect_params_t *
cca_comm_event_take_connect (cca_comm_event_t * event)
{
  cca_connect_params_t * params;
  params = event->event_data.connect;
  event->event_data.connect = NULL;
  return params;
}

cca_config_t *
cca_comm_event_take_config (cca_comm_event_t * event)
{
  cca_config_t * config;
  config = event->event_data.config;
  event->event_data.config = NULL;
  return config;
}

cca_event_t *
cca_comm_event_take_event (cca_comm_event_t * event)
{
  cca_event_t * cca_event;
  cca_event = event->event_data.event;
  event->event_data.event = NULL;
  return cca_event;
}

cca_exec_params_t *
cca_comm_event_take_exec (cca_comm_event_t * event)
{
  cca_exec_params_t * params;
  params = event->event_data.exec;
  event->event_data.exec = NULL;
  return params;
}

long
cca_comm_event_get_number  (const cca_comm_event_t * event)
{
  return event->event_data.number;
}

/* EOF */
