import unittest, shutil, tempfile
from linda.funcs import run_external_cmd
from linda.parser.makefile import MakefileParser
from sets import Set

class MakefileParserTest(unittest.TestCase):
    def setUp(self):
        self.dir = tempfile.mkdtemp()
        run_external_cmd('tar zxCf %s tests/parser_makefile.tar.gz' % \
            self.dir)

    def tearDown(self):
        shutil.rmtree(self.dir)

    def test_simple_makefile(self):
        "Check a simple Makefile parses"
        mfp = MakefileParser('%s/makefile.simple' % self.dir)
        self.assertEqual(mfp.variables(), [])
        targets = Set(('build', 'build-stamp', 'clean', 'real-install', \
            'test', 'install', 'install-stamp', 'binary-indep', 'binary', \
            '.PHONY'))
        self.assertEqual(targets.intersection(mfp.targets()), targets)
        self.assertEqual(mfp['build']['depends'] + mfp['test']['depends'] + \
            mfp['install']['depends'] + mfp['install-stamp']['depends'] + \
            mfp['binary-indep']['depends'] + mfp['binary']['depends'] + \
            mfp['.PHONY']['depends'], \
            ['build-stamp', 'real-install', 'build-stamp', 'install-stamp', \
            'test', 'build', 'install', 'binary-indep', 'build', 'clean', \
            'binary-indep', 'binary', 'install'])
        self.assertEqual(mfp['clean']['commands'] + \
            mfp['install-stamp']['commands'] + \
            mfp['real-install']['commands'] + \
            mfp['binary-indep']['commands'] + mfp['test']['commands'] + \
            mfp['build-stamp']['commands'], \
            ['$(RM) -r build-stamp install-stamp', '$(RM) debian/copyright', \
            '$(MAKE) clean', '$(MAKE) -C po $@', '$(RM) po/*.mo', 'dh_clean', \
            'ln -s $(CURDIR)/copyright $(CURDIR)/debian/copyright', \
            '$(MAKE) -C po install DESTDIR="$(CURDIR)/debian/linda"', \
            'touch install-stamp', \
            '$(MAKE) install DESTDIR="$(CURDIR)/debian/linda"', \
            'dh_testroot', 'dh_installdirs', 'dh_installdocs -XCVS -X.svn', \
            'dh_installman', 'dh_installchangelogs', 'dh_compress', \
            'dh_fixperms', 'dh_installdeb', 'dh_gencontrol', 'dh_md5sums', \
            'dh_builddeb', '$(MAKE) test DESTDIR="$(CURDIR)/debian/linda"', \
            'dh_testdir', 'touch build-stamp'])
    
    def test_complex_makefile(self):
        "Check a complex Makefile parses"
        mfp = MakefileParser('%s/makefile.complex' % self.dir)
        variables = Set(('SHELL', 'PACKAGE', 'SRC_DIR', 'INSTALL_UAG'))
        self.assertEqual(variables.intersection(mfp.variables()), variables)
        targets = Set(('build', 'build-stamp', 'clean', 'build-source', \
            'binary-indep', 'binary', '.PHONY'))
        self.assertEqual(targets.intersection(mfp.targets()), targets)
        self.assertEqual(mfp['binary-arch']['depends'] + \
            mfp['build-source']['depends'], [])
        self.assertEqual(mfp['binary']['commands'] + \
            mfp['binary-arch']['commands'] + mfp['.PHONY']['commands'] + \
            mfp['build']['commands'], [])
        self.assertEqual(mfp['binary']['depends'] + \
            mfp['.PHONY']['depends'] + mfp['build']['depends'] + \
            mfp['clean']['depends'] + mfp['build-stamp']['depends'] + \
            mfp['binary-indep']['depends'], ['binary-indep', \
            'binary-arch'] + ['binary', 'binary-arch', 'binary-indep', \
            'build-source', 'clean', 'patch', 'unpatch'] + ['build-stamp'] +\
            ['unpatch'] + ['patch-stamp'] + ['patch-stamp', 'build-stamp', \
            'build-source'])
        self.assertEqual(mfp['clean']['commands'] + \
            mfp['build-stamp']['commands'] + \
            mfp['binary-indep']['commands'] + \
            mfp['build-source']['commands'], ['dh_testdir', 'dh_testroot', \
                '$(RM) debian/substvars debian/*.debhelper debian/*.substvars', \
                '$(RM) -r debian/alsa-{base,source,headers}', \
                '$(RM) debian/files debian/snddevices', \
                'find . -type f -a \\( -name \\#\\*\\# -o -name \\*\\~ -o -name DEADJOE \\', \
                '-o -name \\*.orig -o -name \\*.rej -o -name \\*.bak -o -name \\', '.\\*.orig -o -name .\\*.rej -o -name .SUMS -o -name TAGS \\', \
                '-o -name .\\#\\* -o -name core -o \\( -path \\*/.deps/\\* -a -name \\', '\\*.P \\) \\) -exec rm -f {} \\;', \
                '-$(RM) build-stamp', '-$(RM) include/isapnp.h', \
                '-$(RM) debian/*substvars', '-$(RM) -r debian/patched', \
                'dh_clean -k'] + ['touch build-stamp'] + ['dh_testdir', \
                'dh_testroot', 'dh_installdirs', \
                'install -d -o root -g root -m 755 debian/alsa-headers/usr/include/sound', \
                'for f in $(shell cat debian/alsa-headers.headers); do \\', \
                'install -p -o root -g root -m 644 alsa-kernel/include/$$f debian/alsa-headers/usr/include/sound/; \\', \
                'done', 'for i in alsa-base alsa-source ; do \\', \
                'mv debian/$$i.templates debian/$$i.templates.old; \\', \
                'cat debian/$$i.templates.old debian/alsa-common.templates > debian/$$i.templates; \\', \
                'done', \
                'install $(INSTALL_UAG) -m 755 debian/alsa-base.apm \\', \
                'debian/alsa-base/etc/apm/event.d/alsa', \
                'install $(INSTALL_UAG) -m 755 snddevices \\', \
                'debian/alsa-base/usr/share/alsa-base/snddevices', \
                'install $(INSTALL_UAG) -m 644 debian/modules-snippet.conf \\', \
                'debian/alsa-base/usr/share/alsa-base/modules-snippet.conf', \
                'install $(INSTALL_UAG) -m 644 debian/alsa-base.conf \\', \
                'debian/alsa-base/usr/share/alsa-base/alsa-base.conf', \
                'install $(INSTALL_UAG) -m 644 debian/alsa-source.conf \\', \
                'debian/alsa-source/usr/share/alsa-source/alsa-source.conf', \
                'install $(INSTALL_UAG) -m 644 debian/devfs.conf \\', \
                'debian/alsa-base/etc/devfs/conf.d/alsa', \
                'install -o root -o root -m 755 debian/program-wrapper \\', \
                'debian/alsa-base/usr/share/alsa-base', \
                'install $(INSTALL_UAG) -m 755 utils/alsaconf \\', \
                'debian/alsaconf/usr/sbin', \
                'install -d -m 755 debian/alsa-base/usr/share/lintian/overrides', \
                'install -d -m 755 debian/alsa-base/usr/share/linda/overrides', \
                'install -m 644 debian/alsa-base.lintian debian/alsa-base/usr/share/lintian/overrides/alsa-base', \
                'install -m 644 debian/alsa-base.linda debian/alsa-base/usr/share/linda/overrides/alsa-base', \
                'dh_installdebconf', \
                'for i in alsa-base alsa-source ; do \\', \
                'if [ -f debian/$$i.templates.old ]; then \\', \
                'mv debian/$$i.templates.old debian/$$i.templates; \\', \
                'fi; \\', 'done', 'dh_installdocs', \
                'mv debian/alsa-source/usr/share/doc/alsa-source/patch-stamp debian/alsa-source/usr/share/doc/alsa-source/PATCHES', \
                'dh_installexamples ', \
                'DH_OPTIONS= dh_installinit --no-start -r --init-script=alsa',\
                'dh_installman', 'dh_installchangelogs ', 'dh_compress', \
                'dh_fixperms', 'dh_installdeb', 'dh_gencontrol', \
                'dh_md5sums', 'dh_builddeb'] + ['mkdir -p $(SRC_DIR)', \
                "find . \\( -path './debian' -o -name 'CVS' -o -name '.cvsignore'  -o -name '*-stamp' \\) -prune -o -print | cpio -admp $(SRC_DIR)", \
                '-$(MAKE) -C $(SRC_DIR) clean', \
                'chown -R root:src debian/alsa-source/usr/src', \
                'install -d $(SRC_DIR)/debian', \
                'install -m755 debian/alsa-source.rules $(SRC_DIR)/debian/rules', \
                'install -m644 debian/alsa-source.control $(SRC_DIR)/debian/control', \
                'install -m644 debian/changelog $(SRC_DIR)/debian/changelog',\
                'install -m644 debian/compat $(SRC_DIR)/debian/compat', \
                'install -m644 debian/alsa-modules.dirs $(SRC_DIR)/debian/dirs', \
                'find debian/alsa-source -type d -exec chmod 775 {} \\;', \
                'find debian/alsa-source/usr/src/modules -type d -exec chmod 2775 {} \\;', \
                'for i in post{inst,rm} ; do \\', \
                'major_ver="$(shell head -1 debian/changelog | cut -d\\( -f2 | cut -d\\) -f1 | cut -d. -f1-2)"; \\', \
                'cat debian/alsa-modules.$$i | sed -e "s/_MAJORVERSION_/$$major_ver/g" > $(SRC_DIR)/debian/$$i; \\', \
                'done', \
                'cd debian/alsa-source/usr/src && tar cf alsa-driver.tar modules', \
               '$(RM) -r debian/alsa-source/usr/src/modules', \
               'bzip2 -9 debian/alsa-source/usr/src/alsa-driver.tar'])

    def test_simple_include(self):
        "Check a simple Makefile using includes"
        mfp = MakefileParser('%s/makefile.includes' % self.dir)
        self.assertEqual(mfp.variables(), [])
        targets = Set(('build', 'build-stamp'))
        self.assertEqual(targets.intersection(mfp.targets()), targets)
        self.assertEqual(mfp.includes, ['/tmp/foo'])

