/*b
 * Copyright (C) 2001,2002  Rick Richardson
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Rick Richardson <rickr@mn.rr.com>
b*/

/*
 * scottrader.com
 *
 * line-oriented ASCII, using custom buffering.
 */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ncurses.h>
#include <panel.h>
#include <errno.h>
#include <time.h>
#include <sys/time.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include "error.h"
#include "rc.h"
#include "streamer.h"
#include "linuxtrade.h"
#include "optchain.h"
#include "info.h"
#include "util.h"
#include "chart.h"

extern int Debug;

#define		BUFLEN		65536
#define		SYMCNT		200
#define		SYMBUFLEN	(SYMCNT * (SYMLEN+1) + 1)

typedef struct streamerpriv
{
	char	buf[BUFLEN+1];
	char	*bufp;
	char	*bufe;

	long	ztime;

	time_t	last_keepalive;
} STREAMERDATA;

typedef struct
{
	char	*canon, *sym;
} SYMMAP;

static SYMMAP SymMap[] =
{
	{	"$DJI",		"DJI",		},
	{	"$DJT",		"TRAN",		},
	{	"$DJU",		"DJU",		},

	{	"$NYA",		"NYA",		},
	{	"$TRIN",	"USN/TRIN",	},
	{	"$TICK",	"USN/TICK",	},

	{	"$COMP",	"COMP",		},
	{	"$NDX",		"NDX",		},
	{	"$SPX",		"SPX",		},
	{	"$OEX",		"OEX",		},
	{	"$MID",		"MID",		},
	{	"$SML",		"SML",		},
	{	"$RLX",		"RLX",		},

	{	"$XAL",		"XAL",		},
	{	"$BTK",		"BTK",		},
	{	"$XBD",		"XBD",		},
	{	"$XAX",		"XAX",		},
	{	"$XCI",		"XCI",		},
	{	"$IIX",		"IIX",		},
	{	"$NWX",		"NWX",		},
	{	"$XOI",		"XOI",		},
	{	"$DRG",		"DRG",		},
	{	"$XTC",		"XTC",		},

	{	"$GSO",		"GSO",		},
	{	"$HWI",		"HWI",		},
	{	"$RUI",		"RUI",		},
	{	"$RUT",		"RUT",		},
	{	"$RUA",		"RUA",		},
	{	"$SOX",		"SOX",		},
	{	"$OSX",		"OSX",		},

	{	"$BKX",		"BKX",		},
	{	"$GOX",		"GOX",		},
	{	"$XAU",		"XAU",		},
	{	"$YLS",		"YLS",		},
	{	"$DDX",		"DDX",		},
	{	"$DOT",		"DOT",		},

	{	"$RXP",		"RXP",		},
	{	"$RXH",		"RXH",		},
	{	"$XNG",		"XNG",		},
	{	"$FPP",		"FPP",		},
	{	"$DJR",		"DJR",		},
	{	"$UTY",		"UTY",		},

	{	"$TYX",		"TYX",		},
	{	"$TRINQ",	"USQ/TRIN",	},
	{	"$TICKQ",	"USQ/TICK",	},
	{	"$TRINA",	"USA/TRIN",	},
	{	"$TICKA",	"USA/TICK",	},
	{	"$VIX",		"VIX",		},
	{	"$VXN",		"VXN",		},
	{	NULL,		NULL		}
};

//
// Convert canonical index names to/from streamer index names
//
static void
streamer_canon2sym(char *out, char *in)
{
	char	*ip, *op;
	char	*p;
	int	len;
	SYMMAP	*map;

	ip = in;
	op = out;
	for (;;)
	{
		p = strchr(ip, '|');
		if (!p) p = strchr(ip, ',');
		if (!p) p = strchr(ip, ' ');
		if (!p) p = strchr(ip, 0);

		len = p - ip;
		if (ip[len-1] != '.')
		{
			memcpy(op, ip, len);
			op[len] = 0;
		}
		else if (len <= (4+1))
		{
			// Short option symbol, 4 or less characters
			sprintf(op, "%.*s.%.2s", len-2-1, ip, ip+len-2-1);
		}
		else
		{
			// Long option symbol
			memcpy(op, ip, len-1);
			op[len-1] = 0;
		}

		for (map = SymMap; map->canon; ++map)
			if (strcmp(op, map->canon) == 0)
			{
				strcpy(op, map->sym);
				break;
			}

		if (*p == 0)
			break;

		ip += len + 1;
		op = strchr(op, 0);
		*op++ = *p;
		*op = 0;
	}
}

static void
streamer_sym2canon(char *out, char *in)
{
	SYMMAP	*map;

	for (map = SymMap; map->sym; ++map)
		if (strcmp(in, map->sym) == 0)
		{
			strcpy(out, map->canon);
			return;
		}

	if (in != out)
		strcpy(out, in);
}

//
// Special routine only in scottrader used to create a stocklist with
// all of the canonical symbols
//
int
index_list(char *buf, int size)
{
	SYMMAP	*map;
	char	*p;
	int	n;

	p = buf; *p = 0;
	for (n = 0, map = SymMap; map->sym; ++map)
	{
		if ( (p-buf+16) >= size)
			break;
		if (n)
			*p++ = ' ';
		strcpy(p, map->canon);
		p = strchr(p, 0);
		++n;
	}

	*p = 0;
	return n;
}

/*
 * Read a line from the streamer.  Discard the \n.  Buffer
 * unused data to be returned later.  Return length of line read.
 */
static int
streamer_gets(STREAMER sr, char *buf, int buflen)
{
	int	rc;
	char	*sp, *ep;

	if (sr->priv->bufp == NULL)
	{
		rc = read_timeout(sr->fd[0], sr->priv->buf, BUFLEN, 20);
		if (rc <= 0)
			return (rc);
		sr->cnt_rx += rc;
		sr->priv->bufp = sr->priv->buf;
		sr->priv->bufe = &sr->priv->buf[rc];
		*sr->priv->bufe = '\0';
		if (StreamerLog)
			fprintf(StreamerLog, "< %s\n", sr->priv->buf);
	}

	sp = sr->priv->bufp;
	ep = strchr(sp, '\n');
	while (!ep)
	{
		rc = read_timeout(sr->fd[0], sr->priv->bufe,
				&sr->priv->buf[BUFLEN] - sr->priv->bufe, 20);
		if (rc <= 0)
			return (rc);
		if (StreamerLog)
		{
			sr->priv->bufe[rc] = '\0';
			fprintf(StreamerLog, "< %s\n", sr->priv->bufe);
		}
		sr->cnt_rx += rc;
		ep = strchr(sr->priv->bufe, '\n');
		sr->priv->bufe += rc;
		*sr->priv->bufe = '\0';
	}
	*ep++ = 0;
	if (ep >= sr->priv->bufe)
		sr->priv->bufp = NULL;
	else
		sr->priv->bufp = ep;

	if ( (ep-sp) > buflen)
		error(1, "Buffer too small for streamer line\n");

	memcpy(buf, sp, ep-sp);

	if (Debug >= 5)
	{
		timestamp(stderr);
		if (ep-sp)
			fprintf(stderr, "< %s\n", buf);
		else
			fprintf(stderr, "< 0 length\n");
	}

	return (ep-sp);
}

static int
streamer_select(STREAMER sr,
		int numfd, fd_set *readfds, fd_set *writefds,
		fd_set *exceptfds, struct timeval *timeout)
{
	if (readfds && FD_ISSET(sr->fd[0], readfds) && sr->priv->bufp)
	{
			FD_ZERO(readfds);
			FD_SET(sr->fd[0], readfds);
			if (writefds)
				FD_ZERO(writefds);
			if (exceptfds)
				FD_ZERO(exceptfds);
			return 1;
	}

	if (0)
		fprintf(stderr, "do select\n");
	return select(numfd, readfds, writefds, exceptfds, timeout);
}

static void
streamer_close(STREAMER sr)
{
	close(sr->fd[0]);
	sr->fd[0] = -1;
}

static void
streamer_record(STREAMER sr, FILE *fp)
{
	sr->writefile = fp;
}

static void
streamer_init(STREAMER sr)
{
	sr->refresh = 60;
	sr->fd[0] = -1;
	sr->nfd = 1;
	sr->id[0] = 0;

	sr->priv->bufp = NULL;
	sr->priv->bufe = NULL;
	sr->priv->ztime = 0;
	time(&sr->priv->last_keepalive);
}

/*
 * Open connection to the streamer
 */
static int
streamer_open(STREAMER sr, RCFILE *rcp, FILE *readfile)
{
	struct hostent		*hep;
	struct sockaddr_in	sockaddr;
	int			rc;
	char			buf[512];
	int			conn_tries;
	int			read_tries;
	char			*password = get_rc_value(rcp, "password");
	char			*hostname = get_rc_value(rcp, "hostname");
	int			port = atoi(get_rc_value(rcp, "port"));

	streamer_init(sr);
	++sr->cnt_opens;
	++sr->cnt_realopens;
	time(&sr->time_open);
	time(&sr->time_realopen);

	if (readfile)
	{
		sr->fd[0] = fileno(readfile);
		sr->readfile = readfile;
		return sr->fd[0];
	}

	sr->readfile = NULL;

	hep = mygethostbyname(hostname);
	if (!hep)
		return (-1);

	memcpy(&sockaddr.sin_addr, hep->h_addr, hep->h_length);
	sockaddr.sin_family = AF_INET;
	sockaddr.sin_port = htons(port);

	conn_tries = 0;
reconnect:
	if (++conn_tries >= 5)
		return -4;

	if (Debug >= 5)
		fprintf(stderr, "Open socket...\n");

	sr->fd[0] = socket(AF_INET, SOCK_STREAM, 0);
	if (sr->fd[0] < 0)
		return -2;

	if (Debug >= 5)
		fprintf(stderr, "Socket fd=%d...\n", sr->fd[0]);

	if (Debug >= 5)
		fprintf(stderr, "Connect to '%s'...\n", hostname);

	rc = connect_timeout(sr->fd[0], (SA *) &sockaddr, sizeof(sockaddr), 15);
	if (rc < 0)
	{
		if (Debug >= 5)
			syserror(0, "Couldn't connect\n");
		streamer_close(sr);
		sleep(1);
		goto reconnect;
	}

	sr->priv->bufp = NULL;

	if (Debug >= 5)
		fprintf(stderr, "Authorize...\n");

	streamer_printf(sr->fd[0], "_0,%s,null, ", password);
	sleep(1);
	streamer_printf(sr->fd[0], "_Z _X ");

	/* get responses */
	if (Debug >= 5)
		fprintf(stderr, "Get handshake...\n");

	read_tries = 0;
	for (;;)
	{
		fd_set		fds;
		struct timeval	tv;

		tv.tv_sec = 3;
		tv.tv_usec = 0;
		FD_ZERO(&fds);
		FD_SET(sr->fd[0], &fds);
		rc = streamer_select(sr, sr->fd[0]+1, &fds, NULL, NULL, &tv);
		if (rc == -1 && errno != EINTR)
			syserror(1, "Select error\n");
		if (rc == 0)
		{
			if (Debug >= 5)
				fprintf(stderr, "Timeout...\n");
			if (++read_tries == 3)
			{
				streamer_close(sr);
				sleep(2);
				goto reconnect;
			}

			streamer_printf(sr->fd[0],
					"_0,%s,null, _Z _X", password);
			continue;
		}
		if (rc <= 0)
		{
			streamer_close(sr);
			sleep(2);
			goto reconnect;
		}

		rc = streamer_gets(sr, buf, sizeof(buf));
		if (rc <= 0)
		{
			if (Debug >= 1)
				syserror(0, "Can't read streamer data\n");
			streamer_close(sr);
			sleep(2);
			goto reconnect;
		}

		if (sr->writefile)
		{
			fputs(buf, sr->writefile);
			putc('\n', sr->writefile);
		}

		if (buf[0] == '@')
			break;

		if (buf[0] != '_')
			continue;

		if (buf[1] == '0')
		{
			if (Debug >= 5)
				fprintf(stderr, "Server: %s\n", buf+5);

			strncpy(sr->id, buf+5, 31);
			sr->id[31] = 0;

			streamer_printf(sr->fd[0], "_3%s ", buf+5);
		}
	}

	return 0;
}

static void
streamer_send_quickquote(STREAMER sr, char *sym)
{
	char	sbuf[32];

	if (sr->fd[0] < 0 || sr->readfile)
		return;

	streamer_canon2sym(sbuf, sym);

	streamer_printf(sr->fd[0], "_E,%s, ", sbuf);
}

static void
streamer_send_livequote(STREAMER sr, char *sym)
{
	streamer_send_quickquote(sr, sym);
}

static void
streamer_send_symbols(STREAMER sr, char *symbols, int add)
{
	char	sbuf[SYMBUFLEN];

	if (sr->fd[0] < 0 || sr->readfile)
		return;

	streamer_canon2sym(sbuf, symbols);

	streamer_printf(sr->fd[0], "_%s,%s, ", add ? "A" : "B", sbuf);
}

static void
streamer_send_symbols_end(STREAMER sr, int add, int all)
{
}

static void
streamer_send_keepalive(STREAMER sr)
{
	if (sr->fd[0] < 0 || sr->readfile)
		return;

	streamer_printf(sr->fd[0], "_X ");
}

static void
streamer_send_disconnect(STREAMER sr)
{
	if (sr->fd[0] < 0 || sr->readfile)
		return;

	streamer_printf(sr->fd[0], "_1 ");
}

static void
streamer_send_top10(STREAMER sr, char market, int type, int add)
{
	if (sr->fd[0] < 0 || sr->readfile)
		return;

	streamer_printf(sr->fd[0], "_T,%d,%c,%d, ",
			add ? 1 : 2,
			market,
			type
			);
}

static void
streamer_send_movers(STREAMER sr, int on)
{
	if (sr->fd[0] < 0 || sr->readfile)
		return;

	streamer_printf(sr->fd[0], "_H%d ", on ? 1 : 2);
}

static void
streamer_send_info(STREAMER sr, char *sym, int type)
{
	char	sbuf[32];

	if (sr->fd[0] < 0 || sr->readfile)
		return;

	streamer_canon2sym(sbuf, sym);

	streamer_printf(sr->fd[0], "_G,%d,%s, ", type, sbuf);
}

static void
streamer_send_optchain(STREAMER sr, char *sym)
{
	if (sr->fd[0] < 0 || sr->readfile)
		return;

	streamer_printf(sr->fd[0], "_K%s ", sym);
}

static int
streamer_send_chart(STREAMER sr, char *sym, int freq, int periods, int days)
{
	char	sbuf[32];

	if (sr->fd[0] < 0 || sr->readfile)
		return SR_ERR;

	streamer_canon2sym(sbuf, sym);

	streamer_printf(sr->fd[0], "_D,%d,%d,%s, ", freq, periods, sbuf);
	return 0;
}

static void
do_F(STREAMER sr, char *buf)
{
	char	exchange;
	int	i1, flag, i3;
	int	rc;
	QUOTE	q;
	STOCK	*sp;

	// _F,INTC,30.000000,30.010000,30.010000,30.510000,29.140000,
	//    1466F30,9AED,A,1,Q,Q,29.950000,64,1,1,Q,0,
	rc = sscanf(buf+3,
		"%[^,],%f,%f,%f,%f,%f,%lx,%lx,%x,%x,%c,%c,%f,%x,%d,%d,%c,%d,",
		q.sym,
		&q.bid,
		&q.ask,
		&q.last,
		&q.high,
		&q.low,

		&q.volume,
		&q.time,
		&q.bid_size,
		&q.ask_size,
		&q.bid_id,
		&q.ask_id,
		&q.close,
		&q.last_size,
		&i1,		//???
		&flag,
		&exchange,
		&i3
		);

	q.delayed = 0;

	if (rc != 18)
	{
		if (Debug)
			error(0, "Bogus _F rc=%d '%s'\n", rc, buf);
		return;
	}

	// Funny time?
	if (q.time < sr->priv->ztime)
	    q.time = sr->priv->ztime;

	streamer_sym2canon(q.sym, q.sym);

	sp = find_stock(q.sym);
	if (!sp)
	{
		// Try canonical option symbol first
		char	buf[32];
		char	*s = q.sym, *d = buf;
		while (*s)
		{
			if (*s != '.')
				*d++ = *s;
			++s;
		}
		*d++ = '.';
		*d = 0;
		sp = find_stock(buf);
		if (sp)
		{
			strcpy(q.sym, buf);
			display_quote(&q, 0);
			return;
		}
	}

	display_quote(&q, 0);
	optchain_quote(&q);
}

//
// Old:
// _P,INTC,31.210000,0,0,31.200000,Q,Q,2,5,1,
// New:
// _Y,INTC,31.200000,31.210000,Q,Q,2,5,1,^M

static void
do_P(char *buf, int freeserver)
{
	int	rc;
	int	i1, i2, i3;
	STOCK	*sp;
	QUOTE	q;

	rc = sscanf(buf+3, "%[^,],", q.sym);
	if (rc != 1)
		return;

	streamer_sym2canon(q.sym, q.sym);

	sp = find_stock(q.sym);
	if (!sp)
	{
		// Try canonical option symbol
		char	buf[32];
		char	*s = q.sym, *d = buf;
		while (*s)
		{
			if (*s != '.')
				*d++ = *s;
			++s;
		}
		*d++ = '.';
		*d = 0;
		sp = find_stock(buf);
		if (!sp)
			return;
		strcpy(q.sym, buf);
	}

	q = sp->cur;

	// Quote refresh message
	//
	//_P,JNPR,24.740000,0,0,24.730000,Q,Q,7,3,1,
	//_P,JNPR,24.750000,0,0,24.730000,Q,Q,1,32,1,
	
	if (freeserver)
	{
		rc = sscanf(buf+3, "%[^,],%f,%d,%d,%f,%c,%c,%x,%x,%d,",
			q.sym,
			&q.ask,
			&i1,		//unused
			&i2,		//unused
			&q.bid,
			&q.bid_id,
			&q.ask_id,
			&q.bid_size,
			&q.ask_size,
			&i3		//???
			);
		if (rc != 10)
		{
			if (Debug)
				error(0, "Bogus _P rc=%d '%s'\n", rc, buf);
			return;
		}
	}
	else
	{
		rc = sscanf(buf+3, "%[^,],%f,%f,%c,%c,%x,%x,%d,",
			q.sym,
			&q.bid,
			&q.ask,
			&q.bid_id,
			&q.ask_id,
			&q.bid_size,
			&q.ask_size,
			&i3		//???
			);
		if (rc != 8)
		{
			if (Debug)
				error(0, "Bogus _Y rc=%d '%s'\n", rc, buf);
			return;
		}
	}

	streamer_sym2canon(q.sym, q.sym);

	display_quote(&q, 0);
}

//
// Old:
// _Y,1,INTC,Q,Q,31.190000,31.210000,31.200000,A,7,ED8,1,23D1C04,CE4C,1,
// New:
// _P,1,INTC,31.210000,Q,Q,31.190000,31.200000,A,7,ED8,1,23D91FC,CE4C,1,^M

static void
do_Y(char *buf, int freeserver)
{
	int	rc;
	int	tradetype;
	int	tick;
	int	i1;
	STOCK	*sp;
	QUOTE	q;

	rc = sscanf(buf+3, "%*d,%[^,],", q.sym);
	if (rc != 1)
		return;

	streamer_sym2canon(q.sym, q.sym);

	sp = find_stock(q.sym);
	if (!sp)
		return;

	q = sp->cur;

	//
	//  If bid is above ask and last, then does that mean
	//  thats a off-hours trade at the bid??? 

	// Quote trade message
	//
	//_Y,1,JNPR,Q,Q,24.730000,24.750000,24.750000,1,32,13EC,1,51EB18,9AF0,1,
	//_Y,2,INTC,Q,Q,31.410000,31.060000,31.200000,9,1,12C,1,2DA78,7F4C,1,
	
	if (freeserver)
	{
		rc = sscanf(buf+3, "%d,%[^,],%c,%c,%f,%f,%f,"
				"%x,%x,%x,%d,%lx,%lx,%d",
				&tradetype,
				q.sym,
				&q.bid_id,
				&q.ask_id,
				&q.bid,
				&q.last,
				&q.ask,
				&q.bid_size,
				&q.ask_size,
				&q.last_size,
				&tick,
				&q.volume,
				&q.time,
				&i1		//???
				);

		if (rc != 14)
		{
			if (Debug)
				error(0, "Bogus _Y rc=%d '%s'\n", rc, buf);
			return;
		}
	}
	else
	{
// _Y,1,INTC,Q,Q,31.190000,31.210000,31.200000,A,7,ED8,1,23D1C04,CE4C,1,
// New:
// _P,1,INTC,31.210000,Q,Q,31.190000,31.200000,A,7,ED8,1,23D91FC,CE4C,1,^M
		rc = sscanf(buf+3, "%d,%[^,],%f,%c,%c,%f,%f,"
				"%x,%x,%x,%d,%lx,%lx,%d",
				&tradetype,
				q.sym,
				&q.last,
				&q.bid_id,
				&q.ask_id,
				&q.bid,
				&q.ask,
				&q.bid_size,
				&q.ask_size,
				&q.last_size,
				&tick,
				&q.volume,
				&q.time,
				&i1		//???
				);

		if (rc != 14)
		{
			if (Debug)
				error(0, "Bogus _P rc=%d '%s'\n", rc, buf);
			return;
		}
	}

	streamer_sym2canon(q.sym, q.sym);

	display_quote(&q, tradetype);
}

static void
do_E(char *buf)
{
	QUICKQUOTE	qq;
	LIVEQUOTE	lq;
	int		rc;

	// sym,bid,ask,last,high,low,
	// volume,timetrade,bidsize,asksize,bidexch,askexch,
	// prevclose,lastsize,lasttick(up/down),CUSIP,
	// fullname,exchange,sharesout,description,
	// annualdiv(.08),divpayyy,divpaymm,divpaydd,
	// hi52,lo52,lasteps(0.12),cureps(0.10),yeareps(1.07),???,???
	// [$chng=UNCHG,%change=UNCH,yield=0.26,PEratio=28.794]
	// > _E,INTC,
	// < _E,INTC,30.700000,30.790000,30.810000,30.840000,29.480000,
	// 34B7118,1ABE7,A,F,Q,Q,
	// 30.810000,E74,-1,458140100,
	// INTEL CORP                              ,Q,669D88,COMMON STOCK ,
	// 800,2001,11,30,
	// 46.750000,18.960000,12,10,107,52,1,

	rc = sscanf(buf+3,
		"%[^,],%f,%f,%f,%f,%f,"
		"%x,%x,%x,%x,%c,%c,"
		"%f,%x,%d,%[^,],"
		"%[^,],%c,%x,%[^,],"
		"%d,%d,%d,%d,"
		"%f,%f,%d,%d,%d,"
		, qq.sym, &qq.bid, &qq.ask, &qq.last, &qq.high, &qq.low
		, &qq.volume, &qq.timetrade, &qq.bid_size, &qq.ask_size
		, &qq.bid_id, &qq.ask_id
		, &qq.prev_close, &qq.last_size, &qq.last_tick, qq.cusip
		, qq.fullname, &qq.exchange, &qq.sharesout, qq.description
		, &qq.annualdiv, &qq.divYY, &qq.divMM, &qq.divDD
		, &qq.high52, &qq.low52, &qq.last_eps, &qq.cur_eps, &qq.year_eps
		);

	if (rc != 29)
	{
		if (Debug)
			error(0, "Bogus _E rc=%d '%s'\n", rc, buf);
		return;
	}

	streamer_sym2canon(qq.sym, qq.sym);

	strcpy(lq.sym, qq.sym);
	lq.last = qq.last;
	lq.close = qq.prev_close;

	info_quickquote(&qq);

	display_livequote(&lq);
}

void
do_Z(STREAMER sr, char *buf)
{
	int	rc;
	long	ztime;

	rc = sscanf(buf+3, "%ld", &ztime);
	if (rc != 1)
		return;

	sr->priv->ztime = ztime;
}

#if 0
void
do_W(char *buf)
{
}

void
do_0(char *buf)
{
	// _0,0 - authentication error
	// _0,1 - connected to ...
}

void
do_S(char *buf)
{
}

void
do_G(char *buf)
{
	// G2 - fundamental dividends data
	// G3 - fundamental split data
	// G4 - fundamental calculated data
	// G5 - fundamental hist calc data
	// G6 - fundamental earnings data
	// G7 - fundamental balance sheet data
}
#endif

static void
do_H(char *buf)
{
	int	rc;
	MKTMOVER mm;
	int	i1;

	// Market Movers message
	rc = sscanf(buf+3, "%[^,],%*[^,],",
			mm.sym
			);		// delete???

	rc = sscanf(buf+3, "%[^,],%lf,%lf,%d,%d,%d",
			mm.sym,
			&mm.last,
			&mm.change,
			&mm.time,
			&mm.hilo,	// 1, 2, 3, or 4
			&i1
			);

	if (rc != 6)
		return;

	if (strncmp(mm.sym, "TEST", 4) == 0)
		return;

	// Unknown with this streamer
	mm.market = 0;
	mm.count = 0;

	display_mktmover(&mm);
}

#if 0
void
do_L(char *buf)
{
	int	rc;
	char	symbol[SYMLEN];
	float	f1, f2, f3;
	int	i1;

	// Watchlist message
	// L1
		rc = sscanf(buf+3, "%[^,],%f,%f,%d",
			symbol,
			&f1,
			&f2,
			&i1
			);
	// L2
		rc = sscanf(buf+3, "%[^,],%f",
			symbol,
			&f3
			);
}
#endif

static void
do_K(char *buf)
{
	OPTCHAIN	oc;
	int		rc;

	// WE SEND _KINTC
	// symbol,index,count,symbol,??,date,price,
	// _K,INTC,1,192,INQAE,0,Jan 2002,25.000000,

	rc = sscanf(buf+3,
		"%[^,],%d,%d,%[^,],%d,%[^,],%f,"
		, oc.sym, &oc.index, &oc.count, oc.optsym
		, &oc.type, oc.date, &oc.price
		);

	if (rc != 7)
	{
		if (Debug)
			error(0, "Bogus _K rc=%d '%s'\n", rc, buf);
		return;
	}

	optchain_optchain(&oc);
}

static void
do_T(char *buf)
{
	int	rc;
	TOP10	top10;

	int	i1,i2,i3;

	// > _T,1,Q,5 	(Add)
	// > _T,2,Q,5 	(Remove)
	// < _T,Q,5,1,MICT,-0.750000,13754,1,1,5,	# NASD top % losers
	//
	// Type...
	// 1 = Top Volume
	// 2 = Top Net Gainers
	// 3 = Top Net Losers
	// 4 = Top % Gainers
	// 5 = Top % Losers

	rc = sscanf(buf+3, "%c,%d,%d,%[^,],%lf,%x,%d,%d,%d",
			&top10.market,
			&top10.type,
			&top10.rank,
			top10.sym,
			&top10.change,
			&top10.volume,
			&i1,
			&i2,
			&i3
			);

	if (rc != 9)
	{
		if (Debug)
			error(0, "Bogus _T rc=%d '%s'\n", rc, buf);
		return;
	}

	display_top10(&top10);
}


static int
streamer_process(STREAMER sr, int fdindex)
{
	int	len;
	char	buf[2048];

	len = streamer_gets(sr, buf, sizeof(buf));
	if (len <= 0)
	{
		if (sr->readfile)
		{
			if (sr->fd[0] >= 0)
			{
				fclose(sr->readfile);
				sr->fd[0] = -1;
			}
			return (0);
		}
		return (-1);
	}

	if (sr->writefile)
	{
		fputs(buf, sr->writefile);
		putc('\n', sr->writefile);
	}

	if (buf[0] == ',')
		return 0;	// crap from _K command

	if (buf[0] == '@')
		return 0;	// delayed handshake?

	if (buf[0] != '_')
		error(1, "bad msg '%s'\n", buf);

	if (Debug >= 5)
		fprintf(stderr, "<%s>\n", buf);

	switch (buf[1])
	{
	case 'Q':
		chart_dataQ(buf);
		break;
	case 'F':
		do_F(sr, buf);
		break;
		//
		// The ST customers streamers seem to reverse the meaning of
		// the _Y and _P commands.
		//
	case 'P':
		if (buf[3] >= '0' && buf[3] <= '9')
			do_Y(buf, FALSE);
		else
			do_P(buf, TRUE);
		break;
	case 'Y':
		if (buf[3] < '0' || buf[3] > '9')
			do_P(buf, FALSE);
		else
			do_Y(buf, TRUE);
		break;
	case 'T':
		do_T(buf);
		break;
	case 'H':
		do_H(buf);
		break;
	case 'E':
		do_E(buf);
		break;
	case 'G':
		info_dataG(buf);
		break;
	case 'K':
		do_K(buf);
		break;
	case 'Z':
		do_Z(sr, buf);
		break;
	case '0':
		// Delayed ID?
		break;
	default:
		fprintf(stderr, "??? %s\n", buf);
		break;
	}

	return (0);
}

static void
streamer_timetick(STREAMER sr, time_t now)
{
	if (now > sr->priv->last_keepalive + 5)
	{
		streamer_send_keepalive(sr);
		sr->priv->last_keepalive = now;

		if (sr->writefile)
			fflush(sr->writefile);
	}
}

static void
nullsr(STREAMER sr)
{
}

STREAMER
scottrader_new(void)
{
	STREAMER	sr;

	sr = (STREAMER) malloc(sizeof(*sr));
	if (!sr)
		return NULL;
	memset(sr, 0, sizeof(*sr));

	sr->open = streamer_open;
	sr->select = streamer_select;
	sr->close = streamer_close;
	sr->record = streamer_record;
	sr->timetick = streamer_timetick;

	sr->send_quickquote = streamer_send_quickquote;
	sr->send_livequote = streamer_send_livequote;
	sr->send_livequote_end = nullsr;
	sr->send_symbols = streamer_send_symbols;
	sr->send_symbols_end = streamer_send_symbols_end;

	sr->send_disconnect = streamer_send_disconnect;
	sr->send_top10 = streamer_send_top10;
	sr->send_movers = streamer_send_movers;
	sr->send_info = streamer_send_info;
	sr->send_optchain = streamer_send_optchain;
	sr->send_chart = streamer_send_chart;

	sr->process = streamer_process;

	sr->priv = (STREAMERDATA *) malloc(sizeof(*sr->priv));
	if (!sr->priv)
	{
		free(sr);
		return NULL;
	}

	time(&sr->time_start);

	streamer_init(sr);

	return (sr);
}
