/*b
 * Copyright (C) 2001,2002  Rick Richardson
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Rick Richardson <rickr@mn.rr.com>
b*/

#include "config.h"
#include <locale.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <ncurses.h>
#include <panel.h>
#include <errno.h>
#include <ctype.h>
#include <time.h>
#include <sys/time.h>
#include <unistd.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <netdb.h>
#include <netinet/in.h>
#include "error.h"
#include "rc.h"
#include "streamer.h"
#include "pref.h"
#include "srpref.h"
#include "linuxtrade.h"
#include "stocklist.h"
#include "toolmode.h"
#include "util.h"

#define	DATEPER	(1*60)

static time_t	LastRefresh;

void
tool_quote(QUOTE *q, int tradetype)
{
	STOCK		*sp;

	sp = find_stock(q->sym);
	if (!sp)
		return;

	if (sp->nquotes++ == 0)
		sp->cur = *q;
	sp->old = sp->cur;

	// Quote update records don't include high/low
	if (q->last > q->high)
		q->high = q->last;
	if (q->last < q->low)
		q->low = q->last;

	sp->cur = *q;

	printf("%s|%s|", tradetype ? "TRADE" : "QUOTE", sp->sym);

	printf("%ld|", q->time);

	printf("%.2f|", q->bid);
	printf("%d|", q->bid_size);
	printf("%c|", q->bid_id ? q->bid_id : '?');

	printf("%.2f|", q->ask);
	printf("%d|", q->ask_size);
	printf("%c|", q->ask_id ? q->ask_id : '?');

	printf("%.2f|", q->last);
	printf("%d|", q->last_size);

	printf("%ld|", q->volume);

	printf("%.2f|", q->high);
	printf("%.2f|", q->low);

	printf("%.2f|", q->close);

	printf("\n");
}

static void
symrec(void)
{
	int	i;

	printf("SYMS|");
	for (i = 0; i < NumStock; ++i)
		printf("%s%s", i ? "," : "", Stock[i].sym);
	printf("|\n");
}

static void
daterec(void)
{
	time_t		now;
	struct tm	*tmp;
	char		buf[64];
	char		*datefmt = "%x|%T|%Z";

	time(&now);
	tmp = localtime(&now);
	strftime(buf, sizeof(buf), datefmt, tmp);

	printf("TIME|");
	printf("%s|%ld|", buf, now);
	printf("\n");
}

int
tool_command(STREAMER sr)
{
	char	buf[512];
	char	*p;
	int	n;

	do
	{
		if (fgets(buf, sizeof(buf), stdin) == NULL)
			return 0;

		p = strchr(buf, '\n');
		if (p) *p = 0;

		for (p = buf+1; *p == ' '; ++p)
			{}

		switch (buf[0])
		{
		case 'a':
			for (p = buf+1; *p == ' '; ++p)
				{}
			if (!*p)
				break;
			add_from_stocklist(p, TRUE);
			symrec();
			(*sr->send_symbols)(sr, p, TRUE);
			(*sr->send_symbols_end)(sr, TRUE, FALSE);
			time(&LastRefresh);
			break;
		case 'D':
			for (p = buf+1; *p == ' '; ++p)
				{}
			if (!*p)
				break;
			n = del_stock(p);
			if (n == -1)
				break;
			symrec();
			(*sr->send_symbols)(sr, p, FALSE);
			(*sr->send_symbols_end)(sr, TRUE, FALSE);
			break;
		case 'q':
			printf("DISC|104|Command request|\n");
			exit(0);
		}
	} while FRcnt(stdin);

	return 1;
}

/*
 * Run in tool mode
 */
void
tool_mode(int mode, char *symbols)
{
	char		*hostname;
	char		*password;
	STREAMER	sr = NULL;
	time_t		last_tick;
	char		*streamer = get_rc_value(RcFile, "streamer");
	int		i, rc;
	int		input = 1;
	time_t		testtime;
	time_t		datetime;
	int		testconn = 0;
	int		refreshmode;

	mode &= 0x0f;
	testconn = 0;

	add_from_stocklist(symbols, FALSE);

newstreamer:
	NewStreamer = FALSE;
	sr = (*SrCur->new)();

restart:
	/*
	 * Open connection to streamer
	 */
	printf("OPEN|%s|\n", streamer);

	hostname = get_rc_value(SrCur->rcfile, "hostname");
	password = get_rc_value(SrCur->rcfile, "password");
	refreshmode = atoi(get_rc_value(SrCur->rcfile, "refreshmode"));

	(*sr->record)(sr, WriteFile);
	rc = (*sr->open)(sr, SrCur->rcfile, ReadFile);

	if (rc < 0)
	{
		// TODO connection failure
		printf("DISC|100|Streamer connection failure|\n");
failure:
		if (mode == 2)
		{
			sleep(2);
			goto restart;
		}
		else if (mode == 3)
		{
			char	*nstreamer = cycle_streamer(1);

			if (strcmp(streamer, nstreamer))
			{
				(*sr->close)(sr);
				streamer_free(sr);
				set_rc_value(RcFile, "streamer", nstreamer);
				streamer = get_rc_value(RcFile, "streamer");
				SrCur = find_streamer(streamer);
				goto newstreamer;
			}
			else
			{
				sleep(2);
				goto restart;
			}
		}
		else
			exit(1);
	}

	/*
	 * Streamer is connected
	 */
	printf("CONNECT|%s|\n", sr->id);
	daterec();

	/*
	 * Send initial requests
	 */
	if (NumStock)
	{
		symrec();
		(*sr->send_symbols)(sr, symbols, TRUE);
		(*sr->send_symbols_end)(sr, TRUE, TRUE);
	}

	printf("COLS|Time|Bid|BidSize|BidID|Ask|AskSize|AskID|");
	printf("Last|LastSize|Volume|High|Low|PrevClose|");
	printf("\n");

	time(&LastRefresh);
	time(&last_tick);
	time(&testtime);
	time(&datetime);
	datetime += DATEPER;
	datetime /= DATEPER;
	datetime *= DATEPER;

	for (;;)
	{
		fd_set		fds;
		int		nfds;
		struct timeval	tv;
		time_t		now;

		time(&now);

		if (now >= datetime)
		{
			daterec();
			datetime = now + DATEPER;
			datetime /= DATEPER;
			datetime *= DATEPER;
		}

		if (StDemoTime && now >= StDemoTime)
		{
			printf("DISC|200|Demo password time limit|\n");
			exit(1);
		}
		if (DemoTime && now >= DemoTime)
		{
			printf("DISC|201|Test drive time limit|\n");
			exit(1);
		}

		if (now >= (last_tick+1))
		{
			if (sr->timetick)
				(*sr->timetick)(sr, now);

			fflush(stdout);

			last_tick = now;
		}

		if (testconn && now >= (testtime+10))
		{
			printf("DISC|104|Conn test|\n");
			goto failure;
		}

		if (sr->refresh && refreshmode
				&& now >= (LastRefresh + sr->refresh))
		{
			// Refresh symbols.  Sometimes the server
			// doesn't automatically send us updates.
			// Scottrader does this with index symbols.
			int	n = 0;
			for (i = 0; i < NumStock; ++i)
				if (refreshmode == 2 || Stock[i].sym[0] == '$')
				{
					++n;
					(*sr->send_symbols)(sr,
						    Stock[i].sym, TRUE);
				}
			if (n)
				(*sr->send_symbols_end)(sr, TRUE, TRUE);
			LastRefresh = now;
		}

		tv.tv_sec = 1;
		tv.tv_usec = 0;
		FD_ZERO(&fds);
		if (input)
			FD_SET(0, &fds);

		if (sr->fd[0] <= 0 && ReadFile)
			break;
		FD_SET(sr->fd[0], &fds);
		nfds = sr->fd[0];

		rc = (*sr->select)(sr, nfds+1, &fds, NULL, NULL, &tv);
		if (rc == -1 && errno != EINTR)
		{
			printf("DISC|101|Select failure|\n");
			goto failure;
		}

		if (rc <= 0)
			continue;

		if (sr->fd[0] <= 0)
			break;

		if (FD_ISSET(0, &fds))
		{
			input = tool_command(sr);
		}

		if (sr->fd[0] > 0 && FD_ISSET(sr->fd[0], &fds))
		{
			rc = (*sr->process)(sr, 0);
			if (rc < 0)
			{
				(*sr->close)(sr);
				if (rc == SR_AUTH)
				{
					printf("DISC|202|Auth failure|\n");
					exit(1);
				}

				printf("DISC|102|Read failure|\n");
				goto failure;
			}
		}

		if (mode == 4)
		{
			int	cnt = 0;

			for (i = 0; i < NumStock; ++i)
				if (Stock[i].nquotes)
					++cnt;
			if (cnt == NumStock)
				break;
		}
	}

	printf("DISC|103|End of file|\n");
	exit(0);
}
