/*
 *   Copyright (C) 2002,2003 by Jonathan Naylor G4KLX/HB9DRD
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <sys/time.h>
#include <math.h>

#include "common/FFT.h"
#include "common/SFFT.h"
#include "common/SoundFile.h"
#include "common/Exception.h"

#include "jt44/JT44Lookups.h"
#include "jt44/JT44Defs.h"

#include "JT44Decoder.h"

int main(int argc, char **argv)
{
	if (argc < 2) {
		::fprintf(stderr, "Usage: JT44Decoder <filename>\n");
		return 1;
	}

	wxString fileName = wxString(argv[1]);

	try {
		CJT44Decoder decoder(fileName);
		decoder.run();
	}
	catch (CException& ex) {
		::fprintf(stderr, "Error: %s\n", ex.getMessage().c_str());
		return 1;
	}
	catch (...) {
		::fprintf(stderr, "An exception has occurred\n");
		return 1;
	}

	return 0;
}

CJT44Decoder::CJT44Decoder(const wxString& fileName) :
m_fileName(fileName),
m_noise()
{
}

CJT44Decoder::~CJT44Decoder()
{
}

void CJT44Decoder::run()
{
	CSoundDev* file = new CSoundFile;

	file->openRead(m_fileName, JT44_SAMPLE_RATE, 16);

//	CSFFT* sfft = new CSFFT(JT44_FFT_LENGTH, JT44_JT44_SYNCBIN_FIRST, JT44_JT44_SYNCBIN_LAST);
	CFFT fft(JT44_FFT_LENGTH);

	double* in = new double[JT44_SOUNDBUF_LENGTH];

	CAverage** correlations = new CAverage*[JT44_SYNCBIN_LAST - JT44_SYNCBIN_FIRST];
	for (int i = 0; i < (JT44_SYNCBIN_LAST - JT44_SYNCBIN_FIRST); i++)
		correlations[i] = new CAverage[(6 * JT44_SAMPLE_RATE) / JT44_SKIP_FACTOR];

	double* audioSamples = new double[31 * JT44_SAMPLE_RATE];

	int samples = 0;

	struct timeval tvStart, tvNow;
	::gettimeofday(&tvStart, NULL);
	tvNow = tvStart;

	while (samples < (30 * JT44_SAMPLE_RATE) && (tvNow.tv_sec - tvStart.tv_sec) < 29) {
		int len = JT44_SOUNDBUF_LENGTH;
		if (!file->read(in, len))
			break;

		for (int i = 0; i < len; i++, samples++) {
			audioSamples[samples] = in[i];

			if (samples >= JT44_FFT_LENGTH && (samples % JT44_SKIP_FACTOR) == 0) {
				double* bins = fft.process(audioSamples + samples - JT44_FFT_LENGTH);

				storeCorrelations(correlations, bins, samples);
				storeNoise(bins);
			}
		}

		::gettimeofday(&tvNow, NULL);
	}

	file->close();

	delete file;
//	delete sfft;
	delete[] in;

	::fprintf(stdout, "Received %d samples of data\n", samples);

	int syncBin;
	int offset;
	findCorrelation(correlations, syncBin, offset);

	for (int i = 0; i < (JT44_SYNCBIN_LAST - JT44_SYNCBIN_FIRST); i++)
		delete[] correlations[i];
	delete[] correlations;

	double quality;
	correlate(audioSamples, samples, syncBin, offset, quality);

	decode(audioSamples, samples, syncBin, offset);

	delete[] audioSamples;
}

#if 0
void CJT44Decoder::storeCorrelations(CAverage** correlations, double* data, int samples) const
{
	CJT44Lookups lookups;

	for (int t = 0; t < (6 * JT44_SAMPLE_RATE) / JT44_SKIP_FACTOR; t++) {
		int timeOffset = (t * JT44_SKIP_FACTOR) - (1 * JT44_SAMPLE_RATE);
		int      diffT = samples - timeOffset;

		int n = diffT / JT44_SYMBOL_LENGTH;

		if (n >= 0 && (diffT % JT44_SYMBOL_LENGTH) == 0 && n < 135) {
			double mult = lookups.lookupSync(n) ? 1.0 : -1.0;

			for (int bin = 0; bin < (JT44_SYNCBIN_LAST - JT44_SYNCBIN_FIRST); bin++)
				correlations[bin][t].addValue(mult * data[bin + JT44_SYNCBIN_FIRST]);
		}
	}
}
#endif

void CJT44Decoder::storeCorrelations(CAverage** correlations, double* data, int samples) const
{
	CJT44Lookups lookups;

	int sampleNo = samples / JT44_SKIP_FACTOR;

	for (int t = 0; t < (6 * JT44_SAMPLE_RATE) / JT44_SKIP_FACTOR; t++) {
		int n = sampleNo - t + (1 * JT44_SAMPLE_RATE) / JT44_SKIP_FACTOR;

		int pos = n / JT44_SKIP_RATE;

		if (pos >= 0 && (n % JT44_SKIP_RATE) == 0 && pos < 135) {
			double mult = lookups.lookupSync(pos) ? 1.0 : -1.0;

//			printf("sample:%d t:%d pos:%d val:%f\n", sampleNo, t, pos, data[236]);

			for (int bin = 0; bin < (JT44_SYNCBIN_LAST - JT44_SYNCBIN_FIRST); bin++)
				correlations[bin][t].addValue(mult * data[bin + JT44_SYNCBIN_FIRST]);
		}
	}
}

void CJT44Decoder::storeNoise(double* data)
{
	// 0 - 2500 Hz
	for (int i = 0; i < 464; i++)
		m_noise.addValue(data[i]);
}

void CJT44Decoder::findCorrelation(CAverage** correlations, int& syncBin, int& offset) const
{
	double value = 0.0;
	offset = 0;
	syncBin = 0;

	for (int i = 0; i < (JT44_SYNCBIN_LAST - JT44_SYNCBIN_FIRST); i++) {
		for (int j = 0; j < (6 * JT44_SAMPLE_RATE) / JT44_SKIP_FACTOR; j++) {
			if (correlations[i][j].getAverage() > value) {
				value   = correlations[i][j].getAverage();
				offset  = j * JT44_SKIP_FACTOR - JT44_SAMPLE_RATE;
				syncBin = i + JT44_SYNCBIN_FIRST;
			}
		}
	}

	::fprintf(stdout, "Best sync bin is %d with time offset of %d and quality of %f\n", syncBin, offset, value);
}

void CJT44Decoder::correlate(double* audioSamples, int samples, int syncBin, int& offset, double& quality) const
{
	quality = 0.0;

	CJT44Lookups lookup;

	double* syncSamples = new double[samples];

	CSFFT* sfft = new CSFFT(JT44_FFT_LENGTH, syncBin, syncBin + 1);

	CAverage level;

	for (int i = 0; i < samples; i++) {
		double* bins = sfft->process(audioSamples[i]);
		syncSamples[i] = bins[syncBin];
		level.addValue(bins[syncBin]);
	}

	delete sfft;

	int newOffset = offset;
//	for (int timeOffset = -(1 * JT44_SAMPLE_RATE); timeOffset < +(5 * JT44_SAMPLE_RATE); timeOffset++) {
	for (int timeOffset = offset - JT44_SKIP_FACTOR; timeOffset < (offset + JT44_SKIP_FACTOR); timeOffset++) {
		CAverage dotProduct;

		for (int i = 0; i < 135; i++) {
			int dataIndex = timeOffset + (i * JT44_SYMBOL_LENGTH);

			if (dataIndex >= 0 && dataIndex < samples) {
				double mult = lookup.lookupSync(i) ? 1.0 : -1.0;
				dotProduct.addValue(mult * syncSamples[dataIndex]);
			}
		}

		if (dotProduct.getAverage() > quality) {
			quality   = dotProduct.getAverage();
			newOffset = timeOffset;
		}
	}

	delete[] syncSamples;

	offset = newOffset;
	double wsjtTime = (offset - JT44_SAMPLE_RATE) / double(JT44_SAMPLE_RATE);

	::fprintf(stdout, "Maximum correlation in bin %d at time offset of %d\n", syncBin, offset);
	::fprintf(stdout, "In WSJT, time offset of %fs\n", wsjtTime);
	::fprintf(stdout, "Stength %.0fdB\n", 15.0 * ::log10(quality / m_noise.getAverage()) - 19.0 + 0.5);
	::fprintf(stdout, "Sync values max/ave/min: %f %f %f\n", quality, level.getAverage(), quality / level.getAverage());
}

void CJT44Decoder::decode(double* audioSamples, int samples, int syncBin, int timeOffset) const
{
	CAverage** aveSymbols = new CAverage*[JT44_ALPHABET_COUNT];
	for (int i = 0; i < JT44_ALPHABET_COUNT; i++)
		aveSymbols[i] = new CAverage[JT44_MESSAGE_LENGTH];

	CFFT* fft = new CFFT(JT44_FFT_LENGTH);

	CJT44Lookups lookups;

	for (int i = 0; i < 135; i++) {
		int index = timeOffset + (i * JT44_SYMBOL_LENGTH) - JT44_FFT_LENGTH;
		int n     = lookups.lookupPosition(i);

		if (n != -1 && index >= 0 && index < (samples - JT44_FFT_LENGTH)) {
			double* bins = fft->process(audioSamples + index);

			for (int j = 0; j < JT44_ALPHABET_COUNT; j++) {
				int binNo = syncBin + (j * 2) + 6;
				aveSymbols[j][n].addValue(bins[binNo]);
			}
		}
	}

	delete fft;

	double correlate = 0.0;

	for (int i = 0; i < JT44_MESSAGE_LENGTH; i++) {
		double highestVal = aveSymbols[0][i].getAverage();
		int           bin = 0;

		for (int j = 1; j < JT44_ALPHABET_COUNT; j++) {
			double val = aveSymbols[j][i].getAverage();

			if (val > highestVal) {
				highestVal = val;
				bin        = j;
			}
		}

		correlate += highestVal;

		int c = lookups.lookupTone(bin + 121);

		if (c != -1)
			::fputc(c, stdout);
		else
			::fputs("<unknown>", stdout);
	}

	::fprintf(stdout, " correlation: %f\n", correlate);
	::fprintf(stdout, "Strength: %f, %fdB\n", m_noise.getMaximum() / m_noise.getAverage(), 10.0 * ::log10(m_noise.getMaximum() / m_noise.getAverage()));

	for (int i = 0; i < JT44_ALPHABET_COUNT; i++)
		delete[] aveSymbols[i];
	delete[] aveSymbols;
}
