#include <Python.h>
#include <pygobject.h>
#include <gdk-pixbuf/gdk-pixbuf.h>

/* Compute the number of bytes used to represent a pixel. */
static guint compute_pixel_size(const GdkPixbuf *pixbuf)
{
  guint n_channels;
  
  n_channels = gdk_pixbuf_get_n_channels(pixbuf);
  if (gdk_pixbuf_get_has_alpha(pixbuf))
    return n_channels + 1;
  else
    return n_channels;
}


/* Compute offset into one-dimesional pixel array. */
#define OFFSET(rowstride, x, y) ((x) * pixel_size + (y) * (rowstride))


#define ROTATE(angle, x_expression, y_expression) \
  static void real_rotate_ ## angle(const GdkPixbuf *orig, GdkPixbuf *dest) \
  { \
    glong x; \
    glong y; \
    glong orig_offset; \
    glong dest_offset; \
    const guchar *orig_pointer; \
    guchar *dest_pointer; \
    glong orig_width; \
    glong orig_height; \
    glong pixel_size; \
    glong orig_rowstride; \
    glong dest_rowstride; \
    guchar *orig_pixels; \
    guchar *dest_pixels; \
    \
    orig_width = gdk_pixbuf_get_width(orig); \
    orig_height = gdk_pixbuf_get_height(orig); \
     \
    orig_pixels = gdk_pixbuf_get_pixels(orig); \
    orig_rowstride = gdk_pixbuf_get_rowstride(orig); \
   \
    dest_pixels = gdk_pixbuf_get_pixels(dest); \
    dest_rowstride = gdk_pixbuf_get_rowstride(dest); \
     \
    g_assert(compute_pixel_size(orig) == compute_pixel_size(dest)); \
    pixel_size = compute_pixel_size(orig); \
   \
    switch (pixel_size) { \
    case 3: \
      for (y = 0; y < orig_height; ++y) { \
	orig_pointer = orig_pixels + OFFSET(orig_rowstride, 0, y); \
	for (x = 0; x < orig_width; ++x) { \
	  dest_offset = OFFSET(dest_rowstride, x_expression, y_expression); \
	  dest_pointer = dest_pixels + dest_offset; \
	  *dest_pointer++ = *orig_pointer++; \
	  *dest_pointer++ = *orig_pointer++; \
	  *dest_pointer++ = *orig_pointer++; \
	} \
      } \
      break; \
   \
    case 4: \
      for (y = 0; y < orig_height; ++y) { \
	orig_pointer = orig_pixels + OFFSET(orig_rowstride, 0, y); \
	for (x = 0; x < orig_width; ++x) { \
	  dest_offset = OFFSET(dest_rowstride, x_expression, y_expression); \
	  dest_pointer = dest_pixels + dest_offset; \
	  *dest_pointer++ = *orig_pointer++; \
	  *dest_pointer++ = *orig_pointer++; \
	  *dest_pointer++ = *orig_pointer++; \
	  *dest_pointer++ = *orig_pointer++; \
	} \
      } \
      break; \
   \
    default: \
      for (y = 0; y < orig_height; ++y) { \
	for (x = 0; x < orig_width; ++x) { \
	  orig_offset = OFFSET(orig_rowstride, x, y); \
	  dest_offset = OFFSET(dest_rowstride, x_expression, y_expression); \
	  memcpy(dest_pixels + dest_offset,  \
	         orig_pixels + orig_offset,  \
	         pixel_size); \
	} \
      } \
      break; \
    } \
  }


ROTATE(90, y, orig_width - x - 1)
ROTATE(180, orig_width - x - 1, orig_height - y - 1)
ROTATE(270, orig_height - y - 1, x)


static PyObject *rotate90(PyObject *self, PyObject *args)
{
    PyObject *py_src;
    PyObject *py_dest;
    GObject *src;
    GObject *dest;

    if (!PyArg_ParseTuple(args, "OO", &py_src, &py_dest))
        return NULL;

    src = pygobject_get(py_src);
    dest = pygobject_get(py_dest);
    real_rotate_90(GDK_PIXBUF(src), GDK_PIXBUF(dest));
    
    Py_INCREF(Py_None);
    return Py_None;
}


static PyObject *rotate180(PyObject *self, PyObject *args)
{
    PyObject *py_src;
    PyObject *py_dest;
    GObject *src;
    GObject *dest;

    if (!PyArg_ParseTuple(args, "OO", &py_src, &py_dest))
        return NULL;

    src = pygobject_get(py_src);
    dest = pygobject_get(py_dest);
    real_rotate_180(GDK_PIXBUF(src), GDK_PIXBUF(dest));
    
    Py_INCREF(Py_None);
    return Py_None;
}


static PyObject *rotate270(PyObject *self, PyObject *args)
{
    PyObject *py_src;
    PyObject *py_dest;
    GObject *src;
    GObject *dest;

    if (!PyArg_ParseTuple(args, "OO", &py_src, &py_dest))
        return NULL;

    src = pygobject_get(py_src);
    dest = pygobject_get(py_dest);
    real_rotate_270(GDK_PIXBUF(src), GDK_PIXBUF(dest));
    
    Py_INCREF(Py_None);
    return Py_None;
}


static PyMethodDef LodjuMethods[] = {
    {"rotate90",  rotate90, METH_VARARGS, "Rotate 90 degrees."},
    {"rotate180",  rotate180, METH_VARARGS, "Rotate 180 degrees."},
    {"rotate270",  rotate270, METH_VARARGS, "Rotate 270 degrees."},
    {NULL, NULL, 0, NULL}        /* Sentinel */
};


void init_lodju(void)
{
    (void) Py_InitModule("_lodju", LodjuMethods);
    init_pygobject();
}
