/*
 *  Linux snipes, a text-based maze-oriented game for linux.
 *  Copyright (C) 1997 Jeremy Boulton.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Jeremy Boulton is reachable via electronic mail at
 *  boultonj@ugcs.caltech.edu.
 */


/* new stuff for maze generation */

/* Method of generating maze:
 * 
 * Determine a maximum line length MAXLEN.
 * 
 * Maintain 4 sets of points which indicate available (unused) points which
 * can be selected for drawing a line N, S, E, or W respectively starting at
 * the point in question.  Also keep a two dimensional array of points
 * indicating for each point what lines are currently drawn from that point
 * and what lines can be drawn starting at the point.  If lines are already
 * drawn through the point, no lines are possible to be drawn from the point
 * and the point should be removed from the NSEW lists.
 * 
 * Draw a random line of length <= MAXLEN.  Add all points either above/below
 * the line (if line is horizontal) or left/right of the line (if line is
 * vertical) within a distance of MAXLEN of the line to the appropriate
 * NSEW set.
 * 
 * While there exists at least one of the NSEW sets which contains points,
 * choose on of the non-empty NSEW sets at random and choose a random
 * point within that set.  Begin drawing at the chosen point in the chosen
 * direction until a line is hit.  Remove all crossed points from the NSEW
 * sets they are members of and add all unused adjacent points to the
 * appropriate sets as described for the initial random line.
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

/*
#define	STANDALONE
*/

#define LINEDROPMIN	2		/* threshhold for dropping lines */
#define LINEDROPMOD	2		/* 1/(frequency of line drops) */

#define	NORTH	0
#define	SOUTH	1
#define	EAST	2
#define	WEST	3

/* WRAP macro for enforcing the range 0 <= t < m.  Assumes that t will
 * always less than m out of range.
 */

#define WRAP(t,m)	( (t>=m)?t-m:((t<0)?t+m:t) )

/* INDEX macro for indexing a one dimensional array as if it were
 * two dimensional.
 */

#define INDEX(x,y,xsz)	( y*xsz+x )

/* NOLINE macro for determining if there is a line through a particular
 * point in the maze
 */

#define NOLINE(xc,yc,m)	( (m->pts[INDEX(xc,yc,m->x)]) == 0 )

/* BOUNDRANDOM macro for getting a number x in 0 <= x < m */

#define BOUNDRANDOM(m)	( random()%m )

typedef void *(*voidfunc)(void *, long, long, long);


typedef struct point {
  int x;
  int y;
} point;

typedef struct set {
  point *ptlist;
  long size;
} set;

void *set_init( long size )
{
  set *s;

  s = (set *) malloc( sizeof(set) );

  s->ptlist = (point *)malloc( size*sizeof(point) );
  s->size=0;

  return (void *)s;
}

void set_finish( void *s )
{
  free( ((set *)s)->ptlist );
  free( s );
}

void set_add_point( void *seta, point p )
{
  set *s=(set *)seta;
  long i, max=s->size;

  for( i=0; i<max; i++ )
    if( s->ptlist[i].x == p.x && s->ptlist[i].y == p.y )
      return;

  s->ptlist[s->size].x = p.x;
  s->ptlist[s->size].y = p.y;
  s->size++;
}

void set_rm_point( void *seta, point p )
{
  set *s=(set *)seta;
  long i, max=s->size;
  for( i=0; i<max; i++ )
    if( s->ptlist[i].x == p.x && s->ptlist[i].y == p.y ) {
      if( i != max-1 ) {
	s->ptlist[i].x = s->ptlist[max-1].x;
	s->ptlist[i].y = s->ptlist[max-1].y;
      }
      s->size--;
      break;
    }
}

int set_get_point_num( void *seta, long i, point *p )
{
  set *s=(set *)seta;

  if( i >= s->size )
    return 0;
  
  p->x=s->ptlist[i].x;
  p->y=s->ptlist[i].y;
  return 1; 
}


long set_size( void *seta )
{
  return ((set *)seta)->size;
}



/* ----------------------------------------- */
/*                    MAZE                   */
/* ----------------------------------------- */


typedef struct maze {
  long x;
  long y;
  long maxlen;
  voidfunc hfptr;
  voidfunc vfptr;
  void *walls;
  char *pts;
  set *nsew_set[4];
} maze;


void maze_gen_vline( maze *m, long x, long y, int direction )
{
  int len=0;
  long ty, tx, c;
  int s;
  point pt;

  for( ty=y; NOLINE(x,ty,m); ty=WRAP( ty+direction, m->y ), len++ )
  {
    /* mark this point as being used */

    m->pts[INDEX(x,ty,(m->x))] = 1;

    /* remove this point from all sets, since we can no
     * longer initiate a line from this point
     */

    pt.y=ty;
    pt.x=x;
    for( s=0; s<4; s++ )
      set_rm_point( m->nsew_set[s], pt );

    /* add points west of here to the east-line-drawing set */

    for( tx=x, c=0; c<m->maxlen; tx=WRAP( tx-1, m->x ), c++ )
      if( NOLINE( tx, ty, m ) ) {
	pt.x=tx;
	set_add_point( m->nsew_set[EAST], pt );
      }

    /* add points east of here to the west-line-drawing set */

    for( tx=x, c=0; c<m->maxlen; tx=WRAP( tx+1, m->x ), c++ )
      if( NOLINE( tx, ty, m ) ) {
	pt.x=tx;
	set_add_point( m->nsew_set[WEST], pt );
      }
  }

  /* now need to mark a maze line existing from (x,y) to (x,ty). */
  
  /* If we have a long line, leave out a section.  This will hopefully
   * make the maze easier to traverse.
   */
  
  if( len > LINEDROPMIN && (random()%LINEDROPMOD)==0 )
    ty=WRAP( ty-direction, m->y );
  
  if( direction == 1 )
    m->vfptr( m->walls, x, y, ty );
  else
    m->vfptr( m->walls, x, ty, y );
}


void maze_gen_hline( maze *m, long x, long y, int direction )
{
  int len=0;
  long ty, tx, c;
  int s;
  point pt;

  for( tx=x; NOLINE(tx,y,m); tx=WRAP( tx+direction, m->x ), len++ )
  {
    /* mark this point as being used */

    m->pts[INDEX(tx,y,m->x)] = 1;

    /* remove this point from all sets, since we can no
     * longer initiate a line from this point
     */

    pt.x=tx;
    pt.y=y;
    for( s=0; s<4; s++ )
      set_rm_point( m->nsew_set[s], pt );

    /* add points north of here to the south-line-drawing set */

    for( ty=y, c=0; c<m->maxlen; ty=WRAP( ty-1, m->y ), c++ )
      if( NOLINE( tx, ty, m ) ) {
	pt.y=ty;
	set_add_point( m->nsew_set[SOUTH], pt );
      }

    /* add points south of here to the north-line-drawing set */

    for( ty=y, c=0; c<m->maxlen; ty=WRAP( ty+1, (m->y) ), c++ )
      if( NOLINE( tx, ty, m ) ) {
	pt.y=ty;
	set_add_point( m->nsew_set[NORTH], pt );
      }
  }
  
  /* now need to mark a maze line existing from (x,y) to (tx,y). */

  /* If we have a long line, leave out a section.  This will hopefully
   * make the maze easier to traverse.
   */
  
  if( len > LINEDROPMIN && (random()%LINEDROPMOD)==0 )
    tx=WRAP( tx-direction, m->x );

  if( direction == 1 )
    m->hfptr( m->walls, y, x, tx );
  else
    m->hfptr( m->walls, y, tx, x );
}

/*
 * typedef struct maze {
 *   long x;
 *   long y;
 *   long maxlen;
 *   voidfunc hfptr;
 *   voidfunc vfptr;
 *   void *walls;
 *   point *pts[];
 *   set *nsew_set[4];
 * }
 */

maze *maze_init( void *wall, voidfunc h, voidfunc v, long x, long y )
{
  int i, dir;
  long sz, ptnum;
  point pt;
  maze *m;

  m = (maze *)malloc( sizeof(maze) );
  m->pts = (char *)malloc( x*y*sizeof(char) );
  memset( m->pts, 0, x*y*sizeof(char) );
  m->x=x;
  m->y=y;
  m->maxlen=6;

  m->hfptr=h;
  m->vfptr=v;
  m->walls=wall;

  for( i=0; i<4; i++ )
    m->nsew_set[i] = set_init( x*y );
  
  /* initial line.  should change this to be random.  */

  m->pts[INDEX(x<7?x-1:6,0,m->x)] = 1;  /* mark line endpoint */
  maze_gen_hline( m, 0, 0, 1 );
    
  while( set_size(m->nsew_set[0]) ||
	  set_size(m->nsew_set[1]) ||
	  set_size(m->nsew_set[2]) ||
	  set_size(m->nsew_set[3]) ) {

    dir=BOUNDRANDOM(4);

    if( (sz=set_size(m->nsew_set[dir])) ) {
      ptnum=BOUNDRANDOM(sz);
      set_get_point_num( m->nsew_set[dir], ptnum, &pt );
      switch(dir) {
	case NORTH:
	  maze_gen_vline( m, pt.x, pt.y, -1 );
	  break;
	case SOUTH:
	  maze_gen_vline( m, pt.x, pt.y, 1 );
	  break;
	case EAST:
	  maze_gen_hline( m, pt.x, pt.y, 1 );
	  break;
	case WEST:
	  maze_gen_hline( m, pt.x, pt.y, -1 );
	  break;
      }
    }
  }

  return m;
}

void maze_finish( maze *m )
{
  int i;

  for( i=0; i<4; i++ )
    set_finish( m->nsew_set[i] );

  free( (void *)m->pts );
  free( (void *)m );
}



#ifdef STANDALONE

void hwall( void *w, long y, long x1, long x2 )
{
  printf("H: (%d,%d) to (%d,%d)\n",x1,y,x2,y);
}

void vwall( void *w, long x, long y1, long y2 )
{
  printf("V: (%d,%d) to (%d,%d)\n",x,y1,x,y2);
}

main( void )
{
  srandom(time(NULL));
  maze_init( NULL, (voidfunc)hwall, (voidfunc)vwall, 40, 40 );
}

#endif
