/*
 *  $Id: cmd-review.c,v 1.29 2001/08/19 04:47:54 linus Exp $
 *		Functions for doing review
 *
 *
 *  Copyright (C) 1994	Lysator Computer Club,
 *			Linkoping University,  Sweden
 *
 *  Everyone is granted permission to copy, modify and redistribute
 *  this code, provided the people they give it to can.
 *
 *
 *  Author:	Linus Tolke
 *		Lysator Computer Club
 *		Linkoping University
 *		Sweden
 *
 *  email:	linus@lysator.liu.se
 *
 *
 *  Any opinions expressed in this code are the author's PERSONAL opinions,
 *  and does NOT, repeat NOT, represent any official standpoint of Lysator,
 *  even if so stated.
 */


/*
 * Idea:
 * simple review commands:
 * 	]terse [av person] [till m|te eller person] 
 * 		[f|re datum|efter datum|fram}t|bak}t]
 *		startar ett }terse-tr{d.
 *      ]terse n{sta
 *		forts{tter ett }terse-tr{d
 *
 * Simplest implementation:
 *	Search only when requested i.e. when writing the previous prompt.
 *
 *
 */

#include <config.h>
#include <time.h>
#include <kom-types.h>
#include <s-string.h>

#include <libintl.h>

#include <zmalloc.h>

#include "conf.h"

#include "xoutput.h"
#include "read-line.h"
#include "parser.h"
#include "misc-parser.h"

#include "services.h"
#include "copy.h"
#include "internal.h"
#include "error.h"
#include "quit.h"
#include "cache.h"
#include "cmds.h"
#include "commands.h"
#include "offline.h"
#include "main.h"

#define Export

/* Static types */
enum direction { forward, backward };
enum commandtypes { number, complex, undefined };

struct parsed_complex_review_info {
    enum commandtypes	cmd;
    Text_no		text_no;
    unsigned int 	has_by	: 1;
    String 		by;
    unsigned int 	has_to : 1;
    String 		to;
    enum direction	direction;
    unsigned int	has_number : 1;
};

static void
parse_complex_review_info(struct parsed_complex_review_info * info, 
			  String arg);
static int
same_Conf_z_info_lists(Conf_z_info_list * lista, Conf_z_info_list * listb);

enum collecting { by, to, col_time, col_number, presentation, not };

void
enter_collected_field(enum collecting collecting,
		      String summan,
		      struct parsed_complex_review_info * pinfo);

/* static structures
 * These are only manipulated by the functions in this file.
 */
static int inreview = 0;	/* This is true if the prompt will be 
				   review next. */
static int doingreview = 0;	/* This is true if the fields below 
				   are filled. */

static Text_no wherewasi;	/* This is always an absolute text number. */
static enum direction direction;

/* These variables keep track of the cached information. */
typedef struct review_text_list
{
    Text_no		last_considered;
    unsigned long	no_of_texts;
    Text_no	      * texts;
} review_text_list;
#define EMPTY_REVIEW_TEXT_LIST_i { 0, 0L, NULL }
static inline Text_no
last_considered_min(struct review_text_list * t1, struct review_text_list * t2)
{
    return t1->last_considered < t2->last_considered 
      ? t1->last_considered
      : t2->last_considered;
}
  
static inline unsigned long
no_of_texts_min(struct review_text_list * t1, struct review_text_list * t2)
{
    return t1->no_of_texts < t2->no_of_texts 
      ? t1->no_of_texts
      : t2->no_of_texts;
}


static int has_written_by;
static Conf_z_info_list written_by_list = EMPTY_CONF_Z_INFO_LIST_i;
static struct review_text_list written_by_texts = EMPTY_REVIEW_TEXT_LIST_i;
static int has_recipients;
static Conf_z_info_list recipients_list = EMPTY_CONF_Z_INFO_LIST_i;
static struct review_text_list recipients_texts = EMPTY_REVIEW_TEXT_LIST_i;
static struct review_text_list prepared_list = EMPTY_REVIEW_TEXT_LIST_i;

 
Text_no find_next_reviewed_text(void);
static struct review_text_list text_list_intersection(struct review_text_list,
						      struct review_text_list);
static Text_no find_last(void);
static struct review_text_list find_all_written_by(Conf_z_info_list *);
static struct review_text_list find_all_sent_to(Conf_z_info_list *);
static struct review_text_list copy_review_text_list(struct review_text_list);


/* Here is the interface function that fills the structures. */ 

Export Success
cmd_review( String argument )
{
    Text_no   tno;
    struct parsed_complex_review_info info;

#define FUNCTION "cmd_review()"
#define CLEAN_UP() do { s_clear( &argument ); } while (0)

    TOPLOOP_SETJMP();

    if (offline)
    {
	/* Try to handle it for the offline case.
	 * In this case we just can do numbers.
	 */
	if (s_empty(argument))
	{
	    xprintf(gettext("Vilket inlgg vill du terse?\n= "));
	    read_line(&argument);		/* BUG! Check retval! */
	    newline();
	}

	if (s_empty(argument))
	{
	    RETURN (FAILURE);
	}
	else
	{
	    String_size		  error_check;

	    argument = s_strip_trailing(argument, command_separators);
    
	    /* This should be a number, and nothing else. */
	    info.text_no = (Text_no) s_strtol(argument, &error_check,
					      DEFAULT_NUMBER_BASE);
	    if (error_check < s_strlen(argument))
	    {
		OFFLINE2();
	    }

	    newline();

	    show_text_no(info.text_no);
	}

	RETURN (OK);
    }

    OFFLINE();

    /* Parse the argument! */ 
    /* We have to move 
       ]terse (det) kommenterade to here. */
    
    if (s_empty(argument))
    {
	xprintf(gettext("Vilket inlgg vill du terse?\n= "));
	read_line (&argument);		/* BUG! Check retval! */
	newline();
    }
    if (s_empty(argument))
    {
	RETURN (FAILURE);
    }

    /* Lets parse it. */
    parse_complex_review_info(&info, argument);

    newline();

    if (info.cmd == number)
    {
	show_text_no(info.text_no);
	RETURN (OK);
    }

    if (info.cmd != complex) 
    {
	/* This is parse error. */
	RETURN (FAILURE);
    }

    /* Ok, here we are. Set it all up. */
    direction = info.direction;

    doingreview = 1;

    /* Determine where we shall start. */
    if (info.has_number) {
	if (direction == backward)
	    wherewasi = info.text_no + 1;
	else
	    wherewasi = info.text_no;
    } else {
	/* Where are we going to start? */
	/* We are going to start first or last. */
	if (direction == backward) {
	    wherewasi = find_last();
	    wherewasi ++;
	} else {		/* direction == forward */
	    wherewasi = 0;
	}
    }

    /* written_by */
    if (info.has_by) {
	release_conf_z_info_list(&written_by_list);
	written_by_list.confs = NULL;

	has_written_by = 1;
	written_by_list = find_person_number(info.by);
	if (written_by_list.no_of_confs == 0)
	{
	    /*xgettext:c-format*/
	    xprintf(gettext("Det finns ingen person %S.\n"), info.by);
	    has_written_by = 0;
	    RETURN (FAILURE);
	}
    } else {
	has_written_by = 0;
    }


    /* The recipients. */
    if (info.has_to) {
	release_conf_z_info_list(&recipients_list);
	recipients_list.confs = NULL;

	has_recipients = 1;
	recipients_list = find_conf_or_pers_no(info.to);
	if (recipients_list.no_of_confs == 0)
	{
	  /*xgettext:c-format*/
	    xprintf(gettext("Det finns inget sdant mte %S.\n"), info.to);
	    has_recipients = 0;
	    RETURN (FAILURE);
	}
    } else {
	has_recipients = 0;
    }

    /* Now the parsing of the command line is done. Lets, build the 
       structures used for this query.
     */
    /* Lets do everything in static variables in this function. */
    {
	static int 		last_has_written_by;
	static Conf_z_info_list	last_written_by_list = 	
	    EMPTY_CONF_Z_INFO_LIST_i;
	static int 		last_has_recipients;
	static Conf_z_info_list	last_recipients_list = 
	    EMPTY_CONF_Z_INFO_LIST_i;

	int changed = FALSE;

	/* Check if the written_by conditions have changed. */
	if (has_written_by)
	{
	    if (last_has_written_by
		&& same_Conf_z_info_lists(&written_by_list,
					  &last_written_by_list))
	    {
		/* We do not need to make this fetch because it was done
		   the last time.
		   */
	    }
	    else
	    {
		if (written_by_texts.texts != NULL)
		    zfree(written_by_texts.texts);
		written_by_texts = find_all_written_by(&written_by_list);
		changed = TRUE;
	    }
	}
	else if (last_has_written_by)
	    changed = TRUE;

	/* Check if the recepients conditions have changed. */
	if (has_recipients)
	{
	    if (last_has_recipients
		&& same_Conf_z_info_lists(&recipients_list,
					  &last_recipients_list))
	    {
		/* Same */
	    }
	    else
	    {
		if (recipients_texts.texts != NULL)
		    zfree(recipients_texts.texts);
		recipients_texts = find_all_sent_to(&recipients_list);
		changed = TRUE;
	    }
	}
	else if (last_has_recipients)
	    changed = TRUE;

	if (changed)
	{
	    if (prepared_list.texts != NULL)
		zfree(prepared_list.texts);
	    prepared_list.texts = NULL;

	    if (has_written_by && has_recipients) {
		prepared_list = text_list_intersection(written_by_texts,
						       recipients_texts);
	    } else if (has_written_by) {
		prepared_list = copy_review_text_list(written_by_texts);
	    } else if (has_recipients) {
		prepared_list = copy_review_text_list(recipients_texts);
	    }
	}
	if (!has_written_by && !has_recipients)
	{
	    /* No discriminators. Lets build a list of all the texts. */
	    /* It is really enough to build until wherewasi. For texts after
	       wherewasi, the searching mechanism, normally used to catch texts
	       that are created after the query is made but before we get there
	       will handle those texts correctly.
	       */
	    if (prepared_list.texts != NULL)
		zfree(prepared_list.texts);
	    prepared_list.texts = NULL;

	    prepared_list.texts = zmalloc(sizeof(Text_no) * wherewasi);
	    for (prepared_list.no_of_texts = 0;
		 prepared_list.no_of_texts < wherewasi;
		 prepared_list.no_of_texts++)
	    {
		prepared_list.texts[prepared_list.no_of_texts] =
		    (Text_no)(prepared_list.no_of_texts + 1);
	    }
	    prepared_list.last_considered = wherewasi;
	}
    }

    /* Now the structure of all texts to consider is built. 
       The starting position is found in find_next_reviewed_text.
       */

#ifdef TRACE
    if (TRACE(3))
        xprintf("%s:%d:DEBUG wherewasi = %d\n", __FILE__, __LINE__, wherewasi);
#endif

    /* Find the first text to be read according to the search */
    tno = find_next_reviewed_text();

#ifdef TRACE
    if (TRACE(3))
        xprintf("%s:%d:DEBUG tno = %d; wherewasi = %d\n", __FILE__, __LINE__, 
		tno, wherewasi);
#endif

    /* Display that text */
    show_text_no (tno);

    wherewasi = tno;

    /* If argument is complex then set inreview */
    inreview = 1;

    RETURN (OK);

#undef CLEAN_UP
#undef FUNCTION
}

Success
cmd_review_again(String argument)
{
#define CLEAN_UP() do { s_clear(&argument); } while (0)

  TOPLOOP_SETJMP();

  CHECK_NO_ARGUMENTS();

  newline();

  RETURN (show_text_no(reading_state.last_viewed));

#undef CLEAN_UP
}

/* This is just a wrapper to provide a beautiful prompt... */
Success
cmd_review_next(String argument)
{
    Text_no	  tno;
    Success	  retval;

#define CLEAN_UP() do { s_clear(&argument); } while (0)

    TOPLOOP_SETJMP();

    CHECK_NO_ARGUMENTS();

#ifdef TRACE
    if (TRACE(3))
        xprintf("%s:%d:DEBUG wherewasi = %d\n", __FILE__, __LINE__, wherewasi);
#endif

    tno = find_next_reviewed_text();

#ifdef TRACE
    if (TRACE(3))
        xprintf("%s:%d:DEBUG tno = %d; wherewasi = %d\n", __FILE__, __LINE__, 
		tno, wherewasi);
#endif

    /* Display that text */
    retval = show_text_no (tno);

    wherewasi = tno;

    if (tno != 0)
	inreview = 1;
    else
	inreview = 0;

    RETURN (retval);
#undef CLEAN_UP
}

/* is_in_review. */
void
set_prompt_if_in_review(Success (**next_command) (String argument))
{
    if (inreview) 
    {
	/* Lets do some searchin here. */
	Text_no	tno;

#ifdef TRACE
	if (TRACE(3))
	    xprintf("%s:%d:DEBUG wherewasi = %d\n", __FILE__, __LINE__, 
		    wherewasi);
#endif

	tno = find_next_reviewed_text();

	if (tno != 0)
	    *next_command = cmd_review_next;

	/* This is an optimization so we don't have to do the same search again
	   in a minute. */
	wherewasi = tno + (direction == forward ? -1 : 1);

#ifdef TRACE
	if (TRACE(3))
	    xprintf("%s:%d:DEBUG tno = %d; wherewasi = %d\n",
		    __FILE__, __LINE__, 
		    tno, wherewasi);
#endif
    }
}



/* review_reset resets the reviewing. 
   This is used to keep track of if we are doing review or not.
   The functions resetting this is next_conf, next_text...
 */
void
review_reset()
{
    inreview = 0;
}





/* Debugging function */
#define DEBUG_PRINT_WHAT_LIST(type)			\
static void debug_print_ ## type(type list);		\
static							\
void							\
debug_print_ ## type(type list)				\
{							\
    xprintf("%d element", list.no_of_texts);		\
    if (list.no_of_texts < 5) {				\
	unsigned int i;					\
							\
	for (i = 0; i < list.no_of_texts; i++)		\
	    xprintf(" %d", list.texts[i]);		\
    } else {						\
	xprintf(" %d %d ... %d %d", 			\
		list.texts[0],				\
		list.texts[1],				\
		list.texts[list.no_of_texts - 2],	\
		list.texts[list.no_of_texts - 1]);	\
    }							\
    xprintf("\n");					\
}

#ifdef TRACE
DEBUG_PRINT_WHAT_LIST(Text_list);
DEBUG_PRINT_WHAT_LIST(review_text_list);
#endif

/*
 * Help functions to find_next_reviewed_text.
 */

/* Copies the argument.
 */
static struct review_text_list
copy_review_text_list(struct review_text_list tl)
{
    struct review_text_list newlist;
    unsigned int i;

    newlist.last_considered = tl.last_considered;
    newlist.no_of_texts = tl.no_of_texts;
    newlist.texts = zmalloc(sizeof(Text_no) * newlist.no_of_texts);
    for (i = 0; i < newlist.no_of_texts; i++)
    {
	newlist.texts[i] = tl.texts[i];
    }
    return newlist;
}

/* Returns a text_list that is the sorted union of all text lists.
   All zeros are removed in the process.

   Since the lists are almost sorted we do this in a very simple way.
   The last_considered field is not set.
 */
static struct review_text_list
union_text_list(Text_list lists[], unsigned int no_of_lists)
{
    struct review_text_list	list = EMPTY_REVIEW_TEXT_LIST_i;
    unsigned int i;
    unsigned int * ind = (unsigned int *) zmalloc(sizeof(int) * no_of_lists);
    int sum;
    int lists_left = no_of_lists;

#ifdef TRACE
    if (TRACE(3))
        xprintf("%s:%d:DEBUG doing union. no_of_lists: %d\n", 
		__FILE__, __LINE__,
		no_of_lists);
#endif
    
    sum = 0;
    for (i = 0; i < no_of_lists; i++) {
	sum += lists[i].no_of_texts;

	ind[i] = 0; 
	while (ind[i] < lists[i].no_of_texts
	       && lists[i].texts[ind[i]] == 0) {
	    ind[i]++;
	    sum --;		/* This is ususally not much. 
				   Probably nothing at all. */
	}

#ifdef TRACE
	if (TRACE(3))
	{
	    xprintf("%s:%d:DEBUG union input: list[%d]: ", __FILE__, __LINE__,
		    lists[i].no_of_texts);
	    debug_print_Text_list(lists[i]);
	}
#endif
    }

    /* This gives us plenty of space. It cannot possibly be to small. */
    list.texts = zrealloc(list.texts, sum * sizeof(Text_no));

    /* Lets do this the brutal way:
       - find the smallest number
       - enter it
       - step forward
       - are there any left?
       */
    while (lists_left > 0) {
	/* Find the smallest number (this really is brutal */
	Text_no smallest;
	int smallpos = 0;

	/* Recalculate lists_left in the process... */
	lists_left = 0;
	smallest = MAX_TEXT_NO;
	for (i = 0; i < no_of_lists; i++) {
	    if (ind[i] < lists[i].no_of_texts) {
		lists_left ++;
		if (lists[i].texts[ind[i]] < smallest) {
		    smallest = lists[i].texts[ind[i]];
		    smallpos = i;
		}
	    }
	}

	if (smallest != MAX_TEXT_NO) {
	    /* We found a text. */
	    if (list.no_of_texts == 0
		|| smallest > list.texts[list.no_of_texts - 1]) {
		list.texts[list.no_of_texts] = smallest;
		list.no_of_texts++;
	    } else if (smallest == list.texts[list.no_of_texts - 1]) {
		/* Do nothing. */
	    } else {
		/* Oups have to enter it somewhere in the middle. */

		for (i = 0; i < list.no_of_texts; i++) {
		    if (list.texts[i] >= smallest) {
			/* Found the position */
			break;
		    }
		}
		if (list.texts[i] == smallest) {
		    /* It was already there. */
		} else {
		    unsigned int j;

		    for (j = list.no_of_texts; j > i; j--) {
			list.texts[j] = list.texts[j - 1];
		    }
		    list.texts[i] = smallest;
		    list.no_of_texts++;
		}
	    }
	    ind[smallpos]++;
	    while (ind[smallpos] < lists[smallpos].no_of_texts
		   && lists[smallpos].texts[ind[smallpos]] == 0) {
		ind[smallpos]++;
		
	    }
	}
    }
    /* We could of coarse do a realloc to "free" some memory but ... */

    zfree(ind);
#ifdef TRACE
    if (TRACE(3))
    {
        xprintf("%s:%d:DEBUG union output: ", __FILE__, __LINE__);
	debug_print_review_text_list(list);
	xprintf("%s:%d:DEBUG union done\n", __FILE__, __LINE__);
    }
#endif
    return list;		
}

/* Now lets do the intersection */
/* The arguments are sorted text_lists. */
static struct review_text_list
text_list_intersection(struct review_text_list tl1, 
		       struct review_text_list tl2)
{
    struct review_text_list	list = EMPTY_REVIEW_TEXT_LIST_i;
    unsigned int		i1, i2;

#ifdef TRACE
    if (TRACE(3))
    {
	xprintf("%s:%d:DEBUG doing intersection between:\n",
		__FILE__, __LINE__);
	xprintf("%s:%d:DEBUG tl1: ", __FILE__, __LINE__); 
	debug_print_review_text_list(tl1);
	xprintf("%s:%d:DEBUG tl2: ", __FILE__, __LINE__); 
	debug_print_review_text_list(tl2);
    }
#endif

    list.last_considered = last_considered_min(&tl1, &tl2);

    /* Plenty of space */
    list.texts = zrealloc(list.texts, sizeof(Text_no) * no_of_texts_min(&tl1,
									&tl2));
    for (i1 = 0, i2 = 0; i1 < tl1.no_of_texts && i2 < tl2.no_of_texts; ) {
	if (tl1.texts[i1] == tl2.texts[i2]) {
	    /* Found one */
	    list.texts[list.no_of_texts++] = tl1.texts[i1];
	    i1++;
	    i2++;
	} else if (tl1.texts[i1] > tl2.texts[i2]) {
	    i2++;
	} else
	    i1++;
    }

#ifdef TRACE
    if (TRACE(3))
    {
        xprintf("%s:%d:DEBUG intersection output: ", __FILE__, __LINE__);
	debug_print_review_text_list(list);
	xprintf("%s:%d:DEBUG intersection done\n", __FILE__, __LINE__);
    }
#endif
    return list;
}


/* Function to find the last text in the lyskom system.
 */
static Text_no
find_last()
{
#define FUNCTION "find_last"
    Text_no res;

    if (FAILURE == kom_find_previous_text_no(MAX_TEXT_NO, &res)) 
    {
	/* This can go wrong if there are no texts in the database. 
	   I mean, no texts at all.
	   BUG: And then we should check for the lost connection.
	   */
	switch (kom_errno)
	{
	case KOM_NO_SUCH_TEXT:
	    /* There are no texts in the database. No texts whatsoever. */
	    res = 0;
	    break;

	default:
	    /* This is not handled */
	    fatal1(CLIENT_SITUATION_NOT_HANDLED,
		   "Cannot get the number of the last text.\n");
	    /*NOTREACHED*/
	}
    }
    return res;
#undef FUNCTION
}


static struct review_text_list
find_all_written_by(Conf_z_info_list * written_by_list_res)
{
#define FUNCTION "find_all_written_by()"
    struct review_text_list list; /* This is released by the main function. */
    Text_list	*all =
        (Text_list *) zmalloc(sizeof (Text_list) 
			      * written_by_list_res->no_of_confs);
    int i;
    Text_no	last = find_last();

    for (i = 0; i < written_by_list_res->no_of_confs; i++) {
	all[i].no_of_texts = 0;
	all[i].texts = NULL;
	if (kom_get_created_texts(written_by_list_res->confs[i].conf_no,
				  0,
				  MAX_LOCAL_TEXT_NO,
				  &all[i]) != OK)
	{
	    switch (kom_errno) {
	    case KOM_NO_SUCH_LOCAL_TEXT:
		/* This is normal when there are no created texts. */
		all[i].no_of_texts = 0;
		break;
	    default:
		fatal4(CLIENT_SERVER_ERROR, 
		       "Cannot get the list of written texts by person %d. "
		       "(kom_errno: %d %s)",
		       written_by_list_res->confs[i].conf_no,
		       (int)kom_errno, kom_errno_string());
	    }
	}
	/* These texts are always gotten in order. */
    }

    list = union_text_list(all, written_by_list_res->no_of_confs);
    for (i = 0; i < written_by_list_res->no_of_confs; i++) {
	release_text_list(&all[i]);
	all[i].texts = NULL;
    }
    zfree(all);
    list.last_considered = last;
    return list;
#undef FUNCTION
}


static struct review_text_list
find_all_sent_to(Conf_z_info_list * sent_to_list)
{
#define FUNCTION "find_all_sent_to()"
    struct review_text_list list; /* This is released by the main function. */
    Text_list	*all = (Text_list *) zmalloc(sizeof (Text_list)
					     * sent_to_list->no_of_confs);
    int i;
    Text_no	last = find_last();

    for (i = 0; i < sent_to_list->no_of_confs; i++) {
	int bubble_sorted;

	all[i].no_of_texts = 0;
	all[i].texts = NULL;
	if (kom_get_map(sent_to_list->confs[i].conf_no,
			0,
			MAX_LOCAL_TEXT_NO,
			&all[i]) != OK)
	{
	    switch (kom_errno) {
	    case KOM_NO_SUCH_LOCAL_TEXT: /* If no texts to the conf */
	    case KOM_ACCESS: /* If not allowed to read that conf */
		all[i].no_of_texts = 0;
		break;
	    default:
		fatal4(CLIENT_SERVER_ERROR,
		       "Cannot get the list of texts sent to conf %d. "
		       "(kom_errno: %d %s)",
		       sent_to_list->confs[i].conf_no,
		       (int)kom_errno, kom_errno_string());
	    }
	}
	/* These texts are not always in order. */
	/* We will have to sort them. */
	/* Since they are almost in order, lets do a bubble sort, it's simple
	   enough. */
	do {
	    int j;

	    bubble_sorted = 1;

	    for (j = all[i].no_of_texts - 1; j > 0 ; j--)
	    {
		if (all[i].texts[j - 1] > all[i].texts[j])
		{
		    Text_no t = all[i].texts[j];
		    all[i].texts[j] = all[i].texts[j - 1];
		    all[i].texts[j - 1] = t;

		    bubble_sorted = 0;
		}
	    }
	} while (!bubble_sorted);
    }

    list = union_text_list(all, sent_to_list->no_of_confs);
    for (i = 0; i < sent_to_list->no_of_confs; i++) {
	release_text_list(&all[i]);
	all[i].texts = NULL;
    }
    zfree(all);
    list.last_considered = last;
    return list;
#undef FUNCTION
}


/*
 * More help functions.
 */
/* This function is to compare two conf_z_info_lists. */
static int
same_Conf_z_info_lists(Conf_z_info_list * lista, Conf_z_info_list * listb)
{
    int i;

    if (lista->no_of_confs != listb->no_of_confs)
	return FALSE;		/* Different size */

    for (i = 0; i < lista->no_of_confs; i++)
    {
	if (!(lista->confs[i].conf_no == listb->confs[i].conf_no))
	    return FALSE;
    }
    return TRUE;
}

/* 
 * find_next_reviewed_text searches for a text according to the rules
 * set up in the variables in this file.
 *
 * It returns the Text_no or 0 if not found.
 * If first_time, then we "reinitiate" the search.
 */
Text_no
find_next_reviewed_text()
{
#define FUNCTION "find_next_reviewed_text()"

    Text_no	thisone;
    int		found;
    Text_stat	textstat = EMPTY_TEXT_STAT;
    volatile unsigned int there = 0;

#define CLEAN_UP() do { release_text_stat(&textstat); } while (0)

    TOPLOOP_SETJMP();

    /* This is simple */
    if (!doingreview)
    {
	RETURN ((Text_no)0);
    }

    /* At this point, list is the list of texts to show */
    /* Now, lets try to find where to start. */
    if (wherewasi == 0 || prepared_list.no_of_texts == 0)
    {
	there = 0;
    }
    else if (prepared_list.no_of_texts > 0
	     && wherewasi >= prepared_list.texts[prepared_list.no_of_texts-1])
    {
	there = prepared_list.no_of_texts;
    } 
    else
    {
	/* We are somewhere in the middle. */
	/* Do a binary search in the list */
	int before = 0;
	int after = prepared_list.no_of_texts;

	while (before + 1 < after)
	{
	    there = (before + after) / 2;
	    if (prepared_list.texts[there] > wherewasi)
		after = there;
	    else if (prepared_list.texts[there] < wherewasi)
		before = there;
	    else
		break; /* Lucky shot */
	}

	if (direction == backward)
	    there += 1;
    }

#define RETURNIFEXIST(posno)			\
    {						\
	Text_stat * ts = NULL;			\
						\
	ts = text_stat(posno);			\
						\
	if (ts == NULL)				\
	{					\
	    /* No such text */			\
	}					\
	else					\
	{					\
	    release_text_stat(ts);		\
	    zfree(ts);				\
	    RETURN(posno);			\
	}					\
    }

    switch (direction)
    {
    case forward:
	while (there < prepared_list.no_of_texts) 
	{
	    Text_no posno = prepared_list.texts[there];
	    if (wherewasi < posno)
	    {
		RETURNIFEXIST(posno);
	    }
	    there++;
	}
	wherewasi = prepared_list.last_considered;
	break;
    case backward:
	while (there > 0) 
	{
	    Text_no posno = prepared_list.texts[there - 1];
	    if (wherewasi > posno)
	    {
		RETURNIFEXIST(posno);
	    }
	    there--;
	}
	wherewasi = 0;
	break;
    }

    /*
     * Now we could not find the text in the list.
     * Lets search forward from where we were.
     */

    /* searching */
    for (found = 0; !found; thisone = wherewasi)
    {
	/* Calculate next number. */ 
	Success retval;

	if (direction == backward)
	{
	    /* If we are reading backwards, then the lists constructed above
	       is complete so we are done.
	       */
	    break;
	}

	retval = kom_find_next_text_no (wherewasi, &thisone);
	if (retval == FAILURE)
	{
	    switch (kom_errno)
	    {
	    case KOM_TEXT_ZERO:	/* This happens the first time. */
		wherewasi ++; /* Get the next one */
		/* This really should never happen... */
		continue;
	    case KOM_NO_SUCH_TEXT: /* We are done. */
		thisone = 0;
		break;
	    default: /* What? */
	        if (direction == forward)
		    fatal4(CLIENT_SERVER_ERROR,
			   "Cannot fetch next text from %ld. "
			   "Kom errno: %d %s\n",
			   wherewasi, kom_errno, kom_errno_string());
		else
		    fatal4(CLIENT_SERVER_ERROR,
			   "Cannot fetch previous text from %ld. "
			   "Kom errno: %d %s\n",
			   wherewasi, kom_errno, kom_errno_string());
	    }
	    break;
	}
	
	wherewasi = thisone;

	/* Check if it is one of those we are looking for. */
	if (FAILURE == kom_get_text_stat(thisone, &textstat))
	{
	    switch(kom_errno)
	    {
	    case KOM_NO_SUCH_TEXT: /* This is the normal error if we are
				      not allowed to read that text.
				      Just ignore.
				    */
		continue;
	    default:
		fatal4(CLIENT_SERVER_ERROR,
		       "Cannot get the textstat of text %ld. "
		       "Komerrno = %d %s\n",
		       thisone, kom_errno, kom_errno_string());
		/*NOTREACHED*/
	    }
	    fatal1(CLIENT_SHOULDNT_HAPPEN,
		   "We should never have gotten here.\n");
	    /*NOTREACHED*/
	}

	/* Now we have the text_stat in textstat. */
	/* Check the author: */
	if (has_written_by)
	{ /* There is an author */
	    int i;
	    int correct_author;
	    
	    correct_author = 0;
	    for (i = 0; i < written_by_list.no_of_confs; i++)
	    {
		if (written_by_list.confs[i].conf_no == textstat.author)
		{
		    correct_author = 1;
		    break;
		}
	    }
	    if (!correct_author) 
	    {
		continue;
	    }
	    /* We have found a text by a mentioned author */
	}
	/* We have found a text by a mentioned author or we are not searching
	   for the authors.
	 */

	/* Check for the recipients: */
	if (has_recipients)
	{
	    Misc_info_group		misc_data;
	    const Misc_info	      * misc_list;
	    int				correct_recipient;
	    
	    correct_recipient = 0;

	    misc_list = textstat.misc_items;

	    while ((misc_data = parse_next_misc (&misc_list,
						 textstat.misc_items
						 + textstat.no_of_misc)).type
		   != m_end_of_list)
	    {
		Conf_no		recp;

		switch (misc_data.type)
		{
		case m_recpt:
		case m_cc_recpt:
		    recp = misc_data.recipient;
		    break;
		default:
		    recp = 0;
		}

		if (recp != 0)
		{
		    int i;

		    for (i = 0; i < recipients_list.no_of_confs; i++)
		    {
			if (recipients_list.confs[i].conf_no == recp) {
			    correct_recipient = 1;
			    break;
			}
		    }
		    if (correct_recipient)
		    {
			break;
		    }
		}
	    }
	    if (!correct_recipient)
		continue;
	}
	/* We have found a text to a mentioned recipient or we are not 
	   searching for the recipient.
	 */

	/* Now, are there any more checks? */
	found = 1;
    }

    if (!found)
    { 
	/* We did break out of the loop without finding anything.
	   This is OK.
	 */
	doingreview = 0;
	inreview = 0;
	RETURN ((Text_no)0);
    }
	
    RETURN (thisone);
#undef FUNCTION
#undef CLEAN_UP
}




/*** Supporting functions ***/

/* parse_complex_review_info is used to parse a line.

   This is really a very language dependent function and
   I have not yet decided how to handle different languages.
 */
void
enter_collected_field(enum collecting collecting,
		      String summan,
		      struct parsed_complex_review_info * pinfo)
{
#define FUNCTION "enter_collected_field()"

    Text_no 		  tno;
    String_size		  error_check;	/* First uncoverted char str->num */
    unsigned int day, month, year, hour, minut;
    struct tm		* current_time;
    struct tm		  parsed_time;
    unsigned int	  found_time;
    char *		  simplesumman;
    String		  collected = EMPTY_STRING;
    Conf_z_info_list	  cl = EMPTY_CONF_Z_INFO_LIST;
    Conference * volatile cs = NULL;
    Kom_err		  error;

#define CLEAN_UP() do { \
			 release_conf(cs); \
			 release_conf_z_info_list(&cl); \
		 } while (0)

    TOPLOOP_SETJMP();

    if (s_substr(&collected, summan, 1, END_OF_STRING) != OK)
    {
	/* This is an error. */
	fatal1(CLIENT_SHOULDNT_HAPPEN, "Not enough collected.");
    }

    s_clear(&summan);

    switch (collecting)
    {
    case presentation:
	{
	    cl = find_conf_or_pers_no(collected);
	    if (cl.no_of_confs == 0 || cl.no_of_confs > 1)
	    {
	        /*xgettext:c-format*/
		xprintf(gettext("Det finns ingen sdan person "
				"och inget sdant mte %S."), collected);
		pinfo->cmd = undefined;
		RETURNVOID;
	    }
	    
	    cs = conf_stat(cl.confs[0].conf_no, &error);
	    /*xgettext:c-format*/
	    xprintf(gettext(" presentation fr %S"), cs->name);
	    pinfo->text_no = cs->presentation;
	    RETURNVOID;
	}
	    

    case by:
	pinfo->by = collected;
	pinfo->has_by = 1;
	/*xgettext:c-format*/
	xprintf(gettext(" av %S"), collected);
	RETURNVOID;
    case to:
	pinfo->to = collected;
	pinfo->has_to = 1;
	/*xgettext:c-format*/
	xprintf(gettext(" till %S"), collected);
	RETURNVOID;
    case col_time:
	parsed_time.tm_wday = 0;
	parsed_time.tm_yday = 0;
	parsed_time.tm_isdst = 0;
	found_time = 0;

	simplesumman = s_crea_c_str(collected);
	
	if (((strlen(simplesumman) == 14
	      && (sscanf(simplesumman,
			 "%2u-%2u-%2u %2u:%2u", 
			 &year, &month, &day, &hour, &minut)
		  == 5
		  || sscanf(simplesumman,
			    "%2u-%2u-%2u %2u.%2u", 
			    &year, &month, &day, &hour, &minut)
		  == 5))
	     || (strlen(simplesumman) == 12
		 && (sscanf(simplesumman,
			    "%2u%2u%2u %2u:%2u", 
			    &year, &month, &day, &hour, &minut)
		     == 5
		     || (sscanf(simplesumman,
				"%2u%2u%2u %2u.%2u", 
				&year, &month, &day, &hour, &minut)
			 == 5))))
	    && year < 100)
	{
	    /* closest year */
	    time_t now = time(NULL);
	    current_time = localtime(&now);
	    parsed_time.tm_year = year;
	    while (parsed_time.tm_year > current_time->tm_year + 50)
		parsed_time.tm_year += 100;
	    parsed_time.tm_mon = month - 1;
	    parsed_time.tm_mday = day;
	    parsed_time.tm_hour = hour;
	    parsed_time.tm_min = minut;
	    parsed_time.tm_sec = 0;
	    found_time = 1;
	}
	else if (((strlen(simplesumman) == 16
		   && (sscanf(simplesumman,
			      "%4u-%2u-%2u %2u:%2u",
			      &year, &month, &day, &hour, &minut)
		       == 5
		       || sscanf(simplesumman,
				 "%4u-%2u-%2u %2u.%2u",
				 &year, &month, &day, &hour, &minut)
		       == 5))
		  || (strlen(simplesumman) == 14
		      && (sscanf(simplesumman,
				 "%4u%2u%2u %2u:%2u", 
				 &year, &month, &day, &hour, &minut)
			  == 5
			  || sscanf(simplesumman,
				    "%4u%2u%2u %2u.%2u", 
				    &year, &month, &day, &hour, &minut)
			  == 5)))
		 && year >= 100)
	{
	    parsed_time.tm_year = year - 1900;
	    parsed_time.tm_mon = month - 1;
	    parsed_time.tm_mday = day;
	    parsed_time.tm_hour = hour;
	    parsed_time.tm_min = minut;
	    parsed_time.tm_sec = 0;
	    found_time = 1;
	}
	else if ((strlen(simplesumman) == 10
		  && sscanf(simplesumman,
			    "%4u-%2u-%2u", &year, &month, &day)
		  == 3)
		 || ((strlen(simplesumman) == 8
		      && sscanf(simplesumman,
				"%4u%2u%2u", &year, &month, &day)
		      == 3)
		     && year >= 100))
	{
	    parsed_time.tm_year = year - 1900;
	    parsed_time.tm_mon = month - 1;
	    parsed_time.tm_mday = day;
	    parsed_time.tm_hour = 0;
	    parsed_time.tm_min = 0;
	    parsed_time.tm_sec = 0;
	    found_time = 1;
	}
	else if ((sscanf(simplesumman,
			 "%2u-%2u-%2u", &year, &month, &day)
		  == 3
		  || sscanf(simplesumman,
			    "%2u%2u%2u", &year, &month, &day) 
		  == 3)
		 && year < 100)
	{
	    /* closest year */
	    time_t now = time(NULL);
	    current_time = localtime(&now);
	    parsed_time.tm_year = year;
	    while (parsed_time.tm_year > current_time->tm_year + 50)
		parsed_time.tm_year += 100;
	    parsed_time.tm_mon = month - 1;
	    parsed_time.tm_mday = day;
	    parsed_time.tm_hour = 0;
	    parsed_time.tm_min = 0;
	    parsed_time.tm_sec = 0;
	    found_time = 1;
	}

	zfree(simplesumman);

	if (found_time)
	{
	    /* We have a time. Lets map it to a textno. */
	    print_time(&parsed_time);
	    if (FAILURE == kom_get_last_text(&parsed_time, &pinfo->text_no)) {
		fatal1(CLIENT_CONNECTION_LOST /* This is just assumed */,
		       "Cannot get the number of a text.\n");
	    }
	    pinfo->has_number = 1;
	    RETURNVOID; 
	}
	/*FALLTHROUGH*/
    case col_number:
	tno = (Text_no) s_strtol (collected, &error_check, DEFAULT_NUMBER_BASE);
	if (error_check < s_strlen(collected))
	{
	    if (collecting == col_number)
	    {
		xprintf(gettext("Felaktigt textnummer.\n"));
		pinfo->cmd = undefined;
	    }
	    else
	    {
		xprintf(gettext("Icke tydbar datumangivelse.\n"));
		pinfo->has_number = 0;
	    }
	    RETURNVOID;
	}
	xprintf(gettext(" nummer %d"), tno); 
	pinfo->text_no = tno;
	pinfo->has_number = 1;
	RETURNVOID;
    case not:
	fatal1(CLIENT_SHOULDNT_HAPPEN,
	       "Called with incorrect argument.\n");
    }
#undef FUNCTION
#undef CLEAN_UP
}



/* Yes
 * The parsing algorithm implemented here below is very simple.
 * There is no backtracking, the keywords must be given in full...
 *
 * Feel free to supply a better one!
 */
void
parse_complex_review_info(struct parsed_complex_review_info * info, String arg)
{
#define FUNCTION "parse_complex_review_info"

    Parse_token			      * working;
    Parse_token * volatile		start_of_working = NULL;
    String				summan = EMPTY_STRING;
    enum collecting			collecting;
    int					collected;
    Bool	 			direction_given;
    Text_no				tno;
    String_size				fic;
    Text_stat				text_stat_val = EMPTY_TEXT_STAT;


#define CLEAN_UP() do { s_clear(&summan); \
	    free_tokens(start_of_working); \
	    release_text_stat(&text_stat_val); \
      } while (0)

    TOPLOOP_SETJMP();

    info->cmd = undefined;
    info->has_by = 0;
    info->by = EMPTY_STRING;
    info->has_to = 0;
    info->to = EMPTY_STRING;
    direction_given = FALSE;
    info->direction = forward;
    info->has_number = 0;

    working = start_of_working = tokenize(arg, s_fcrea_str(" \t"));

    collecting = not;
    collected = 0;
    while (working && !s_empty(working->word))
    {
	if (s_usr_streq(working->word,
			s_fcrea_str(gettext("av")),
			DEFAULT_COLLAT_TAB)
	    && (collecting == not || collected > 0))
	{
	    if (info->cmd != complex && info->cmd != undefined)
	    {
		xprintf(gettext("Fr mnga argument\n"));
		info->cmd = undefined;
		RETURNVOID;
	    }
	    if (info->has_by)
	    {
		xprintf(gettext("Du kan bara ha ett \"av\" direktiv.\n"));
		info->cmd = undefined;
		RETURNVOID;
	    }
	    info->cmd = complex;

	    if (collecting != not)
	    {
		/* enter_collected_field takes care of "freeing/clearing"
		   summan */
		enter_collected_field(collecting, summan, info);
		summan = EMPTY_STRING;
	    }

	    collecting = by;
	} 
	else if (s_usr_streq(working->word,
			     s_fcrea_str(gettext("till")),
			     DEFAULT_COLLAT_TAB)
		 && (collecting == not || collected > 0))
	{
	    if (info->cmd != complex && info->cmd != undefined)
	    {
		xprintf(gettext("Fr mnga argument\n"));
		info->cmd = undefined;
		RETURNVOID;
	    }
	    if (info->has_to)
	    {
		xprintf(gettext("Du kan bara ha ett \"av\" direktiv.\n"));
		info->cmd = undefined;
		RETURNVOID;
	    }
	    info->cmd = complex;

	    if (collecting != not)
	    {
		enter_collected_field(collecting, summan, info);
		summan = EMPTY_STRING;
	    }

	    collecting = to;

	} 
	else if (s_usr_streq(working->word,	
			     s_fcrea_str(gettext("fre")),
			     DEFAULT_COLLAT_TAB)
		 && (collecting == not || collected > 0))
	{
	    if (info->cmd != complex && info->cmd != undefined)
	    {
		xprintf("\n");
		xprintf(gettext("Fr mnga argument\n"));
		info->cmd = undefined;
		RETURNVOID;
	    }
	    if (info->has_number)
	    {
		xprintf("\n");
		xprintf(gettext("Du kan bara ha en tidsangivelse.\n"));
		info->cmd = undefined;
		RETURNVOID;
	    }
	    if (direction_given == TRUE)
	    {
		xprintf("\n");
		xprintf(gettext("Du kan bara anvnda en av framt, bakt, fre eller efter i taget.\n"));
		info->cmd = undefined;
		RETURNVOID;
	    }
	    info->cmd = complex;

	    if (collecting != not)
	    {
		enter_collected_field(collecting, summan, info);
		summan = EMPTY_STRING;
	    }

	    xprintf(gettext(" fre "));

	    info->direction = backward;
	    direction_given = TRUE;
	    collecting = col_time;
	} 
	else if (s_usr_streq(working->word,	
			     s_fcrea_str(gettext("efter")),
			     DEFAULT_COLLAT_TAB)
		 && (collecting == not || collected > 0))
	{
	    if (info->cmd != complex && info->cmd != undefined)
	    {
		xprintf(gettext("Fr mnga argument\n"));
		info->cmd = undefined;
		RETURNVOID;
	    }
	    if (info->has_number)
	    {
		xprintf(gettext("Du kan bara ha en tidsangivelse.\n"));
		info->cmd = undefined;
		RETURNVOID;
	    }
	    if (direction_given == TRUE)
	    {
		xprintf(gettext("Du kan bara anvnda en av framt, bakt, fre eller efter i taget.\n"));
		info->cmd = undefined;
		RETURNVOID;
	    }
	    info->cmd = complex;

	    if (collecting != not)
	    {
		enter_collected_field(collecting, summan, info);
		summan = EMPTY_STRING;
	    }

	    xprintf(gettext(" efter "));

	    info->direction = forward;
	    direction_given = TRUE;
	    collecting = col_time;
	} 
	else if ((collecting == not)
		 && (info->cmd == undefined)
		 && s_usr_strhead(working->word,
				  s_fcrea_str(gettext("presentation")),
				  DEFAULT_COLLAT_TAB)
		 )
	{
	    info->cmd = number;
	    collecting = presentation;
	}
	else if (s_usr_streq(working->word,	
			     s_fcrea_str(gettext("framt")),
			     DEFAULT_COLLAT_TAB)
		 && (collecting == not || collected > 0))
	{
	    if (info->cmd != complex && info->cmd != undefined)
	    {
		xprintf(gettext("Fr mnga argument\n"));
		info->cmd = undefined;
		RETURNVOID;
	    }
	    if (direction_given == TRUE)
	    {
		xprintf(gettext("Du kan bara anvnda en av "
				"framt, bakt, fre eller efter i taget.\n"));
		info->cmd = undefined;
		RETURNVOID;
	    }
	    info->cmd = complex;

	    if (collecting != not)
	    {
		enter_collected_field(collecting, summan, info);
		summan = EMPTY_STRING;
	    }

	    xprintf(gettext(" framt"));

	    info->direction = forward;
	    direction_given = TRUE;
	    collecting = not;
	} 
	else if ((s_usr_streq(working->word,	
			      s_fcrea_str(gettext("bakt")),
			      DEFAULT_COLLAT_TAB)
		  && (collecting == not || collected > 0))
		 || (collecting == not
		     && s_usr_strhead(working->word,	
				      s_fcrea_str(gettext("bakt")),
				      DEFAULT_COLLAT_TAB)))
	{
	    if (info->cmd != complex && info->cmd != undefined)
	    {
		xprintf(gettext("Fr mnga argument\n"));
		info->cmd = undefined;
		RETURNVOID;
	    }
	    if (direction_given == TRUE)
	    {
		xprintf(gettext("Du kan bara anvnda en av framt, bakt, fre eller efter i taget.\n"));
		info->cmd = undefined;
		RETURNVOID;
	    }
	    info->cmd = complex;

	    if (collecting != not)
	    {
		enter_collected_field(collecting, summan, info);
		summan = EMPTY_STRING;
	    }

	    xprintf(gettext(" bakt"));

	    info->direction = backward;
	    direction_given = TRUE;
	    collecting = not;
	}
	else if ((collecting == not)
		 && (s_usr_strhead(working->word,
				   s_fcrea_str(gettext("nummer")),
				   DEFAULT_COLLAT_TAB)
		     || s_usr_strhead(working->word,
				      s_fcrea_str(gettext("inlgg")),
				      DEFAULT_COLLAT_TAB)))
	{
	    info->cmd = number;
	    collecting = col_number;
	}
	else if ((collecting == not)
		 && (info->cmd == undefined)
		 && (s_usr_strhead(working->word,
				   s_fcrea_str(gettext("kommenterade")),
				   DEFAULT_COLLAT_TAB)))
	{
	    Text_no	text_to_search_from;
	    Text_no	text_to_read = 0;

	    info->cmd = number;

	    text_to_search_from = reading_state.last_viewed;

	    if ( text_to_search_from
		&& kom_get_text_stat (text_to_search_from,
				      &text_stat_val ) != FAILURE)
	    {
		Misc_info     * misc;
		unsigned short	r;

		for (misc = text_stat_val.misc_items , r = 0;
		     r < text_stat_val.no_of_misc && !text_to_read;
		     r++, misc++)
		{
		    if (misc->type == comm_to)
		    {
			text_to_read = misc->datum.comment_to;
		    }
		}
	    }

	    info->text_no = text_to_read;
	    RETURNVOID;

	}
	else if (collecting == not
		 && (info->cmd == undefined)
		 && ((tno = (Text_no) s_strtol(working->word, 
					       &fic,
					       DEFAULT_NUMBER_BASE)) != 0
		     && fic == s_strlen(working->word)))
	{
	    info->cmd = number;
	    info->text_no = tno;
	    RETURNVOID;
	} 
	else if (collecting != not) 
	{
	    s_strcat(&summan, s_fcrea_str(" "));
	    s_strcat(&summan, working->word);
	    collected++;
	}
	else 
	{
	    /*xgettext:c-format*/
	    xprintf(gettext("Oknt kommando %S.\n"), working->word);
	    info->cmd = undefined;
	    RETURNVOID;
	}

	working ++;
    }
    if (collecting != not)
    {
	if (s_strlen(summan) == 0) /* We have not yet collected anything */
	{
	    String			 re = EMPTY_STRING;

	    s_strcat(&summan, s_fcrea_str(" "));

	    switch (collecting)
	    {
	    case by:
		xprintf(gettext("Vem inlgg vill du se?\n"));
		break;
	    case to:
		xprintf(gettext("Vilket mtes inlgg vill du se?\n"));
		break;
	    case col_time:
		xprintf(gettext("Ange tid?\n"));
		break;
	    case col_number:
		xprintf(gettext("Inlgg nummer?\n"));
		break;
	    case presentation:
		xprintf(gettext("Vad/vem vill du se presentationen fr?\n"));
		break;
	    case not:
		fatal1(CLIENT_SHOULDNT_HAPPEN,
		       "You have a buggy compiler or something is very weird.\n");
	    }
	    xprintf("= ");
	    read_line (&re);		/* BUG! Check retval! */
	    newline();

	    re = s_strip_trailing(re, s_fcrea_str(" \t"));

	    if (s_strlen(re) == 0) /* Nothing was given. */
	    {
		/* Break out of here. */
		xprintf(gettext("Nehej.\n"));
		info->cmd = undefined;
		RETURNVOID;
	    }

	    s_strcat(&summan, re);

	    s_clear(&re);
	}	

	enter_collected_field(collecting, summan, info);
	summan = EMPTY_STRING;
    }
    RETURNVOID;
#undef CLEAN_UP
#undef FUNCTION
}

