/* -*- indent-tabs-mode: t; tab-width: 8; c-basic-offset: 8; -*- */

#ifndef __LIBISOFS
#define __LIBISOFS

#include <sys/time.h>
#include "libburn/libburn.h"

#ifndef DOXYGEN

#if defined(__cplusplus)
#define ISO_BEGIN_DECLS \
	namespace iso { \
		extern "C" {
#define ISO_END_DECLS \
		} \
	}
#else
#define ISO_BEGIN_DECLS
#define ISO_END_DECLS
#endif

ISO_BEGIN_DECLS

#endif

struct iso_tree;
struct iso_volumeset;

enum iso_file_name_version {
	/** The actual desired name for the file */
	ISO_FILE_NAME_FULL,
	/** The name that the file will be refered to as in the ISO9660
	    filesystem. The name may change when options for generation of the
	    filesystem change. */
	ISO_FILE_NAME_ISO,
	/** The name that the file wil be refered to as in the Joliet
	    filesystem. The name may change when options for generation of the
	    filesystem change. */
	ISO_FILE_NAME_JOLIET,
	/** The name that the file will be refered to as in the Rock-Ridge
	    filesystem. The name may change when options for generation of the
	    filesystem change. */
	ISO_FILE_NAME_ROCKRIDGE
};

struct iso_tree_dir** iso_tree_add_dir (struct iso_tree_dir **parent,
					const char *name,
					int volume);
/* volume should usually be 0 :) */
struct iso_tree_file** iso_tree_add_file (struct iso_tree_dir **parent,
					  const char *path,
					  int volume);

/** Get the name of a file in the volume
    @param file The file to get the name for
    @param ver The file system type for which to get the name. Different
               systems impose different limitations and so different names will
	       result for each of them. The names may change when options
	       for generation of the filesystem are changed, and should be
	       updated visually when options are changed.
*/
const char* iso_tree_file_get_name (struct iso_tree_file **file,
				    enum iso_file_name_version ver);
/** Sets the desired name for a file in the volume.
    After changing a file's name with this function, iso_tree_file_get_name
    should be used to update the display for the user to the new name.
    @param file The file to rename
    @param name The name to give the file (name.extension)
    @return Returns a non zero value if the name was set successfully, returns
            zero if the specified name was invalid.
*/
int iso_tree_file_set_name (struct iso_tree_file **file,
			    const char *name);

const char* iso_tree_dir_get_name (struct iso_tree_dir **dir,
				   enum iso_file_name_version ver);
int iso_tree_dir_set_name (struct iso_tree_dir **dir,
			   const char *name);

/** Create a new volume set
    @param volumeset_size The number of volumes in the set
    @param volumeset_id The name of the volume set. Only the first 128 valid
                        characters will be used
    @param volume_id An array of names for the volumes (one for each up to
                     the volumeset_size). Only the first 32 valid characters
		     will be used
    @param publisher_id The name of the publisher of the volume set. Only the
                        first 128 valid characters will be used
    @return A new volume set
*/
struct iso_volumeset* iso_volumeset_new (int              volumeset_size,
					 const char      *volumeset_id,
					 char      *const*volume_id,
					 const char      *publisher_id,
					 const char      *data_preparer_id,
					 time_t          *expiration_time,
					 time_t          *effective_time);
/** Free the volume set */
void iso_volumeset_free (struct iso_volumeset *volumeset);

/* XXX functions to set the copyright/etc files for the volumeset */

/** Returns the root directory node for the volume set */
struct iso_tree_dir** iso_volumeset_get_root (struct iso_volumeset *volumeset);

int  iso_volumeset_get_iso_level (struct iso_volumeset *volumeset);
void iso_volumeset_set_iso_level (struct iso_volumeset *volumeset, int level);

int  iso_volumeset_get_joliet (struct iso_volumeset *volumeset);
void iso_volumeset_set_joliet (struct iso_volumeset *volumeset, int joliet);

/** Create a burn_source which can be used as a data source for a track

    The volume set used to create the libburn_source can _not_ be modified
    until the libburn_source is freed.

    @param volumeset The volume set from which you want to write
    @param volume The volume in the set which you want to write (usually 0)
    @param sector_size The size of the physical sectors on the target disc.
           This value must match the value used to write in libburn or the
	   process will fail.
    @return A burn_source to be used for the data source for a track
*/
struct burn_source* iso_source_new (struct iso_volumeset *volumeset,
				    int volume);

/** Write a portion of the iso output to the buffer
    @param src The data source
    @param buffer The target for the output
    @param size The size of the buffer
*/
int iso_source_generate (struct burn_source *src,
			 unsigned char *buffer,
			 int size);


/* XXX debugging funcs will be removed */
void iso_tree_print (struct iso_tree_dir **root);

#ifndef DOXYGEN

ISO_END_DECLS

#endif

#endif /* __LIBISOFS */
