/*
  CoreLinux++ 
  Copyright (C) 1999,2000 CoreLinux Consortium
  
   The CoreLinux++ Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   The CoreLinux++ Library Library is distributed in the hope that it will 
   be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the GNU C Library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  
*/   

#if   !defined(__COMMON_HPP)
#include <Common.hpp>
#endif

#if   !defined(__COMMANDFRAME_HPP)
#include <CommandFrame.hpp>
#endif

namespace corelinux
{
   // Default constructor

   CommandFrame::CommandFrame( bool autoReverse )
      :
      Synchronized(),
      theAutoReverseFlag( autoReverse ),
      theWorkState( BUILDING ),
      theCommands()
   {
      ;  // do nothing, all in the initialization
   }

   // Copy constructor

   CommandFrame::CommandFrame( CommandFrameCref aCommandFrame )
      throw( CommandFrameException )
      :
      Synchronized(),
      theAutoReverseFlag( aCommandFrame.theAutoReverseFlag ),
      theWorkState( BUILDING ),
      theCommands()
   {

      if( aCommandFrame.getState() == BUILDING ||
          aCommandFrame.getState() == COMPLETED ||
          aCommandFrame.getState() == REVERSED )
      {
         aCommandFrame.getCommands( theCommands );
      }
      else
      {
         throw CommandFrameException( LOCATION );
      }

   }

   // Destructor

   CommandFrame::~CommandFrame( void )
   {
      theAutoReverseFlag = false;
      theWorkState = NEVERCOMPLETED;
      theCommands.clear();
   }

   // Operator assignment

   CommandFrameRef CommandFrame::operator=( CommandFrameCref aCommandFrame )
      throw( CommandFrameException )
   {
      if( this != &aCommandFrame )
      {

         if( this->getState() == BUILDING  && 
             ( aCommandFrame.getState() == BUILDING ||
             aCommandFrame.getState() == COMPLETED ||
             aCommandFrame.getState() == REVERSED ) )
         {
            GUARD;
            theCommands.clear();
            aCommandFrame.getCommands( theCommands );
            theAutoReverseFlag = aCommandFrame.theAutoReverseFlag;
         }
         else
         {
            throw CommandFrameException( LOCATION );
         }

      }
      else
      {
         ;  // do nothing
      }

      return ( *this );
   }

   // Equality 

   bool  CommandFrame::operator==( CommandFrameCref aCommandFrame ) const
   {
      return ( this == &aCommandFrame );
   }

   // Get the state of the frame

   WorkState CommandFrame::getState( void ) const
   {
      GUARD;
      return theWorkState;
   }

   bool CommandFrame::getReverseFlag( void ) const
   {
      return theAutoReverseFlag;
   }

   // Get the command list

   void  CommandFrame::getCommands( CommandsRef aCommandList ) const
   {
      GUARD;
      if( aCommandList != theCommands )
      {
         aCommandList = theCommands;
      }
      else
      {
         ;  // do nothing
      }
   }

   // Add a command to the list

   CommandFrameRef  CommandFrame::operator+=( AbstractCommandPtr aCommand )
      throw( CommandFrameException )
   {
      GUARD;
      if( aCommand != NULLPTR )
      {
         theCommands.push_back( aCommand );
      }
      else
      {
         throw CommandFrameException( LOCATION );
      }

      return ( *this );
   }

   // Appends the commands from the other list

   CommandFrameRef CommandFrame::operator+=( CommandFrameCref aCommandFrame )
      throw( CommandFrameException )
   {
      if( this != &aCommandFrame )
      {
         GUARD;
         Commands temp;
         aCommandFrame.getCommands( temp );
         for( Dword x = 0; x < temp.size(); ++x )
         {
            theCommands.push_back( temp[x] );
         }
      }
      else
      {
         ;  // do nothing
      }
      return ( *this );
   }

   // Add a command to the list

   void  CommandFrame::addCommand( AbstractCommandPtr aCommand )
      throw( CommandFrameException )
   {
      GUARD;
      if( aCommand != NULLPTR )
      {
         theCommands.push_back( aCommand );
      }
      else
      {
         throw CommandFrameException( LOCATION );
      }
   }

   // Set the autoreverse flag

   void CommandFrame::setAutoReverse( bool autoReverse )
      throw( CommandFrameException )
   {
      GUARD;
      if( theWorkState != BUILDING )
      {
         throw CommandFrameException( LOCATION );
      }
      else
      {
         theAutoReverseFlag = autoReverse;
      }
   }

   // execute

   void  CommandFrame::execute( void )
      throw( CommandFrameException )
   {
      GUARD;
      if( theWorkState == BUILDING )
      {
         theWorkState = EXECUTING;
         try
         {
            theWorkState = executeCommands();
         }
         catch( CommandFrameException aExcp )
         {
            theWorkState = NEVERCOMPLETED;
            throw ;
         }
         catch( ExceptionRef aExcp )
         {
            theWorkState = NEVERCOMPLETED;
            aExcp.addUnwindInfo( "CommandFrame::execute Exception handler" );
            throw ;
         }
         catch( ... )
         {
            theWorkState = NEVERCOMPLETED;
            throw CommandFrameException("Unhandled exception received",LOCATION);
         }
      }
      else
      {
         throw CommandFrameException( LOCATION );
      }

   }

   // execute reverse

   void CommandFrame::executeReverse( void )
      throw( CommandFrameException )
   {
      GUARD;
      if( theWorkState == COMPLETED )
      {
         theWorkState = REVERSING;
         try
         {
            theWorkState = executeReverseCommands();
         }
         catch( CommandFrameException aExcp )
         {
            theWorkState = NEVERCOMPLETED;
            throw ;
         }
         catch( ExceptionRef aExcp )
         {
            theWorkState = NEVERCOMPLETED;
            aExcp.addUnwindInfo( "CommandFrame::executeReverse Exception handler" );
            throw ;
         }
         catch( ... )
         {
            theWorkState = NEVERCOMPLETED;
            throw CommandFrameException("Unhandled exception received",LOCATION);
         }
      }
      else
      {
         throw CommandFrameException( LOCATION );
      }

   }

   // Do the default work

   WorkState CommandFrame::executeCommands( void )
   {
      bool              runException( false );
      Int               lastValidPosition( 0 );
      Int               maxPosition( theCommands.size() );
      WorkState         aState( COMPLETED );

      //
      // Attempt to run through the commands
      //

      for( lastValidPosition = 0; 
           ( lastValidPosition < maxPosition) && (runException == false) ; 
            ++lastValidPosition 
         )
      {
         try
         {
            AbstractCommandPtr   aCommand( theCommands[lastValidPosition] );
            aCommand->execute();
         }

         //
         // If we get an exception for any reason set our indicator
         //

         catch( ExceptionRef aExcp )
         {
            runException = true;
         }
         catch( ... )
         {
            runException = true;
         }

         //
         // If we need to get out, first check for reverse and if set
         // execute the completed commands reverses until we get back
         // to a valid state
         //

         if( runException == true )
         {
            if( this->getReverseFlag() == true )
            {
               for(--lastValidPosition ; lastValidPosition >= 0; --lastValidPosition )
               {
                  theCommands[lastValidPosition]->executeReverse();
               }

               aState = REVERSED;
            }
            else
            {
               throw CommandFrameException( LOCATION );
            }
         }
         else
         {
            ;  // do nothing, keep executing
         }
      }
      return aState;
   }

   // Do the default reverse

   WorkState CommandFrame::executeReverseCommands( void )
   {
      Int               maxPosition( theCommands.size() - 1 );

      for( ; maxPosition >= 0; --maxPosition )
      {
         theCommands[maxPosition]->executeReverse();
      }

      return REVERSED;
   }
}

/*
   Common rcs information do not modify
   $Author: frankc $
   $Revision: 1.3 $
   $Date: 2000/06/06 12:04:12 $
   $Locker:  $
*/


