/* 
 * $Id: ctkoptionmenu.c,v 1.31 2000/07/12 00:44:10 cbond Exp $
 *
 * CTK - Console Toolkit
 *
 * Copyright (C) 1998-2000 Stormix Technologies Inc.
 *
 * License: LGPL
 *
 * Authors: Kevin Lindsay, Wesley Terpstra
 *  
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2 of the License, or (at your option) any later version.
 *    
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *    
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <glib.h>

#include "ctk.h"
#include "ctkcolor.h"

void ctk_option_menu_release_signals(CtkOptionMenu* option_menu)
{
      ctk_signal_disconnect(CTK_OBJECT(option_menu->menu), option_menu->lost_focus);
      ctk_signal_disconnect(CTK_OBJECT(option_menu->menu), option_menu->select_row);
      ctk_signal_disconnect(CTK_OBJECT(option_menu->menu), option_menu->key_press);
      ctk_signal_disconnect(CTK_OBJECT(option_menu->menu), option_menu->button_press);
}

gboolean ctk_option_menu_menu_lost_focus(CtkWidget* widget, CdkEventFocus* event, gpointer data)
{
      /* If the list loses focus, it must die */
      CTK_OPTION_MENU(data)->doClose = TRUE;
      ctk_signal_emit_by_name(CTK_OBJECT(widget), "select_row");

      return FALSE;
}

void ctk_option_menu_colour_bar(CtkOptionMenu* om)
{
      CtkMenu* menu = CTK_MENU(om->menu);
      
      if (CTK_WIDGET(om)->inversed)
      {
	      ctk_table_colour_row(CTK_TABLE(menu), menu->selected_item+1, 
		   ctk_calculate_palette(CTK_COLOR_WHITE, CTK_COLOR_RED));
      }
      else
      {
	      ctk_table_colour_row(CTK_TABLE(menu), menu->selected_item+1, 
		   ctk_calculate_palette(CTK_COLOR_WHITE, CTK_COLOR_BLUE));
      }
}

gboolean ctk_option_menu_menu_button_press(CtkWidget* widget, CdkEventButton* event, gpointer data)
{
	/* If a button is pressed, we hear the click and murder the popup */
	CTK_OPTION_MENU(data)->doClose = TRUE;
	/* Rely on option menu to trigger select_row */
	return FALSE;
}

gboolean ctk_option_menu_menu_key_press(CtkWidget* widget, CdkEventKey* event, gpointer data)
{
	if ((event->keyval == AK_ENTER) || (event->keyval == '\r') ||
	    (event->keyval == ' '))
	{
		CTK_OPTION_MENU(data)->doClose = TRUE;
		/* Rely on option menu to trigger select_row */
	}
	
	return FALSE;
}

gboolean ctk_option_menu_menu_select_row(CtkWidget* widget, gint row, gint col, 
					 CdkEventButton* event, gpointer data)
{
      /* If the list has a row selected, it must die */
      CtkOptionMenu* om = CTK_OPTION_MENU(data);
      CtkMenuItem*   mi;
      
      if (!om->open || !om->doClose) 
	    return TRUE;

      /* Remove the menu from the window and put it back in us. */
      ctk_widget_set_sensitive(om->menu, FALSE);
      ctk_container_remove(CTK_CONTAINER(om->scroll), widget);
      ctk_container_add(CTK_CONTAINER(om), widget);

      CTK_MENU(om->menu)->selected_item = row;

      ctk_widget_hide(om->window);
      om->open = FALSE;
      ctk_option_menu_colour_bar(om);
      
      mi = CTK_MENU_ITEM(ctk_menu_get_active(CTK_MENU(om->menu)));
      if (mi)
      {
      	ctk_menu_item_activate(mi);
      }

      return TRUE;
}

void ctk_option_menu_hook_signals(CtkOptionMenu* option_menu)
{
      option_menu->select_row = 
	    ctk_signal_connect(CTK_OBJECT(option_menu->menu), "select_row",
			       CTK_SIGNAL_FUNC(&ctk_option_menu_menu_select_row), option_menu);
      option_menu->lost_focus = 
	    ctk_signal_connect(CTK_OBJECT(option_menu->menu), "focus_out_event",
			       CTK_SIGNAL_FUNC(&ctk_option_menu_menu_lost_focus), option_menu);
      option_menu->key_press = 
	    ctk_signal_connect(CTK_OBJECT(option_menu->menu), "key_press_event",
			       CTK_SIGNAL_FUNC(&ctk_option_menu_menu_key_press), option_menu);
      option_menu->button_press = 
	    ctk_signal_connect(CTK_OBJECT(option_menu->menu), "button_press_event",
			       CTK_SIGNAL_FUNC(&ctk_option_menu_menu_button_press), option_menu);
}

void ctk_option_menu_open(CtkOptionMenu* om)
{
	CtkWidget* self = CTK_WIDGET(om);
	
	if (om->open) 
	      return;

	om->wasDragged = FALSE;
	om->open       = TRUE;
	om->doClose    = FALSE;
	
	/* Minimum size the window */
	ctk_size_set_minimum(om->window);
	
	/* Ok, now, we will try to position it vertically */
	if (om->menu->min_height + 2 < ctk_scr_r - self->row)
	{ /* If the thing fits below us, that's where it goes */
		om->window->height = om->menu->min_height + 2;
		om->window->row = self->row;
	}
	else if (om->menu->min_height + 2 < ctk_scr_r)
	{ /* If the thing fits vertically, position at bottom */
		om->window->height = om->menu->min_height + 2;		  	
		om->window->row = ctk_scr_r - om->window->height;
	}
	else
	{ /* It won't fit, so use whole screen and let the scrolled window work */
		om->window->height = ctk_scr_r;
		om->window->row = 0;
	}
	
	om->window->width = self->width - 1;
	om->window->col   = self->col;

	/* Move the menu into the window */
	ctk_container_remove(CTK_CONTAINER(om), om->menu);
	ctk_container_add(CTK_CONTAINER(om->scroll), om->menu);

	ctk_window_set_focus(CTK_WINDOW(om->window), om->menu);
	ctk_widget_set_sensitive(om->menu, TRUE);
	
	ctk_widget_show(om->window);
}

gboolean ctk_option_menu_do_click(CtkWidget* widget, CdkEventButton* event, gpointer data)
{
      CTK_BUTTON(widget)->pressed = FALSE;
      ctk_signal_emit_by_name(CTK_OBJECT(widget), "clicked");
      return TRUE;
}

gboolean ctk_option_menu_click(CtkWidget* widget, CdkEventButton* event, gpointer data)
{
      ctk_option_menu_open(CTK_OPTION_MENU(widget));
      return TRUE;
}

gboolean ctk_option_menu_drag(CtkWidget* widget, CdkEventButton* event, gpointer data)
{
      /* If the user dragged their mouse on the combo box, move selection */
      CtkOptionMenu* om = CTK_OPTION_MENU(widget);
      om->wasDragged = TRUE;

      ctk_clist_move_cursor(CTK_CLIST(om->menu), event->dy);

      return TRUE;
}

gboolean ctk_option_menu_button_release(CtkWidget* widget, CdkEventButton* event, gpointer data)
{
      /* If the button was released after dragging, we're done. */
      CtkOptionMenu* om = CTK_OPTION_MENU(widget);

      if (om->wasDragged)
      {	
      	om->doClose = TRUE;
	    ctk_signal_emit_by_name(CTK_OBJECT(om->menu), "select_row");
	  }

      return TRUE;
}

gboolean ctk_option_menu_key_press(CtkWidget* widget, CdkEventKey* event, gpointer data)
{
      /* This is bound onto the text entry widget to catch an up/down keypress
       */

      if ((event->keyval == AK_ARROW_DOWN) || (event->keyval == AK_ARROW_UP)) {
	    ctk_signal_emit_by_name(CTK_OBJECT(widget), "clicked");
	    return TRUE;
      }

      return FALSE;
}

gboolean ctk_option_menu_focus_in(CtkWidget* widget, CdkEventFocus* event, gpointer data)
{
	if (widget->sensitive)
	{
		widget->inversed = TRUE;
		ctk_option_menu_colour_bar(CTK_OPTION_MENU(widget));
		return TRUE;
	}
	
	return FALSE;
}

gboolean ctk_option_menu_focus_out(CtkWidget* widget, CdkEventFocus* event, gpointer data)
{
	if (widget->sensitive)
	{
		widget->inversed = FALSE;
		ctk_option_menu_colour_bar(CTK_OPTION_MENU(widget));
		return TRUE;
	}
	
	return FALSE;
}

gpointer ctk_option_menu_destroy(CtkObject* object);
void ctk_option_menu_min_size (CtkWidget* widget);
void ctk_option_menu_real_size(CtkWidget* widget);

/* Initialize the Option Menu Structure */
void ctk_option_menu_init(CtkOptionMenu* optionmenu)
{
	ctk_button_init(&optionmenu->button);

	optionmenu->open = FALSE;
	optionmenu->menu = NULL;
	
	ctk_signal_new("activate", CtkTypeOptionMenu);

	((CtkObject *)optionmenu)->type = CtkTypeOptionMenu;
	((CtkWidget *)optionmenu)->width = 3;
	((CtkWidget *)optionmenu)->height = 1;
	((CtkWidget *)optionmenu)->orig_width = 3;
	((CtkWidget *)optionmenu)->orig_height = 1;

	((CtkObject *)optionmenu)->destroy_func = ctk_option_menu_destroy;

	optionmenu->clicked = FALSE;

	optionmenu->window = ctk_window_new(CTK_WINDOW_TOPLEVEL);
	CTK_WINDOW(optionmenu->window)->OPTIONS_MASK = 0;

	optionmenu->scroll = ctk_scrolled_window_new(
	      CTK_ADJUSTMENT(ctk_adjustment_new(0, 0, 0, 5, 0, 0)),
	      CTK_ADJUSTMENT(ctk_adjustment_new(0, 0, 0, 5, 0, 0)));
	ctk_container_add(CTK_CONTAINER(optionmenu->window), optionmenu->scroll);
	ctk_widget_show(optionmenu->scroll);

	CTK_WIDGET(optionmenu)->set_min_size = &ctk_option_menu_min_size;
	CTK_WIDGET(optionmenu)->set_real_size = &ctk_option_menu_real_size;

	ctk_signal_connect(CTK_OBJECT(optionmenu), "pressed", 
			   CTK_SIGNAL_FUNC(&ctk_option_menu_do_click), NULL);
	ctk_signal_connect(CTK_OBJECT(optionmenu), "clicked", 
			   CTK_SIGNAL_FUNC(&ctk_option_menu_click), NULL);
	ctk_signal_connect(CTK_OBJECT(optionmenu), "ctk_drag",
			   CTK_SIGNAL_FUNC(&ctk_option_menu_drag), NULL);
	ctk_signal_connect(CTK_OBJECT(optionmenu), "released",
			   CTK_SIGNAL_FUNC(&ctk_option_menu_button_release), NULL);
	ctk_signal_connect(CTK_OBJECT(optionmenu), "key_press_event",
			   CTK_SIGNAL_FUNC(&ctk_option_menu_key_press), NULL);
	ctk_signal_connect(CTK_OBJECT(optionmenu), "focus_in_event", 
			   CTK_SIGNAL_FUNC(&ctk_option_menu_focus_in), NULL);
	ctk_signal_connect(CTK_OBJECT(optionmenu), "focus_out_event", 
			   CTK_SIGNAL_FUNC(&ctk_option_menu_focus_out), NULL);
}

CtkWidget* ctk_option_menu_new(void)
{
	CtkOptionMenu *optionmenu;
	
	optionmenu = g_malloc(sizeof(CtkOptionMenu));
	
	ctk_option_menu_init(optionmenu);
	
	return ((CtkWidget *)optionmenu);
}

/* Set a menu */
void ctk_option_menu_set_menu(CtkOptionMenu* option_menu, CtkWidget* menu)
{
	if (!option_menu) 
	{
		ctk_close();
		g_error("ctk_option_menu_set_menu: CtkOptionMenu *option_menu == NULL!");
	}

	ctk_option_menu_remove_menu(option_menu);
	option_menu->menu = menu;
	ctk_size_mark_changed(CTK_WIDGET(option_menu));

	if (menu)
	{
	      ctk_container_add(CTK_CONTAINER(option_menu), menu);
	      ctk_option_menu_hook_signals(option_menu);
	      ctk_widget_set_sensitive(menu, FALSE);
	      ctk_option_menu_colour_bar(option_menu);
	      ctk_widget_show(menu);
	}
}

/* Set Option Menu History */
void ctk_option_menu_set_history(CtkOptionMenu* option_menu, guint index)
{
	CtkMenu* menu;
	
	if (!option_menu)
	    return;
	    
	menu = CTK_MENU(option_menu->menu);
	if (!menu)
		return;
		
	if (index < 0)
	    index = 0;
	
    ctk_table_colour_row(CTK_TABLE(menu), menu->selected_item+1, 
			   ctk_calculate_palette(CTK_COLOR_BLACK, CTK_COLOR_LIGHT_GRAY));
	ctk_menu_set_active(menu, index);
    ctk_option_menu_colour_bar(option_menu);
	
	ctk_size_mark_changed(CTK_WIDGET(option_menu));
}

gint ctk_option_menu_get_history(CtkOptionMenu* option_menu)
{
	CtkMenu* menu;
	
	if (!option_menu)
	    return -1;
	    
	menu = CTK_MENU(option_menu->menu);
	if (!menu)
		return -1;
		
	return menu->selected_item;
}
	
/* Remove Menu from Option Menu */
void  ctk_option_menu_remove_menu(CtkOptionMenu* option_menu)
{
	if (!option_menu)
	    return;

	if (option_menu->menu)
	{
	      ctk_option_menu_release_signals(option_menu);
	      ctk_container_remove(CTK_CONTAINER(option_menu), option_menu->menu);
	      ctk_widget_destroy(option_menu->menu);
	      option_menu->menu = NULL;
	}
	
	ctk_size_mark_changed(CTK_WIDGET(option_menu));
}

/* Get Menu */
CtkWidget* ctk_option_menu_get_menu(CtkOptionMenu* option_menu)
{
	if (!option_menu)
	    return NULL;

	return option_menu->menu;
}

/* Destroy Option Menu */
gpointer ctk_option_menu_destroy(CtkObject* object)
{
	CtkOptionMenu *optionmenu;

	optionmenu = CTK_OPTION_MENU(object);

	if (optionmenu->window)
	{
	    ctk_widget_destroy(optionmenu->window);
	}

	return NULL;
}

void ctk_option_menu_min_size(CtkWidget* widget)
{
      CtkOptionMenu* optionmenu = CTK_OPTION_MENU(widget);

      if (optionmenu->menu)
      {
	    CtkMenu* menu = CTK_MENU(optionmenu->menu);
	    gint row;

	    widget->min_height = 0;
	    for (row = 0; row < menu->num_items; row++)
	    {
		  gint delta = 
			ctk_table_get_row_row(CTK_TABLE(menu), row+2) -
			ctk_table_get_row_row(CTK_TABLE(menu), row+1);

		  if (delta > widget->min_height)
			widget->min_height = delta;
	    }

	    widget->min_width = optionmenu->menu->min_width + 3;
      }
      else
      {
	    widget->min_width = 1;
	    widget->min_height = 1;
      }
}

void ctk_option_menu_real_size(CtkWidget* widget)
{
      CtkOptionMenu* optionmenu = CTK_OPTION_MENU(widget);

      if (optionmenu->menu)
      {
	    optionmenu->menu->width  = widget->width - 1;
	    optionmenu->menu->height = optionmenu->menu->min_height;

	    optionmenu->menu->col = widget->col;
	    optionmenu->menu->row = widget->row;
	    
	    optionmenu->menu->row = widget->row 
		  - (ctk_table_get_row_row(CTK_TABLE(optionmenu->menu),
					CTK_MENU (optionmenu->menu)->selected_item+1)
		      - optionmenu->menu->row);
      }
}
