#include "basic-gconf-app.h"
#include <inti/gtk/button.h>
#include <inti/bind.h>

/*  ConfigEntry
 */

ConfigEntry::ConfigEntry(const char *config_key)
: Gtk::HBox(false, 5), key(config_key)
{
	Gtk::Label *label = new Gtk::Label(config_key);
	entry = new Gtk::Entry;
	pack_start(*label, false, false);
	pack_start(*entry, false, false);
}

ConfigEntry::~ConfigEntry()
{
}

const String& 
ConfigEntry::get_key()
{ 
	return key; 
}
	
Gtk::Entry* 
ConfigEntry::get_entry()
{ 
	return entry; 
}

/*  ConfigurableFrame
 */

ConfigurableFrame::ConfigurableFrame(GConf::Client& client, const char *config_key)
: client_(client)
{
	String s;
	s.format("Value of \"%s\"", config_key);
	set_label(s);
	
	label = new Gtk::Label;
	add(*label);

	s = client.get_string(config_key);
	if (!s.null())
		label->set_text(s);

	// Notification slot for our label widgets that monitor the current value of a gconf key.
	// i.e. we are conceptually "configuring" the label widgets. A smart pointer is used to
	// unreference notify_slot when it goes out of scope. Remember, a slot not connected to
	// a signal must be explicitly unreferenced.
	notify_slot = slot(this, &ConfigurableFrame::on_frame_notify);

	// Note that notify_id will be 0 if there was an error, so we handle that in the destructor.
	notify_id = client.notify_add(config_key, notify_slot);
}

ConfigurableFrame::~ConfigurableFrame()
{
	if (notify_id)
		client_.notify_remove(notify_id);
}

void
ConfigurableFrame::on_frame_notify(unsigned int /*cnxn_id*/, const GConf::Entry& entry)
{
	GConf::Value value = entry.get_value();

	// Note that value can be null (unset) or it can have the wrong type! Need
	// to check that to survive gconftool --break-key
 	if (!value.is_set())
	{
		label->set_text("");
	}
	else if (value.get_type() == GConf::VALUE_STRING)
	{
		label->set_text(value.get_string());
	}
	else
	{
		label->set_text("!type error!");
	}
}

/*  PreferenceDialog
 */

PreferenceDialog::PreferenceDialog(GConf::Client& client, Gtk::Window& parent)
: Gtk::Dialog("Basic GConf App Preferences", &parent, 0, GTK_STOCK_CLOSE, Gtk::RESPONSE_ACCEPT, 0),
  client_(client)
{
	set_default_response(Gtk::RESPONSE_ACCEPT);

	// resizing doesn't grow the entries anyhow
	set_resizable(false);
	
	Gtk::VBox *vbox = new Gtk::VBox(false, 5);
	vbox->set_border_width(5);
	client_area()->pack_start(*vbox);

	ConfigEntry *entry = create_config_entry("/apps/basic-gconf-app/foo", true);
	vbox->pack_start(*entry, false, false);

	entry = create_config_entry("/apps/basic-gconf-app/bar");
	vbox->pack_start(*entry, false, false);

	entry = create_config_entry("/apps/basic-gconf-app/baz");
	vbox->pack_start(*entry, false, false);

	entry = create_config_entry("/apps/basic-gconf-app/blah");
	vbox->pack_start(*entry, false, false);
	vbox->show_all();
}

PreferenceDialog::~PreferenceDialog()
{
}

ConfigEntry*
PreferenceDialog::create_config_entry(const char *config_key, bool focus)
{
	ConfigEntry *config_entry = new ConfigEntry(config_key);
	Gtk::Entry *entry = config_entry->get_entry();

	// This will print an error via default error handler if the key isn't set to a string
	String s = client_.get_string(config_key);
	if (!s.null())
		entry->set_text(s);

	// Commit changes if the user focuses out, or hits enter; we don't do this on "changed" since
	// it'd probably be a bit too slow to round-trip to the server on every "changed" signal.
	sig_focus_out_event().connect(bind(slot(this, &PreferenceDialog::on_entry_focus_out), config_entry));
 	entry->sig_activate().connect(bind(slot(this, &PreferenceDialog::on_entry_activate), config_entry));
	
	// Set the entry insensitive if the key it edits isn't writable. Technically, we should update
	// this sensitivity if the key gets a change notify, but that's probably overkill.
	entry->set_sensitive(client_.key_is_writable(config_key));

	if (focus)
		entry->grab_focus();

	return config_entry;
}

namespace {

void config_entry_commit(GConf::Client& client, ConfigEntry *entry)
{
	// Commit changes to the GConf database.
	String s = entry->get_entry()->get_chars(0, -1);
	
	// Unset if the string is zero-length, otherwise set
	if (!s.empty())
		client.set_string(entry->get_key(), s);
	else
		client.unset(entry->get_key());
}

} // namespace

bool 
PreferenceDialog::on_entry_focus_out(GdkEventFocus */*event*/, ConfigEntry *entry)
{
	config_entry_commit(client_, entry);
	return false;
}

void 
PreferenceDialog::on_entry_activate(ConfigEntry *entry)
{
	config_entry_commit(client_, entry);
}

void
PreferenceDialog::on_response(int /*response_id*/)
{
	dispose();
}

/*  BasicGConfApp
 */

BasicGConfApp::BasicGConfApp()
{
	set_title("Basic GConf App");

	// Get the default client
	client = GConf::Client::get_default();

	// Tell GConf::Client that we're interested in the given directory. This means 
	// GConf::Client will receive notification of changes to this directory, and
	// cache keys under this directory. So _don't_ add "/" or something silly 
	// like that or you'll end up with a copy of the whole GConf database. ;-)
	// We use the default error handler; and use PRELOAD_NONE to avoid loading all
	// config keys on startup. If your app pretty much reads all config keys on 
	// startup, then preloading the cache may make sense.
	client->add_dir("/apps/basic-gconf-app", GConf::Client::PRELOAD_NONE);

	Gtk::VBox *vbox = new Gtk::VBox(false, 5);
	vbox->set_border_width(5);
	add(*vbox);

	ConfigurableFrame *frame = new ConfigurableFrame(*client, "/apps/basic-gconf-app/foo");
	vbox->pack_start(*frame);

	frame = new ConfigurableFrame(*client, "/apps/basic-gconf-app/bar");
	vbox->pack_start(*frame);

	frame = new ConfigurableFrame(*client, "/apps/basic-gconf-app/baz");
	vbox->pack_start(*frame);

	frame = new ConfigurableFrame(*client, "/apps/basic-gconf-app/blah");
	vbox->pack_start(*frame);

	Gtk::Button *button = new Gtk::Button("_Preferences", true);
	button->sig_clicked().connect(slot(this, &BasicGConfApp::on_button_clicked));
	vbox->pack_start(*button);
	vbox->show_all();
}

BasicGConfApp::~BasicGConfApp()
{
}

void
BasicGConfApp::on_button_clicked()
{
	Gtk::Dialog *dialog = new PreferenceDialog(*client, *this);
	dialog->show();
}

int main (int argc, char *argv[])
{
	using namespace Main;

	init(&argc, &argv);

	BasicGConfApp window;
	window.sig_destroy().connect(slot(&Inti::Main::quit));
	window.show();

	run();
	return 0;
}

