/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002-2003 The Inti Development Team.
 *
 *  menushell.cc - GtkMenuShell C++ wrapper implemenation
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
#include "menushell.h"
#include "private/menushell_p.h"
#include "menu.h"
#include "imagemenuitem.h"

using namespace Inti;

/*  Gtk::MenuShell
 */

Gtk::MenuShell::MenuShell(GtkMenuShell *menu_shell, bool reference)
: Container((GtkContainer*)menu_shell, reference)
{
}

Gtk::MenuShell::MenuShell()
: Container((GtkContainer*)MenuShellClass::create())
{
}

Gtk::MenuShell::~MenuShell()
{
}

GtkMenuShellClass*
Gtk::MenuShell::gtk_menu_shell_class() const 
{ 
	return get_class<GtkMenuShellClass>(); 
}

Gtk::MenuShell::operator GtkMenuShell* () const
{ 
	return this ? gtk_menu_shell() : 0; 
}
	
bool 
Gtk::MenuShell::children(std::vector<MenuItem*>& child_list) const
{
	g_return_val_if_fail(child_list.empty(), false);
	GList *next = gtk_menu_shell()->children;
	while (next != 0)
	{
		child_list.push_back(G::Object::wrap<MenuItem>((GtkMenuItem*)next->data));
		next = g_list_next(next);
	}
	return !child_list.empty();
}

void
Gtk::MenuShell::append(MenuItem& item, const Slot0<void> *callback)
{
	insert(item, -1, callback);
}

void
Gtk::MenuShell::append(MenuItem& item, const AccelKey& accel_key, const Slot0<void> *callback)
{
	insert(item, -1, accel_key, callback);
}
	
void
Gtk::MenuShell::append(MenuItem& item, const AccelKey& accel_key, AccelGroup& accel_group, const Slot0<void> *callback)
{
	insert(item, -1, accel_key, accel_group, callback);
}

void
Gtk::MenuShell::append(const char *stock_id, const Slot0<void> *callback)
{
	insert(stock_id, -1, callback);
}

void
Gtk::MenuShell::prepend(MenuItem& item, const Slot0<void> *callback)
{
	insert(item, 0, callback);
}

void
Gtk::MenuShell::prepend(MenuItem& item, const AccelKey& accel_key, const Slot0<void> *callback)
{
	insert(item, 0, accel_key, callback);
}
	
void
Gtk::MenuShell::prepend(MenuItem& item, const AccelKey& accel_key, AccelGroup& accel_group, const Slot0<void> *callback)
{
	insert(item, 0, accel_key, accel_group, callback);
}

void
Gtk::MenuShell::prepend(const char *stock_id, const Slot0<void> *callback)
{
	insert(stock_id, 0, callback);
}

void
Gtk::MenuShell::insert(MenuItem& item, int position, const Slot0<void> *callback)
{
	gtk_menu_shell_insert(gtk_menu_shell(), item.gtk_widget(), position);
	if (callback)
		item.sig_activate().connect(callback);
}

void
Gtk::MenuShell::insert(MenuItem& item, int position, const AccelKey& accel_key, const Slot0<void> *callback)
{
	Menu *menu = dynamic_cast<Menu*>(this);
	if (menu)
	{
		Gtk::AccelGroup *accel_group = menu->get_accel_group();
		g_return_if_fail(accel_group != 0);
		insert(item, position, accel_key, *accel_group, callback);
		return;
	}
	insert(item, position, callback);
}

void
Gtk::MenuShell::insert(MenuItem& item, int position, const AccelKey& accel_key, AccelGroup& accel_group, const Slot0<void> *callback)
{
	item.add_accelerator("activate", accel_group, accel_key);
	gtk_menu_shell_insert(gtk_menu_shell(), item.gtk_widget(), position);
	if (callback)
		item.sig_activate().connect(callback);
}

void
Gtk::MenuShell::insert(const char *stock_id, int position, const Slot0<void> *callback)
{
	Menu *menu = dynamic_cast<Menu*>(this);
	if (menu)
	{
		Gtk::AccelGroup *accel_group = menu->get_accel_group();
		insert(*(new StockMenuItem(stock_id, accel_group)), position, callback);
		return;
	}
	insert(*(new MenuItem(stock_id)), position, callback);
}

void
Gtk::MenuShell::select_item(MenuItem& item)
{
	gtk_menu_shell_select_item(gtk_menu_shell(), item.gtk_widget());
}

void 
Gtk::MenuShell::deselect()
{
	gtk_menu_shell_deselect(gtk_menu_shell());
}

void 
Gtk::MenuShell::activate_item(MenuItem& item, bool force_deactivate)
{
	gtk_menu_shell_activate_item(gtk_menu_shell(), item.gtk_widget(), force_deactivate);
}

void
Gtk::MenuShell::deactivate()
{
	gtk_menu_shell_deactivate(gtk_menu_shell());
}

void 
Gtk::MenuShell::select_first(bool search_sensitive)
{
	gtk_menu_shell_select_first(gtk_menu_shell(), search_sensitive);
}

/*  Gtk::MenuShellClass
 */

void
Gtk::MenuShellClass::init(GtkMenuShellClass *g_class)
{
	ContainerClass::init((GtkContainerClass*)g_class);
	g_class->deactivate = &deactivate_proxy;
	g_class->selection_done = &selection_done_proxy;
	g_class->select_item = &select_item_proxy;
	g_class->insert = &insert_proxy;
	g_class->get_popup_delay = &get_popup_delay_proxy;
}

GType
Gtk::MenuShellClass::get_type()
{
	static GType type = 0;
	if (!type)
	{
		type = G::TypeInstance::register_type(GTK_TYPE_MENU_SHELL, (GClassInitFunc)init);
	}
	return type;
}

void*
Gtk::MenuShellClass::create()
{
	return g_object_new(get_type(), 0);
}

void
Gtk::MenuShellClass::select_item_proxy(GtkMenuShell *menu_shell, GtkWidget *menu_item)
{
	MenuShell *tmp_menu_shell = G::Object::pointer<MenuShell>(menu_shell);
	if (tmp_menu_shell)
		tmp_menu_shell->do_select_item(menu_item);
	else
	{
		GtkMenuShellClass *tmp_class = GTK_MENU_SHELL_GET_CLASS(menu_shell);
		GtkMenuShellClass *g_class = G::TypeInstance::class_peek_parent<GtkMenuShellClass>(tmp_class);
		if (g_class->select_item)
			g_class->select_item(menu_shell, menu_item);
	}
}


void
Gtk::MenuShellClass::insert_proxy(GtkMenuShell *menu_shell, GtkWidget *child, gint position)
{
	MenuShell *tmp_menu_shell = G::Object::pointer<MenuShell>(menu_shell);
	if (tmp_menu_shell)
		tmp_menu_shell->do_insert(child, position);
	else
	{
		GtkMenuShellClass *tmp_class = GTK_MENU_SHELL_GET_CLASS(menu_shell);
		GtkMenuShellClass *g_class = G::TypeInstance::class_peek_parent<GtkMenuShellClass>(tmp_class);
		if (g_class->insert)
			g_class->insert(menu_shell, child, position);
	}
}

int
Gtk::MenuShellClass::get_popup_delay_proxy(GtkMenuShell *menu_shell)
{
	int result = 0;
	MenuShell *tmp_menu_shell = G::Object::pointer<MenuShell>(menu_shell);
	if (tmp_menu_shell)
		result = tmp_menu_shell->do_get_popup_delay();
	else
	{
		GtkMenuShellClass *tmp_class = GTK_MENU_SHELL_GET_CLASS(menu_shell);
		GtkMenuShellClass *g_class = G::TypeInstance::class_peek_parent<GtkMenuShellClass>(tmp_class);
		if (g_class->get_popup_delay)
			result = g_class->get_popup_delay(menu_shell);
	}
	return result;
}

void
Gtk::MenuShellClass::deactivate_proxy(GtkMenuShell *menu_shell)
{
	MenuShell *tmp_menu_shell = G::Object::pointer<MenuShell>(menu_shell);
	if (tmp_menu_shell)
		tmp_menu_shell->on_deactivate();
	else
	{
		GtkMenuShellClass *tmp_class = GTK_MENU_SHELL_GET_CLASS(menu_shell);
		GtkMenuShellClass *g_class = G::TypeInstance::class_peek_parent<GtkMenuShellClass>(tmp_class);
		if (g_class->deactivate)
			g_class->deactivate(menu_shell);
	}
}

void
Gtk::MenuShellClass::selection_done_proxy(GtkMenuShell *menu_shell)
{
	MenuShell *tmp_menu_shell = G::Object::pointer<MenuShell>(menu_shell);
	if (tmp_menu_shell)
		tmp_menu_shell->on_selection_done();
	else
	{
		GtkMenuShellClass *tmp_class = GTK_MENU_SHELL_GET_CLASS(menu_shell);
		GtkMenuShellClass *g_class = G::TypeInstance::class_peek_parent<GtkMenuShellClass>(tmp_class);
		if (g_class->selection_done)
			g_class->selection_done(menu_shell);
	}
}

/*  Overridable methods
 */

void
Gtk::MenuShell::do_select_item(GtkWidget *menu_item)
{
	GtkMenuShellClass *g_class = class_peek_parent<GtkMenuShellClass>(gtk_menu_shell_class());
	if (g_class->select_item)
		g_class->select_item(gtk_menu_shell(), menu_item);
}

void
Gtk::MenuShell::do_insert(GtkWidget *child, int position)
{
	GtkMenuShellClass *g_class = class_peek_parent<GtkMenuShellClass>(gtk_menu_shell_class());
	if (g_class->insert)
		g_class->insert(gtk_menu_shell(), child, position);
}

int
Gtk::MenuShell::do_get_popup_delay()
{
	int result = 0;
	GtkMenuShellClass *g_class = class_peek_parent<GtkMenuShellClass>(gtk_menu_shell_class());
	if (g_class->get_popup_delay)
		result = g_class->get_popup_delay(gtk_menu_shell());
	return result;
}

/*  Signal handlers
 */

void
Gtk::MenuShell::on_deactivate()
{
	GtkMenuShellClass *g_class = class_peek_parent<GtkMenuShellClass>(gtk_menu_shell_class());
	if (g_class->deactivate)
		g_class->deactivate(gtk_menu_shell());
}

void
Gtk::MenuShell::on_selection_done()
{
	GtkMenuShellClass *g_class = class_peek_parent<GtkMenuShellClass>(gtk_menu_shell_class());
	if (g_class->selection_done)
		g_class->selection_done(gtk_menu_shell());
}

/*  Signals
 */

const Gtk::MenuShell::DeactivateSignalType Gtk::MenuShell::deactivate_signal("deactivate");

const Gtk::MenuShell::SelectionDoneSignalType Gtk::MenuShell::selection_done_signal("selection_done");

