#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <stdlib.h>
#include <pthread.h>
#include <unistd.h> /* getpid */
#ifdef HAVE_ERRNO_H
#include <errno.h>
#endif

#include <jack/jack.h>

/* Local header Files */

#include "libjackasyn.h"

#define JACKASYN_THREAD

int debug = 0;
int jack_running = 0;

void libjackasyn_version(int* major,int* minor) {
	fprintf(stderr,"Version = %d.%d\n",MAJOR,MINOR);
        if (major) *major = MAJOR;
	if (minor) *minor = MINOR;
	return;
}


char* process_name(int num) {
  int fd;
  int pid;
  char path[256];
  char namefield[256];
  char* n;
  int len;
  char* pname;

  pid =  getpid();
  
  sprintf(path,"/proc/%d/status",pid);

  fd = open(path,O_RDONLY);
  
  
  if (read(fd,namefield,256)<0) {
    fprintf(stderr,"libjackasyn: cannot get application name\n");
    return NULL;
  }
  n = namefield + strlen("Name: ");
  len=0;
  while (*(n+len) != '\n' && len < 255) len++;
  if (len >= 255) {
    fprintf(stderr,"libjackasyn: cannot get process name\n");
    return NULL;
  }

  pname = malloc(len+2);
  n[len]='\0';
  sprintf(pname,"%s_%d",n,num);
  
  return pname;
}


/*
 *   Simple FIFO.
 *
 */


static fifo_t* fifo_allocate(fifo_t* f, int size,int msize) {
  if (f->mem) fprintf(stderr,"fifo: resizing is not thread safe");
  f->mem = (void*)realloc(f->mem,(size+1)*sizeof(msize));
  f->size = size+1;
  f->maxsize = size+1;
  f->readp = 0;
  f->writep = 0;
  f->framecount = 0;
  return f;
}


int fifo_setsize(fifo_t* f,int size)
{
  if (size > f->maxsize || size < 2) return f->size = f->maxsize;
  f->size = size+1;
  return f->size;
}

static void fifo_free(fifo_t* f) 
{
  free(f->mem);
}
  
int fifo_filled(fifo_t* f) {
  int readp = atomic_read(f->readp);
  int writep = atomic_read(f->writep);
  int size = f->size;

  if (writep >= readp)
    return (writep - readp);
  else
    return (size + writep - readp);
}


int fifo_empty(fifo_t* f) {  
  return f->size - fifo_filled(f)-1;
}



#define FIFO_DEBUG(f)\
{\
  fprintf(stderr," write %d, read %d, space %d, size %d\n",f->writep,f->readp,\
	  fifo_empty(f),f->size);\
}



/*
 *  the process copies, does deinterleaving and samplerate conversion 
 *  if necessary
 */


static int
jack_process (jack_nframes_t nframes, long handle)
{
  int i;
  int j;
  virdev* h = (virdev*) handle;
  int readp = h->ofifo.readp;
  int writep =  h->ififo.writep;
  static float cbuffer[16536]; 
  int ninframes = (double)nframes*h->rate/(double)h->jackrate ;


  DEBUGx(2,fprintf(stderr,"process called\n"));

  /*
   *   OUTPUT SECTION
   */


  /* check for underruns */
  
  if (h->ochans) {
    if (fifo_filled(&h->ofifo) < ninframes*h->ochans) {
      h->underruns++;
      //    fprintf(stderr,__FUNCTION__" underrun %d\n",fifo_filled(&h->ofifo)); 
      for (i=0;i<h->ochans;i++) {
        jack_default_audio_sample_t *out = jack_port_get_buffer (h->out[i], nframes);      
        j= nframes;
        while (j--) *out++ = 0.;
      }
    }
    else {
      for (i=0;i<h->ochans;i++) {
        jack_default_audio_sample_t *jackout = jack_port_get_buffer (h->out[i], nframes);
        float* source = h->ofifo.mem;
        float* out;
        DEBUG(fprintf(stderr,"process: channel %d frames i:%d %d\n",i,ninframes,nframes));
        
        out = jackout;
        
        if (h->convert && h->rate != h->jackrate) {
          out = cbuffer;
          DEBUG(fprintf(stderr,"converting\n"));
        }      
        
        readp = atomic_read(h->ofifo.readp);
        readp+=i;
        readp %= h->ofifo.size;
	
        /* deinterleave */
        
        for (j=0;j<ninframes;j++) {
          out[j] = source[readp];
          readp += h->ochans;
          readp %= h->ofifo.size;
        }
        
        /* samplerate conversion */
        
        if (h->convert && h->rate != h->jackrate) {
          static SRC_DATA srcdata;
          srcdata.data_in = out;
          srcdata.data_out = jackout;
          srcdata.src_ratio = (double)nframes/(double)ninframes;
          srcdata.input_frames = ninframes;
          srcdata.output_frames = nframes;
          srcdata.end_of_input = 0;
          
          src_process(h->converter[i],&srcdata);
          if (srcdata.input_frames_used != ninframes || 
              srcdata.output_frames_gen != nframes) 
            DEBUG(fprintf(stderr,"rate %f: used %ld generated %ld\n",srcdata.src_ratio,srcdata.input_frames_used,srcdata.output_frames_gen));
        }      
        
      }
      
      atomic_write(h->ofifo.readp,readp-h->ochans + 1);
      h->ofifo.framecount+=ninframes;
      DEBUG(fprintf(stderr,"process: output done\n"));
    }
  }

  /*
   *   INPUT SECTION
   */


  /* check for overruns */
  
  if (h->ichans) {
    if (fifo_empty(&h->ififo) < ninframes*h->ichans) {
      float* dest = h->ififo.mem;
      /* fprintf(stderr,__FUNCTION__" underrun %d\n",fifo_filled(&h->ofifo)); */
      for (i=0;i<h->ichans;i++) {
        j= ninframes;
        while (j--) *dest++ = 0.;
      }
    }
    else {
      for (i=0;i<h->ichans;i++) {
        jack_default_audio_sample_t *in = jack_port_get_buffer (h->in[i], nframes);
        float* dest;
        
        /* sample rate conversion */
        
        if (h->convert && h->rate != h->jackrate) {
          static SRC_DATA srcdata;
          dest = cbuffer;
          srcdata.data_in = in;
          srcdata.data_out = dest;
          srcdata.src_ratio = (double)ninframes/(double)nframes;
          srcdata.input_frames = nframes;
          srcdata.output_frames = ninframes;
          srcdata.end_of_input = 0;
          
          src_process(h->iconverter[i],&srcdata);
          if (srcdata.input_frames_used != nframes || 
              srcdata.output_frames_gen != ninframes) 
            DEBUG(fprintf(stderr,"rate %f: used %ld generated %ld\n",srcdata.src_ratio,srcdata.input_frames_used,srcdata.output_frames_gen));
          in = dest;
        }      
        
        /* deinterleave */
        
        dest = h->ififo.mem;
        writep = atomic_read(h->ififo.writep);
        writep+=i;
        writep %= h->ififo.size;
        
        for (j=0;j<ninframes;j++) {
          dest[writep] = in[j];
          writep += h->ichans;
          writep %= h->ififo.size;
        }
      }
      atomic_write(h->ififo.writep,writep-h->ichans+1);
      h->ififo.framecount+=ninframes;
      DEBUG(fprintf(stderr,"process: input done\n"));
    }
  }

  /* wake them up */

#ifdef JACKASYN_THREAD
  pthread_cond_broadcast(&h->sem);
#endif

  return 0;      
}
static int
jack_bufsize (jack_nframes_t nframes, void *arg)

{
        printf ("the maximum buffer size is now %lu\n",(unsigned long)nframes);
        return 0;
}

static int
jack_srate (jack_nframes_t nframes, void *arg)

{
        return 0;
}


int jack_running;

static void
jack_shutdown (void *arg)
{
        virdev* h = (virdev*) arg;
        DEBUG(fprintf (stderr,"shutdown callback\n"));
        jack_running = 0;
#ifdef JACKASYN_THREAD
  pthread_cond_broadcast(&h->sem);
#endif
        DEBUG(fprintf (stderr,"shutdown callback done\n"));
}

static void 
jack_error(const char* err) {
  DEBUG(fprintf(stderr,"libjackasyn: %s\n",err);)
}


/*
 *  Common sequence of calls:
 *  - virdev_connect .. connect to jack
 *  - virdev_start .. starts the  jack connection
 *  - virdev_output .. output interleaved 16bit native endian data
 *  - virdev_input .. input interleaved 16 bit native endian data
 *  - virdev_shutdown .. shutdown connection
 * other:
 *  - virdev_new .. create a new structure (used in virdev_connect)
 *  - virdev_free ... free the  struct
 */ 

static virdev* virdev_new(jack_client_t* client,int nin, int nout) 
{
  int nfrags = jack_get_buffer_size(client);

  virdev* v = (virdev*)malloc(sizeof(virdev));

  DEBUG(fprintf(stderr,"%s\n",__FUNCTION__);)

  v->client = client;
  v->flags = 0;
  v->state = JACK_INACTIVE;

  v->ichans = nin;
  v->ochans = nout;
  if (client) v->rate = v->jackrate = jack_get_sample_rate(client);
  v->fragnum = 4;
  v->adjustedfragsize = v->fragsize = nfrags;
  v->underruns = 0;

  v->ififo.mem = NULL;
  v->ofifo.mem = NULL;
  if (v->ochans) fifo_allocate(&v->ofifo,v->fragnum*v->fragsize*v->ochans,sizeof(float));
  if (v->ichans) fifo_allocate(&v->ififo,v->fragnum*v->fragsize*v->ichans,sizeof(float));

  v->converter[0] = src_new(SRC_SINC_FASTEST,1,NULL);
  v->converter[1] = src_new(SRC_SINC_FASTEST,1,NULL);
  v->iconverter[0] = src_new(SRC_SINC_FASTEST,1,NULL);
  v->iconverter[1] = src_new(SRC_SINC_FASTEST,1,NULL);
  if (getenv("JACKASYN_NOCONVERT")) v->convert = 0;
  else v->convert = 1;
  
  DEBUG(fprintf(stderr,"%s done\n",__FUNCTION__);)

  return v;
}


static void virdev_free(virdev* v) {
  fifo_free(&v->ififo);
  fifo_free(&v->ofifo);
  src_delete(v->converter[0]);
  src_delete(v->converter[1]);
  src_delete(v->iconverter[0]);
  src_delete(v->iconverter[1]);
  free(v);
}


void  virdev_reset(virdev* virdev)
{
  virdev->ofifo.writep = 0;
  virdev->ofifo.readp = 0;
  virdev->ififo.writep = 0;
  virdev->ififo.readp = 0;
  virdev->underruns = 0;
}


virdev* virdev_connect(char* name,int ichans,int ochans)
{
  virdev* handle = NULL;
  jack_client_t* client=NULL;
  char* pname;
  int i;

  DEBUG(fprintf(stderr,"%s\n",__FUNCTION__);)


  jack_set_error_function (jack_error);

  for (i=0;i < 5;i++) {
     if ((pname = process_name(i))) {
        client = jack_client_new(pname);
        free(pname);
     }
     if (client) break;
  }
 
  if (!client)
      client = jack_client_new(name);

  if (!client) {
    fprintf (stderr, "jack_open: jack server not running?\n");
    jack_running = 0;
    return handle;
  }
  else {
    handle = virdev_new(client,ichans,ochans);
#ifdef JACKASYN_THREAD
    pthread_mutex_init(&handle->mutex,NULL);
    pthread_cond_init(&handle->sem,NULL);
#endif
  }

  if (!handle) {
    fprintf(stderr,"jack_open: out of memory\n");
    jack_running = 0;
    return handle;
  }

  jack_running = 1;
  jack_set_process_callback(client, (JackProcessCallback)jack_process,
			    (void*)handle);
  jack_set_buffer_size_callback (client, jack_bufsize, 0);
  jack_set_sample_rate_callback (client, jack_srate, 0);
  jack_on_shutdown (client, jack_shutdown, handle);
  
  return handle;
}


/* Default connection to  ports */

#if 0

int  jackasyn_connect_defaultports(virdev* h)		
{
  char** ports;
  
  ports = jack_get_ports (h->client, NULL, NULL,JackPortIsPhysical|JackPortIsOutput);
  
  for (i=0;i<h->ichans && ports[i];i++) {
    fprintf(stderr,"connecting port %s --> %s\n",ports[i],jack_port_name(h->in[i]));
    jack_connect(h->client, ports[i],jack_port_name(h->in[i])); 
    free(ports[i]);
  }
  
  free(ports);
  
  ports = jack_get_ports (h->client, NULL, NULL,JackPortIsPhysical|JackPortIsOutput);
  
  for (i=0;i<h->ochans && ports[i];i++) {
    fprintf(stderr,"connecting port %s --> %s\n",ports[i],jack_port_name(h->in[i]));
    jack_connect(h->client, jack_port_name(h->out[i]),ports[i]); 
    free(ports[i]);
  }
  free(ports);
}
#else

static char defaultaudioinput[] = "alsa_pcm:capture_%d";
static char defaultaudiooutput[] = "alsa_pcm:playback_%d";

int  jackasyn_connect_defaultports(virdev* h)		
{
    /* Default connection to ALSA ports */
    char portname[255];
    char* audioinput;
    char* audiooutput;
    int i;

    audioinput = getenv("JACKASYN_DEFAULT_INPUT");
    audiooutput = getenv("JACKASYN_DEFAULT_OUTPUT");

    if (!audioinput) audioinput = defaultaudioinput;
    if (!audiooutput) audiooutput = defaultaudiooutput;

    for (i=0;i<h->ichans;i++) {
      sprintf(portname,audioinput,i+1);
      jack_connect(h->client, portname,jack_port_name(h->in[i])); 
    }

    for (i=0;i<h->ochans;i++) {
      sprintf(portname,audiooutput,i+1);
      jack_connect(h->client, jack_port_name(h->out[i]),portname); 
    }
    h->state = JACK_ACTIVE;
    return 1;
}

#endif


int virdev_start(virdev* h) 
{
  
  if (h->state != JACK_ACTIVE) {
    int i;
    char portname[255];

    for (i=0;i<h->ichans;i++) {
      sprintf(portname,"in_%d",i+1);
      h->in[i] = jack_port_register (h->client, portname, 
				     JACK_DEFAULT_AUDIO_TYPE,
				     JackPortIsInput,0);
      if (!h->in[i]) fprintf(stderr,"port_register failed for %s\n",portname);
    }

    for (i=0;i<h->ochans;i++) {
      sprintf(portname,"out_%d",i+1);
      h->out[i] = jack_port_register (h->client, portname, 
				     JACK_DEFAULT_AUDIO_TYPE,
				     JackPortIsOutput,0);
      if (!h->out[i]) fprintf(stderr,"port_register failed for %s\n",portname);
    }
    if (jack_activate(h->client)) {
      fprintf(stderr,"Cannot activate client\n");
      return 0;
    }
    jackasyn_connect_defaultports(h);
  }
  return 1;
}



/* returns true if wait occured */

int virdev_wait(virdev* h) 
{
  int space;
  int avail;
  
  if (h->ochans) {
    space = fifo_empty(&h->ofifo);
    avail = fifo_filled(&h->ofifo);
  }
  else {
    space = fifo_empty(&h->ififo);
    avail = fifo_filled(&h->ififo);
  }

  DEBUGx(1,fprintf(stderr,"space %d %d\n",space,avail));

#ifdef JACKASYN_THREAD
  if (!jack_running) return 0;
  if ((h->ochans && space == 0) ||(h->ichans && avail == 0)) {
    pthread_mutex_lock(&h->mutex);
    pthread_cond_wait(&h->sem,&h->mutex);
    pthread_mutex_unlock(&h->mutex);
    return 1;    
  }
  else
    return 0;  
#else
  if (space < 64 || avail < 64) {
    while (space < 64 || avail < 64) {
      space = fifo_empty(&h->ofifo);
      avail = fifo_filled(&h->ififo);
      usleep(100);
    }
    return 1;    
  }
  else return 0;  
#endif

}


int virdev_output16i(virdev* h,const char* buf,int len)
{
  int i;
  int space = 0;
  short* sbuf = (short*)buf;
  int len1 = len;

  /* 16 bit representation */

  len >>= 1;
  while (jack_running && len) {    
    int writep = atomic_read(h->ofifo.writep);
    float* dbuf = h->ofifo.mem;
    space = fifo_empty(&h->ofifo);

    if (virdev_wait(h)) continue;

    space = space > len ? len : space;
    for (i=0;i<space;i++) {
      writep %= h->ofifo.size;
      dbuf[writep] = sbuf[i]/32768.;
      writep++;
    }

    atomic_write(h->ofifo.writep,writep);

    sbuf += space;
    len -= space;
  }    

  if (!jack_running) return -1;
  return len1;

}

int virdev_input16i(virdev* h,char* buf,int len)
{
  int i;
  int space = 0;
  short* dbuf = (short*)buf;
  int len1 = len;

  /* 16 bit representation */

  len >>= 1;
  while (jack_running && len) {    
    int readp = h->ififo.readp;
    float* sbuf = h->ififo.mem;

    space = fifo_filled(&h->ififo);

    if (virdev_wait(h)) continue;

    space = space > len ? len : space;
    for (i=0;i<space;i++) {
      readp %= h->ififo.size;
      dbuf[i] = sbuf[readp]*32768;
      readp++;
    }

    atomic_write(h->ififo.readp,readp);

    sbuf += space;
    len -= space;
  }    

  if (!jack_running) return -1;
  return len1;

}


int virdev_shutdown(virdev* h)
{
  if (h->client) {
      jack_deactivate(h->client);
      jack_client_close(h->client);
  }

  virdev_free(h);
  return 1;
}



