/***************************************************************************
 $RCSfile: accountsegs.cpp,v $
                             -------------------
    cvs         : $Id: accountsegs.cpp,v 1.37 2003/06/25 17:48:04 aquamaniac Exp $
    begin       : Mon Nov 26 2001
    copyright   : (C) 2001 by Martin Preuss
    email       : openhbci@aquamaniac.de

 ***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/


/*
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#ifdef __declspec
# if BUILDING_DLL
#  define DLLIMPORT __declspec (dllexport)
# else /* Not BUILDING_DLL */
#  define DLLIMPORT __declspec (dllimport)
# endif /* Not BUILDING_DLL */
#else
# define DLLIMPORT
#endif



#include <stdio.h> // for dump
#include <algorithm>
#include <list>

#include "accountsegs.h"
#include "bankimpl.h"
#include "accountimpl.h"
#include "hbcistring.h"
#include "error.h"


namespace HBCI {

/*AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
 * SEGGetBalance
 *AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
 */


SEGGetBalance::SEGGetBalance(Pointer<Customer> cust):Seg(cust){
}


SEGGetBalance::~SEGGetBalance(){
}


string SEGGetBalance::toString(int segnr){
    const bpdJob *jp;
    string result;

    BankImpl &bank = 
        dynamic_cast<BankImpl&> (_bank.ref());

    /* internally store the current segment number
     */
    _segnumber=segnr;

    /* As you know the HBCI class is the father of most other classes,
     * and it is the center of information (see APIDOC for HBCI to learn
     * what you can find in class HBCI)
     */

    /* determine the segment-number we need (sets versionMin/versionMax)
     * this is a macro which retrieves the minimum and maximum segment
     * version number for each HBCI version. As you maybe know each
     * kind of segment has a version number specified in the HBCI specs.
     * Whenever the layout of a segment changes the version number is
     * increased. This makes it possible to have multiple possible version
     * for one segment in each HBCI version.
     * In this example (Get Balance) HBCI version 2.01 used segment version
     * 3 (only 3, so we have 3 as minimum and as maximum). Version 2.10 used
     * 4 and so on.
     * first parameter is the current HBCI version, next are:
     * <ul>
     * <li>minumum version for HBCI 2.01</li>
     * <li>maximum version for HBCI 2.01</li>
     * <li>minimum version for HBCI 2.10</li>
     * <li>maximum version for HBCI 2.10</li>
     * <li>minimum version for HBCI 2.20</li>
     * <li>maximum version for HBCI 2.20</li>
     * </ul>
     */
    int versionMin = -1; 
    int versionMax = -1; 
    Seg::segment_number(versionMin, versionMax, 
			    bank.hbciVersion(),3, 3, 4, 4, 5, 5);

    /* get the job paramters for that job
     * the server has its own understanding about which segment versions
     * it allows, and the institute parameters we just retrieved from the
     * HBCI class have these information stored in them. The next method
     * tries to get additional information about this segment here, but
     * it only asks for information concerning the given versions ( between
     * versionmin and versionmax, depending on the HBCI version, see above)
     */
    jp=bank.findJob("HISALS",versionMin,versionMax);
    if (jp==0)
        /* now additional information available, that in most cases means
         * that this segment is not supported, so abort */
        throw Error("SEGGetBalance::toString()",
                        "job not supported by your institute",0);
    /* ok, now really start creating the segment
     * let's start with the segment head
     * the rest here only builds the segment according to the HBCI specs
     * and returns the segment just built in a string.
     */
    result="HKSAL:" + String::num2string(segnr) + ":";
    // segment version
    result+=String::num2string(jp->segmentVersion()) + "+";
    // ktv
    result+=_acc.ref().accountId()+":";
    if (bank.hbciVersion() > HBCI_VERSION_210)
        result+=_acc.ref().accountSuffix()+":";
    result+=String::num2string(_acc.ref().bank().ref().countryCode())+":";
	// we need to use the institute id that is stored along with the account
	// (it may differ from the id that comes with the bank...!)
	//    result+=_acc.ref().bank().ref().bankCode()+"+";
    result+=_acc.cast<AccountImpl>().ref().instituteCode()+"+";
    // all available accounts ? No.
    result += "N";
    // segment end
    result += "'";
    return result;
}



/*AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
 * SEGBalance
 *AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
 */


SEGBalance::SEGBalance(Pointer<Customer> cust):Seg(cust){
}


SEGBalance::~SEGBalance(){
}


Balance SEGBalance::_parseBalance(string deg) {
    unsigned int spos;
    string tmp;
    Balance b;
    Value val;

    spos=0;
    // read sign
    b.setDebit(String::nextDEG(deg,spos).at(0)=='D');
    spos+=String::nextDEG(deg,spos).length()+1;
    // read value
    val=Value(String::nextDEG(deg,spos));
    spos+=String::nextDEG(deg,spos).length()+1;
    // read currency
    b.setValue(Value(val.getValue(),String::nextDEG(deg,spos)));
    spos+=String::nextDEG(deg,spos).length()+1;
    // read date
    b.setDate(Date(String::nextDEG(deg,spos)));
    spos+=String::nextDEG(deg,spos).length()+1;
    // read time, if any
    tmp=String::nextDEG(deg,spos);
    if (!tmp.empty()) {
        b.setTime(Time(String::nextDEG(deg,spos)));
    }
    return b;
}


bool SEGBalance::parse(const string& segment, unsigned int pos){
    string tmp;
    string dep;

    /* The job of this method is very easy, it just takes
     * a segment received from the server as input and tries to parse all
     * informative out of it. In this case the interesting thing is
     * the account balance, as the name of this class implies ;-)
     */

    // skip segment head
    pos += String::nextDE(segment, pos).length() + 1;
    /* skip account (we only get data for the account we specified with
     * SEGGetBalance, and this segment here is a direct response to it
     */
    pos += String::nextDE(segment, pos).length() + 1;
    // skip account name
    pos += String::nextDE(segment, pos).length() + 1;
    // get currency
    _balance._currency=String::nextDE(segment, pos);
    pos += String::nextDE(segment, pos).length() + 1;

    // get booked saldo
    _balance._bookedBalance=_parseBalance(String::nextDE(segment, pos));
    pos += String::nextDE(segment, pos).length() + 1;

    // get noted balance (if any)
    tmp=String::nextDE(segment, pos);
    if (!tmp.empty()) {
        _balance._notedBalance=_parseBalance(tmp);
        pos += String::nextDE(segment, pos).length() + 1;
    }

    // get credit line (if any)
    tmp=String::nextDE(segment, pos);
    if (!tmp.empty()) {
        _balance._bankLine=Value(tmp);
        pos += String::nextDE(segment, pos).length() + 1;
    }
    // get disposeable money (if any)
    tmp=String::nextDE(segment, pos);
    if (!tmp.empty()) {
        _balance._disposable=Value(tmp);
        pos += String::nextDE(segment, pos).length() + 1;
    }

    // get disposed money (if any)
    tmp=String::nextDE(segment, pos);
    if (!tmp.empty()) {
        _balance._disposed=Value(tmp);
        pos += String::nextDE(segment, pos).length() + 1;
    }
    // get date of booking
    tmp=String::nextDE(segment, pos);
    if (!tmp.empty()) {
        _balance._date=Date(tmp);
        pos += String::nextDE(segment, pos).length() + 1;
    }
    // get time of booking
    tmp=String::nextDE(segment, pos);
    if (!tmp.empty()) {
        _balance._time=Time(tmp);
        pos += String::nextDE(segment, pos).length() + 1;
    }
    return true;
}


/*AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
 * SEGSingleTransferBaseBase
 *AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
 */


SEGSingleTransferBase::SEGSingleTransferBase(Pointer<Customer> cust,
                                             string segName,
                                             string segParamName)
:Seg(cust)
,_segName(segName)
,_segParamName(segParamName)
,_versionMin(-1)
,_versionMax(-1)
{
}


SEGSingleTransferBase::~SEGSingleTransferBase(){
}


string SEGSingleTransferBase::toString(int segnr) {
    const bpdJob *jp;
    string result;
    string purpose;
    string param;
    string tmp;
    unsigned int pos;
    unsigned int maxpurp;
    bool textkeyIsValid;
    list<string>::const_iterator it;

    BankImpl &bank = 
        dynamic_cast<BankImpl&> (_bank.ref());

    _segnumber=segnr;

    // get the job paramters for that job
    jp=bank.findJob(_segParamName,_versionMin,_versionMax);
    if (jp==0)
        throw Error("SEGSingleTransferBase::toString()",
		    ERROR_LEVEL_NORMAL,
		    HBCI_ERROR_CODE_JOB_NOT_SUPPORTED,
		    ERROR_ADVISE_DONTKNOW,
		    "job is not supported by your institute");

    // basic parameter checks
    param=jp->parameter();
    pos=0;
    // get number of allowed purpose lines
    tmp=String::nextDEG(param,pos);
    if (tmp.empty())
        throw Error("SEGSingleTransferBase::toString()",
                        ERROR_LEVEL_NORMAL,
                        0,
                        ERROR_ADVISE_DONTKNOW,
                        "could not get maximum number of allowed purpose lines");
    maxpurp=atoi(tmp.c_str());
    if (!maxpurp)
        maxpurp=2;
    if (_xaction.description().size()>maxpurp)
        throw Error("SEGSingleTransferBase::toString()",
                        ERROR_LEVEL_NORMAL,
                        0,
                        ERROR_ADVISE_DONTKNOW,
                        "too many purpose lines.");
    // check if the textkey is supported
    pos+=tmp.length()+1;
    textkeyIsValid=false;
    while (pos<param.length()){
        tmp=String::nextDEG(param,pos);
        pos+=tmp.length()+1;
        if (tmp.empty())
            break;
        if (atoi(tmp.c_str())==_xaction.transactionCode()) {
            textkeyIsValid=true;
            break;
        }
    }
    if (!textkeyIsValid)
        throw Error("SEGSingleTransferBase::toString()",
                        ERROR_LEVEL_NORMAL,
                        HBCI_ERROR_CODE_JOB_NOT_SUPPORTED,
                        ERROR_ADVISE_DONTKNOW,
                        "given textkey is not supported by your institute",
                        String::num2string(_xaction.transactionCode()));

    // ok, now create the segment
    // segment head
    result=_segName+":" + String::num2string(segnr) + ":";
    // segment version
    result+=String::num2string(jp->segmentVersion())+ "+";
    // our ktv
    result+=_xaction.ourAccountId()+":";
    if (bank.hbciVersion() > HBCI_VERSION_210)
        result+=_xaction.ourSuffix()+":";
    result+=String::num2string(_xaction.ourCountryCode())+":"; // countryCode
    result+=_xaction.ourBankCode()+"+";
    // their ktv
    result+=_xaction.otherAccountId()+":";
    if (bank.hbciVersion() > HBCI_VERSION_210)
        result+=_xaction.otherSuffix()+":";
    result+=String::num2string(_xaction.otherCountryCode())+":";
    result+=_xaction.otherBankCode()+"+";
    // their names 1 and 2
    if (_xaction.otherName().empty() ||
        _xaction.otherName().size()>2)
        throw Error("SEGSingleTransferBase::toString()",
                        ERROR_LEVEL_NORMAL,
                        0,
                        ERROR_ADVISE_DONTKNOW,
                        "invalid number of otherNames",
                        String::num2string(_xaction.otherName().size()));
    // their name 1
    result+=String::transformToDTAUS0(_xaction.otherName().front());
    result+="+";
    // their name 2 (if given)
    if (!_xaction.otherName().size()>1)
        result+=String::transformToDTAUS0(_xaction.otherName().back());
    result+="+";
    // value
    result += _xaction.value().toString() + "+";
    // text key
    result += String::num2string(_xaction.transactionCode(), true, 2)+"+";
    // text key add
    result+="000+";
    // create the purpose-string (DTAUS0-format) and cut off everything after the 27th byte
    purpose="";
    for (it=_xaction.description().begin();
         it != _xaction.description().end(); it++) {
        purpose+=String::transformToDTAUS0((*it)).substr(0,27)+":";
    }
    // delete the last ":"
    if (purpose.length() > 0)
        purpose = purpose.substr(0, purpose.length() - 1);
    result+=purpose;

    // finished
    result+="'";
    return result;
}


void SEGSingleTransferBase::setData(Transaction xa){
    _xaction=xa;
}








/*AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
 * SEGSingleTransfer
 *AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
 */


SEGSingleTransfer::SEGSingleTransfer(Pointer<Customer> cust)
:SEGSingleTransferBase(cust,"HKUEB","HIUEBS")
{
    // determine the segment-number we need (sets versionMin/versionMax)
    int versionMin = -1; 
    int versionMax = -1; 
    Seg::segment_number(versionMin, versionMax, 
                            _bank.ref().hbciVersion(),2,2,3,3,4,4);
    _versionMin=versionMin;
    _versionMax=versionMax;
}


SEGSingleTransfer::~SEGSingleTransfer(){
}


/*AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
 * SEGGetTurnover
 *AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
 */


SEGGetTurnover::SEGGetTurnover(Pointer<Customer> cust):Seg(cust){
}


SEGGetTurnover::~SEGGetTurnover(){
}


string SEGGetTurnover::toString(int segnr){
    const bpdJob *jp;
    string result;
    string purpose;
    string segid;

    BankImpl &bank =
      dynamic_cast<BankImpl&> (_bank.ref());

    _segnumber=segnr;

    // determine the segment-number we need (sets versionMin/versionMax)
    int versionMin = -1; 
    int versionMax = -1; 
    Seg::segment_number(versionMin, versionMax, 
			    bank.hbciVersion(),4,4,4,4,5,5);

    // get the job paramters for that job, first try "HKKAZ"
    segid="HKKAZ";
    jp=bank.findJob("HIKAZS",versionMin,versionMax);
    if (!jp) {
      // if not found try "HKKANS"
      segid="HKKAN";
      jp=bank.findJob("HIKANS",versionMin,versionMax);
    }
    if (jp==0)
      throw Error("SEGGetTurnover::getJobData()",
		  "job not supported",0);
    // ok, now create the segment
    // segment head
    result=segid+":";
    // segment number
    result+=String::num2string(segnr) + ":";
    // segment version
    result+=String::num2string(jp->segmentVersion()) + "+";
    // ktv
    result+=_acc.ref().accountId()+":";
    if (bank.hbciVersion() > HBCI_VERSION_210)
      result+=_acc.ref().accountSuffix()+":";
    result+=String::num2string(_acc.ref().bank().ref().countryCode())+":";
    // we need to use the institute id that is stored along with the account
    // (it may differ from the id that comes with the bank...!)
    result+=_acc.ref().instituteCode();

    // all available accounts? NO! (comes with hbci 2.2)
    if (bank.hbciVersion() > HBCI_VERSION_210)
      result += "+N";

    // date
    if (segid=="HKKAZ") {
      if (_fromdate.isValid() ||
	  _todate.isValid() ||
	  !_attachPoint.empty()) {
	if (jp->segmentVersion()<5)
	  // for versions < 2.2 we have to supply currency here
	  // that took me half an hour to find out !
	  result+="+";
	result+="+";
      }
      if (_fromdate.isValid()) {
	result+=_fromdate.toString();
      }

      if (_todate.isValid() || !_attachPoint.empty())
	result+="+";
      if (_todate.isValid()) {
	result+=_todate.toString();
      }
    }
    else {
      if (jp->segmentVersion()<5)
	// for versions < 2.2 we have to supply currency here
	// that took me half an hour to find out !
	if (!_attachPoint.empty())
	  result+="+";
    }

    // check the attachpoint
    if (!_attachPoint.empty()) {
      // don't set number of entries to be returned (thus the two "+")
      result+="++";
      result += _attachPoint;
    }

    // segment end
    result += "'";
    return result;
}


void SEGGetTurnover::setData(Pointer<Account> acc,
                             Date fromDate,
                             Date toDate,
			     string attachPoint/* Default: "" */){
    _acc=acc;
    _fromdate=fromDate;
    _todate=toDate;
	_attachPoint = attachPoint;
}



/*AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
 * SEGDebitNote
 *AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
 */


SEGDebitNote::SEGDebitNote(Pointer<Customer> cust)
:SEGSingleTransferBase(cust,"HKLAS","HILASS")
{
    // determine the segment-number we need (sets versionMin/versionMax)
    int versionMin = -1; 
    int versionMax = -1; 
    Seg::segment_number(versionMin, versionMax, 
                            _bank.ref().hbciVersion(),2,2,2,2,4,4);
    _versionMin=versionMin;
    _versionMax=versionMax;
}


SEGDebitNote::~SEGDebitNote(){
}


/*AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
 * SEGGetStandingOrders
 *AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
 */


SEGGetStandingOrders::SEGGetStandingOrders(Pointer<Customer> cust):
    Seg(cust) {
}


SEGGetStandingOrders::~SEGGetStandingOrders(){
}


string SEGGetStandingOrders::toString(int segnr){
    const bpdJob *jp;
    string result;
    string purpose;
    string segid;

    _segnumber=segnr;

    BankImpl &bank = 
        dynamic_cast<BankImpl&> (_bank.ref());

    // determine the segment-number we need (sets versionMin/versionMax)
    int versionMin = -1; 
    int versionMax = -1; 
    Seg::segment_number(versionMin, versionMax, 
                            _bank.ref().hbciVersion(),
			    1,1,2,2,3,3);
    // get the job paramters for that job
    segid="HKDAB";
    jp=bank.findJob("HIDABS",versionMin,versionMax);
    if (jp==0)
        throw Error("SEGGetStandingOrders::toString()",
                        "job not supported",0);

    // ok, now create the segment
    // segment head
    result=segid+":";
    // segment number
    result+=String::num2string(segnr) + ":";
    // segment version
    result+=String::num2string(jp->segmentVersion()) + "+";
    // ktv
    result+=_acc.ref().accountId()+":";
    if (bank.hbciVersion() > HBCI_VERSION_210)
        result+=_acc.ref().accountSuffix()+":";
    result+=String::num2string(_acc.ref().bank().ref().countryCode())+":";
    // we need to use the institute id that is stored along with the account
    // (it may differ from the id that comes with the bank...!)
    //    result+=_acc.ref().bank().ref().bankCode()+"+";
    result+=_acc.cast<AccountImpl>().ref().instituteCode();

    if ("" != _attachPoint)
	// we have to set the attach point
	result += "+++" + _attachPoint;
    else
	// skip the rest, we don't need it
	;

    // segment end
    result += "'";
    return result;
}

void SEGGetStandingOrders::setData(Pointer<Account> acc, 
								   string attachPoint) {
    _acc=acc;
	_attachPoint = attachPoint;
}

bool SEGGetStandingOrders::parse(const string& segment, unsigned int pos) {
  SEGStandingOrder order;
  return order.parse(segment, pos);
}


SEGStandingOrder::SEGStandingOrder(Pointer<Customer> cust):
    Seg(cust) {
}

SEGStandingOrder::SEGStandingOrder() : Seg() {}
SEGStandingOrder::~SEGStandingOrder(){}

string SEGStandingOrder::toString(int segnr, const string& segId) {
	int versionMin = -1;
	int versionMax = -1;
	string tmp;
	unsigned int i = 0;
	const bpdJob *jp;
        BankImpl &bank =
            dynamic_cast<BankImpl&> (_bank.ref());

	if (segId == "HKDAE") {
	  // new order
	  Seg::segment_number(versionMin, versionMax, 
							  bank.hbciVersion(), 2, 2, 2, 2, 4, 4);
	  jp=bank.findJob("HIDAES",versionMin,versionMax);
	} else if (segId == "HKDAN") {
	  // modify
	  Seg::segment_number(versionMin, versionMax, 
							  bank.hbciVersion(), 2, 2, 2, 2, 4, 4);
	  jp=bank.findJob("HIDANS",versionMin,versionMax);
	} else if (segId == "HKDAL") {
	  // delete
	  Seg::segment_number(versionMin, versionMax,
							  bank.hbciVersion(), 1, 1, 1, 1, 3, 3);
	  jp = bank.findJob("HIDALS", versionMin, versionMax);
	} else {
	    throw Error("SEGStandingOrder::toString()",
			ERROR_LEVEL_INTERNAL,
			0,
			ERROR_ADVISE_ABORT,
			"Unknown SegId for StandingOrder.");
	}


	// seghead
    string result = segId + ":" + String::num2string(segnr) + ":" +
	  String::num2string(jp->segmentVersion()) + "+";

	// our account data
	result += _myAccountNumber + ":";
	if (_bank.ref().hbciVersion() > HBCI_VERSION_210)
	  result += _myAccountSuffix + ":";
	result += String::num2string(_myCountryCode) + ":";
	result += _myInstituteCode + "+";

	// the recipient's account data
	result += _recAccountNumber + ":";
        if (_bank.ref().hbciVersion() > HBCI_VERSION_210)
	  result += _recAccountSuffix + ":";
	result += String::num2string(_recCountryCode) + ":";
	result += _recInstituteCode + "+";

	// recipient's name (to upper case and escape)
	result += String::transformToDTAUS0(_recName1) + "+";
	result += String::transformToDTAUS0(_recName2) + "+";

	// value and textkeys
	result += _value.toString() + "+" + _textKey1 + "+" + _textKey2 + "+";

	// purpose
	list<string>::const_iterator iter;
	i = 0;
	for (iter = _purpose.begin(); iter != _purpose.end(); iter++) {
	  i++;
	  tmp = String::transformToDTAUS0((*iter));
	  result += tmp + ((_purpose.size() == i)?"+":":");
	}

	// if this is a new order, we must not set
	// execdate and jobid
	if (segId == "HKDAE") 
	  result += "++";
	else {
	  // only set the exec-date if it is valid
	  if (_date.isValid())
		result += _date.toString();
	  result += "+" + _jobId + "+";
	}
	
	// date of first execution
	result += _firstExecution.toString() + ":";

	// period & cycle
	result = result + (_monthly?"M":"W") + ":" + 
	  String::num2string(_cycle) + ":";

	// day of execution within month/week
	result += String::num2string(_execDay);

	if (_lastExecution.isValid())
	  result += ":" + _lastExecution.toString();

	// segtail
	result += "'";

	return result; 
}

bool SEGStandingOrder::parse(const string& segment, unsigned int pos) {
    string tmp;
    string dep;
	string deg;

	unsigned int spos = 0;
	//bool hasSuffix = false;

    // skip segment head
    pos += String::nextDE(segment, pos).length() + 1;


	// my account
    // check if there is a suffix given
    deg = String::nextDE(segment, pos);
	pos += deg.length() + 1;
	parse_ktv(deg, _myAccountNumber, _myAccountSuffix, _myInstituteCode, 
		  _myCountryCode);

	// the reciever's account
    // check if there is a suffix given
    deg = String::nextDE(segment, pos);
	pos += deg.length() + 1;
	parse_ktv(deg, _recAccountNumber, _recAccountSuffix, _recInstituteCode,
		  _recCountryCode);

	// rec. name 1
	_recName1 = String::nextDE(segment, pos);
	pos += _recName1.length() + 1;

	// rec. name 2
	_recName2 = String::nextDE(segment, pos);
	pos += _recName2.length() + 1;

	// value
	tmp = String::nextDE(segment, pos);
	pos += tmp.length() + 1;
	_value = Value(tmp);

	// key 1
	_textKey1 = String::nextDE(segment, pos);
	pos += _textKey1.length() + 1;

	// key 2
	_textKey2 = String::nextDE(segment, pos);
	pos += _textKey2.length() + 1;

	// purpose
	tmp = String::nextDE(segment, pos);
	spos = 0;
	while (spos < tmp.length()) {
	  string purposeLine = String::nextDEG(tmp, spos);
	  purposeLine = String::unEscape(purposeLine);
	  _purpose.push_back(purposeLine);
	  spos += String::nextDEG(tmp, spos).length() + 1;
	}
	pos += tmp.length() + 1;

	// execution date
	tmp = String::nextDE(segment, pos);
	pos += tmp.length() + 1;
	_date = Date(tmp);

	// job id
	_jobId = String::nextDE(segment, pos);
	pos += _jobId.length() + 1;

	// details
	deg = String::nextDE(segment, pos);
	pos += deg.length();
	spos = 0;
	tmp = String::nextDEG(deg, spos);
	spos += tmp.length() + 1;
	_firstExecution = Date(tmp);
	
	tmp = String::nextDEG(deg, spos);
	spos += tmp.length() + 1;
	_monthly = ("M" == tmp);

	tmp = String::nextDEG(deg, spos);
	spos += tmp.length() + 1;
	_cycle = atoi(tmp.c_str());

	tmp = String::nextDEG(deg, spos);
	spos += tmp.length() + 1;
	_execDay = atoi(tmp.c_str());	

	tmp = String::nextDEG(deg, spos);
	spos += tmp.length() + 1;
	_lastExecution = Date(tmp);

	// suspend
	tmp = String::nextDE(segment, pos);

    return true;
}

Pointer<StandingOrder> SEGStandingOrder::getOrder() {
  Pointer<StandingOrder> order = new StandingOrder();

  list<string>::const_iterator iter;
  for (iter = _purpose.begin(); iter != _purpose.end(); iter++) 
	order.ref().addDescription((*iter));

  order.ref().setJobIdentification(_jobId);
  order.ref().setOtherAccountId(_recAccountNumber);
  order.ref().setOtherSuffix(_recAccountSuffix);
  order.ref().setOtherBankCode(_recInstituteCode);
  order.ref().setOtherCountryCode(_recCountryCode);
  order.ref().addOtherName(_recName1);
  if ("" != _recName2)
	order.ref().addOtherName(_recName2);
  order.ref().setValue(Value(_value));
  order.ref().setTransactionCode(atoi(_textKey1.c_str()));
  order.ref().setFirstExecutionDate(Date(_firstExecution));
  order.ref().setLastExecutionDate(Date(_lastExecution));
  order.ref().setExecutionDate(Date(_date));

  order.ref().setCycle(_cycle);
  order.ref().setPeriod((_monthly?StandingOrder::EXECUTE_MONTHLY:
					StandingOrder::EXECUTE_WEEKLY));
  order.ref().setExecDay(_execDay);

  return order;
}

} // namespace HBCI
