/***************************************************************************
 $RCSfile: file.cpp,v $
 -------------------
 cvs         : $Id: file.cpp,v 1.4 2003/04/08 09:38:00 aquamaniac Exp $
 begin       : Fri Dec 14 2001
 copyright   : (C) 2001 by Martin Preuss
 email       : martin@aquamaniac.de
 */


/***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/

/*
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#ifdef __declspec
# if BUILDING_DLL
#  define DLLIMPORT __declspec (dllexport)
# else /* Not BUILDING_DLL */
#  define DLLIMPORT __declspec (dllimport)
# endif /* Not BUILDING_DLL */
#else
# define DLLIMPORT
#endif


#include "error.h"
#include "file.h"

namespace HBCI {

File::File(const string &name): _handle(FILE_NO_FILEHANDLE){
  _name=name;
  _convPath(_name);
};


File::File(StdType typ): _handle(FILE_NO_FILEHANDLE){
  switch(typ) {
  case StdTypeStdIn:
    _handle=0;
    break;
  case StdTypeStdOut:
    _handle=1;
    break;
  case StdTypeStdErr:
    _handle=2;
    break;
  default:
    break;
  }
}


File::~File(){
}


void File::operator=(File &f){
  _name=f._name;
  _handle=FILE_NO_FILEHANDLE;
}


void File::_convPath(string &p){
    unsigned int pos=0;

    while(pos<p.length()) {
      if (p.at(pos)=='\\')
        p[pos]='/';
      pos++;
    }
}


Error File::accessFile(int mode){
    int am;

    am=0;
    if (mode & FILE_AM_RDONLY) am|=O_RDONLY;
    if (mode & FILE_AM_WRONLY) am|=O_WRONLY;
    if (mode & FILE_AM_RDWR) am|=O_RDWR;

    switch (mode & FILE_INTERNAL_AM_MASK) {
    case FILE_AM_CREATE_NEW:
        am|=(O_CREAT | O_EXCL);
        break;
    case FILE_AM_TRUNCATE_EXISTING:
        am|=O_TRUNC;
        break;
  case FILE_AM_OPEN_EXISTING:
      break;
    case FILE_AM_OPEN_ALWAYS:
        am|=O_CREAT;
        break;
    case FILE_AM_CREATE_ALWAYS:
    am|=(O_CREAT | O_TRUNC);
    break;
    default:
        return Error("File::accessFile()",
                       ERROR_LEVEL_NORMAL,
                       0,
                       ERROR_ADVISE_DONTKNOW,
                       "unknown access mode "+_name);
        break;
    } // switch
    if (access(_name.c_str(),am))
        return Error("File::accessFile()",
                       ERROR_LEVEL_NORMAL,
                       0,
                       ERROR_ADVISE_DONTKNOW,
                       strerror(errno),
                       "error on access "+_name);
    return Error();
}


Error File::openFile(int access,
                         int mode){
  int am;

  am=0;
  if (access & FILE_AM_RDONLY) am|=O_RDONLY;
  if (access & FILE_AM_WRONLY) am|=O_WRONLY;
  if (access & FILE_AM_RDWR) am|=O_RDWR;
  if (access & FILE_AM_APPEND) am|=O_APPEND;

  switch (access & FILE_INTERNAL_AM_MASK) {
  case FILE_AM_CREATE_NEW:
    am|=(O_CREAT | O_EXCL);
    break;
  case FILE_AM_TRUNCATE_EXISTING:
    am|=O_TRUNC;
    break;
  case FILE_AM_OPEN_EXISTING:
    break;
  case FILE_AM_OPEN_ALWAYS:
    am|=O_CREAT;
    break;
  case FILE_AM_CREATE_ALWAYS:
    am|=(O_CREAT | O_TRUNC);
    break;
  default:
    return Error("File::openFile()",
                   ERROR_LEVEL_NORMAL,
                   0,
                   ERROR_ADVISE_DONTKNOW,
                   "unknown access mode "+_name);
    break;
  } // switch

  _handle=open(_name.c_str(),am,mode);
  if (_handle==-1)
      return Error("File::openFile()",
                     ERROR_LEVEL_NORMAL,
                     0,
                     ERROR_ADVISE_DONTKNOW,
                     strerror(errno),
                     "error on open() "+_name);
  return Error();
}


Error File::closeFile(){
  if (close(_handle))
    return Error("File::closeFile()",
                   ERROR_LEVEL_NORMAL,
                   0,
                   ERROR_ADVISE_DONTKNOW,
                   strerror(errno),
                   "error on close() "+_name);
  return Error();
}


Error File::createFile(int access,
                           int mode){
  return openFile(access,mode);
}


Error File::statFile(s_filestat &s){
  struct stat statbuffer;

  if (stat(_name.c_str(),&statbuffer))
    return Error("File::statFile()",
                   ERROR_LEVEL_NORMAL,
                   0,
                   ERROR_ADVISE_DONTKNOW,
                   strerror(errno),
                   "error on stat() "+_name);
  s.size=(unsigned long)statbuffer.st_size;
  s.mode=(unsigned int)statbuffer.st_mode;
  // access time
  s.atime=DateTime(statbuffer.st_atime);
  // modification time
  s.mtime=DateTime(statbuffer.st_mtime);
  // change time
  s.ctime=DateTime(statbuffer.st_ctime);
  return Error();
}


Error File::renameFile(string newname){
  _convPath(newname);

  if (rename(_name.c_str(),newname.c_str()))
    return Error("File::renameFile()",
                   ERROR_LEVEL_NORMAL,
                   0,
                   ERROR_ADVISE_DONTKNOW,
                   strerror(errno),
                   "error on rename()"+_name+" to "+newname);
  _name=newname;
  return Error();
}


Error File::deleteFile(){
  if (unlink(_name.c_str()))
    return Error("File::deleteFile()",
                   ERROR_LEVEL_NORMAL,
                   0,
                   ERROR_ADVISE_DONTKNOW,
                   strerror(errno),
                   "error on delete() "+_name);
  return Error();
}


Error File::lockFile(int mode,
		     int whence,
		     t_offset start,
		     t_offset length,
		     bool wait){
  struct flock fl;
  int rv;

  switch (mode) {
  case FILE_LOCK_SHARED:
    fl.l_type=F_RDLCK;
    break;

  case FILE_LOCK_EXCLUSIVE:
    fl.l_type=F_WRLCK;
    break;

  default:
    return Error("File::lockFile()",
                   ERROR_LEVEL_NORMAL,
                   0,
                   ERROR_ADVISE_DONTKNOW,
                   "unknown lock mode "+_name);
    break;
  } // switch
  fl.l_whence=whence;
  fl.l_start=start;
  fl.l_len=length;
  if (wait)
    rv=fcntl(_handle,F_SETLKW,&fl);
  else rv=fcntl(_handle,F_SETLK,&fl);
  if (rv)
    return Error("File::lockFile()",
                   ERROR_LEVEL_NORMAL,
                   0,
                   ERROR_ADVISE_DONTKNOW,
                   strerror(errno),
                   "error on fcntl() "+_name);
  return Error();
}


Error File::unlockFile(int whence,
                           t_offset start,
                           t_offset length){
  struct flock fl;

  fl.l_type=F_UNLCK;
  fl.l_whence=whence;
  fl.l_start=start;
  fl.l_len=length;
  if (fcntl(_handle,F_SETLK,&fl))
    return Error("File::unlockFile()",
                   ERROR_LEVEL_NORMAL,
                   0,
                   ERROR_ADVISE_DONTKNOW,
                   strerror(errno),
                   "error on fcntl() "+_name);
  return Error();
}


Error File::filePos(t_offset &p){
  p=lseek(_handle,0,SEEK_CUR);
  if (p==(off_t)-1)
    return Error("File::filePos()",
                   ERROR_LEVEL_NORMAL,
                   0,
                   ERROR_ADVISE_DONTKNOW,
                   strerror(errno),
                   "error on lseek() "+_name);
  return Error();
}


Error File::setFilePos(t_offset p, int whence){
  if (lseek(_handle,p,whence)==(off_t)-1)
    return Error("File::setFilePos()",
                   ERROR_LEVEL_NORMAL,
                   0,
                   ERROR_ADVISE_DONTKNOW,
                   strerror(errno),
                   "error on lseek() "+_name);
  return Error();
}


Error File::changeMode(int mode){
  if (chmod(_name.c_str(),mode))
    return Error("File::changeMode()",
                   ERROR_LEVEL_NORMAL,
                   0,
                   ERROR_ADVISE_DONTKNOW,
                   strerror(errno),
                   "error on chmod() "+_name);
  return Error();
}


Error File::readData(string &data, unsigned int size){
  int bytesread;
  char buffer[8192];

  // Now read the bytes
  if (size>sizeof(buffer))
    size=sizeof(buffer);

  bytesread=read(_handle,buffer,size);
  if (bytesread==-1)
    return Error("File::readData()",
                   ERROR_LEVEL_NORMAL,
                   0,
                   ERROR_ADVISE_DONTKNOW,
                   strerror(errno),
                   "error on read() "+_name);
  data.assign(buffer,bytesread);
  return Error();
}


Error File::writeData(const string &buffer){
  int bytesleft, byteswritten;
  const char *bytespos;

  bytesleft=buffer.length();
  bytespos=buffer.c_str();

  while (bytesleft) {
    // Now write the bytes
    byteswritten=write(_handle,bytespos,bytesleft);
    if (byteswritten==-1)
      return Error("File::writeData()",
                     ERROR_LEVEL_NORMAL,
                     0,
                     ERROR_ADVISE_DONTKNOW,
                     strerror(errno),
                     "error on write() "+_name);
    if (!byteswritten)
      return Error("File::writeData()",
                     ERROR_LEVEL_NORMAL,
                     0,
                     ERROR_ADVISE_DONTKNOW,
                     strerror(errno),
                     "no bytes written on write() "+_name);
    bytesleft-=byteswritten;
    bytespos+=byteswritten;
  } // while bytesleft
  return Error();
}

} /* namespace HBCI */


