/***************************************************************************
  $RCSfile: libloader.cpp,v $
                             -------------------
    cvs         : $Id: libloader.cpp,v 1.6 2003/05/17 06:34:04 aquamaniac Exp $
    begin       : Sun Mar 10 2002
    copyright   : (C) 2002 by Martin Preuss
    email       : martin@libchipcard.de


 ***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/

/*
 Changes

 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef __declspec
# if BUILDING_DLL
#  define DLLIMPORT __declspec (dllexport)
# else /* Not BUILDING_DLL */
#  define DLLIMPORT __declspec (dllimport)
# endif /* Not BUILDING_DLL */
#else
# define DLLIMPORT
#endif

#include <string>
#include <stdio.h>
#ifndef OS_WIN32
# ifdef HAVE_DLFCN_H
#  include <dlfcn.h>
# endif
#endif


#include "error.h"
#include "libloader.h"


namespace HBCI {


LibLoader::LibLoader(const string &libname, const string &prefix)
:_handle(0)
,_name(libname)
,_prefix(prefix){
#if DEBUGMODE>0
  fprintf(stderr,"LibLoader::LibLoader\n");
#endif

}


LibLoader::~LibLoader(){
#if DEBUGMODE>0
  fprintf(stderr,"LibLoader::~LibLoader\n");
#endif
  closeLibrary();
}


Error LibLoader::openLibrary(){
#if DEBUGMODE>0
  fprintf(stderr,"LibLoader::openLibrary\n");
#endif

#if defined (OS_WIN32)
  _handle=(void*)LoadLibrary(_name.c_str());
  if (!_handle) {
# if DEBUGMODE>0
    fprintf(stderr,
	    "OpenHBCI: Error loading library \"%s\"\n",
	    _name.c_str());
# endif
    return Error("LibLoader::openLibrary()",
		 ERROR_LEVEL_NORMAL,
		 HBCI_ERROR_CODE_LIBRARY,
		 ERROR_ADVISE_ABORT,
		 "could not load library");

  }
#else // for linux and others
  _handle=dlopen(_name.c_str(),RTLD_LAZY);
    if (!_handle)
      return Error("LibLoader::openLibrary()",
		   ERROR_LEVEL_NORMAL,
		   HBCI_ERROR_CODE_LIBRARY,
		   ERROR_ADVISE_ABORT,
		   "could not load library",
		   dlerror());
#endif
    return Error();
}


Error LibLoader::closeLibrary(){
#if DEBUGMODE>0
  fprintf(stderr,"LibLoader::closeLibrary\n");
#endif

  if (!_handle)
    return Error("LibLoader::closeLibrary()",
		 ERROR_LEVEL_NORMAL,
		 HBCI_ERROR_CODE_LIBRARY,
		 ERROR_ADVISE_ABORT,
		 "library not open");
#ifndef OS_WIN32
  /* hmm, linux does not like this with libtowitoko.so and qlcsetup */
  //result=(dlclose(_handle)==0);
  if (dlclose(_handle)!=0)
    return Error("LibLoader::closeLibrary()",
		 ERROR_LEVEL_NORMAL,
		 HBCI_ERROR_CODE_LIBRARY,
		 ERROR_ADVISE_ABORT,
		 "could not close library",
		 dlerror());
#else
  if (!FreeLibrary((HINSTANCE)_handle))
    return Error("LibLoader::closeLibrary()",
		 ERROR_LEVEL_NORMAL,
		 HBCI_ERROR_CODE_LIBRARY,
		 ERROR_ADVISE_ABORT,
		 "could not close library");
#endif
  _handle=0;
  return Error();
}


void *LibLoader::_resolve(const string &symbol){
  void *result;
  string localsymbol;

#if DEBUGMODE>0
  fprintf(stderr,"LibLoader::_resolve\n");
#endif

  if (!_handle)
    return 0;
#ifndef OS_WIN32
  result=dlsym(_handle,symbol.c_str());
  // Use return value of dlerror() to check for errors, according to
  // dlsym(3)
  const char *err = dlerror();
  if (err != NULL) {
    result=0;
    fprintf(stderr,"LibLoader::resolve: Error on resolving \"%s\": %s\n",
	    symbol.c_str(),err);
    throw Error("LibLoader::resolve()",
		ERROR_LEVEL_NORMAL,
		HBCI_ERROR_CODE_LIBRARY,
		ERROR_ADVISE_ABORT,
		"Error on resolving" + symbol,
		err);
  }
#else
  result=(void*)GetProcAddress((HINSTANCE)_handle,symbol.c_str());
#endif

#if DEBUGMODE>1
  fprintf(stderr,"Resolve \"%s\": %08x\n",
	  symbol.c_str(),(int)result);
#endif
  return result;
}


void *LibLoader::resolve(const string &symbol){
  void *result;
  string localsymbol;
  Error err;

#if DEBUGMODE>0
  fprintf(stderr,"LibLoader::resolve\n");
#endif

  if (_prefix.empty())
    localsymbol=symbol;
  else
    localsymbol=_prefix+"_"+symbol;

  /* OpenBSD has the additional prefix "_", and maybe other systems
   * have, too. So we try both here */
  err=Error();
  try {
    result=_resolve(localsymbol);
  }
  catch (Error xerr) {
    err=xerr;
  }
  if (!err.isOk()) {
    try {
      result=_resolve("_"+localsymbol);
    }
    catch (Error xerr) {
      err=xerr;
    }
  }
  if (!err.isOk())
    throw Error("LibLoader::resolve()",err);
  return result;
}


string LibLoader::libraryName() const{
#if DEBUGMODE>0
  fprintf(stderr,"LibLoader::libraryName\n");
#endif

  return _name;
}


}; // namespace



