/***************************************************************************
                          hbcirsakey.cpp  -  description
                             -------------------
    begin                : Sun Jul 1 2001
    copyright            : (C) 2001 by fabian kaiser
    email                : fabian.kaiser@gmx.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/

/*
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef __declspec
# if BUILDING_DLL
#  define DLLIMPORT __declspec (dllexport)
# else /* Not BUILDING_DLL */
#  define DLLIMPORT __declspec (dllimport)
# endif /* Not BUILDING_DLL */
#else
# define DLLIMPORT
#endif


#include <string>
#include <stdio.h>
#include <list>

#include "rsakey.h"
#include "hbcistring.h"

#include <openssl/bn.h>
#include <openssl/ripemd.h>
#include <openssl/rsa.h>

namespace HBCI {

RSAKey::RSAKey(bool publicKey){/* Default param: false */
    isPublic = publicKey;
}


string RSAKey::toString() 
{
    string result = string((isCryptKey ? "crypt" : "sign")) + "'";

    if (isPublicKey()) {
	result += string("pub") + "'"
	    + String::num2string(exponent) + "'"
	    + String::transformToHBCIBinData(modulus)  + "'";
    } else {
        result += string("priv") + "'"
	    + String::transformToHBCIBinData(n)  + "'"
	    + String::transformToHBCIBinData(p)  + "'"
	    + String::transformToHBCIBinData(q)  + "'"
	    + String::transformToHBCIBinData(d)  + "'"
	    + String::transformToHBCIBinData(dmp1)  + "'"
	    + String::transformToHBCIBinData(dmq1)  + "'"
	    + String::transformToHBCIBinData(iqmp)  + "'";
    }

    result += myUserId + "'"
	+ String::num2string(_number) + "'"
	+ String::num2string(_version) + "'";

    return result;
}

/** Note that the argument tpos is a reference, i.e. the calling
    variable will be changed through this method. */
string RSAKey::loadDataNext(const string &keystring,
				unsigned int &tpos) 
{
    string ttmp;
    ttmp = String::nextSEG(keystring, tpos); 
    tpos += ttmp.length() + 1;  // will be stored in tpos!
    if (ttmp[0] == '@') 
	ttmp = String::extractBinData(ttmp);
    return ttmp;
}

void RSAKey::loadDataFromString(const string &keyData) 
{
    unsigned int pos = 0;
    string tmp;

    tmp = loadDataNext(keyData, pos);
    isCryptKey = (tmp == "crypt");
    tmp = loadDataNext(keyData, pos);
    isPublic = (tmp == "pub");

    if (isPublic) {
        tmp = loadDataNext(keyData, pos);
        exponent = atoi(tmp.c_str());
        modulus = loadDataNext(keyData, pos);
    } else {
        n = loadDataNext(keyData, pos);
        p = loadDataNext(keyData, pos);
        q = loadDataNext(keyData, pos);
        d = loadDataNext(keyData, pos);
        dmp1 = loadDataNext(keyData, pos);
        dmq1 = loadDataNext(keyData, pos);
        iqmp = loadDataNext(keyData, pos);
    }

    myUserId = loadDataNext(keyData, pos);
    tmp = loadDataNext(keyData, pos);
    _number = atoi(tmp.c_str());
    tmp = loadDataNext(keyData, pos);
    _version = atoi(tmp.c_str());
}


RSAKey::RSAKey(RSAKey::keyData *kd) 
    : isPublic(kd->isPublic)
    , isCryptKey(kd->isCrypt)
    , _number(kd->number)
    , _version(kd->version)
    , myUserId(kd->owner)
    , modulus(kd->modulus)
    , exponent(kd->exponent)
    , n(kd->n)
    , p(kd->p)
    , q(kd->q)
    , d(kd->d)
    , dmp1(kd->dmp1)
    , dmq1(kd->dmq1)
    , iqmp(kd->iqmp)
{
}


bool RSAKey::getKeyData(RSAKey::keyData *kd) {
    kd->isPublic=isPublic;
    kd->isCrypt=isCryptKey;
    kd->number=_number;
    kd->version=_version;
    kd->owner=myUserId;
    kd->modulus=modulus;
    kd->exponent=exponent;
    kd->n=n;
    kd->p=p;
    kd->q=q;
    kd->d=d;
    kd->dmp1=dmp1;
    kd->dmq1=dmq1;
    kd->iqmp=iqmp;
    return true;
}


RSAKey::~RSAKey(){
}


bool RSAKey::decrypt() {
    unsigned int res;
    unsigned char source[intData.length()];
    unsigned char dest[intData.length()];

    for (unsigned int i=0; i<intData.length(); i++)
        source[i]=intData[i];

    RSA *myRSA = RSA_new();
    fillRSAStruct(myRSA);

    if (isPublicKey())
        res = RSA_public_decrypt(intData.length(), (unsigned char*) &source, (unsigned char*) &dest, myRSA, RSA_NO_PADDING);
    else
        res = RSA_private_decrypt(intData.length(), (unsigned char*) &source, (unsigned char*) &dest, myRSA, RSA_NO_PADDING);

    string s2 = "";
    for (unsigned int i=0; i< intData.length(); i++)
        s2 += dest[i];
    intData = s2;
    RSA_free(myRSA);

    return (res == intData.length());
}


bool RSAKey::encrypt() {
    //padding wiht 80 * '\0' to DEFAULT_KEY_LENGTH / 8 byte
    unsigned int res;
    RSA *myRSA = RSA_new();
    fillRSAStruct(myRSA);
	//	string s =  string((DEFAULT_KEY_LENGTH / 8) - intData.length(),'\0') + intData;
	// This might be a fix for the HBV
	// we don't padd to 768 bit but to the length of the key we use for
	// encryption
	int keylen = (modulus.empty())?n.length():modulus.length();
	if (keylen > DEFAULT_KEY_LENGTH / 8)
	  keylen = DEFAULT_KEY_LENGTH / 8;
	string s = string(keylen - intData.length(), '\0') + intData;

    unsigned char dest[s.length()];
    unsigned char *source;
    source = (unsigned char *) s.c_str();

    if (isPublicKey())
        res = RSA_public_encrypt(s.length(), source, (unsigned char*) &dest, myRSA, RSA_NO_PADDING);
    else
        res = RSA_private_encrypt(s.length(), source, (unsigned char*) &dest, myRSA, RSA_NO_PADDING);

    string s2 = "";
    for (unsigned int i=0; i < s.length(); i++)
        s2 += dest[i];
    intData = s2;
    RSA_free(myRSA);

    return (res = intData.length());
}


bool RSAKey::verify(const string &signature) {
    int res;
    string result;
    BIGNUM *myModulus = BN_new();
    BIGNUM *myExponent = BN_new();
    BIGNUM *myMsg = BN_new();
    BIGNUM *myResult = BN_new();
    BN_CTX *myCTX = BN_CTX_new();

    // decrypt the institutes signature
    BN_set_word(myExponent, DEFAULT_EXPONENT);
    myModulus = BN_bin2bn((unsigned char*) modulus.data(), modulus.length(), myModulus);
    myMsg = BN_bin2bn((unsigned char*) signature.data(), signature.length(), myMsg);
    BN_CTX_start(myCTX);
    res = BN_mod_exp(myResult, myMsg, myExponent, myModulus, myCTX);
    // transform to string
    unsigned char buffer[1024];
    res = BN_bn2bin(myResult, (unsigned char*) &buffer);
    result = string((char*) &buffer, res);

    // create my own ripe and padd it
    string s = paddWithISO9796(ripe(intData));
    BIGNUM *myCalcSig = BN_new();
    myCalcSig = BN_bin2bn((unsigned char*) s.data(), s.length(), myCalcSig);
    res = BN_bn2bin(myCalcSig, (unsigned char*) &buffer);
    s = string((char*) &buffer, res);

    // check if inst-sig is equal to what we have calculated
    // else use the iso9796-appendix and check again
    if (s != result) {
        res = BN_sub(myResult, myResult, myModulus);
        res = BN_bn2bin(myResult, (unsigned char*) &buffer);
        result = string((char*) &buffer, res);
    }

    return (s == result);
}


bool RSAKey::sign() {

    int res;
    BIGNUM *myModulus  = BN_new();
    BIGNUM *myExponent = BN_new();
    BIGNUM *myMsg = BN_new();
    BIGNUM *myResult = BN_new();
    BIGNUM *myResult2 = BN_new();
    BN_CTX *myCTX = BN_CTX_new();

    string hash = paddWithISO9796(ripe(intData));

    myMsg = BN_bin2bn((unsigned char*) hash.data(), hash.length(), myMsg);
    myModulus = BN_bin2bn((unsigned char*) n.data(), n.length(), myModulus);
    myExponent = BN_bin2bn((unsigned char*) d.data(), d.length(), myExponent);

    BN_CTX_start(myCTX);
    res = BN_mod_exp(myResult, myMsg, myExponent, myModulus, myCTX);

    // the iso9796-appendix is as follows:
    // if (the calculated signature - the modulus) < (the calculated signature)
    // use (the calculated signature - the modulus) as signature
    BN_sub(myResult2, myModulus, myResult);
    if (BN_cmp(myResult2, myResult) < 0) {
        BN_free(myResult);
        myResult = myResult2;
    }

    // transform to string
    unsigned char dest[hash.length()];
    res = BN_bn2bin(myResult, (unsigned char*) &dest);
    intData = string((char*) &dest, res);

    // pad to a multiple of 8 (add \0 at the beginning)
    if (0 != intData.length() % 8) {
      int pad = 8 - (intData.length() % 8);
      intData = string(pad, '\0') + intData;
    }

    // free BNs
    BN_free(myResult);
    BN_free(myModulus);
    BN_free(myMsg);
    BN_free(myExponent);

    // successfull?
    return (intData.length() == hash.length());
}


string RSAKey::getExpData() const {
    unsigned char dest[10];
    string result;
    int length;

    BIGNUM *myBN = BN_new();
    BN_set_word(myBN, exponent);

    length=BN_bn2bin(myBN, dest);
    result = string((const char*) dest, length);

    return result;
}


void RSAKey::fillRSAStruct(RSA *rsaStruct) {
#define NEXT(tstring) { \
    myBIGNUM = BN_new(); \
    myBIGNUM = BN_bin2bn((unsigned char*) tstring.data(), tstring.length(), myBIGNUM); \
    rsaStruct->tstring = myBIGNUM; }

  // since openssl uses blinding, we have to set 'e' even if this is the
  // private key
  BIGNUM *myExponent = BN_new();
  BN_set_word(myExponent,DEFAULT_EXPONENT);
  rsaStruct->e = myExponent;

    if (isPublicKey()) {
        BIGNUM *myModulus = BN_new();
        myModulus = BN_bin2bn((unsigned char*) modulus.data(), 
							  modulus.length(), myModulus);
        rsaStruct->n = myModulus;
    } else {
        BIGNUM *myBIGNUM;
        NEXT(n);
        NEXT(p);
        NEXT(q);
        NEXT(dmp1);
        NEXT(dmq1);
        NEXT(iqmp);
        NEXT(d);
    }
#undef NEXT
}


string RSAKey::paddWithISO9796(string source) {
    unsigned char c;
    string dest(DEFAULT_KEY_LENGTH / 8, '\0');
    string d2(DEFAULT_KEY_LENGTH / 8,'0');

    // we don't need to padd, 'cause length of signature is already a multiple of 8

    //extending
    dest = source + source + source;
    dest = dest.substr(20, 40);

    // append redundancy
    for (int i=0; i <= 47 ; i++) {
        d2[1 + 96 - (2*i)] = dest[40 - i];
        d2[96 - (2*i)] = pi(dest[40 - i]);
    }
    string tmp = d2.substr(d2.length() - 16, 16);
    d2.replace(0,16,tmp);

    // finish
    c = d2[d2.length() - 1];
    c = (c & 15) * 16;
    c += 6;
    d2[d2.length() - 1] = c;
    d2[0] = d2[0] & 127;
    d2[0] = d2[0] | 64;
    d2[d2.length() - 40] = d2[d2.length() - 40] ^ 1;

    return d2;
}


string RSAKey::ripe(string source) const {
    string result="";
    unsigned char hash[20];
    RIPEMD160((unsigned char*) source.data(), source.length(), (unsigned char*) &hash);

    for (int i=0; i<20; i++)
        result += hash[i];

    return result;
}


void RSAKey::generateKeyPair(unsigned int keyLength, RSAKey **privKey, RSAKey **pubKey) {
#define NEXT(tstring) { \
    length = BN_bn2bin(myRSA->tstring, (unsigned char*) &dest); \
    (*privKey)->tstring = string((const char*) &dest, length); }

    unsigned char dest[keyLength / 8];
    int length;
    *privKey = new RSAKey();
    *pubKey = new RSAKey();

    (*privKey)->isPublic = false;
    (*pubKey)->isPublic = true;
    RSA *myRSA = RSA_generate_key(keyLength, DEFAULT_EXPONENT, NULL, NULL);

    length=BN_bn2bin(myRSA->n, dest);
    (*pubKey)->modulus=string((const char*) &dest, length);
    (*pubKey)->exponent = DEFAULT_EXPONENT;

    NEXT(n);
    NEXT(p);
    NEXT(q);
    NEXT(dmp1);
    NEXT(dmq1);
    NEXT(iqmp);
    NEXT(d);

    (*privKey)->_number = 1;
    (*privKey)->_version = 1;
    (*pubKey) ->_number  = 1;
    (*pubKey) ->_version  = 1;
    RSA_free(myRSA);
#undef NEXT
}


// fw-declaration
string bin2hex(string &bin, int length);


string RSAKey::getIniLetterModulus() const {
    string result = getModulusData();
    return bin2hex(result, DEFAULT_KEY_LENGTH / 8);
}


string RSAKey::getIniLetterExponent() const {
    string result = getExpData();
    return bin2hex(result, DEFAULT_KEY_LENGTH / 8);
}


string RSAKey::getIniLetterHash() const {
    string result = string(128 - getExpData().length(), 0x0) + getExpData();
    result += string(128 - getModulusData().length(), 0x0) + getModulusData();
    result = ripe(result);
    return bin2hex(result, 20);
}


unsigned char RSAKey::lookUp[2][16] =
{{0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15},
{14,3,5,8,9,4,2,15,0,13,11,6,7,10,12,1}};


unsigned char RSAKey::pi(unsigned char input) {
    unsigned char leftNibble;
    unsigned char rightNibble;

    rightNibble = input & 15;
    leftNibble = input & 240;
    leftNibble = leftNibble / 16;
    rightNibble = lookUp[1][rightNibble];
    leftNibble = lookUp[1][leftNibble];
    leftNibble = leftNibble * 16;

    return leftNibble + rightNibble;
}

string bin2hex(string &bin, int length) {
    // use the openssl hex-transformer
    BIGNUM *myBIGNUM;
    myBIGNUM = BN_new();
    myBIGNUM = BN_bin2bn((unsigned char*) bin.data(), bin.length(), myBIGNUM);

    // transform
    char *hexChar = BN_bn2hex(myBIGNUM);
    BN_free(myBIGNUM);
    string hexCode = string((const char*) hexChar);
    delete hexChar;
    // pad
    hexCode = string(2 * length - hexCode.length(), '0') + hexCode;
    return hexCode;
}

} /* namespace HBCI */
