(* Postgres: OCaml bindings for PostgreSQL
   Copyright (C) 2001  Alain Frisch         <Alain.Frisch@ens.fr>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License; see the file LGPL.
*)

(* Low-level stubs *)

module Result :
sig
  type t
    
  type status =  
    | Empty_query
    | Command_ok
    | Tuples_ok
    | Copy_out
    | Copy_in
    | Bad_response
    | Nonfatal_error
    | Fatal_error

  type oid = int
  external string_of_status: status -> string = "stub_PQresStatus" 

  (* Below this: low-level; use at your own risk ! *)
  external is_null: t -> bool = "res_isnull"
  external status: t -> status = "stub_PQresultStatus"
  external error: t -> string = "stub_PQresultErrorMessage"
  external ntuples: t -> int = "stub_PQntuples"
  external nfields: t -> int = "stub_PQnfields"
  external binary_tuples: t -> bool = "stub_PQbinaryTuples"
  external fname: t -> int -> string = "stub_PQfname"
  external fnumber: t -> string -> int ="stub_PQfnumber"
  external ftype: t -> int -> oid = "stub_PQftype"
  external fsize: t -> int -> int = "stub_PQfsize"
  external fmod: t -> int -> int = "stub_PQfmod"
  external getvalue: t -> int -> int -> string = "stub_PQgetvalue"
  external getlength: t -> int -> int -> int = "stub_PQgetlength"
  external getisnull: t -> int -> int -> bool = "stub_PQgetisnull"
  external cmd_status: t -> string = "stub_PQcmdStatus"
  external cmd_tuples: t -> string = "stub_PQcmdTuples"
  external oid_value: t -> oid = "stub_PQoidValue"
end

module Connection :
sig
  type t
  type status = Ok | Bad

  external connect: string -> t = "stub_PQconnectdb"
  external is_null: t -> bool = "conn_isnull"
  external finish: t -> unit = "stub_PQfinish"
  external reset: t -> unit = "stub_PQreset"
  external db: t -> string = "stub_PQdb"
  external user: t -> string = "stub_PQuser"
  external pass: t -> string = "stub_PQpass"
  external host: t -> string = "stub_PQhost"
  external port: t -> string = "stub_PQport"
  external tty: t -> string = "stub_PQtty"
  external options: t -> string = "stub_PQoptions"
  external status: t -> status = "stub_PQstatus"
  external error_message: t -> string = "stub_PQerrorMessage"
  external backend_pid: t -> int = "stub_PQbackendPID"

  external notifies: t -> (string * int) option = "stub_PQnotifies"
  external set_notice_processor: t -> (string -> unit) -> unit 
		       = "stub_PQsetNoticeProcessor"

  external set_nonblocking: t -> bool -> int = "stub_PQsetnonblocking"
  external is_nonblocking: t -> bool = "stub_PQisnonblocking"
  external consume_input: t -> int = "stub_PQconsumeInput"

  external is_busy: t -> bool = "stub_PQisBusy"
  external flush: t -> int = "stub_PQflush"
  external socket: t -> int = "stub_PQsocket"
  external request_cancel: t -> int = "stub_PQrequestCancel"

  external getline: t -> string -> int -> int -> int = "stub_PQgetline"
  external getline_async: t -> string -> int -> int -> int = "stub_PQgetlineAsync"
  external putline: t -> string -> int = "stub_PQputline"
  external putnbytes: t -> string -> int -> int -> int = "stub_PQputnbytes"
  external endcopy: t -> int = "stub_PQendcopy"

  external exec: t -> string -> Result.t = "stub_PQexec"
  external send_query: t -> string -> int = "stub_PQsendQuery"
  external get_result: t -> Result.t = "stub_PQgetResult"
  external make_empty: t -> Result.status -> Result.t = "stub_PQmakeEmptyPGresult"
end

module LargeObjects :
sig
  type t
  type conn = Connection.t
  type oid = Result.oid
  external lo_open: conn -> oid -> t = "stub_lo_open"
  external close: conn -> t -> int = "stub_lo_close"
  external read: conn -> t -> string -> int -> int -> int = "stub_lo_read"
  external write: conn -> t -> string -> int -> int -> int = "stub_lo_write"
  external seek: conn -> t -> int -> int = "stub_lo_lseek"
  external tell: conn -> t -> int = "stub_lo_tell"
  external create: conn -> oid = "stub_lo_creat"
  external unlink: conn -> oid -> oid = "stub_lo_unlink"
  external import: conn -> string -> oid = "stub_lo_import"
  external export: conn -> oid -> string -> int = "stub_lo_export"
end

(* High-level encapsulation *)

type error =
  | Field_out_of_range of int*int
  | Tuple_out_of_range of int*int
  | Binary
  | ConnectionFailure of string
  | UnexpectedStatus of Result.status * string * (Result.status list)

exception Error of error

val string_of_error: error -> string


class type result =
object
  method internal: Result.t

  method status: Result.status
  method error: string

  method ntuples: int
  method nfields: int
  method binary: bool

  method fname: int -> string
  method fnumber: string -> int
  method ftype: int -> Result.oid
  method fsize: int -> int
  method fmod: int -> int
  method getvalue: int -> int -> string
  method getlength: int -> int -> int
  method getisnull: int -> int -> bool

  method cmd_status: string
  method cmd_tuples: string
  method oid_value: int

  method get_fields_list: string list

  method get_tuple_list: int -> string list
  method get_tuple_array: int -> string array

  method get_list: string list list
  method get_array: string array array
end

class connection : string ->
object
  method internal: Connection.t

  method close : unit
  method may_reset : unit
  method reset : unit
    
  method notification : (string * int) option

  method set_notice_callback : (string -> unit) -> unit

  (* Accessors *)

  method db      : string
  method user    : string
  method pass    : string
  method host    : string
  method port    : string
  method tty     : string
  method options : string
  method backend : int
  method socket  : int


  (* Queries *)

  method empty_result : Result.status -> result (* Dummy result *)
  method exec : string -> result (* Synchronous *)
  method send : string -> unit  (* Asynchronous *)
  method get_result : result option
    
  method exec_expect : string -> Result.status list -> result

  (* Copy operations *)
    
  (* low level *)
  method getline : string -> int -> int -> int
  method getline_async : string -> int -> int -> int
  method putline : string -> unit
  method putnbytes : string -> int -> int -> unit
  method endcopy : unit

  (* high level *)
  method copy_out : (string -> unit) -> unit
  method copy_out_channel : out_channel -> unit
  method copy_in_channel : in_channel -> unit

  (* Asynchronous operations and non blocking mode *)
  method set_nonblocking : bool -> unit
  method is_nonblocking : bool
  method consume_input : unit
  method is_busy : bool
  method flush : unit
  method request_cancel : unit


  (* (Inversion) Large objects *)
  method lo_create : LargeObjects.oid
  method lo_import : string -> LargeObjects.oid
  method lo_export : LargeObjects.oid -> string -> unit
  method lo_open : LargeObjects.oid -> LargeObjects.t
  method lo_write : LargeObjects.t -> string -> int -> int -> unit
  method lo_write_string : LargeObjects.t -> string -> unit
  method lo_read : LargeObjects.t -> string -> int -> int -> int
  method lo_seek : LargeObjects.t -> int -> unit
  method lo_tell : LargeObjects.t -> int
  method lo_close : LargeObjects.t -> unit
  method lo_unlink : LargeObjects.oid -> unit
 end

val conninfo :
  ?host:string ->
  ?hostaddr:string ->
  ?port:string  ->
  ?dbname:string ->
  ?user:string ->
  ?password:string ->
  ?options:string -> 
  ?tty:string -> 
  ?requiressl:string -> 
  unit -> string

val escape_string: string -> string
val escape_substring: string -> int -> int -> string

(* direct stub - may be used to avoid creation of useless buffer *)
external escapeString: dest:string -> int -> src:string -> int -> int -> int = "stub_PQescapeString"

