/*
 * Copyright (c) 2001-2003 Shiman Associates Inc. All Rights Reserved.
 * 
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */
/*
 * $Id: mas_codec_ulaw_device.c,v 1.2 2003/06/26 21:16:34 rocko Exp $
 *
 * Copyright (c) 2000, 2001 by Shiman Associates Inc. and Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions: The above
 * copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the names of the authors or
 * copyright holders shall not be used in advertising or otherwise to
 * promote the sale, use or other dealings in this Software without
 * prior written authorization from the authors or copyright holders,
 * as applicable.
 *
 * All trademarks and registered trademarks mentioned herein are the
 * property of their respective owners. No right, title or interest in
 * or to any trademark, service mark, logo or trade name of the
 * authors or copyright holders or their licensors is granted.
 *
 */

/* 2 OCT 2002 - rocko - verified reentrant
 * 2 OCT 2002 - rocko - verified timestamp clean
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include "mas/mas_dpi.h"
#include "profile.h"

extern int16 ulaw_to_lin16_table[];
extern uint8 lin16_to_ulaw_table[];

#define ULAW  1
#define LIN16 2
#define ULAW_2_LIN16 1
#define LIN16_2_ULAW 2

struct codec_ulaw_state
{
    int32 reaction;
    int32 sink;
    int32 source;
    int   from_fmt;
    int   to_fmt;
};

/* standard actions ****************************************************/
int32 mas_dev_init_instance( int32 , void* );
int32 mas_dev_configure_port( int32 , void* );
int32 mas_dev_show_state( int32 device_instance, void* predicate );

/* device specific actions *********************************************/
int32 mas_codec_ulaw_convert( int32 , void* );

int32
mas_dev_init_instance( int32 device_instance, void* predicate )
{
    struct codec_ulaw_state*  state;
    
    /* Allocate state holder and cast it so we can work on it */
    state       = MAS_NEW( state );

    masd_set_state(device_instance, state); /* set device state */
    
    masd_get_port_by_name( device_instance, "sink",
			   &state->sink );
    masd_get_port_by_name( device_instance, "source",
			   &state->source );
    masd_get_port_by_name( device_instance, "reaction",
			   &state->reaction );

    return 0;
}

int32
mas_dev_configure_port( int32 device_instance, void* predicate )
{
    struct codec_ulaw_state*  state;
    struct mas_data_characteristic* dc;
    struct mas_data_characteristic* odc; /*other data characteristic*/
    int32*                dataflow_port_dependency;
    int32                 fk, resk, chk, ratek;
    int                   tfmt;
    int32 err;

    MASD_GET_STATE(device_instance, state);
    err = masd_get_data_characteristic( *(int32*)predicate, &dc );
    
    fk = masc_get_index_of_key(dc, "format");
    resk = masc_get_index_of_key(dc, "resolution");
    chk = masc_get_index_of_key( dc, "channels");
    ratek = masc_get_index_of_key( dc, "sampling rate");
    
    if ( *(int32*)predicate == state->sink )
    {
        if ( strcmp( dc->values[fk], "linear" ) == 0 )
            state->from_fmt = LIN16;
        else if ( strcmp( dc->values[fk], "ulaw" ) == 0 )
            state->from_fmt = ULAW;
        tfmt = state->from_fmt;
    }
    else if ( *(int32*)predicate == state->source )
    {
        if ( strcmp( dc->values[fk], "linear" ) == 0 )
            state->to_fmt = LIN16;
        else if ( strcmp( dc->values[fk], "ulaw" ) == 0 )
            state->to_fmt = ULAW;
        tfmt = state->to_fmt;
    }
    else return mas_error( MERR_NOTDEF );

    /*
     * make a stab at the dc of the other port, given the port we've
     * configured
     */
    if ( ( ( state->from_fmt != 0 ) && ( state->to_fmt == 0 ) ) ||
         ( ( state->from_fmt == 0 ) && ( state->to_fmt != 0 ) ) )
    {
        odc = MAS_NEW( odc );
        masc_setup_dc( odc, 5 );

        /* these parameters are independent of the format */
        masc_append_dc_key_value( odc, "endian", "host" );
        masc_append_dc_key_value( odc, "sampling rate", dc->values[ratek] );
        masc_append_dc_key_value( odc, "channels", dc->values[chk] );

        /* resolution and format depend on the format of the port
         * we're configuring in this call. */
        if ( tfmt == LIN16 )
        {
            masc_append_dc_key_value( odc, "resolution", "8" );
            masc_append_dc_key_value( odc, "format", "ulaw" );
        }
        if ( tfmt == ULAW )
        {
            masc_append_dc_key_value( odc, "resolution", "16" );
            masc_append_dc_key_value( odc, "format", "linear" );
        }

        /* set the other data characteristic */
        if ( state->from_fmt )
            masd_set_data_characteristic( state->source, odc );
        else
            masd_set_data_characteristic( state->sink, odc );
    } 
    else if ( ( state->from_fmt != 0 ) && ( state->to_fmt != 0 ) )
    {
        /* If both ports are configured... */
        /* Make sure we're actually doing work.  If we're not, bail.*/
        if ( state->from_fmt == state->to_fmt )
            return mas_error(MERR_INVALID);

        /* use the from_fmt from now on to decide how to decode. */
        
        /* schedule our dataflow dependency on data_sink */
        dataflow_port_dependency = masc_rtalloc( sizeof (int32) );
        *dataflow_port_dependency = state->sink;
        err = masd_reaction_queue_action(state->reaction, device_instance, 
                                         "mas_codec_ulaw_convert", 0, 0, 0, 0, 0,
                                         MAS_PRIORITY_DATAFLOW, 1, 1, 
                                         dataflow_port_dependency);
        if ( err < 0 ) return err;
    }
    

    return 0;
}

int32
mas_dev_disconnect_port( int32 device_instance, void* predicate )
{
    /* nothing to do? */
    return 0;
}

int32
mas_dev_exit_instance( int32 device_instance, void* predicate )
{
    struct codec_ulaw_state*  state;
    
    MASD_GET_STATE(device_instance, state);
    masc_rtfree( state );
    
    return 0;
}

int32
mas_dev_terminate( int32 device_instance, void* predicate )
{
    return 0;
}

int32
mas_dev_show_state( int32 device_instance, void* predicate )
{
    return mas_error(MERR_NOTIMP);
}

int32
mas_codec_ulaw_convert( int32 device_instance, void* predicate )
{
    struct codec_ulaw_state* state;
    struct mas_data*     data;
    int                  i;
    uint8*               b_ulaw;
    int16*               b_lin16;
    
    
    MASD_GET_STATE(device_instance, state);
    masd_get_data( state->sink, &data );

    if ( state->from_fmt == ULAW )
    {
        /* do ulaw to linear 16 */
        /** create a new data segment with 2x the memory.  We need to
         ** free the old when we're done. */
        b_ulaw = (uint8*) data->segment;
        b_lin16 = masc_rtalloc( data->length * 2 );

        for (i=0; i<data->length; i++)
            b_lin16[i] = ulaw_to_lin16_table[b_ulaw[i]];
        data->allocated_length = data->length*2;
        data->length = data->allocated_length;
        data->segment = (char*) b_lin16;
        masc_rtfree( b_ulaw );
    }
    else
    {
        /* do linear 16 to ulaw */
        /** reuse the allocated data - no sense in changing it, since
         ** we're going to use half the memory. */
        b_ulaw = (uint8*) data->segment;
        b_lin16 = (int16*) data->segment;

        for (i=0; i<data->length/2; i++)
        {
            b_ulaw[i] = lin16_to_ulaw_table[((uint16)b_lin16[i]) >> 2];
        }

        data->length = data->length/2;
    }
    
    masd_post_data( state->source, data );

    return 0;
}
