/*
 * Copyright (c) 2001-2003 Shiman Associates Inc. All Rights Reserved.
 * 
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */

#include <string.h>
#include <stdlib.h>
#include "mas/mas_common.h"

/***************************************************************************
 * masc_trim_string
 *
 * arguments:
 *  1. char* string
 *
 *  Trim the whitespace from the head and tail of the supplied string.
 *  This does in-place operation with shifting; it can't be used on
 *  const strings
 *
 * returns: void
 *
 ***************************************************************************/
void
masc_trim_string(char* string)
{
    int i = 0, j = 0;

    /* trim end first */
    while ( string[i] != 0 ) i++; /* find end */
    j = i;

    /* find last non-whitespace char, double-quote or single-quote */
    while ( ( string[--i] == ' ' || string[i] == '"'
	      || string[i] == '\'' ) && i > 0);
    string[i+1] = 0; /* and terminate after it */

    /* now, trim the beginning */
    i = 0; j = 0;
    while ( ( string[i] == ' ' || string[i] == '"'
	      || string[i] == '\'' ) && string[i++] != 0 );
    
    if (i) /* if there's whitespace to trim */
    {
	while ( string[j] != 0 ) /* copy */
	{
	    string[j] = string[i];
	    i++; j++;
	}
    }
}

/* at some point, this will have to handle atoms */
/* or we could require "strings" to be a sorted list */
int32
masc_get_string_index( char* sut, char** strings, int num )
{
    int32 i;

    for (i=0; i<num; i++)
    {
        if ( strncmp( sut, strings[i], MAX_STRING_LENGTH) == 0 )
            return i;
    }

    return mas_error(MERR_NOTDEF);
}

size_t
masc_strlcpy(char *dst, const char *src, size_t size)
{
    int i;

    for (i=0; i<size-1 && *src != 0; i++)
        *(dst++) = *(src++);

    *dst = 0;

    for ( ; *src != 0; i++ )
        src++;
    
    return i;
}

size_t
masc_strlcat(char *dst, const char *src, size_t size)
{
    int i;
    
    for (i=0; i<size && *dst != 0; i++)
        dst++;

    if ( *dst != 0 )
        return i;

    for ( ; i<size-1 && *src != 0; i++)
        *(dst++) = *(src++);

    *dst = 0;

    for ( ; *src != 0; i++ )
        src++;

    return i;
}


char*
masc_construct_title( const char *app_name )
{
    char *mas_host;
    char *host;
    char *title;
    int len;
    
    mas_host = getenv( "MAS_HOST" );
    host = getenv( "HOSTNAME" );

    if(!host)
    {
        host = "localhost";
    }
    
    if(!mas_host)
    {
        mas_host = host;
    }

    len = 1 + strlen( mas_host ) + strlen( host ) + strlen( app_name ) + strlen( "@ - " );
    title = masc_rtalloc( len );

    if( title )
    {
        masc_strlcpy( title, mas_host, len );
        masc_strlcat( title, " - ", len );
        masc_strlcat( title, app_name, len );
        masc_strlcat( title, "@", len );
        masc_strlcat( title, host, len );
    }

    return title;
}
