/*!**************************************************************************

    module      : vkb641.cpp

    -------------------------------------------------------------------------

    author      : JuergenA
    responsible : UweH

    special area: Logging
    description : KB_CreateDropFile_interface

    last changed: 2001-03-12

    -------------------------------------------------------------------------

    copyright:    (c) 2001-2004 SAP AG


    ========== licence begin  GPL
    Copyright (c) 2001-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end



*****************************************************************************/

/*==========================================================================*
*  INCLUDES                                                                 *
*===========================================================================*/

#include "SAPDBCommon/SAPDB_Types.hpp"
#include "Logging/Log_ActionCreateDropFile.hpp"
#include "Logging/Log_ActionReorganizeFile.hpp"
#include "Logging/Log_ActionUnloadFile.hpp"
#include "Logging/Log_ActionReloadFile.hpp"
#include "Logging/Log_ActionSetFileState.hpp"
#include "Logging/Log_InvDescMap.hpp"
#include "Logging/Log_Transaction.hpp"
#include "gsp00.h"
#include "ggg91.h"
#include "ggg00.h"
#include "hbd01.h"
#include "hkb560.h"
#include "hkb641.h" // check of PASCAL header vkb641


/*==========================================================================*
* local functions                                                           *
*===========================================================================*/

/*-----------------------------------------------------------------------------------*/

static inline Log_Transaction&
GetLogTrans (tgg00_TransContext &Trans,
                   const char *        Msg)
{
    Log_Transaction* LogTrans = REINTERPRET_CAST(Log_Transaction*,Trans.trLogContext_gg00);
    if ( LogTrans == NULL )
        RTE_Crash( SAPDBErr_Exception(__FILE__, __LINE__,
                                      SAPDBERR_ASSERT_STATE_FAILED, Msg) );
    return *LogTrans;
}

/*==========================================================================*
* PASCAL INTERFACE                                                          *
*===========================================================================*/

//----------------------------------------------------------------------------

externPascal void kb641BeginLoad (
    tgg00_TransContext    VAR_VALUE_REF  trans,
    tgg00_FileId          VAR_VALUE_REF  fileid,
    tgg00_Lkey            VAR_VALUE_REF  lastkey)
{
    SAPDB_Byte* pKey = reinterpret_cast<SAPDB_Byte*>( &(lastkey.keyVal_gg00()) );
    Log_ActionReorganizeFile beforeimage ( fileid.fileName_gg00(),
                                           lastkey.keyLen_gg00(),
                                           pKey );
    beforeimage.WriteBeforeImage (trans);
}

//----------------------------------------------------------------------------

externPascal void kb641BeginLoadLong (
    tgg00_TransContext    VAR_VALUE_REF  trans,
    tgg00_FileId          VAR_VALUE_REF  fileid)
{
    Log_ActionDropFile action ( fileid.fileName_gg00() );
    action.WriteBeforeImage (trans);
}

//----------------------------------------------------------------------------

externPascal void kb641ReloadFile (
    tgg00_TransContext    VAR_VALUE_REF  TransContext,
    tgg00_FileId          VAR_VALUE_REF  FileId)
{
    Log_Transaction&  logtrans = GetLogTrans (TransContext, "kb641ReloadFile");
    tgg00_BasisError &trError  = TransContext.trError_gg00;

    logtrans.BeginSubtrans();

    Log_ActionUnloadFile beforeimage ( FileId.fileName_gg00() );
    
    beforeimage.WriteBeforeImage (TransContext);

    if ( trError != e_ok )
    {
        if ( ! logtrans.EndSubtrans() )
            trError = e_no_open_subtrans;
        return;
    }
    
    Log_ActionReloadFile afterimage ( FileId.fileName_gg00() );

    afterimage.ExecuteAction (TransContext, &FileId );

    if  ( trError != e_ok )
    {
        tgg00_BasisError auxError = trError;
        trError = e_ok;

        Log_Transaction::RollbackResult result =
            logtrans.Rollback (Log_Transaction::subTrans, TransContext.trSubtransId_gg00);
        
        trError = kb560ConvertRollbackResultToBasisError (result);

        if ( e_ok == trError )
            trError = auxError;
        return;
    }
    
    if ( ! logtrans.EndSubtrans() )
    {
        trError = e_no_open_subtrans;
        return;
    }

    afterimage.WriteAfterImage(TransContext);
}

//----------------------------------------------------------------------------

externPascal void kb641UnloadFile (
    tgg00_TransContext    VAR_VALUE_REF  TransContext,
    tgg00_FileId          VAR_VALUE_REF  FileId,
    tgg00_FileId          VAR_VALUE_REF  AuxFileId)
{
    Log_Transaction&  logtrans = GetLogTrans (TransContext, "kb641UnloadFile");
    tgg00_BasisError &trError  = TransContext.trError_gg00;

    Log_ActionDropFile beforeimage (FileId.fileName_gg00());

    logtrans.BeginSubtrans();

    beforeimage.WriteBeforeImage (TransContext);

    if ( trError != e_ok )
    {
        if ( ! logtrans.EndSubtrans() )
            trError = e_no_open_subtrans;
        return;
    }
    
    Log_ActionUnloadFile afterimage (FileId.fileName_gg00(),
                                     AuxFileId.fileName_gg00() );
    
    afterimage.ExecuteAction( TransContext, &AuxFileId, &FileId );
    
    if  ( trError != e_ok )
    {
        tgg00_BasisError auxError = trError;
        trError = e_ok;

        Log_Transaction::RollbackResult result =
            logtrans.Rollback (Log_Transaction::command, LOG_NIL_SUBTRANS_NO);
        
        trError = kb560ConvertRollbackResultToBasisError (result);

        if ( e_ok == trError )
            trError = auxError;
        return;
    }
    
    if ( ! logtrans.EndSubtrans() )
    {
        trError = e_no_open_subtrans;
        return;
    }
    
    afterimage.WriteAfterImage (TransContext);
}

//----------------------------------------------------------------------------

externPascal void kb641ChangeFileType (tgg00_TransContext      &TransContext,
                                       tgg00_FileId            &FileId,
                                       const tgg00_FiletypeSet &NewFiletypeSet)
{
    Log_ActionChangeFileType
        BeforeImage (FileId.fileName_gg00(), NewFiletypeSet, FileId.fileType_gg00());
    
    Log_ActionChangeFileType
        AfterImage (FileId.fileName_gg00(), FileId.fileType_gg00(), NewFiletypeSet);
    
    BeforeImage.WriteBeforeImage (TransContext);
    
    if (e_ok != TransContext.trError_gg00) return;
    
    AfterImage.ExecuteAction (TransContext, &FileId);
    
    if (e_ok != TransContext.trError_gg00) return;
    
    AfterImage.WriteAfterImage (TransContext);
}

//----------------------------------------------------------------------------

externPascal void kb641CreateFile (tgg00_TransContext &TransContext,
                                   tgg00_FileId       &FileId)
{
    Log_ActionDropFile BeforeImage (FileId.fileName_gg00());
    
    Log_ActionCreateFile
        AfterImage (FileId.fileName_gg00(), FileId.fileVersion_gg00(), FileId.fileType_gg00());
    
    BeforeImage.WriteBeforeImage (TransContext);
    
    if (e_ok != TransContext.trError_gg00) return;
    
    AfterImage.ExecuteAction (TransContext, &FileId);
    
    if (e_ok != TransContext.trError_gg00) return;
    
    AfterImage.WriteAfterImage (TransContext);
}

//----------------------------------------------------------------------------

externPascal void kb641CreateIndex (tgg00_TransContext    &TransContext,
                                    tgg00_FileId          &PrimFileId,
                                    tgg00_FileId          &InvFileId,
                                    const tgg00_StackDesc &StackDesc,
                                    tgg00_IndexCounter    &IndexCounter)
{
    Log_InvDescMap InvDescMap;
    
    InvDescMap.Assign (
        &((*StackDesc.mst_addr())[StackDesc.mmult_pos-1]),
        StackDesc.mmult_cnt, 
        true); // LookForUniqueEntries
    
    Log_ActionDropFile BeforeImage (InvFileId.fileName_gg00());
    
    Log_ActionCreateIndex AfterImage (
        InvFileId.fileName_gg00(),
        &InvDescMap,
        PrimFileId.fileTfn_gg00(),
        PrimFileId.fileTfnNo_gg00()[0]);
    
    BeforeImage.WriteBeforeImage (TransContext);
    
    if (e_ok != TransContext.trError_gg00) return;
    
    AfterImage.ExecuteAction (TransContext, &PrimFileId, &InvFileId, &IndexCounter);
    
    if (e_ok != TransContext.trError_gg00) return;
    
    AfterImage.WriteAfterImage (TransContext);
}

//----------------------------------------------------------------------------

externPascal void kb641CreateTable (tgg00_TransContext &TransContext,
                                    tgg00_FileId       &FileId,
                                    pasbool             NoTreeIndex,
                                    tsp00_Int2          StrColCnt,
                                    tsp00_Int2          VarColCntOff)
{
    Log_ActionDropFile    BeforeImage (FileId.fileName_gg00());
    
    Log_ActionCreateTable AfterImage (
        FileId.fileName_gg00(),
        FileId.fileVersion_gg00(),
        FileId.fileType_gg00(),
        NoTreeIndex,
        StrColCnt,
        VarColCntOff);
    
    BeforeImage.WriteBeforeImage (TransContext);
    
    if (e_ok != TransContext.trError_gg00) return;
    
    AfterImage.ExecuteAction (TransContext, &FileId);
    
    if (e_ok != TransContext.trError_gg00) return;
    
    AfterImage.WriteAfterImage (TransContext);
} 

//----------------------------------------------------------------------------

externPascal void kb641RenameFile (tgg00_TransContext &TransContext,
                                   tgg00_FileId       &OldFileId,
                                   tgg00_FileId       &NewFileId)
{
    Log_ActionRenameFile BeforeImage 
        (NewFileId.fileName_gg00(), OldFileId.fileName_gg00(), OldFileId.fileVersion_gg00());
    
    Log_ActionRenameFile AfterImage
        (OldFileId.fileName_gg00(), NewFileId.fileName_gg00(), NewFileId.fileVersion_gg00());
    
    BeforeImage.WriteBeforeImage (TransContext);
    
    if (e_ok != TransContext.trError_gg00) return;
    
    AfterImage.ExecuteAction (TransContext, &OldFileId, &NewFileId);
    
    if (e_ok != TransContext.trError_gg00) return;
    
    AfterImage.WriteAfterImage (TransContext);
}

//----------------------------------------------------------------------------

externPascal void kb641SetFileState (
    tgg00_TransContext    VAR_VALUE_REF  TransContext,
    tgg00_FileId          VAR_VALUE_REF  FileId,
    pasbool                              readonly)
{
    Log_ActionSetFileState BeforeImage (FileId.fileName_gg00(), 1!=readonly);
    Log_ActionSetFileState AfterImage  (FileId.fileName_gg00(), 1==readonly);
    
    BeforeImage.WriteBeforeImage (TransContext);
    
    if (e_ok != TransContext.trError_gg00) return;
    
    AfterImage.ExecuteAction (TransContext, &FileId, 1==readonly);

    if (e_ok != TransContext.trError_gg00) return;
    
    AfterImage.WriteAfterImage (TransContext);
}

//----------------------------------------------------------------------------

externPascal void kb641SuccFileVersion (tgg00_TransContext      &TransContext,
                                        tgg00_FileId            &FileId,
                                        const tgg91_FileVersion &NewFileVersion,
                                        tsp00_Int4               OldLongCnt,
                                        tsp00_Int4               NewLongCnt)
{
    Log_ActionSuccFileVersion
        BeforeImage (FileId.fileName_gg00(), FileId.fileVersion_gg00(), OldLongCnt);
    
    Log_ActionSuccFileVersion
        AfterImage (FileId.fileName_gg00(), NewFileVersion, NewLongCnt);
    
    BeforeImage.WriteBeforeImage (TransContext);
    
    if (e_ok != TransContext.trError_gg00) return;
    
    AfterImage.ExecuteAction (TransContext, &FileId);
    
    if (e_ok != TransContext.trError_gg00) return;
    
    AfterImage.WriteAfterImage (TransContext);
}

//----------------------------------------------------------------------------

externPascal void kb641UpdFileStatistic (tgg00_TransContext &TransContext,
                                         tgg00_FileId       &FileId,
                                         tsp00_Int4          OldLeafNodes)
{
    Log_ActionUpdFileStatistic BeforeImage (FileId.fileName_gg00(), OldLeafNodes);
    
    Log_ActionUpdFileStatistic AfterImage  (FileId.fileName_gg00(), FileId.fileLeafNodes_gg00());
    
    BeforeImage.WriteBeforeImage (TransContext);
    
    if (e_ok != TransContext.trError_gg00) return;
    
    AfterImage.ExecuteAction (TransContext, &FileId);
    
    if (e_ok != TransContext.trError_gg00) return;
    
    AfterImage.WriteAfterImage (TransContext);
}

/*===========================================================================*
*  END OF CODE                                                               *
*============================================================================*/
