/*      Copyright (C) 2001, 2002, 2003, 2004 Stijn van Dongen
 *
 * This file is part of MCL.  You can redistribute and/or modify MCL under the
 * terms of the GNU General Public License; either version 2 of the License or
 * (at your option) any later version.  You should have received a copy of the
 * GPL along with MCL, in the file COPYING.
*/


#include <string.h>
#include <stdio.h>

#include "report.h"

#include "impala/matrix.h"
#include "impala/vector.h"
#include "impala/scan.h"
#include "impala/io.h"
#include "impala/app.h"
#include "impala/iface.h"
#include "mcl/interpret.h"
#include "mcl/clm.h"

#include "util/io.h"
#include "util/err.h"
#include "util/types.h"
#include "util/opt.h"
#include "util/minmax.h"


const char* usagelines[] =
{  "Usage: clminfo [options] <mx file> <cl file> [<cl file>*]"
,  "  where mx file encodes a graph (in MCL matrix format)"
,  "  and each cl file encodes a clustering of it (in MCL matrix format)."
,  "Options:"
,  "  -do <str>      <str> in {header, rule} (output mode)"
,  "  -ap            append performance measures to cluster file"
,  "  -ag            append granularity measures to file"
,  "  --adapt        adapt matrix if domains don't fit"
,  "  -tag <str>     write 'tag=<str>' in performance measure"
,  "  -pi <num>      apply inflation to input matrix before measuring"
,  NULL
}  ;


int main
(  int                  argc
,  const char*          argv[]
)
   {  mcxIO *xfmx    =  NULL
   ;  mclMatrix *cl  =  NULL
   ;  mclMatrix *mx  =  NULL
   ;  const char* me =  "clminfo"
   ;  const char* tag=  NULL
   ;  double inflation = 0.0
   ;  int preprune = 0
   ;  int do_header = 0
   ;  int do_rule = 0

   ;  int append_granul = 0
   ;  int append_perf = 0

   ;  int a =  1
   ;  int n_err = 0
   ;  int adapt = 0
   ;  int duh = 0
   ;  const char* arg
   ;  mcxTing* ginfo = mcxTingEmpty(NULL, 40)

   ;  mclxIOsetQMode("MCLXIOVERBOSITY", MCL_APP_VB_YES)

   ;  if (argc == 1)
      goto help

   ;  while (a < argc)
      {  if (!strcmp(argv[a], "-h"))
         {  help
         :  mcxUsage(stdout, me, usagelines)
         ;  mcxExit(0)
      ;  }

         else if (!strcmp(argv[a], "--adapt"))
         adapt = 1

      ;  else if (!strcmp(argv[a], "-ap"))
         append_perf = 1

      ;  else if (!strcmp(argv[a], "-ag"))
         append_granul = 1

      ;  else if
         (  ((!strcmp(argv[a], "-do")) && (duh = 1))
         || ((duh = 0) || (!strcmp(argv[a], "-no")))
         )
         {  if (a++ + 1 >= argc)
            goto arg_missing
         ;  arg = argv[a]
         ;  if (!strcmp(arg, "header"))
            do_header = duh
         ;  else if (!strcmp(arg, "rule"))
            do_rule = duh
      ;  }
         else if (!strcmp(argv[a], "-tag"))
         {  if (a++ + 1 >= argc)
            goto arg_missing
         ;  tag = argv[a]
      ;  }
         else if (!strcmp(argv[a], "-pi"))
         {  if (a++ + 1 >= argc)
            goto arg_missing
         ;  inflation = atof(argv[a])
      ;  }
         else if (!strcmp(argv[a], "--version"))
         {  app_report_version(me)
         ;  exit(0)
      ;  }
         else if (!strcmp(argv[a], "-foo"))
         {  if (a++ + 1 >= argc)
            goto arg_missing
         ;  if (strcmp(argv[a], "bar"))
               mcxErr(me, "foo needs bar")
            ,  mcxExit(1)
      ;  }
         else if (0)
         {  arg_missing:
            mcxErr
            (  me
            ,  "flag <%s> needs argument (see -h for usage)"
            ,  argv[argc-1]
            )
         ;  mcxExit(1)
      ;  }
         else if (a >= argc-1)
         {  mcxErr(me, "Need at least matrix file and cluster file")
         ;  mcxExit(1)
      ;  }
         else
         {  break
      ;  }
         a++
   ;  }

      xfmx     =  mcxIOnew(argv[a++], "r")
   ;  mx       =  mclxRead(xfmx, EXIT_ON_FAIL)
   ;  report_graph_or_exit(me, mx, xfmx->fn)

   ;  if (tag)
      mcxTingPrint
      (  ginfo
      ,  "tag=%s\n"
      ,  tag
      )

   ;  mcxTingPrintAfter
      (  ginfo
      ,  "target-name=%s\n"
      ,  xfmx->fn->str
      )

   ;  if (inflation)
      {  mclxInflate(mx, inflation)
      ;  mcxTingPrintAfter(ginfo, "inflation=%.2f\n", (double) inflation)
   ;  }

      if (preprune)
      {  mclxMakeSparse(mx, preprune)
      ;  mcxTingPrintAfter(ginfo, "preprune=%df\n", (int) preprune)
   ;  }

      mcxIOfree(&xfmx)

   ;  while(a < argc)
      {  mcxIO *xfcl =  mcxIOnew(argv[a], "r")
      ;  mclMatrix* tmx = NULL
      ;  int o, m, e
      ;  cl =  mclxRead(xfcl, EXIT_ON_FAIL)

      ;  if (!mcldEquate(mx->dom_cols, cl->dom_rows, MCLD_EQ_EQUAL))
         {  mclVector* meet = mcldMeet(mx->dom_cols, cl->dom_rows, NULL)
         ;  mcxErr
            (  me
            ,  "Domain mismatch for matrix ('left') and clustering ('right')"
            )
         ;  report_domain(me, N_ROWS(mx), N_ROWS(cl), meet->n_ivps)
         ;  if (adapt)
            {  mclMatrix* tcl = mclxSub(cl, cl->dom_cols, meet)
            ;  report_fixit(me, n_err++)
            ;  mclxFree(&cl)
            ;  cl  = tcl
            ;  tmx = mx
            ;  mx  = mclxSub(tmx, meet, meet)
         ;  }
            else
            report_exit(me, SHCL_ERROR_DOMAIN)
      ;  }

         if (mclcEnstrict(cl, &o, &m, &e, ENSTRICT_TRULY))
         {  report_partition(me, cl, xfcl->fn, o, m, e)
         ;  if (adapt)
            report_fixit(me, n_err++)
         ;  else
            report_exit(me, SHCL_ERROR_PARTITION)
      ;  }

         mcxIOclose(xfcl)

      ;  {  double vals[8]
         ;  mcxTing* linfo = mcxTingNew(ginfo->str)
         ;  mcxTingPrintAfter(linfo, "source-name=%s\n", xfcl->fn->str)

         ;  if (do_rule)
            {  mcxTing* rule = mcxTingKAppend(NULL, "=", 78)
            ;  mcxTingPrintSplice(rule, 0, TING_INS_CENTER, " %s ", xfcl->fn->str)
            ;  fprintf(stdout, "%s\n", rule->str)
            ;  mcxTingFree(&rule)
         ;  }

            if (append_granul || append_perf)
            {  mcxIOrenew(xfcl, NULL, "a")
            ;  if (mcxIOopen(xfcl, RETURN_ON_FAIL))
               {  mcxErr(me, "cannot open file %s for appending", xfcl->fn->str)
               ;  append_granul = 0
               ;  append_perf = 0
            ;  }
               else
               mcxTell
               (me, "appending performance measures to file %s", xfcl->fn->str)
         ;  }

            clmGranularity(cl, vals)
         ;  clmGranularityPrint
            (stdout, do_header ? linfo->str : NULL, vals, do_header)
         ;  if (append_granul)
            clmGranularityPrint (xfcl->fp, linfo->str, vals, 1)

         ;  clmPerformance(mx, cl, vals)
         ;  clmPerformancePrint
            (stdout, do_header ? linfo->str : NULL, vals, do_header)
         ;  if (append_perf)
            clmPerformancePrint (xfcl->fp, linfo->str, vals, 1)
      ;  }

         mclxFree(&cl)
      ;  if (tmx)
         {  mclxFree(&mx)
         ;  mx = tmx
      ;  }

         mcxIOfree(&xfcl)
      ;  a++
   ;  }

      mclxFree(&mx)
   ;  return 0
;  }


