/*
 * medussa - a distributed cracking system
 * Copyright (C) 1999 Kostas Evangelinos <kos@bastard.net>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*
 * $Id: binary.c,v 1.2 2003/02/05 04:38:37 kos Exp $
 *
 * Bruteforce keyspace generator, binary flavor
 */

#include <stdio.h>
#include <string.h>
#include <math.h>
#include <stdlib.h>

#include "common.h"
#include "keyspace.h"
#include "xmalloc.h"
#include "generator.h"
#include "binary.h"

key_index_t *
binary_minindex(binary_t *b) {
  return &b->minindex;
}

key_index_t *
binary_maxindex(binary_t *b) {
  return &b->maxindex;
}

key_index_t *
binary_curindex(binary_t *b) {
  return &b->index;
}

int
binary_minlen(binary_t *b) {
  return b->len;
}

int
binary_maxlen(binary_t *b) {
  return b->len;
}

int
binary_destroy(binary_t *k) {
  keyspace_destroy(k->divisor);
  keyspace_destroy(k->temp);
  keyspace_destroy(k->minindex);
  keyspace_destroy(k->maxindex);
  keyspace_destroy(k->index);
  xfree(k);
  return 0;
}

binary_t *
binary_init(char *params) {
  binary_t *k;

  k = (binary_t *)xcalloc(1, sizeof(binary_t));

  keyspace_init(k->minindex);
  keyspace_init(k->maxindex);
  keyspace_init(k->index);
  keyspace_init(k->divisor);
  keyspace_init(k->temp);
  keyspace_fromint(0x100, k->divisor);

  param_get_int(params, "len", &k->len, BIN_DEFLEN);

  k->minlen = k->maxlen = k->len;
  keyspace_fromint(0, k->minindex);
  keyspace_fromint(2, k->maxindex);
  keyspace_pow(k->maxindex, k->len*8, k->maxindex);
  keyspace_set(k->index, k->minindex);  
  binary_set(k, k->index);

  return k;
}

/* Efficient next index operation */
int
binary_fetch(binary_t *k, kchar *buf, int userlen, int *len) {
  int i;
  int j;

  for(i=0; i<k->len; i++) {
    keyspace_pow(k->temp, i, k->divisor);
    keyspace_div(k->temp, k->index, k->temp);
    j = keyspace_toint(k->temp) & 0xff;
    buf[i] = (kchar)j;
  }
  
  *len = k->len;
  keyspace_inc(k->index);
  return 0;
}

int
binary_set(binary_t *k, key_index_t index) {

  if(keyspace_cmp(index, k->minindex) < 0 || keyspace_cmp(index, k->maxindex) > 0)
    return 1;

  keyspace_set(k->index, index);
  return 0;
}

int 
binary_done(binary_t *k) {
  return (keyspace_cmp(k->index, k->maxindex) == 0);
}

generator_impl_t binary_impl = {
  "binary",
  (gen_init)binary_init,
  (gen_set)binary_set,
  (gen_fetch)binary_fetch,
  (gen_done)binary_done,
  (gen_destroy)binary_destroy,
  (gen_minindex)binary_minindex,
  (gen_maxindex)binary_maxindex,
  (gen_curindex)binary_curindex,
  (gen_minlen)binary_minlen,
  (gen_maxlen)binary_maxlen
};

#ifdef BINARY_DEBUG
int
main(int argc, char **argv) {
  binary_t *k;
  char buffer[1024];
  
  k = binary_init(1, 5, "test01");

  while(!binary_done(k)) {
    binary_fetch(k, buffer, 1024);
    printf("%d ", k->len);
    printf("%s ", keyspace_ntoa(*binary_curindex(k)));
    printf("%s ", keyspace_ntoa(*binary_minindex(k)));
    printf("%s ", keyspace_ntoa(*binary_maxindex(k)));
    printf("%s\n", buffer);
  }
  exit(0);
}

#endif
