/*
 * medussa - a distributed cracking system
 * Copyright (C) 1999 Kostas Evangelinos <kos@bastard.net>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*
 * $Id: method.c,v 1.14 2004/06/09 01:59:28 kos Exp $
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <sys/types.h>
#include <stdio.h>
#include <string.h>

#include "method.h"
#include "common.h"
#include "xmalloc.h"

extern method_impl_t unixcrypt_impl;
extern method_impl_t md5raw_impl;
extern method_impl_t md2raw_impl;
extern method_impl_t md5_impl;
extern method_impl_t sharaw_impl;
extern method_impl_t domino_impl;

method_impl_t *method_impl[] = {
  &unixcrypt_impl,
  &md5raw_impl,
  &md2raw_impl,
  &md5_impl,
  &sharaw_impl,
  &domino_impl,
  (method_impl_t *)NULL
};

method_t *
method_init(char *name, char *opts) {
  method_t *m;
  int i;

  m = (method_t *)xcalloc(1, sizeof(method_t));

  for(i=0; method_impl[i]; i++) {
    if(!strcmp(method_impl[i]->name, name)) {
      m->impl = method_impl[i];
      if(!(m->context = m->impl->init(opts))) {
	xfree(m);
	return (method_t *)NULL;
      }
      strncpy(m->name, name, METH_LINELEN);
      if(opts)
	strncpy(m->opts, opts, METH_LINELEN);
      else
	strcpy(m->opts, "");
      return m;
    }
  }
  xfree(m);
  return (method_t *)NULL;
}

int
method_query(int index, char *buf, int len) {
  int i;

  i = 0;
  while(method_impl[i] && i < index)
    i++;

  if(!method_impl[i])
    return 1;

  strncpy(buf, method_impl[i]->name, len);
  return 0;
}

#ifdef METHOD_DEBUG

#include <stdio.h>
#include <getopt.h>
#include <string.h>
#include "llog.h"

void
usage(char *argv0, int code) {
  fprintf(stderr, "%s: Insert usage here\n", argv0);
  exit(code);
}

int
dostuff(char *type, char *opts) {
  method_t *m;
  char key[METH_LINELEN];
  int i;
  int limit;
  time_t now;
  time_t then;
  int cps;
  
  if(!(m=method_init(type, opts))) {
    llog(1, "method_init(%s, %s): failed\n", type, opts);
    return 1;
  }

  strcpy(key, "test");
  limit = method_siz(m);
  method_sethash(m, "noway");

  llog(1, "preparing for %d crypts...\n", limit);
  cps=0;
  now = time(NULL);
  for(i=0; i<limit; i++)
    method_add(m, key);
  if(method_crypt(m)) {
    llog(1, "strange: crypt() actually found the key\n");
  } else {
    then = time(NULL);
    cps = limit / ((int)(then-now));
    llog(1, "crypt() finished. %d crypts per second\n", cps);
  }
  return 0;
}

int
main(int argc, char **argv) {
  char c;
  int verbose;
  char type[METH_LINELEN];
  char opts[METH_LINELEN];

  strcpy(type, "unixcrypt");
  strcpy(opts, "");
  verbose = 9;

  while((c = getopt(argc, argv, "v:o:t:")) != EOF) {
    switch(c) {
    case 'v':
      verbose = atoi(optarg);
      break;
    case 't':
      strncpy(type, optarg, METH_LINELEN);
      break;
    case 'o':
      strncpy(opts, optarg, METH_LINELEN);
      break;
    case '?':
      usage(argv[0], 1);
      break;
    }
  }

  llog_init(LLOG_STDERR);
  llog_level(9);
  llog(1, "method %s, options %s\n", type, opts);
  dostuff(type, opts);  
  exit(0);
}

#endif /* METHOD_DEBUG */
