/* Mergeant
 *
 * Copyright (C) 1999 - 2004 Vivien Malerba
 * Copyright (C) 2002 - 2003 Rodrigo Moya
 *
 * Authors:
 *       Rodrigo Moya <rodrigo@gnome-db.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <bonobo/bonobo-i18n.h>
#include <gtk/gtklabel.h>
#include <gtk/gtkmenuitem.h>
#include <gtk/gtkstock.h>
#include <gtk/gtktable.h>
#include <gtk/gtkvpaned.h>
#include <libgnomedb/gnome-db-gray-bar.h>
#include <libgnomedb/gnome-db-util.h>
#include <libgnomedb/gnome-db-editor.h>
#include <libmergeant/libmergeant.h>
#include "workspace.h"
#include <string.h>
#include "query-druid.h"
#include "query-editor.h"

#include "workspace-page.h"
#include "ws-tables.h"
#include "ws-datatypes.h"
#include "ws-queries.h"
#include "ws-dbrels.h"


struct _WorkspacePrivate {
	MgServer  *server;
	
	guint      timeout_event_conf_save; /* returned by g_timeout_add() */
	guint      idle_event_conf_save; /* returned from g_idle_add() */

	/* main page */
	GtkWidget *main_page_pane;
	GtkWidget *main_page_object_type;     /* option menu */
	GtkWidget *main_page_object_list;     /* notebook for selectors */
	GtkWidget *main_page_detail_table;
	GtkWidget *main_page_detail_label;    /* gray bar */
	GtkWidget *main_page_detail_contents; /* notebook for the work areas */
	guint      main_page_detail_contents_object_type;

	GSList    *pages_objects;
};

static void workspace_class_init (WorkspaceClass *klass);
static void workspace_init (Workspace *wk, WorkspaceClass *klass);
static void workspace_dispose (GObject *object);
static void workspace_finalize (GObject *object);

static void create_widgets (Workspace *wk);


static GObjectClass *parent_class = NULL;

static void
workspace_class_init (WorkspaceClass *klass)
{
	GObjectClass *object_class;

	parent_class = g_type_class_peek_parent (klass);

	object_class = G_OBJECT_CLASS (klass);
	object_class->dispose = workspace_dispose;
	object_class->finalize = workspace_finalize;
}

static void
workspace_init (Workspace *wk, WorkspaceClass *klass)
{
	/* allocate private structure */
	wk->priv = g_new0 (WorkspacePrivate, 1);
}

static void conf_changed_ch (MgConf *conf, Workspace *wk);
static void
workspace_dispose (GObject *object)
{
	Workspace *wk = (Workspace *) object;

	/* free memory */
	if (wk->priv) {
		GSList *list = wk->priv->pages_objects;
		while (list ) {
			g_object_unref (G_OBJECT (list->data));
			list = g_slist_next (list);
		}

		if (wk->priv->pages_objects) {
			g_slist_free (wk->priv->pages_objects);
			wk->priv->pages_objects = NULL;
		}
	}

	if (parent_class->dispose)
		parent_class->dispose (object);
}

static void
workspace_finalize (GObject *object)
{
	Workspace *wk = (Workspace *) object;

	/* free memory */
	if (wk->priv) {
		MgConf *conf = mg_server_get_conf (wk->priv->server);
		g_signal_handlers_disconnect_by_func (G_OBJECT (conf),
						      G_CALLBACK (conf_changed_ch), wk);

		g_free (wk->priv);
		wk->priv = NULL;
	}
	
	if (parent_class->finalize)
		parent_class->finalize (object);
}

GType
workspace_get_type (void)
{
        static GType type = 0;
                                                                                    
        if (!type) {
                static const GTypeInfo info = {
                        sizeof (WorkspaceClass),
                        (GBaseInitFunc) NULL,
                        (GBaseFinalizeFunc) NULL,
                        (GClassInitFunc) workspace_class_init,
                        NULL,
                        NULL,
                        sizeof (Workspace),
                        0,
                        (GInstanceInitFunc) workspace_init
                };
                type = g_type_register_static (GTK_TYPE_VBOX, "Workspace", &info, 0);
        }
        return type;
}

/**
 * workspace_new
 * @server:
 *
 * Returns:
 */
GtkWidget *
workspace_new (MgServer *server)
{
	Workspace *wk;
	WorkspacePage *page;

	wk = g_object_new (workspace_get_type (), NULL);
	workspace_set_server (wk, server);

	/* create the pages' objects */
	wk->priv->pages_objects = NULL;
	page = WORKSPACE_PAGE (ws_tables_new (mg_server_get_conf (server)));
	wk->priv->pages_objects = g_slist_append (wk->priv->pages_objects, page);

	page = WORKSPACE_PAGE (ws_datatypes_new (mg_server_get_conf (server)));
	wk->priv->pages_objects = g_slist_append (wk->priv->pages_objects, page);

	page = WORKSPACE_PAGE (ws_queries_new (wk));
	wk->priv->pages_objects = g_slist_append (wk->priv->pages_objects, page);

	page = WORKSPACE_PAGE (ws_dbrels_new (mg_server_get_conf (server)));
	wk->priv->pages_objects = g_slist_append (wk->priv->pages_objects, page);

	/* initialize all the widgets */
	create_widgets (wk);

	return GTK_WIDGET (wk);
}

MgServer *
workspace_get_server (Workspace *wk)
{
	return wk->priv->server;
}

void
workspace_set_server (Workspace *wk, MgServer *server)
{
	MgConf *conf;
	wk->priv->server = server;

	/* connecting to MgConf's signals to handle auto saving */
	conf = mg_server_get_conf (server);
	g_signal_connect (G_OBJECT (conf), "changed",
			  G_CALLBACK (conf_changed_ch), wk);
}


static void description_changed_cb (WorkspacePage *page, gchar *description, Workspace *wk);
static void mitem_activate_cb (GtkMenuItem *mitem, Workspace *wk);

/*
 * Creates the widgets composing the different 'pages', and sets
 * up any required signal callback
 */
static void
create_widgets (Workspace *wk)
{
	GtkWidget *table, *mitem, *wid;
	gchar *str;
	GSList *list;
	gint i;

	/* add main page */
	wk->priv->main_page_pane = gnome_db_new_hpaned_widget ();
	gtk_box_pack_start (GTK_BOX (wk), wk->priv->main_page_pane, TRUE, TRUE, 0);

	/* left part of main page, option menu */
	table = gnome_db_new_table_widget (2, 3, FALSE);

	wk->priv->main_page_object_type = gnome_db_new_option_menu_widget ();
	gtk_table_attach (GTK_TABLE (table), wk->priv->main_page_object_type, 0, 3, 0, 1,
			  0, 0, 0, 0);
	
	list = wk->priv->pages_objects;
	i = 0;
	while (list) {
		str = workspace_page_get_name (WORKSPACE_PAGE (list->data));
		mitem = gnome_db_option_menu_add_item (GTK_OPTION_MENU (wk->priv->main_page_object_type), str);
		g_free (str);
		g_signal_connect (G_OBJECT (mitem), "activate", G_CALLBACK (mitem_activate_cb), wk);
		g_object_set_data (G_OBJECT (mitem), "pageno", GINT_TO_POINTER (i));

		i++;
		list = g_slist_next (list);
	}

	str = workspace_page_get_name (WORKSPACE_PAGE (wk->priv->pages_objects->data));
	gnome_db_option_menu_set_selection (GTK_OPTION_MENU (wk->priv->main_page_object_type), str);
	g_free (str);

	/* left part, selectors */
	wk->priv->main_page_object_list = gtk_notebook_new ();
	gtk_notebook_set_show_tabs (GTK_NOTEBOOK (wk->priv->main_page_object_list), FALSE);
	gtk_notebook_set_show_border (GTK_NOTEBOOK (wk->priv->main_page_object_list), FALSE);
	gtk_widget_show (wk->priv->main_page_object_list);

	list = wk->priv->pages_objects;
	while (list) {
		wid = workspace_page_get_selector (WORKSPACE_PAGE (list->data));
		gtk_notebook_append_page (GTK_NOTEBOOK (wk->priv->main_page_object_list), wid, NULL);
		gtk_widget_show (wid);
		list = g_slist_next (list);
	}
	gtk_widget_show (wk->priv->main_page_object_list);
	gtk_table_attach (GTK_TABLE (table), wk->priv->main_page_object_list, 0, 3, 1, 2,
			  GTK_FILL | GTK_EXPAND | GTK_SHRINK,
			  GTK_FILL | GTK_EXPAND | GTK_SHRINK, 0, 0);

	gtk_paned_pack1 (GTK_PANED (wk->priv->main_page_pane), table, TRUE, FALSE);
	

	/* right part of main page, gray bar */
	wk->priv->main_page_detail_table = gnome_db_new_table_widget (2, 1, FALSE);

	wk->priv->main_page_detail_label = gnome_db_gray_bar_new ("<b>---</b>");
	gnome_db_gray_bar_set_show_icon (GNOME_DB_GRAY_BAR (wk->priv->main_page_detail_label), TRUE);
	gnome_db_gray_bar_set_icon_from_stock (GNOME_DB_GRAY_BAR (wk->priv->main_page_detail_label),
					       GTK_STOCK_NEW, GTK_ICON_SIZE_BUTTON);
	gnome_db_gray_bar_set_show_icon (GNOME_DB_GRAY_BAR (wk->priv->main_page_detail_label), FALSE);
	gtk_widget_show (wk->priv->main_page_detail_label);
	gtk_table_attach (GTK_TABLE (wk->priv->main_page_detail_table),
			  wk->priv->main_page_detail_label, 0, 1, 0, 1,
			  GTK_FILL | GTK_EXPAND | GTK_SHRINK, GTK_FILL, 0, 0);
	list = wk->priv->pages_objects;
	while (list) {
		g_signal_connect (G_OBJECT (list->data), "description_changed", 
				  G_CALLBACK (description_changed_cb), wk);
		list = g_slist_next (list);
	}
	str = workspace_page_get_description (WORKSPACE_PAGE (wk->priv->pages_objects->data));
	gnome_db_gray_bar_set_text (GNOME_DB_GRAY_BAR (wk->priv->main_page_detail_label), str);
	g_free (str);

	/*  right part, work area */
	wk->priv->main_page_detail_contents = gtk_notebook_new ();
	gtk_notebook_set_show_border (GTK_NOTEBOOK (wk->priv->main_page_detail_contents), FALSE);
	gtk_notebook_set_show_tabs (GTK_NOTEBOOK (wk->priv->main_page_detail_contents), FALSE);
	gtk_widget_show (wk->priv->main_page_detail_contents);
	gtk_table_attach_defaults (GTK_TABLE (wk->priv->main_page_detail_table),
				   wk->priv->main_page_detail_contents, 0 ,1, 1, 2);

	gtk_paned_pack2 (GTK_PANED (wk->priv->main_page_pane), wk->priv->main_page_detail_table, TRUE, FALSE);

	list = wk->priv->pages_objects;
	while (list) {
		wid = workspace_page_get_work_area (WORKSPACE_PAGE (list->data));
		gtk_notebook_append_page (GTK_NOTEBOOK (wk->priv->main_page_detail_contents), wid, NULL);
		gtk_widget_show (wid);
		list = g_slist_next (list);
	}
}

static void
description_changed_cb (WorkspacePage *page, gchar *description, Workspace *wk)
{
	gnome_db_gray_bar_set_text (GNOME_DB_GRAY_BAR (wk->priv->main_page_detail_label), description);
}

static void
mitem_activate_cb (GtkMenuItem *mitem, Workspace *wk)
{
	gchar *title;
	gint pageno = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (mitem), "pageno"));
	WorkspacePage *page;

	gtk_notebook_set_current_page (GTK_NOTEBOOK (wk->priv->main_page_object_list), pageno);
	gtk_notebook_set_current_page (GTK_NOTEBOOK (wk->priv->main_page_detail_contents), pageno);

	page = g_slist_nth_data (wk->priv->pages_objects, pageno);
	title = workspace_page_get_description (page);
	gnome_db_gray_bar_set_text (GNOME_DB_GRAY_BAR (wk->priv->main_page_detail_label), title);
	g_free (title);
}

static void
conf_changed_ch (MgConf *conf, Workspace *wk)
{
	workspace_queue_conf_save (wk);	
}

static gboolean timeout_conf_save_cb (Workspace *wk);
static gboolean idle_conf_save_cb (Workspace *wk);
void
workspace_queue_conf_save (Workspace *wk)
{
	if (wk->priv->timeout_event_conf_save)
		g_source_remove (wk->priv->timeout_event_conf_save);
	
	wk->priv->timeout_event_conf_save = g_timeout_add (5000, (GSourceFunc) timeout_conf_save_cb, wk);
}

static gboolean
timeout_conf_save_cb (Workspace *wk)
{
	if (wk->priv->idle_event_conf_save == 0) 
		wk->priv->idle_event_conf_save = g_idle_add ((GSourceFunc) idle_conf_save_cb, wk);
	wk->priv->timeout_event_conf_save = 0;

	return FALSE;
}

static gboolean
idle_conf_save_cb (Workspace *wk)
{
	mg_conf_save_xml (mg_server_get_conf (wk->priv->server), NULL);
	wk->priv->idle_event_conf_save = 0;

	return FALSE;
}
