/* mg-graph.c
 *
 * Copyright (C) 2004 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <string.h>
#include "mg-graph.h"
#include <libmergeant/mg-ref-base.h>
#include <libmergeant/mg-xml-storage.h>
#include "mg-graph-item.h"
#include "marshal.h"

/* 
 * Main static functions 
 */
static void mg_graph_class_init (MgGraphClass *class);
static void mg_graph_init (MgGraph *graph);
static void mg_graph_dispose (GObject *object);
static void mg_graph_finalize (GObject *object);

static void mg_graph_set_property (GObject              *object,
				   guint                 param_id,
				   const GValue         *value,
				   GParamSpec           *pspec);
static void mg_graph_get_property (GObject              *object,
				   guint                 param_id,
				   GValue               *value,
				   GParamSpec           *pspec);


static void nullified_item_cb (MgGraphItem *item, MgGraph *graph);

static void graph_item_moved_cb (MgGraphItem *item, MgGraph *graph);


/* XML storage interface */
static void        mg_graph_xml_storage_init (MgXmlStorageIface *iface);
static gchar      *mg_graph_get_xml_id (MgXmlStorage *iface);
static xmlNodePtr  mg_graph_save_to_xml (MgXmlStorage *iface, GError **error);
static gboolean    mg_graph_load_from_xml (MgXmlStorage *iface, xmlNodePtr node, GError **error);


#ifdef debug
static void        mg_graph_dump                (MgGraph *graph, guint offset);
#endif

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass  *parent_class = NULL;

/* signals */
enum
{
	ITEM_ADDED,
	ITEM_DROPPED,
	ITEM_MOVED,
	LAST_SIGNAL
};

static gint mg_graph_signals[LAST_SIGNAL] = { 0, 0 };

/* properties */
enum
{
	PROP_0,
	PROP_REF_OBJECT
};


struct _MgGraphPrivate
{
	MgGraphType   type;
	MgRefBase    *ref_object;
	GSList       *graph_items;
};

/* module error */
GQuark mg_graph_error_quark (void)
{
	static GQuark quark;
	if (!quark)
		quark = g_quark_from_static_string ("mg_graph_error");
	return quark;
}


guint
mg_graph_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (MgGraphClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) mg_graph_class_init,
			NULL,
			NULL,
			sizeof (MgGraph),
			0,
			(GInstanceInitFunc) mg_graph_init
		};
		
		static const GInterfaceInfo xml_storage_info = {
                        (GInterfaceInitFunc) mg_graph_xml_storage_init,
                        NULL,
                        NULL
                };

		type = g_type_register_static (MG_BASE_TYPE, "MgGraph", &info, 0);
		g_type_add_interface_static (type, MG_XML_STORAGE_TYPE, &xml_storage_info);
	}
	return type;
}

static void
mg_graph_xml_storage_init (MgXmlStorageIface *iface)
{
        iface->get_xml_id = mg_graph_get_xml_id;
        iface->save_to_xml = mg_graph_save_to_xml;
        iface->load_from_xml = mg_graph_load_from_xml;
}


static void
mg_graph_class_init (MgGraphClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);
	
	mg_graph_signals[ITEM_ADDED] =
		g_signal_new ("item_added",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (MgGraphClass, item_added),
			      NULL, NULL,
			      marshal_VOID__POINTER, G_TYPE_NONE,
			      1, G_TYPE_POINTER);
	mg_graph_signals[ITEM_DROPPED] =
		g_signal_new ("item_dropped",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (MgGraphClass, item_dropped),
			      NULL, NULL,
			      marshal_VOID__POINTER, G_TYPE_NONE,
			      1, G_TYPE_POINTER);
	mg_graph_signals[ITEM_MOVED] =
		g_signal_new ("item_moved",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (MgGraphClass, item_moved),
			      NULL, NULL,
			      marshal_VOID__POINTER, G_TYPE_NONE,
			      1, G_TYPE_POINTER);
	
	class->item_added = NULL;
	class->item_dropped = NULL;
	class->item_moved = NULL;

	object_class->dispose = mg_graph_dispose;
	object_class->finalize = mg_graph_finalize;

	/* Properties */
	object_class->set_property = mg_graph_set_property;
	object_class->get_property = mg_graph_get_property;

	g_object_class_install_property (object_class, PROP_REF_OBJECT,
					 g_param_spec_pointer ("ref_object", NULL, NULL, 
							       (G_PARAM_READABLE | G_PARAM_WRITABLE)));

	/* virtual functions */
#ifdef debug
        MG_BASE_CLASS (class)->dump = (void (*)(MgBase *, guint)) mg_graph_dump;
#endif
}

static void
mg_graph_init (MgGraph *graph)
{
	graph->priv = g_new0 (MgGraphPrivate, 1);
	graph->priv->type = MG_GRAPH_DB_RELATIONS;
	graph->priv->ref_object = NULL;
	graph->priv->graph_items = NULL;
}

/**
 * mg_graph_new
 * @conf: a #MgConf object
 * @type: the graph type (one of #MgGraphType)
 *
 * Creates a new #MgGraph object. The graph type is used only to be able to sort out the
 * different types of graphs. It brings no special functionnality.
 *
 * Returns: the newly created object
 */
GObject   *
mg_graph_new (MgConf *conf, MgGraphType type)
{
	GObject *obj = NULL;
	MgGraph *graph;
	guint id;

	g_return_val_if_fail (conf && IS_MG_CONF (conf), NULL);

	obj = g_object_new (MG_GRAPH_TYPE, "conf", conf, NULL);
	graph = MG_GRAPH (obj);

	g_object_get (G_OBJECT (conf), "graph_serial", &id, NULL);
	mg_base_set_id (MG_BASE (obj), id);

	graph->priv->type = type;
	graph->priv->ref_object = MG_REF_BASE (mg_ref_base_new (conf));

	mg_conf_declare_graph (conf, graph);

	return obj;
}

static void
mg_graph_dispose (GObject *object)
{
	MgGraph *graph;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_MG_GRAPH (object));

	graph = MG_GRAPH (object);
	if (graph->priv) {		
		if (graph->priv->ref_object) {
			g_object_unref (G_OBJECT (graph->priv->ref_object));
                        graph->priv->ref_object = NULL;
                }
		
		while (graph->priv->graph_items)
			nullified_item_cb (MG_GRAPH_ITEM (graph->priv->graph_items->data), graph);
	}

	/* parent class */
	parent_class->dispose (object);
}

static void
nullified_item_cb (MgGraphItem *item, MgGraph *graph)
{
	g_assert (g_slist_find (graph->priv->graph_items, item));
	g_signal_handlers_disconnect_by_func (G_OBJECT (item),
					      G_CALLBACK (nullified_item_cb) , graph);
	g_signal_handlers_disconnect_by_func (G_OBJECT (item),
					      G_CALLBACK (graph_item_moved_cb) , graph);

	graph->priv->graph_items = g_slist_remove (graph->priv->graph_items, item);
#ifdef debug_signal
	g_print (">> 'ITEM_DROPPED' from %s::%s()\n", __FILE__, __FUNCTION__);
#endif
	g_signal_emit (G_OBJECT (graph), mg_graph_signals [ITEM_DROPPED], 0, item);
#ifdef debug_signal
	g_print ("<< 'ITEM_DROPPED' from %s::%s()\n", __FILE__, __FUNCTION__);
#endif
	g_object_unref (G_OBJECT (item));
}


static void
mg_graph_finalize (GObject   * object)
{
	MgGraph *graph;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_MG_GRAPH (object));

	graph = MG_GRAPH (object);
	if (graph->priv) {
		g_free (graph->priv);
		graph->priv = NULL;
	}

	/* parent class */
	parent_class->finalize (object);
}


static void 
mg_graph_set_property (GObject              *object,
			   guint                 param_id,
			   const GValue         *value,
			   GParamSpec           *pspec)
{
	MgGraph *graph;
	gpointer ptr;

	graph = MG_GRAPH (object);
	if (graph->priv) {
		switch (param_id) {
		case PROP_REF_OBJECT:
			g_assert (graph->priv->ref_object);
			if (graph->priv->ref_object) {
				ptr = g_value_get_pointer (value);
				mg_ref_base_set_ref_object (graph->priv->ref_object, ptr);
			}
			break;
		}
	}
}

static void
mg_graph_get_property (GObject              *object,
			   guint                 param_id,
			   GValue               *value,
			   GParamSpec           *pspec)
{
	MgGraph *graph;

	graph = MG_GRAPH (object);
        if (graph->priv) {
                switch (param_id) {
                case PROP_REF_OBJECT:
			g_assert (graph->priv->ref_object);
			g_value_set_pointer (value, 
					     mg_ref_base_get_ref_object (graph->priv->ref_object));
                        break;
                }
        }
}


/**
 * mg_graph_get_graph_type
 * @graph: a #MgGraph object
 *
 * Get the graph type of @graph.
 *
 * Returns: the type
 */
MgGraphType
mg_graph_get_graph_type (MgGraph *graph)
{
	g_return_val_if_fail (graph && IS_MG_GRAPH (graph), MG_GRAPH_DB_RELATIONS);
	g_return_val_if_fail (graph->priv, MG_GRAPH_DB_RELATIONS);

	return graph->priv->type;
}

/**
 * mg_graph_get_items
 * @graph: a #MgGraph object
 *
 * Get a list of #MgGraphItem objects which are items of @graph
 *
 * Returns: a new list of #MgGraphItem objects
 */
GSList *
mg_graph_get_items (MgGraph *graph)
{
	g_return_val_if_fail (graph && IS_MG_GRAPH (graph), NULL);
	g_return_val_if_fail (graph->priv, NULL);

	if (graph->priv->graph_items)
		return g_slist_copy (graph->priv->graph_items);
	else
		return NULL;
}



/**
 * mg_graph_get_item_from_obj
 * @graph: a #MgGraph object
 * @ref_obj: the #MgBase the returned item references
 * @create_if_needed:
 *
 * Get a pointer to a #MgGraphItem item from the object is represents.
 * If the searched #MgGraphItem is not found and @create_if_needed is TRUE, then a new
 * #MgGraphItem is created.
 *
 * Returns: the #MgGraphItem object, or %NULL if not found
 */
MgGraphItem *
mg_graph_get_item_from_obj (MgGraph *graph, MgBase *ref_obj, gboolean create_if_needed)
{
	MgGraphItem *item = NULL;
	GSList *list;
	MgBase *obj;

	g_return_val_if_fail (graph && IS_MG_GRAPH (graph), NULL);
	g_return_val_if_fail (graph->priv, NULL);
	g_return_val_if_fail (ref_obj, NULL);

	list = graph->priv->graph_items;
	while (list && !item) {
		g_object_get (G_OBJECT (list->data), "ref_object", &obj, NULL);
		if (obj == ref_obj)
			item = MG_GRAPH_ITEM (list->data);
		list = g_slist_next (list);
	}

	if (!item && create_if_needed) {
		item = MG_GRAPH_ITEM (mg_graph_item_new (mg_base_get_conf (MG_BASE (graph)), ref_obj));
		
		mg_graph_add_item (graph, item);
		g_object_unref (G_OBJECT (item));
	}

	return item;
}


/**
 * mg_graph_add_item
 * @graph: a #MgGraph object
 * @item: a #MgGraphItem object
 *
 * Adds @item to @graph.
 */
void
mg_graph_add_item (MgGraph *graph, MgGraphItem *item)
{
	g_return_if_fail (graph && IS_MG_GRAPH (graph));
	g_return_if_fail (graph->priv);
	g_return_if_fail (item && IS_MG_GRAPH_ITEM (item));

	g_object_ref (G_OBJECT (item));
	
	graph->priv->graph_items = g_slist_append (graph->priv->graph_items, item);
	g_signal_connect (G_OBJECT (item), "nullified",
			  G_CALLBACK (nullified_item_cb), graph);
	g_signal_connect (G_OBJECT (item), "moved",
			  G_CALLBACK (graph_item_moved_cb), graph);
#ifdef debug_signal
	g_print (">> 'ITEM_ADDED' from %s::%s()\n", __FILE__, __FUNCTION__);
#endif
	g_signal_emit (G_OBJECT (graph), mg_graph_signals [ITEM_ADDED], 0, item);
#ifdef debug_signal
	g_print ("<< 'ITEM_ADDED' from %s::%s()\n", __FILE__, __FUNCTION__);
#endif
}

static void
graph_item_moved_cb (MgGraphItem *item, MgGraph *graph)
{
#ifdef debug_signal
	g_print (">> 'ITEM_MOVED' from %s::%s()\n", __FILE__, __FUNCTION__);
#endif
	g_signal_emit (G_OBJECT (graph), mg_graph_signals [ITEM_MOVED], 0, item);
#ifdef debug_signal
	g_print ("<< 'ITEM_MOVED' from %s::%s()\n", __FILE__, __FUNCTION__);
#endif
}

/**
 * mg_graph_del_item
 * @graph: a #MgGraph object
 * @item: a #MgGraphItem object
 *
 * Removes @item from @graph
 */
void
mg_graph_del_item (MgGraph *graph, MgGraphItem *item)
{
	g_return_if_fail (graph && IS_MG_GRAPH (graph));
	g_return_if_fail (graph->priv);
	g_return_if_fail (item && IS_MG_GRAPH_ITEM (item));

	nullified_item_cb (item, graph);
}

#ifdef debug
static void
mg_graph_dump (MgGraph *graph, guint offset)
{
	gchar *str;
	gint i;

	g_return_if_fail (graph && IS_MG_GRAPH (graph));
	
        /* string for the offset */
        str = g_new0 (gchar, offset+1);
        for (i=0; i<offset; i++)
                str[i] = ' ';
        str[offset] = 0;

        /* dump */
        if (graph->priv) {
		GSList *items;
		if (mg_base_get_name (MG_BASE (graph)))
			g_print ("%s" D_COL_H1 "MgGraph" D_COL_NOR "\"%s\" (%p) ",
				 str, mg_base_get_name (MG_BASE (graph)), graph);
		else
			g_print ("%s" D_COL_H1 "MgGraph" D_COL_NOR " (%p) ", str, graph);
		g_print ("\n");
		items = graph->priv->graph_items;
		while (items) {
			mg_base_dump (MG_BASE (items->data), offset+5);
			items = g_slist_next (items);
		}
	}
        else
                g_print ("%s" D_COL_ERR "Using finalized object %p" D_COL_NOR, str, graph);
}
#endif

/* 
 * MgXmlStorage interface implementation
 */
static gchar *
mg_graph_get_xml_id (MgXmlStorage *iface)
{
        gchar *xml_id;
	
        g_return_val_if_fail (iface && IS_MG_GRAPH (iface), NULL);
        g_return_val_if_fail (MG_GRAPH (iface)->priv, NULL);
	
        xml_id = g_strdup_printf ("GR%d", mg_base_get_id (MG_BASE (iface)));

        return xml_id;
}

static xmlNodePtr
mg_graph_save_to_xml (MgXmlStorage *iface, GError **error)
{
        xmlNodePtr node = NULL;
	MgGraph *graph;
        gchar *str = NULL;
	MgBase *base;
	GSList *list;

        g_return_val_if_fail (iface && IS_MG_GRAPH (iface), NULL);
        g_return_val_if_fail (MG_GRAPH (iface)->priv, NULL);

        graph = MG_GRAPH (iface);

        node = xmlNewNode (NULL, "MG_GRAPH");

        str = mg_graph_get_xml_id (iface);
        xmlSetProp (node, "id", str);
        g_free (str);
	xmlSetProp (node, "name", mg_base_get_name (MG_BASE (graph)));
        xmlSetProp (node, "descr", mg_base_get_description (MG_BASE (graph)));

	switch (graph->priv->type) {
	case MG_GRAPH_DB_RELATIONS:
		str = "R";
		break;
	case MG_GRAPH_QUERY_JOINS:
		str = "Q";
		break;
	case MG_GRAPH_MODELLING:
		str = "M";
		break;
	default:
		g_assert_not_reached ();
		break;
	}
        xmlSetProp (node, "type", str);

	g_assert (graph->priv->ref_object);
	base = mg_ref_base_get_ref_object (graph->priv->ref_object);
	if (base) {
		str = mg_xml_storage_get_xml_id (MG_XML_STORAGE (base));
		xmlSetProp (node, "object", str);
		g_free (str);
	}

	/* graph items */
	list = graph->priv->graph_items;
	while (list) {
		xmlNodePtr sub = mg_xml_storage_save_to_xml (MG_XML_STORAGE (list->data), error);
		if (sub)
                        xmlAddChild (node, sub);
                else {
                        xmlFreeNode (node);
                        return NULL;
                }
		list = g_slist_next (list);
	}

        return node;
}

static gboolean
mg_graph_load_from_xml (MgXmlStorage *iface, xmlNodePtr node, GError **error)
{
	MgGraph *graph;
        gchar *prop;
	xmlNodePtr children;
	gboolean id=FALSE;

        g_return_val_if_fail (iface && IS_MG_GRAPH (iface), FALSE);
        g_return_val_if_fail (MG_GRAPH (iface)->priv, FALSE);
        g_return_val_if_fail (node, FALSE);

	graph = MG_GRAPH (iface);

	if (strcmp (node->name, "MG_GRAPH")) {
                g_set_error (error,
                             MG_GRAPH_ERROR,
                             MG_GRAPH_XML_LOAD_ERROR,
                             _("XML Tag is not <MG_GRAPH>"));
                return FALSE;
        }

	prop = xmlGetProp (node, "id");
        if (prop) {
                if (strlen (prop) <= 2) {
                        g_set_error (error,
                                     MG_GRAPH_ERROR,
                                     MG_GRAPH_XML_LOAD_ERROR,
                                     _("Wrong 'id' attribute in <MG_GRAPH>"));
                        return FALSE;
                }
                mg_base_set_id (MG_BASE (graph), atoi (prop+2));
		id = TRUE;
                g_free (prop);
        }

	prop = xmlGetProp (node, "name");
        if (prop) {
                mg_base_set_name (MG_BASE (graph), prop);
                g_free (prop);
        }

        prop = xmlGetProp (node, "descr");
        if (prop) {
                mg_base_set_description (MG_BASE (graph), prop);
                g_free (prop);
        }

	prop = xmlGetProp (node, "type");
        if (prop) {
		switch (*prop) {
		case 'R':
			graph->priv->type = MG_GRAPH_DB_RELATIONS;
			break;
		case 'Q':
			graph->priv->type = MG_GRAPH_QUERY_JOINS;
			break;
		case 'M':
			graph->priv->type = MG_GRAPH_MODELLING;
			break;
		default:
			g_set_error (error,
                                     MG_GRAPH_ERROR,
                                     MG_GRAPH_XML_LOAD_ERROR,
                                     _("Wrong 'type' attribute in <MG_GRAPH>"));
                        return FALSE;
			break;
		}
                g_free (prop);
        }

	prop = xmlGetProp (node, "object");
	if (prop) {
		g_assert (graph->priv->ref_object);
		mg_ref_base_set_ref_name (graph->priv->ref_object, 0/* FIXME */, REFERENCE_BY_XML_ID, prop);
		g_free (prop);
	}

	/* items nodes */
	children = node->children;
	while (children) {
		if (!strcmp (children->name, "MG_GRAPH_ITEM")) {
			MgGraphItem *item;
			
			item = MG_GRAPH_ITEM (mg_graph_item_new (mg_base_get_conf (MG_BASE (graph)), NULL));
			if (mg_xml_storage_load_from_xml (MG_XML_STORAGE (item), children, error)) {
				mg_graph_add_item (graph, item);
				g_object_unref (G_OBJECT (item));
			}
			else
				return FALSE;
                }

		children = children->next;
	}

	if (!id) {
		g_set_error (error,
			     MG_GRAPH_ERROR,
			     MG_GRAPH_XML_LOAD_ERROR,
			     _("Missing Id attribute in <MG_GRAPH>"));
		return FALSE;
        }

        return TRUE;
}

