/* mg-work-grid.c
 *
 * Copyright (C) 2002 - 2004 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <string.h>
#include <gdk/gdkkeysyms.h>
#include <libgnomedb/libgnomedb.h>
#include "marshal.h"
#include "mg-server.h"
#include "mg-server-data-type.h"
#include "mg-data-handler.h"
#include "mg-work-grid.h"
#include "mg-work-core.h"
#include "mg-work-widget.h"
#include "mg-query.h"
#include "mg-target.h"
#include "mg-entity.h"
#include "mg-renderer.h"
#include "mg-result-set.h"
#include "mg-form.h"
#include "mg-parameter.h"
#include "mg-context.h"
#include "mg-field.h"
#include "mg-qfield.h"
#include "handlers/mg-data-cell-renderer-info.h"
#include "handlers/mg-data-cell-renderer-combo.h"
#include "mg-data-entry.h" /* only the MG_DATA_ENTRY_* enum is used here */
#include "utility.h"
#include "mg-util.h"

#ifdef debug
#include "mg-graphviz.h"
#endif

static void mg_work_grid_class_init (MgWorkGridClass * class);
static void mg_work_grid_init (MgWorkGrid * wid);
static void mg_work_grid_dispose (GObject   * object);

static void mg_work_grid_set_property (GObject              *object,
				       guint                 param_id,
				       const GValue         *value,
				       GParamSpec           *pspec);
static void mg_work_grid_get_property (GObject              *object,
				       guint                 param_id,
				       GValue               *value,
				       GParamSpec           *pspec);

static void mg_work_grid_initialize (MgWorkGrid *grid);

static void nullified_core_cb (MgWorkCore *core, MgWorkGrid *grid);

static GtkWidget *modif_buttons_make (MgWorkGrid *grid);
static void       modif_buttons_update (MgWorkGrid *grid);
static void       modif_actions_real_do (MgWorkGrid *grid, gchar action);

static void refresh_all_rows (MgWorkGrid *grid, gboolean keep_old_modifications);
static void update_simple_grid (MgWorkGrid *grid, gboolean keep_user_modifs);
static void arg_param_changed_cb (MgParameter *param, MgWorkGrid *grid);
static void work_param_changed_cb (MgParameter *param, MgWorkGrid *grid);



/* MgWorkWidget interface */
static void            mg_work_grid_widget_init         (MgWorkWidgetIface *iface);
static void            mg_work_grid_run                 (MgWorkWidget *iface, guint mode);
static void            mg_work_grid_set_mode            (MgWorkWidget *iface, guint mode);
static void            mg_work_grid_set_entry_editable  (MgWorkWidget *iface, MgQfield *field, gboolean editable);
static void            mg_work_grid_show_entry_actions  (MgWorkWidget *iface, MgQfield *field, gboolean show_actions);
static void            mg_work_grid_show_global_actions (MgWorkWidget *iface, gboolean show_actions);
static MgParameter    *mg_work_grid_get_param_for_field (MgWorkWidget *iface, MgQfield *field, const gchar *field_name, 
							 gboolean in_exec_context);
static gboolean        mg_work_grid_has_been_changed    (MgWorkWidget *iface);
static MgContext      *mg_work_grid_get_exec_context    (MgWorkWidget *iface);
static GtkActionGroup *mg_work_grid_get_actions_group   (MgWorkWidget *iface);

typedef struct {
	MgContextNode   *node;
	GtkCellRenderer *data_cell;
	GtkCellRenderer *info_cell;
	gboolean         info_shown;
	gboolean         data_locked;
} ColumnData;
#define COLUMN_DATA(x) ((ColumnData *)(x))


struct _MgWorkGridPriv
{
	MgWorkCore        *core;
	gboolean           has_run;
	GSList            *modified_rows; /* list of ModelUserModifiedRow, all data's memory is handled there */

	guint              mode;
	GtkTooltips       *tooltips;
	gboolean           default_show_info_cell;
	GSList            *columns_data; /* list of ColumnData */
	gboolean           internal_params_change;

	gint               sample_first_row;
	gint               sample_last_row;
	gint               sample_size;

	GtkWidget         *title;
	GtkWidget         *scroll;
	GtkWidget         *treeview;

	GtkUIManager      *uimanager;
	GtkActionGroup    *actions_group;
	GtkWidget         *modif_all;
	GtkWidget         *nav_current_sample;

	gchar             *current_selection; /* Selection as a GtkTreePath string representation */
};

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *parent_class = NULL;

/* signals */
enum
{
        SELECTION_CHANGED,
        LAST_SIGNAL
};
                                                                                                                                                                              
static gint mg_work_grid_signals[LAST_SIGNAL] = { 0 };


/* properties */
enum
{
        PROP_0,
        PROP_TITLE_VISIBLE,
	PROP_TITLE_STRING,
	PROP_ACTIONS_VISIBLE,
	PROP_INFO_CELL_VISIBLE
};

guint
mg_work_grid_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (MgWorkGridClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) mg_work_grid_class_init,
			NULL,
			NULL,
			sizeof (MgWorkGrid),
			0,
			(GInstanceInitFunc) mg_work_grid_init
		};		

		static const GInterfaceInfo work_widget_info = {
                        (GInterfaceInitFunc) mg_work_grid_widget_init,
                        NULL,
                        NULL
                };
		
		type = g_type_register_static (GTK_TYPE_VBOX, "MgWorkGrid", &info, 0);
		g_type_add_interface_static (type, MG_WORK_WIDGET_TYPE, &work_widget_info);
	}

	return type;
}

static void
mg_work_grid_widget_init (MgWorkWidgetIface *iface)
{
	iface->run = mg_work_grid_run;
	iface->set_mode = mg_work_grid_set_mode;
	iface->set_entry_editable = mg_work_grid_set_entry_editable;
	iface->show_entry_actions = mg_work_grid_show_entry_actions;
	iface->show_global_actions = mg_work_grid_show_global_actions;
	iface->get_param_for_field = mg_work_grid_get_param_for_field;
        iface->has_been_changed = mg_work_grid_has_been_changed;
	iface->get_exec_context = mg_work_grid_get_exec_context;
	iface->get_actions_group = mg_work_grid_get_actions_group;
}

static void
mg_work_grid_class_init (MgWorkGridClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);
	
	parent_class = g_type_class_peek_parent (class);

	mg_work_grid_signals[SELECTION_CHANGED] = 
		g_signal_new ("selection_changed",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET (MgWorkGridClass, selection_changed),
                              NULL, NULL,
                              marshal_VOID__BOOLEAN, G_TYPE_NONE,
                              1, G_TYPE_BOOLEAN);

	object_class->dispose = mg_work_grid_dispose;

	/* Properties */
        object_class->set_property = mg_work_grid_set_property;
        object_class->get_property = mg_work_grid_get_property;
	g_object_class_install_property (object_class, PROP_TITLE_VISIBLE,
                                         g_param_spec_boolean ("title_visible", NULL, NULL, FALSE,
                                                               G_PARAM_WRITABLE));
	g_object_class_install_property (object_class, PROP_TITLE_STRING,
                                         g_param_spec_string ("title_string", NULL, NULL, NULL,
							      G_PARAM_WRITABLE));
	g_object_class_install_property (object_class, PROP_ACTIONS_VISIBLE,
                                         g_param_spec_boolean ("actions_visible", NULL, NULL, FALSE,
                                                               G_PARAM_WRITABLE));
	g_object_class_install_property (object_class, PROP_INFO_CELL_VISIBLE,
                                         g_param_spec_boolean ("info_cell_visible", NULL, NULL, FALSE,
                                                               G_PARAM_WRITABLE));
}

static void
mg_work_grid_init (MgWorkGrid * wid)
{
	wid->priv = g_new0 (MgWorkGridPriv, 1);
	wid->priv->core = NULL;
	wid->priv->has_run = FALSE;
	wid->priv->modified_rows = NULL;
	wid->priv->default_show_info_cell = TRUE;
	wid->priv->columns_data = NULL;
	wid->priv->internal_params_change = FALSE;

	wid->priv->sample_first_row = 0;
	wid->priv->sample_last_row = 0;
	wid->priv->sample_size = 100;

	wid->priv->treeview = NULL;

	wid->priv->mode = 0;
	wid->priv->tooltips = NULL;

	wid->priv->current_selection = NULL;
}

/**
 * mg_work_grid_new
 * @query: a #MgQuery object
 * @modified: a #MgTarget object, or %NULL
 *
 * Creates a new #MgWorkGrid widget.
 *
 * @query must be a SELECT query (no union, etc selection query)
 *
 * The @modified target must belong to @query and represent
 * modifiable entity (a #MgDbTable for example). If @modified is %NULL then
 * no modification will be allowed.
 *
 * Returns: the new widget
 */
GtkWidget *
mg_work_grid_new (MgQuery *query, MgTarget *modified)
{
	GObject *obj;
	MgWorkGrid *grid;

	g_return_val_if_fail (query && IS_MG_QUERY (query), NULL);
	g_return_val_if_fail (mg_query_get_query_type (query) == MG_QUERY_TYPE_SELECT, NULL);
	
	if (modified) {
		g_return_val_if_fail (IS_MG_TARGET (modified), NULL);
		g_return_val_if_fail (mg_target_get_query (modified) == query, NULL);
		g_return_val_if_fail (mg_entity_is_writable (mg_target_get_represented_entity (modified)), NULL);
	}

	obj = g_object_new (MG_WORK_GRID_TYPE, NULL);
	grid = MG_WORK_GRID (obj);

	grid->priv->core = MG_WORK_CORE (mg_work_core_new (query, modified));
	g_signal_connect (G_OBJECT (grid->priv->core), "nullified",
			  G_CALLBACK (nullified_core_cb), grid);

	mg_work_grid_initialize (grid);

	return GTK_WIDGET (obj);
}

static void
nullified_core_cb (MgWorkCore *core, MgWorkGrid *grid)
{
	g_signal_handlers_disconnect_by_func (G_OBJECT (core),
					      G_CALLBACK (nullified_core_cb), grid);
	
	if (grid->priv->core->work_context) {
		GSList *list = grid->priv->core->work_context->parameters;
		while (list) {
			g_signal_handlers_disconnect_by_func (G_OBJECT (list->data),
							      G_CALLBACK (work_param_changed_cb), grid);
			list = g_slist_next (list);
		}
	}


	if (grid->priv->has_run && grid->priv->core->args_context)
		g_signal_handlers_disconnect_by_func (G_OBJECT (grid->priv->core->args_context),
						      G_CALLBACK (arg_param_changed_cb), grid);
	
	g_object_unref (G_OBJECT (grid->priv->core));
	grid->priv->core = NULL;
	gtk_widget_set_sensitive (GTK_WIDGET (grid), FALSE);
}

static void clean_user_modif_rows (MgWorkGrid *grid);
static void user_modifs_row_free (ModelUserModifiedRow *user_modifs);
static void
mg_work_grid_dispose (GObject *object)
{
	MgWorkGrid *grid;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_MG_WORK_GRID (object));
	grid = MG_WORK_GRID (object);

	if (grid->priv) {
		if (grid->priv->current_selection)
			g_free (grid->priv->current_selection);

		clean_user_modif_rows (grid);

		/* info cells */
		if (grid->priv->columns_data) {
			GSList *list = grid->priv->columns_data;
			while (list) {
				g_free (list->data);
				list = g_slist_next (list);
			}
			g_slist_free (grid->priv->columns_data);
			grid->priv->columns_data = NULL;
		}

		/* core */
		if (grid->priv->core)
			nullified_core_cb (grid->priv->core, grid);

		/* tooltips */
		if (grid->priv->tooltips) {
			gtk_object_destroy (GTK_OBJECT (grid->priv->tooltips));
			grid->priv->tooltips = NULL;
		}

		/* UI */
		if (grid->priv->actions_group) 
			g_object_unref (G_OBJECT (grid->priv->actions_group));

		if (grid->priv->uimanager)
			g_object_unref (G_OBJECT (grid->priv->uimanager));
		
		/* the private area itself */
		g_free (grid->priv);
		grid->priv = NULL;
	}

	/* for the parent class */
	parent_class->dispose (object);
}

/*
 * Free any memory associated with the user modified rows, as nothing is
 * managed by the data model
 */ 
static void
clean_user_modif_rows (MgWorkGrid *grid)
{
	if (grid->priv->modified_rows) {
		GSList *list;
		
		list = grid->priv->modified_rows;
		while (list) {
			user_modifs_row_free (MODEL_USER_MODIFIED_ROW (list->data));
			list = g_slist_next (list);
		}
		g_slist_free (grid->priv->modified_rows);
		grid->priv->modified_rows = NULL;
	}
}

static void
user_modifs_row_free (ModelUserModifiedRow *user_modifs)
{
	GSList *list;
			
	/* pkey values */
	if (user_modifs->pkey) {
		list = user_modifs->pkey;
		while (list) {
			if (list->data)
				gda_value_free ((GdaValue *) list->data);
			list = g_slist_next (list);
		}
		g_slist_free (user_modifs->pkey);
	}
	
	/* user modified values */
	if (user_modifs->user_values) {
		list = user_modifs->user_values;
		while (list) {
			if (MODEL_USER_MODIFIED_VALUE (list->data)->value)
				gda_value_free (MODEL_USER_MODIFIED_VALUE (list->data)->value);
			g_free (list->data);
			list = g_slist_next (list);
		}
		g_slist_free (user_modifs->user_values);
	}
	g_free (user_modifs);
}

static void
mg_work_grid_set_property (GObject              *object,
			   guint                 param_id,
			   const GValue         *value,
			   GParamSpec           *pspec)
{
	MgWorkGrid *grid;

        grid = MG_WORK_GRID (object);
        if (grid->priv) {
                switch (param_id) {
                case PROP_TITLE_VISIBLE:
			if (g_value_get_boolean (value))
				gtk_widget_show (grid->priv->title);
			else
				gtk_widget_hide (grid->priv->title);
                        break;
                case PROP_TITLE_STRING:
			gnome_db_gray_bar_set_text (GNOME_DB_GRAY_BAR (grid->priv->title), g_value_get_string (value));
			gtk_widget_show (grid->priv->title);
                        break;
		case PROP_ACTIONS_VISIBLE:
			if (g_value_get_boolean (value))
				gtk_widget_show (grid->priv->modif_all);
			else
				gtk_widget_hide (grid->priv->modif_all);
			break;
		case PROP_INFO_CELL_VISIBLE: 
		{
			GSList *list = grid->priv->columns_data;
			gboolean show = g_value_get_boolean (value);
			grid->priv->default_show_info_cell = show;

			while (list) {
				COLUMN_DATA (list->data)->info_shown = show;
				g_object_set (G_OBJECT (COLUMN_DATA (list->data)->info_cell), "visible", show, NULL);
				list = g_slist_next (list);
			}
		}
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
			break;
                }
        }
}

static void
mg_work_grid_get_property (GObject              *object,
			   guint                 param_id,
			   GValue               *value,
			   GParamSpec           *pspec)
{
	MgWorkGrid *grid;

        grid = MG_WORK_GRID (object);
        if (grid->priv) {
                switch (param_id) {
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
			break;
                }
        }	
}




/*
 * this function (re)computes the SELECT query and (re)runs it, resulting
 * in a change in the displayed values
 */
static void
refresh_all_rows (MgWorkGrid *grid, gboolean keep_old_modifications)
{
	GError *error = NULL;

	if (mg_work_core_run_select_query (grid->priv->core, &error))
		update_simple_grid (grid, keep_old_modifications);
	else {
		/* we don't want non relevant error messages so we check that the args context is valid before */
		if ((grid->priv->mode & MG_ACTION_REPORT_ERROR) && mg_context_is_valid (grid->priv->core->args_context)) {
			GtkWidget *dlg;
			gchar *message;
			GtkWidget *parent;
			
			/* find the top level window of the grid */
			parent = gtk_widget_get_parent (GTK_WIDGET (grid));
			while (parent && !GTK_IS_WINDOW (parent)) 
				parent = gtk_widget_get_parent (parent);
			
			if (error) {
				message = g_strdup (error->message);
				g_error_free (error);
			}
			else
				message = g_strdup_printf (_("An unknown error occurred while executing the query."));
			
			dlg = gtk_message_dialog_new (GTK_WINDOW (parent), 0,
						      GTK_MESSAGE_ERROR,
						      GTK_BUTTONS_CLOSE,
						      message);
			g_free (message);
			gtk_dialog_run (GTK_DIALOG (dlg));
			gtk_widget_destroy (dlg);
		}
		update_simple_grid (grid, FALSE);
	}
}

static void
arg_param_changed_cb (MgParameter *param, MgWorkGrid *grid)
{
	refresh_all_rows (grid, FALSE);
}

static ModelUserModifiedRow *grid_model_make_user_modifs (MgWorkGrid *grid, gint row);
static ModelUserModifiedValue *grid_model_get_user_modifs (MgWorkGrid *grid, GtkTreeModel *tree_model, GtkTreeIter *iter,
							   MgContextNode *context_node, gboolean create_if_needed);
static void grid_set_user_modifs_to_default_if_notnull (MgWorkGrid *grid, GtkTreeModel *tree_model, GtkTreeIter *iter, 
							MgWorkCore *core);
static void grid_modif_struct_set_value (MgWorkGrid *grid, ModelUserModifiedValue *user_modif, 
					 GtkTreeModel *tree_model, GtkTreeIter *iter,
					 MgWorkCore *core, const GdaValue *value);
static void grid_clean_data_model_row (MgWorkGrid *grid, GtkTreeModel *tree_model, GtkTreeIter *iter);
static void grid_modif_struct_set_status (MgWorkGrid *grid, ModelUserModifiedValue *user_modif,
					  GtkTreeModel *tree_model, GtkTreeIter *iter,
					  MgWorkCore *core, MgDataEntryAttribute requested_status);

static void
work_param_changed_cb (MgParameter *param, MgWorkGrid *grid)
{
	GSList *list;
	GtkTreeSelection *selection;
	GtkTreeModel *model;
	GtkTreeIter iter;

	/* take care of param dependencies */
	if (!grid->priv->treeview)
		return;

	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (grid->priv->treeview));
	if (!grid->priv->internal_params_change && gtk_tree_selection_get_selected (selection, &model, &iter)) {
		GtkTreePath *path;

		path = gtk_tree_model_get_path (model, &iter);
		list = grid->priv->core->work_context->parameters;
		while (list) {
			GSList *depend = mg_parameter_get_dependencies (MG_PARAMETER (list->data));
			while (depend) {
				if (depend->data == (gpointer) param) {
					ModelUserModifiedValue *user_modif;
					MgContextNode *context_node;
					
					context_node = mg_context_find_node_for_param (grid->priv->core->work_context, 
										       MG_PARAMETER (list->data));

					user_modif = grid_model_get_user_modifs (grid, model, &iter, context_node, TRUE);
					
					grid_modif_struct_set_status (grid, user_modif, model, &iter, grid->priv->core, 
								      MG_DATA_ENTRY_IS_NULL);
				}
				depend = g_slist_next (depend);
			}
			list = g_slist_next (list);
		}
		gtk_tree_path_free (path);
	}
}


/*
 * Real initialization
 */
static void 
mg_work_grid_initialize (MgWorkGrid *grid)
{
	GtkWidget *group, *sw;

	/* title */
	grid->priv->title = gnome_db_gray_bar_new (_("No title"));
        gtk_box_pack_start (GTK_BOX (grid), grid->priv->title, FALSE, TRUE, 2);
	gtk_widget_show (grid->priv->title);

	/*
	 * TreeView preparation
	 */
	sw = gtk_scrolled_window_new (NULL, NULL);
        gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
        gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (sw), GTK_SHADOW_IN);
	gtk_box_pack_start (GTK_BOX (grid), sw, TRUE, TRUE, 0);
	gtk_widget_show (sw);
	grid->priv->scroll = sw;

	/* signals connecting */
	if (grid->priv->core->work_context) {
		GSList *list = grid->priv->core->work_context->parameters;
		while (list) {
			g_signal_connect (G_OBJECT (list->data), "changed",
					  G_CALLBACK (work_param_changed_cb), grid);
			list = g_slist_next (list);
		}
	}

	/* 
	 * the grid's attributes
	 */
	if (mg_base_get_name (MG_BASE (grid->priv->core->query_select)))
		gnome_db_gray_bar_set_text (GNOME_DB_GRAY_BAR (grid->priv->title),
					    mg_base_get_name (MG_BASE (grid->priv->core->query_select)));
	else
		gtk_widget_hide (grid->priv->title);

	/*
	 * the actions part
	 */
	group = modif_buttons_make (grid);
	gtk_box_pack_start (GTK_BOX (grid), group, FALSE, FALSE, 0);
	gtk_widget_show (group);

	/* tooltips */
	grid->priv->tooltips = gtk_tooltips_new ();
}


static void
treeview_show_param_column (MgWorkGrid *grid, MgParameter *param, gboolean showit)
{
	gint pos = -1;
	GtkTreeViewColumn *viewcol;
	MgContextNode *cnode;

	cnode = mg_context_find_node_for_param (grid->priv->core->work_context, param);
	
	pos = g_slist_index (grid->priv->core->work_context->nodes, cnode);	
	g_assert (pos >= 0);

	viewcol = gtk_tree_view_get_column (GTK_TREE_VIEW (grid->priv->treeview), pos);

	/* Sets the column's visibility */
	gtk_tree_view_column_set_visible (viewcol, showit);
}

static void compute_row_boundaries (MgWorkGrid *grid);
static void tree_view_selection_changed_cb (GtkTreeSelection *selection, MgWorkGrid *grid);
static void create_new_tree_view (MgWorkGrid *grid, GtkTreeModel *tree_model);
static void tree_model_row_changed_cb (GtkTreeModel *treemodel, GtkTreePath *arg1, GtkTreeIter *arg2, MgWorkGrid *grid);
static gboolean compare_pk_values (GSList *list1, GSList *list2);

/*
 * Creates a GtkTreeView if necessary, and adds or update the data in grid->priv->core->data_rs in it.
 * if @keep_user_modifs is TRUE, then the update process tries to keep data as modified by the user
 */
static void
update_simple_grid (MgWorkGrid *grid, gboolean keep_user_modifs)
{
	GtkTreeModel *tree_model;

	/* REM: if a treeview already exists, then we don't need to create a new one as the
	 * SELECT query will have returned the same columns (number and type), we just need
	 * to update the model.
	 */

	/* new treeview if necessary */
	if (!grid->priv->treeview) {
		/* Model creation: 
		 * --> 1 column which refers to the row number to be displayed at the selected row
		 *     (or -1 for a new row, or -2 as a temporary value for a row which will soon be deleted)
		 * --> 1 column which holds the changes the user has made to the model; the original data
		 *     is not modified. This column is a ModelUserModifiedRow structure (all
		 *     the data in each of these structures is allocated on demand, and managed in priv->modified_rows).
		 */
		tree_model = GTK_TREE_MODEL (gtk_list_store_new (2, G_TYPE_INT, G_TYPE_POINTER));
		create_new_tree_view (grid, tree_model);
		g_object_unref (G_OBJECT (tree_model));

		gtk_container_add (GTK_CONTAINER (grid->priv->scroll), grid->priv->treeview);
		gtk_widget_show (grid->priv->treeview);
		
	}
	else
		tree_model = gtk_tree_view_get_model (GTK_TREE_VIEW (grid->priv->treeview));

	g_object_set_data (G_OBJECT (tree_model), "grid", grid);

	/* temporary disabling signal to update the buttons status */
	g_signal_handlers_block_by_func (G_OBJECT (tree_model),
					 G_CALLBACK (tree_model_row_changed_cb), grid);

	/* Filling or updating the data model */
	compute_row_boundaries (grid);
	if (grid->priv->core->data_rs) {
		GtkTreeIter current_iter;
		gint current_row;
		gint row_count;
		gboolean mode_new_data = TRUE; /* TRUE if there is not yet any data */
		GSList *kept_user_modif_rows = NULL;
		GSList *query_pk_fields = NULL;

		if (keep_user_modifs) 
			query_pk_fields = mg_query_get_target_pkfields (grid->priv->core->query_select, 
									grid->priv->core->modif_target);
		
		/* setting the current pointers (in the model and in the resultset) */
		row_count = mg_resultset_get_nbtuples (grid->priv->core->data_rs);
		current_row = grid->priv->sample_first_row;
#ifdef debug
		g_print ("-> %d row(s)\n", row_count);
#endif
		
		while (current_row < grid->priv->sample_last_row) {
			gboolean model_row = -1;
			
			/* set the iter position for this round, while keeping the new rows (created by the user) */
			while (model_row == -1) {
				if (current_row == grid->priv->sample_first_row) 
					mode_new_data = !gtk_tree_model_get_iter_first (tree_model, &current_iter);
				else
					mode_new_data = !gtk_tree_model_iter_next (tree_model, &current_iter);
				if (mode_new_data)
					gtk_list_store_append (GTK_LIST_STORE (tree_model), &current_iter);
				
				if (mode_new_data || !keep_user_modifs)
					model_row = current_row;
				else 
					gtk_tree_model_get (tree_model, &current_iter, COLUMN_ROW_NUM, &model_row, -1);

				if (model_row == -1) {
					ModelUserModifiedRow *stored_row;
					gtk_tree_model_get (tree_model, &current_iter, COLUMN_USER_MODIFS_ROW, &stored_row, -1);
					if (stored_row) {
						kept_user_modif_rows = g_slist_append (kept_user_modif_rows, stored_row);
						grid->priv->modified_rows = g_slist_remove (grid->priv->modified_rows, 
											    stored_row);
					}
				}
			}

			/* set model's columns */
			gtk_list_store_set (GTK_LIST_STORE (tree_model), &current_iter, 
					    COLUMN_ROW_NUM, current_row, COLUMN_USER_MODIFS_ROW, NULL, -1);
			
			/* see if we have a user modified value for that row that we want to keep */
			if (keep_user_modifs && !mode_new_data) {
				GSList *stored_list = grid->priv->modified_rows;
				ModelUserModifiedRow *stored_row = NULL;

				while (stored_list && !stored_row) {
					ModelUserModifiedRow *tmp = MODEL_USER_MODIFIED_ROW (stored_list->data);
					gboolean equal = TRUE;
					
					if (tmp->pkey) {
						GSList *pk_values = NULL, *list;
						list = query_pk_fields;
						while (list) {
							const GdaValue *value;
							gint col = mg_entity_get_field_index (MG_ENTITY (grid->priv->core->query_select),
											      MG_FIELD (list->data));
							value = mg_resultset_get_gdavalue (grid->priv->core->data_rs, current_row, col);
							pk_values = g_slist_append (pk_values, value);
							list = g_slist_next (list);
						}
						
						equal = compare_pk_values (tmp->pkey, pk_values);
						g_slist_free (pk_values);
					}
					else
						equal = FALSE;

					if (equal) {
						/* keep that ModelUserModifiedRow */
						stored_row = tmp;
						kept_user_modif_rows = g_slist_append (kept_user_modif_rows, stored_row);
						stored_list = g_slist_next (stored_list);
						grid->priv->modified_rows = g_slist_remove (grid->priv->modified_rows, 
											    stored_row);
						gtk_list_store_set (GTK_LIST_STORE (tree_model), &current_iter, 
								    COLUMN_USER_MODIFS_ROW, stored_row, -1);
					}
					else 
						stored_list = g_slist_next (stored_list);
				}
			}
			current_row ++;
		}
		
		/* remove extra remaining rows */
		if (row_count == 0)
			gtk_list_store_clear (GTK_LIST_STORE (tree_model));
		else {
			mode_new_data = !gtk_tree_model_iter_next (tree_model, &current_iter);			
			if (!mode_new_data) 
				while (gtk_list_store_remove (GTK_LIST_STORE (tree_model), &current_iter));
		}

		clean_user_modif_rows (grid);
		grid->priv->modified_rows = kept_user_modif_rows;

		if (keep_user_modifs) 
			g_slist_free (query_pk_fields);
	}
	else {
		clean_user_modif_rows (grid);
		gtk_list_store_clear (GTK_LIST_STORE (tree_model));
	}

	/* re-enabling signal to update the buttons status */
	g_signal_handlers_unblock_by_func (G_OBJECT (tree_model),
					   G_CALLBACK (tree_model_row_changed_cb), grid);

	/* make sure the new selected row is taken into account */
	if (grid->priv->core->data_rs) {
		GtkTreeSelection *selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (grid->priv->treeview));
		tree_view_selection_changed_cb (selection, grid);
	}

	modif_buttons_update (grid);
}

static void
compute_row_boundaries (MgWorkGrid *grid)
{
	/* Note: sample_first_row and sample_last_row evolve by complete truncks of size sample_size if sample_size != 0 */
	if (grid->priv->core->data_rs) {
		gint row_count;

		row_count = mg_resultset_get_nbtuples (grid->priv->core->data_rs);
		if (grid->priv->sample_size) { /* sample size is set */
			if (grid->priv->sample_first_row > row_count) {
				while (grid->priv->sample_first_row > row_count) {
					grid->priv->sample_first_row -= grid->priv->sample_size;
					if (grid->priv->sample_first_row <= 0)
						grid->priv->sample_first_row = 0;
				}
				
			}

			grid->priv->sample_last_row = grid->priv->sample_first_row + grid->priv->sample_size;
			if (grid->priv->sample_last_row > row_count)
				grid->priv->sample_last_row = row_count;
		}
		else { /* sample size is not set */
			grid->priv->sample_first_row = 0;
			grid->priv->sample_last_row = row_count - 1;
		}
	}
	else {
		grid->priv->sample_first_row = 0;
		grid->priv->sample_last_row = 0;
	}
}

static gboolean
compare_pk_values (GSList *list1, GSList *list2)
{
	gboolean equal = TRUE;
	GSList *l1, *l2;

	g_return_val_if_fail ((list1 && list2) || (!list1 && !list2), FALSE);
	if (!list1)
		return TRUE;

	l1 = list1;
	l2 = list2;

	while (l1 && l2 && equal) {
		if ((!l1->data || gda_value_is_null ((GdaValue *)l1->data)) &&
		    l2->data && !gda_value_is_null ((GdaValue *)l2->data))
			equal = FALSE;
		if ((!l2->data || gda_value_is_null ((GdaValue *)l2->data)) &&
		    l1->data && !gda_value_is_null ((GdaValue *)l1->data))
			equal = FALSE;
		if (l1->data && l2->data && 
		    (gda_value_get_type ((GdaValue *)l1->data) == gda_value_get_type ((GdaValue *)l2->data)))
			equal = !gda_value_compare ((GdaValue *)l1->data, (GdaValue *)l2->data);
		l1 = g_slist_next (l1);
		l2 = g_slist_next (l2);
	}

	g_return_val_if_fail ((list1 && list2) || (!list1 && !list2), FALSE);
	return equal;
}

/*
 * creates a new string where underscores '_' are replaced by double underscores '__'
 * WARNING: the old string is free'ed so it is possible to do "str=double_underscores(str);"
 */
static gchar *
replace_double_underscores (gchar *str)
{
        gchar **arr;
        gchar *ret;
	
        arr = g_strsplit (str, "_", 0);
        ret = g_strjoinv ("__", arr);
        g_strfreev (arr);
	g_free (str);
	
        return ret;
}

static ColumnData *get_column_data (MgWorkGrid *grid, MgContextNode *node);
static gboolean tree_view_event_cb (GtkWidget *treeview, GdkEvent *event, MgWorkGrid *grid);
static void data_cell_value_changed (GtkCellRenderer *renderer, const gchar *path, const GdaValue *new_value,
				     MgWorkGrid *grid);
static void data_cell_status_changed (GtkCellRenderer *renderer, const gchar *path, 
				      MgDataEntryAttribute requested_action, MgWorkGrid *grid);
static void cell_render_set_attributes (GtkTreeViewColumn *tree_column,
					GtkCellRenderer *cell,
					GtkTreeModel *tree_model,
					GtkTreeIter *iter, MgWorkGrid *grid);
static gint tree_sortable_sort_values (GtkTreeModel *model, GtkTreeIter *itera, GtkTreeIter *iterb, MgContextNode *node);
static void
create_new_tree_view (MgWorkGrid *grid, GtkTreeModel *tree_model)
{
	gint i;
	GtkTreeView *tree_view;
	GtkTreeSelection *selection;
	GSList *list;

	/* Actual GtkTreeView widget creation */
	tree_view = GTK_TREE_VIEW (gtk_tree_view_new_with_model (tree_model));
	gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (tree_view), TRUE);
	gtk_tree_view_set_enable_search (GTK_TREE_VIEW (tree_view), TRUE);

	g_assert (!grid->priv->treeview);
	grid->priv->treeview = GTK_WIDGET (tree_view);
	g_signal_connect (G_OBJECT (tree_view), "event",
			  G_CALLBACK (tree_view_event_cb), grid);

	/* selection and signal handling */
	selection = gtk_tree_view_get_selection (tree_view);
	g_signal_connect (G_OBJECT (selection), "changed",
			  G_CALLBACK (tree_view_selection_changed_cb), grid);
	g_signal_connect (G_OBJECT (tree_model), "row_changed",
			  G_CALLBACK (tree_model_row_changed_cb), grid);

	/* Creation of the columns in the treeview, to fit the parameters in the context */
	list = grid->priv->core->work_context->nodes;
	i = 0;
	while (list) {
		MgParameter *param;
		MgContextNode *node;
		GtkTreeViewColumn *column;
		GtkCellRenderer *renderer;
		ColumnData *column_data;

		node = MG_CONTEXT_NODE (list->data);

		/* update the list of columns data */
		column_data = get_column_data (grid, node);
		if (!column_data) {
			column_data = g_new0 (ColumnData, 1);
			column_data->node = node;
			column_data->info_shown = grid->priv->default_show_info_cell;
			column_data->data_locked = grid->priv->core->modif_target ? FALSE : TRUE;
			grid->priv->columns_data = g_slist_append (grid->priv->columns_data, column_data);
		}

		/* create renderers */
		if ((param = node->param)) { /* single direct parameter */
			MgConf *conf;
			MgServerDataType *type;
			MgDataHandler *dh = NULL;
			gchar *plugin = NULL;
			gchar *title;
			
			type = mg_parameter_get_data_type (param);
			g_assert (type);

			title = replace_double_underscores (g_strdup (mg_base_get_name (MG_BASE (param))));
			if (!title)
				title = g_strdup (_("No title"));

			conf = mg_base_get_conf (MG_BASE (type));
			g_object_get (G_OBJECT (param), "handler_plugin", &plugin, NULL);
			if (plugin) {
				dh = mg_server_get_handler_by_name (mg_conf_get_server (conf), plugin);

				/* test if plugin can handle the parameter's data type */
				if (!mg_data_handler_accepts_gda_type (dh, mg_server_data_type_get_gda_type (type)))
					dh = NULL;
			}

			if (!dh)
				dh = mg_server_data_type_get_handler (type);
						
			renderer = mg_data_handler_get_cell_renderer (dh, mg_server_data_type_get_gda_type (type));
			column_data->data_cell = renderer;
			gtk_tree_view_insert_column_with_data_func (tree_view, i, title, renderer,
								    (GtkTreeCellDataFunc) cell_render_set_attributes, 
								    grid, NULL);
			column = gtk_tree_view_get_column (tree_view, i);
			g_free (title);
		}
		else { /* parameters depending on a sub query */
			gchar *title = NULL;
			
			title = replace_double_underscores (g_strdup (mg_base_get_name (MG_BASE (node->query))));
			/* FIXME: find a better label */
			if (!title)
				title = g_strdup (_("Value"));

			renderer = mg_data_cell_renderer_combo_new (mg_base_get_conf (MG_BASE (grid->priv->core)),
								    grid->priv->core->work_context, node);
			column_data->data_cell = renderer;
			gtk_tree_view_insert_column_with_data_func (tree_view, i, title, renderer,
								    (GtkTreeCellDataFunc) cell_render_set_attributes, 
								    grid, NULL);
			column = gtk_tree_view_get_column (tree_view, i);
			g_free (title);
		}
		g_object_set_data (G_OBJECT (column), "data_renderer", renderer);

		/* settings and signals */
		g_signal_connect (G_OBJECT (renderer), "changed", 
				  G_CALLBACK (data_cell_value_changed), grid);
		g_object_set (G_OBJECT (renderer), "editable", !column_data->data_locked, NULL);
		if (g_object_class_find_property (G_OBJECT_GET_CLASS (renderer), "set_default_if_invalid"))
			g_object_set (G_OBJECT (renderer), "set_default_if_invalid", TRUE, NULL);
		g_object_set_data (G_OBJECT (renderer), "context-node", node);
		g_object_set_data (G_OBJECT (renderer), "tree-model", tree_model);
		g_object_set_data (G_OBJECT (renderer), "work-core", grid->priv->core);
		g_object_set_data (G_OBJECT (column), "context-node", node);

		/* Adding the GdaValue's information cell as another GtkCellRenderer */
		renderer = mg_data_cell_renderer_info_new ();
		column_data->info_cell = renderer;
		gtk_tree_view_column_pack_end (column, renderer, FALSE);
		gtk_tree_view_column_set_cell_data_func (column, renderer, 
							 (GtkTreeCellDataFunc) cell_render_set_attributes, 
							 grid, NULL);
		g_signal_connect (G_OBJECT (renderer), "status_changed",
				  G_CALLBACK (data_cell_status_changed), grid);
		g_object_set (G_OBJECT (renderer), "visible", column_data->info_shown, NULL);
		g_object_set_data (G_OBJECT (renderer), "context-node", node);
		g_object_set_data (G_OBJECT (renderer), "tree-model", tree_model);
		g_object_set_data (G_OBJECT (renderer), "work-core", grid->priv->core);


		/* Sorting data */
		gtk_tree_view_column_set_sort_column_id (column, i);
		gtk_tree_sortable_set_sort_func (GTK_TREE_SORTABLE (tree_model),
                                                 i, (GtkTreeIterCompareFunc) tree_sortable_sort_values, node, NULL);
		list = g_slist_next (list);
		i++;
	}

	/*
	 * Hiding some entries of the grid for which correspond to alias parameters
	 */
	list = grid->priv->core->no_show_params;
	while (list) {
		treeview_show_param_column (grid, MG_PARAMETER (list->data), FALSE);
		
		list = g_slist_next (list);
	}
}

static gboolean 
tree_view_event_cb (GtkWidget *treeview, GdkEvent *event, MgWorkGrid *grid)
{
	gboolean done = FALSE;

	if (event->type == GDK_KEY_PRESS) {
		GdkEventKey *ekey = (GdkEventKey *) event;
		guint modifiers = gtk_accelerator_get_default_mod_mask ();		

		/* Tab to move one column left or right */
		if (ekey->keyval == GDK_Tab) {
			GtkTreeViewColumn *column;
			GtkTreePath *path;

			/* FIXME: if a column is currently edited, then make sure the editing of that cell is not cancelled */
			gtk_tree_view_get_cursor (GTK_TREE_VIEW (treeview), &path, &column);
			if (column && path) {
				GList *columns = gtk_tree_view_get_columns (GTK_TREE_VIEW (treeview));
				GList *col;
				GtkCellRenderer *renderer;

				/* change column */
				col = g_list_find (columns, column);
				g_return_val_if_fail (col, FALSE);

				if (((ekey->state & modifiers) == GDK_SHIFT_MASK) || ((ekey->state & modifiers) == GDK_CONTROL_MASK))
					col = g_list_previous (col); /* going to previous column */
				else 
					col = g_list_next (col); /* going to next column */

				if (col) {
					renderer = g_object_get_data (G_OBJECT (col->data), "data_renderer");
					gtk_tree_view_set_cursor_on_cell (GTK_TREE_VIEW (treeview), path, 
									  GTK_TREE_VIEW_COLUMN (col->data), 
									  renderer, FALSE);
					gtk_widget_grab_focus (treeview);
					done = TRUE;
				}
				g_list_free (columns);
			}
			if (path)
				gtk_tree_path_free (path);
		}
		
		/* DELETE to delete the selected row */
		if (ekey->keyval == GDK_Delete) {
			if (((ekey->state & modifiers) == GDK_SHIFT_MASK) || ((ekey->state & modifiers) == GDK_CONTROL_MASK))
				modif_actions_real_do (grid, 'u');
			else
				modif_actions_real_do (grid, 'd');
			done = TRUE;
		}
	}

	return done;
}

static gint
tree_sortable_sort_values (GtkTreeModel *model, GtkTreeIter *itera, GtkTreeIter *iterb, MgContextNode *node)
{
        GdaValue *valuea, *valueb;
        gint retval = 1;
	MgWorkGrid *grid;

	grid = 	g_object_get_data (G_OBJECT (model), "grid");

	valuea = utility_grid_model_get_value (model, itera, grid->priv->core, node, FALSE, NULL);
	valueb = utility_grid_model_get_value (model, iterb, grid->priv->core, node, FALSE, NULL);

	if (gda_value_get_type (valuea) == gda_value_get_type (valueb)) {
		if (gda_value_get_type (valuea) == GDA_VALUE_TYPE_LIST) {
			const GList *lista, *listb;

			lista = gda_value_get_list (valuea);
			listb = gda_value_get_list (valueb);
			retval = 0;

			g_assert (g_list_length (lista) == g_list_length (listb));
			while (lista && listb && !retval) {
				GdaValue *vala = (GdaValue*) lista->data, *valb = (GdaValue*) listb->data;
				if (gda_value_is_null (vala)) {
					if (gda_value_is_null (valb))
						retval = 0;
					else
						retval = 1;
				}
				else {
					if (gda_value_is_null (valb))
						retval = -1;
					else
						retval = gda_value_compare (vala, valb);;
				}
				lista = g_list_next (lista);
				listb = g_list_next (listb);
			}
		}
		else
			retval = gda_value_compare (valuea, valueb);
	}
	else {
		if (gda_value_is_null (valuea))
			retval = -1;
		else
			retval = 1;
	}

	gda_value_free (valuea);
	gda_value_free (valueb);

        return retval;
}

static ColumnData *
get_column_data (MgWorkGrid *grid, MgContextNode *node)
{
	ColumnData *retval = NULL;
	GSList *list = grid->priv->columns_data;
	while (list && !retval) {
		if (COLUMN_DATA (list->data)->node == node)
			retval = COLUMN_DATA (list->data);

		list = g_slist_next (list);
	}

	return retval;
}

static void data_cell_update_context_params (MgContextNode *node, GdaValue *value);

static void
data_cell_value_changed (GtkCellRenderer *renderer, const gchar *path, const GdaValue *new_value, MgWorkGrid *grid)
{
	ModelUserModifiedValue *user_modif;
	GtkTreePath *treepath;
	GtkTreeIter iter;
	MgContextNode *context_node;
	GtkTreeModel *tree_model;
	MgWorkCore *core;
	
	context_node = g_object_get_data (G_OBJECT (renderer), "context-node");
	tree_model = g_object_get_data (G_OBJECT (renderer), "tree-model");
	if (path) {
		treepath = gtk_tree_path_new_from_string (path);
		if (! gtk_tree_model_get_iter (tree_model, &iter, treepath)) {
			gtk_tree_path_free (treepath);
			g_warning ("Can't get iter for path %s", path);
			return;
		}
		gtk_tree_path_free (treepath);
	}
	else {
		GtkTreeSelection *selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (grid->priv->treeview));
		if (!gtk_tree_selection_get_selected (selection, NULL, &iter)) {
			g_warning ("Can't get iter for current selection");
			return;
		}
	}

	core = g_object_get_data (G_OBJECT (renderer), "work-core");
	user_modif = grid_model_get_user_modifs (grid, tree_model, &iter, context_node, TRUE);

	if (!new_value || gda_value_is_null (new_value)) 
		grid_modif_struct_set_status (grid, user_modif, tree_model, &iter, core, MG_DATA_ENTRY_IS_NULL);
	else 
		grid_modif_struct_set_value (grid, user_modif, tree_model, &iter, core, new_value);

	data_cell_update_context_params (context_node, new_value);
}

static void
data_cell_update_context_params (MgContextNode *node, GdaValue *value)
{
	if (node->param)
		mg_parameter_set_value (node->param, value);
	else {
		const GList *values = NULL;
		GSList *params = node->params;
		
		if (value && !gda_value_is_null (value)) {
			g_assert (gda_value_isa (value, GDA_VALUE_TYPE_LIST));
			values = gda_value_get_list (value);
			g_assert (g_list_length (values) >= g_slist_length (params));
		}

		while (params) {
			if (values) 
				mg_parameter_set_value (MG_PARAMETER (params->data),
							g_list_nth_data (values, 
									 GPOINTER_TO_INT (g_hash_table_lookup 
											  (node->params_pos_in_query, params->data))));
			else
				mg_parameter_set_value (MG_PARAMETER (params->data), NULL);
			params = g_slist_next (params);
		}
	}
}

static void
data_cell_status_changed (GtkCellRenderer *renderer, const gchar *path, MgDataEntryAttribute requested_action,
			  MgWorkGrid *grid)
{
	ModelUserModifiedValue *user_modif;
	GtkTreePath *treepath;
	GtkTreeIter iter;
	MgContextNode *context_node;
	GtkTreeModel *tree_model;
	MgWorkCore *core;
	GdaValue *new_value;

	context_node = g_object_get_data (G_OBJECT (renderer), "context-node");
	tree_model = g_object_get_data (G_OBJECT (renderer), "tree-model");
	treepath = gtk_tree_path_new_from_string (path);
	if (! gtk_tree_model_get_iter (tree_model, &iter, treepath)) {
		gtk_tree_path_free (treepath);
		g_warning ("Can't get iter for path %s", path);
		return;
	}
	gtk_tree_path_free (treepath);

	core = g_object_get_data (G_OBJECT (renderer), "work-core");
	user_modif = grid_model_get_user_modifs (grid, tree_model, &iter, context_node, TRUE);
	grid_modif_struct_set_status (grid, user_modif, tree_model, &iter, core, requested_action);

	/* updating the parameters */
	new_value = utility_grid_model_get_value (tree_model, &iter, core, context_node, FALSE, NULL);
	data_cell_update_context_params (context_node, new_value);
	gda_value_free (new_value);
}


static void
cell_render_set_attributes (GtkTreeViewColumn *tree_column,
			    GtkCellRenderer *cell,
			    GtkTreeModel *tree_model,
			    GtkTreeIter *iter, MgWorkGrid *grid)
{
	MgContextNode *node;
	ModelUserModifiedRow *user_modifs;
	guint attributes;
	GdaValue *value;
	gboolean to_be_deleted = FALSE;
	ColumnData *column_data;
		
	if (!grid->priv->core->data_rs) 
		return;

	gtk_tree_model_get (tree_model, iter, COLUMN_USER_MODIFS_ROW, &user_modifs, -1);
	if (user_modifs)
		to_be_deleted = user_modifs->to_be_deleted;
	node = g_object_get_data (G_OBJECT (tree_column), "context-node");
	column_data = get_column_data (grid, node);
	
	value = utility_grid_model_get_value (tree_model, iter, grid->priv->core, node, FALSE, &attributes);
	if (node->param) { /* single parameter */
		g_object_set (G_OBJECT (cell), 
			      "value", value,
			      "editable", !column_data->data_locked,
			      "value_attributes", attributes,
			      "cell_background", MG_COLOR_NORMAL_MODIF,
			      "cell_background-set", user_modifs ? TRUE : FALSE,
			      "to_be_deleted", to_be_deleted, 
			      NULL);
	}
	else { /* multiple parameters depending on a query */
		const GList *values = NULL;

		if (!gda_value_is_null (value))
			values = (GList *) gda_value_get_list (value);

		g_object_set (G_OBJECT (cell), 
			      "values_complete", values,
			      "editable", !column_data->data_locked,
			      "value_attributes", attributes,
			      "cell_background", MG_COLOR_NORMAL_MODIF,
			      "cell_background-set", user_modifs ? TRUE : FALSE,
			      "to_be_deleted", to_be_deleted, 
			      NULL);
	}
	gda_value_free (value);
}

static gboolean commit_do_work (MgWorkGrid *grid, GtkTreeIter *iter);
static void mg_work_grid_set_params_from_iter (MgWorkGrid *grid, GtkTreeIter *iter);
static void
tree_view_selection_changed_cb (GtkTreeSelection *selection, MgWorkGrid *grid)
{
	GtkTreeIter iter;
	GtkTreeModel *model;
	gboolean row_selected = FALSE;
	gboolean has_selection = TRUE;

	/* set all the parameters from the values of the current row */
	if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
		GtkTreePath *path;
		gchar *str;

		path = gtk_tree_model_get_path (model, &iter);
		str = gtk_tree_path_to_string (path);
		gtk_tree_path_free (path);

		if ((grid->priv->mode & MG_ACTION_MODIF_AUTO_COMMIT) &&
		    grid->priv->current_selection && 
		    (strcmp (str, grid->priv->current_selection))) {
			GtkTreeIter olditer;

			path = gtk_tree_path_new_from_string (grid->priv->current_selection);
			if (gtk_tree_model_get_iter (model, &olditer, path)) {
				/* save the previously modified row */
				if (commit_do_work (grid, &olditer)) {
					refresh_all_rows (grid, FALSE);
					has_selection = gtk_tree_selection_get_selected (selection, NULL, &iter);
				}
			}
			gtk_tree_path_free (path);
		}
		g_free (str);
		
		if (has_selection) {
			path = gtk_tree_model_get_path (model, &iter);
			if (grid->priv->current_selection)
				g_free (grid->priv->current_selection);
			grid->priv->current_selection = gtk_tree_path_to_string (path);
			gtk_tree_path_free (path);

			grid->priv->internal_params_change = TRUE;
			mg_work_grid_set_params_from_iter (grid, &iter);
			grid->priv->internal_params_change = FALSE;

			row_selected = TRUE;
		}
	}
	else {
		if (grid->priv->current_selection) {
			g_free (grid->priv->current_selection);
			grid->priv->current_selection = NULL;
		}
		has_selection = FALSE;
	}

	if (!has_selection) {
		/* render all the parameters invalid */
		GSList *list = grid->priv->core->work_context->parameters;
		while (list) {
			MgParameter *tmp;
			g_object_get (G_OBJECT (list->data), "full_bind", &tmp, NULL);
			if (! tmp) 
				mg_parameter_declare_invalid (MG_PARAMETER (list->data));

			list = g_slist_next (list);
		}
	}

	/* update the actions buttons */
	modif_buttons_update (grid);

#ifdef debug_signal
	g_print (">> 'SELECTION_CHANGED' from %s %p\n", G_OBJECT_TYPE_NAME (grid), grid);
#endif
	g_signal_emit (G_OBJECT (grid), mg_work_grid_signals[SELECTION_CHANGED], 0, row_selected);
#ifdef debug_signal
	g_print ("<< 'SELECTION_CHANGED' from %s %p\n", G_OBJECT_TYPE_NAME (grid), grid);
#endif
}

static void
tree_model_row_changed_cb (GtkTreeModel *treemodel, GtkTreePath *arg1, GtkTreeIter *arg2, MgWorkGrid *grid)
{
	/* update the actions buttons */
	modif_buttons_update (grid);
}

/*
 * mg_work_grid_set_params_from_iter
 * @grid:
 * @iter:
 *
 * Sets the parameters in 'core->work_context' from the values stored for the row provided by @iter.
 */
static void
mg_work_grid_set_params_from_iter (MgWorkGrid *grid, GtkTreeIter *iter)
{
	GSList *list;

	list = grid->priv->core->work_context->nodes;
	while (list) {
		MgContextNode *node;
		GdaValue *value;
		GtkTreeModel *model;
		guint attributes;

		model = gtk_tree_view_get_model (GTK_TREE_VIEW (grid->priv->treeview));
		node = MG_CONTEXT_NODE (list->data);
		value = utility_grid_model_get_value (model, iter, grid->priv->core, node, TRUE, &attributes);

		if (node->param)  /* single direct parameter */ {
			MgParameter *tmp;
			g_object_get (G_OBJECT (node->param), "full_bind", &tmp, NULL);
			if (! tmp) {
				mg_parameter_set_value (node->param, value);
				g_object_set (G_OBJECT (node->param), "use_default_value", 
					      attributes & MG_DATA_ENTRY_IS_DEFAULT ? TRUE : FALSE, NULL);
			}
		}
		else { /* parameters depending on a sub query */
			GSList *plist;
			GList *vlist = NULL;

			g_return_if_fail (!value || gda_value_is_null (value) || 
								       gda_value_isa (value, GDA_VALUE_TYPE_LIST));
			
			plist = node->params;
			if (value && !gda_value_is_null (value)) {
				vlist = (GList *) gda_value_get_list (value);
				g_assert (g_slist_length (plist) == g_list_length (vlist));
			}

			while (plist && (!value || gda_value_is_null (value) || vlist)) {
				if (value && !gda_value_is_null (value)) {
					mg_parameter_set_value (MG_PARAMETER (plist->data), (GdaValue *)(vlist->data));
					if (attributes & MG_DATA_ENTRY_IS_DEFAULT)
						g_object_set (G_OBJECT (plist->data), "use_default_value", TRUE, NULL);
					vlist = g_list_next (vlist);
				}
				else {
					mg_parameter_set_value (MG_PARAMETER (plist->data), NULL);
					if (attributes & MG_DATA_ENTRY_IS_DEFAULT)
						g_object_set (G_OBJECT (plist->data), "use_default_value", TRUE, NULL);
				}
				plist = g_slist_next (plist);
			}
		}

		gda_value_free (value);
		list = g_slist_next (list);
	}
}

/*
 *
 * Modification buttons (Commit changes, Reset grid, New entry, Delete)
 *
 */
static void action_new_cb (GtkAction *action, MgWorkGrid *grid);
static void action_delete_cb (GtkAction *action, MgWorkGrid *grid);
static void action_undelete_cb (GtkAction *action, MgWorkGrid *grid);
static void action_commit_cb (GtkAction *action, MgWorkGrid *grid);
static void action_reset_cb (GtkAction *action, MgWorkGrid *grid);
static void action_first_chunck_cb (GtkAction *action, MgWorkGrid *grid);
static void action_prev_chunck_cb (GtkAction *action, MgWorkGrid *grid);
static void action_next_chunck_cb (GtkAction *action, MgWorkGrid *grid);
static void action_last_chunck_cb (GtkAction *action, MgWorkGrid *grid);


static GtkActionEntry ui_actions[] = {
	{ "WorkWidgetNew", GTK_STOCK_NEW, "_New", NULL, "Create a new data entry", G_CALLBACK (action_new_cb)},
	{ "WorkWidgetDelete", GTK_STOCK_DELETE, "_Delete", NULL, "Delete the selected entry", G_CALLBACK (action_delete_cb)},
	{ "WorkWidgetUndelete", GTK_STOCK_UNDELETE, "_Undelete", NULL, "Cancels the deletion of the selected entry", 
	  G_CALLBACK (action_undelete_cb)},
	{ "WorkWidgetCommit", GTK_STOCK_SAVE, "_Commit", NULL, "Commit the latest changes", G_CALLBACK (action_commit_cb)},
	{ "WorkWidgetReset", GTK_STOCK_REFRESH, "_Reset", NULL, "Reset the data", G_CALLBACK (action_reset_cb)},
	{ "WorkWidgetFirstChunck", GTK_STOCK_GOTO_FIRST, "_First chunck", NULL, "Go to first chunck of records", 
	  G_CALLBACK (action_first_chunck_cb)},
	{ "WorkWidgetLastChunck", GTK_STOCK_GOTO_LAST, "_Last chunck", NULL, "Go to last chunck of records", 
	  G_CALLBACK (action_last_chunck_cb)},
	{ "WorkWidgetPrevChunck", GTK_STOCK_GO_BACK, "_Previous chunck", NULL, "Go to previous chunck of records", 
	  G_CALLBACK (action_prev_chunck_cb)},
	{ "WorkWidgetNextChunck", GTK_STOCK_GO_FORWARD, "Ne_xt chunck", NULL, "Go to next chunck of records",
	  G_CALLBACK (action_next_chunck_cb)}
};

static const gchar *ui_actions_info =
"<ui>"
"  <toolbar  name='ToolBar'>"
"    <toolitem action='WorkWidgetNew'/>"
"    <toolitem action='WorkWidgetDelete'/>"
"    <toolitem action='WorkWidgetUndelete'/>"
"    <toolitem action='WorkWidgetCommit'/>"
"    <toolitem action='WorkWidgetReset'/>"
"    <separator/>"
"    <toolitem action='WorkWidgetFirstChunck'/>"
"    <toolitem action='WorkWidgetPrevChunck'/>"
"    <toolitem action='WorkWidgetNextChunck'/>"
"    <toolitem action='WorkWidgetLastChunck'/>"
"  </toolbar>"
"</ui>";


static GtkWidget *
modif_buttons_make (MgWorkGrid *grid)
{
	GtkActionGroup *actions;
	GtkUIManager *ui;
	GtkWidget *table, *wid;

	table = gtk_table_new (1, 2, FALSE);

	/* action buttons */
	actions = gtk_action_group_new ("Actions");
	grid->priv->actions_group = actions;

	gtk_action_group_add_actions (actions, ui_actions, G_N_ELEMENTS (ui_actions), grid);

	ui = gtk_ui_manager_new ();
	gtk_ui_manager_insert_action_group (ui, actions, 0);
	gtk_ui_manager_add_ui_from_string (ui, ui_actions_info, -1, NULL);
	grid->priv->uimanager = ui;
	grid->priv->modif_all = gtk_ui_manager_get_widget (ui, "/ToolBar");
	gtk_table_attach_defaults (GTK_TABLE (table), grid->priv->modif_all, 0, 1, 0, 1);
	gtk_widget_show (grid->priv->modif_all);

	/* samples counter */
	wid = gtk_label_new ("? - ? / ?");
	gtk_widget_show (wid);
	grid->priv->nav_current_sample = wid;
	gtk_table_attach (GTK_TABLE (table), wid, 1, 2, 0, 1, 0, 0, 5, 0);

	return table;
}

static void
action_new_cb (GtkAction *action, MgWorkGrid *grid)
{
	modif_actions_real_do (grid, 'i');
}

static void
action_delete_cb (GtkAction *action, MgWorkGrid *grid)
{
	modif_actions_real_do (grid, 'd');
}

static void
action_undelete_cb (GtkAction *action, MgWorkGrid *grid)
{
	modif_actions_real_do (grid, 'u');
}

static void
action_commit_cb (GtkAction *action, MgWorkGrid *grid)
{
	modif_actions_real_do (grid, 'c');
}

static void
action_reset_cb (GtkAction *action, MgWorkGrid *grid)
{
	modif_actions_real_do (grid, 'r');
}

static void
action_first_chunck_cb (GtkAction *action, MgWorkGrid *grid)
{
	mg_work_grid_set_sample_start (grid, 0);	
}

static void
action_prev_chunck_cb (GtkAction *action, MgWorkGrid *grid)
{
	mg_work_grid_set_sample_start (grid, grid->priv->sample_first_row - grid->priv->sample_size);
}

static void
action_next_chunck_cb (GtkAction *action, MgWorkGrid *grid)
{
	mg_work_grid_set_sample_start (grid, grid->priv->sample_first_row + grid->priv->sample_size);
}

static void
action_last_chunck_cb (GtkAction *action, MgWorkGrid *grid)
{
	mg_work_grid_set_sample_start (grid, G_MAXINT - grid->priv->sample_size);
}

static void
modif_buttons_update (MgWorkGrid *grid)
{
	GtkTreeSelection *select;
	GtkTreeIter iter;
	GtkTreeModel *model;
	gboolean changed=FALSE;
	gboolean has_selection;
	gchar *str;
	GtkAction *action;
	gint row_count = 0;

	if (!grid->priv->treeview || !grid->priv->core) {
		action = gtk_ui_manager_get_action (grid->priv->uimanager, "/ToolBar/WorkWidgetDelete");
		g_object_set (G_OBJECT (action), "visible", TRUE, NULL);
		action = gtk_ui_manager_get_action (grid->priv->uimanager, "/ToolBar/WorkWidgetUndelete");
		g_object_set (G_OBJECT (action), "visible", FALSE, NULL);
		action = gtk_ui_manager_get_action (grid->priv->uimanager, "/ToolBar/WorkWidgetCommit");
		g_object_set (G_OBJECT (action), "sensitive", FALSE, NULL);
		action = gtk_ui_manager_get_action (grid->priv->uimanager, "/ToolBar/WorkWidgetReset");
		g_object_set (G_OBJECT (action), "sensitive", FALSE, NULL);
		action = gtk_ui_manager_get_action (grid->priv->uimanager, "/ToolBar/WorkWidgetNew");
		g_object_set (G_OBJECT (action), "sensitive", FALSE, NULL);
		action = gtk_ui_manager_get_action (grid->priv->uimanager, "/ToolBar/WorkWidgetDelete");
		g_object_set (G_OBJECT (action), "sensitive", FALSE, NULL);
		
		grid->priv->sample_first_row = 0;
		grid->priv->sample_last_row = 0;

		action = gtk_ui_manager_get_action (grid->priv->uimanager, "/ToolBar/WorkWidgetFirstChunck");
		g_object_set (G_OBJECT (action), "sensitive", FALSE, NULL);
		action = gtk_ui_manager_get_action (grid->priv->uimanager, "/ToolBar/WorkWidgetPrevChunck");
		g_object_set (G_OBJECT (action), "sensitive", FALSE, NULL);
		action = gtk_ui_manager_get_action (grid->priv->uimanager, "/ToolBar/WorkWidgetNextChunck");
		g_object_set (G_OBJECT (action), "sensitive", FALSE, NULL);
		action = gtk_ui_manager_get_action (grid->priv->uimanager, "/ToolBar/WorkWidgetLastChunck");
		g_object_set (G_OBJECT (action), "sensitive", FALSE, NULL);

		gtk_label_set_text (GTK_LABEL (grid->priv->nav_current_sample), "? - ? / ?");
		gtk_widget_set_sensitive (grid->priv->nav_current_sample, FALSE);

		return;
	}

	changed = mg_work_grid_has_been_changed (MG_WORK_WIDGET (grid));
	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (grid->priv->treeview));
	has_selection = gtk_tree_selection_get_selected (select, &model, &iter);

	/* FIXME: if the working mode is to allow the edition of one line at a time, then 
	 * we should check that the corresponding INSERT query is valid (or UPDATE, if INSERT is valid then
	 * UPDATE will also be valid, by construction) */
	action = gtk_ui_manager_get_action (grid->priv->uimanager, "/ToolBar/WorkWidgetCommit");
	g_object_set (G_OBJECT (action), "sensitive", changed ? TRUE : FALSE, NULL);

	action = gtk_ui_manager_get_action (grid->priv->uimanager, "/ToolBar/WorkWidgetReset");
	g_object_set (G_OBJECT (action), "sensitive", TRUE, NULL);

	action = gtk_ui_manager_get_action (grid->priv->uimanager, "/ToolBar/WorkWidgetNew");
	g_object_set (G_OBJECT (action), "sensitive", 
		      grid->priv->core->modif_target && 
		      mg_context_is_valid (grid->priv->core->args_context) ? TRUE : FALSE, NULL);

	if (has_selection && grid->priv->core->modif_target) {
		gboolean to_be_deleted = FALSE;
		ModelUserModifiedRow *user_modifs;
		gtk_tree_model_get (model, &iter, COLUMN_USER_MODIFS_ROW, &user_modifs, -1);
		if (user_modifs)
			to_be_deleted = user_modifs->to_be_deleted;

		action = gtk_ui_manager_get_action (grid->priv->uimanager, "/ToolBar/WorkWidgetDelete");
		g_object_set (G_OBJECT (action), "sensitive", !to_be_deleted, NULL);
		
		action = gtk_ui_manager_get_action (grid->priv->uimanager, "/ToolBar/WorkWidgetUndelete");
		g_object_set (G_OBJECT (action), "sensitive", to_be_deleted, NULL);
	}
	else {
		action = gtk_ui_manager_get_action (grid->priv->uimanager, "/ToolBar/WorkWidgetDelete");
		g_object_set (G_OBJECT (action), "sensitive", FALSE, NULL);
		action = gtk_ui_manager_get_action (grid->priv->uimanager, "/ToolBar/WorkWidgetUndelete");
		g_object_set (G_OBJECT (action), "sensitive", FALSE, NULL);
	}

	if (grid->priv->sample_size) 
		row_count = grid->priv->core->data_rs ? mg_resultset_get_nbtuples (grid->priv->core->data_rs) : 0;

	action = gtk_ui_manager_get_action (grid->priv->uimanager, "/ToolBar/WorkWidgetFirstChunck");
	g_object_set (G_OBJECT (action), "sensitive", 
		      grid->priv->sample_size ? (grid->priv->sample_first_row > 0 ? TRUE : FALSE) : FALSE, 
		      "visible", grid->priv->mode & MG_ACTION_NAVIGATION_ARROWS ? TRUE : FALSE, NULL);
	action = gtk_ui_manager_get_action (grid->priv->uimanager, "/ToolBar/WorkWidgetPrevChunck");
	g_object_set (G_OBJECT (action), "sensitive", 
		      grid->priv->sample_size ? (grid->priv->sample_first_row > 0 ? TRUE : FALSE) : FALSE,  
		      "visible", grid->priv->mode & MG_ACTION_NAVIGATION_ARROWS ? TRUE : FALSE, NULL);
	action = gtk_ui_manager_get_action (grid->priv->uimanager, "/ToolBar/WorkWidgetNextChunck");
	g_object_set (G_OBJECT (action), "sensitive", 
		      grid->priv->sample_size ? (grid->priv->sample_last_row < row_count ? TRUE : FALSE) : FALSE,  
		      "visible", grid->priv->mode & MG_ACTION_NAVIGATION_ARROWS ? TRUE : FALSE, NULL);
	action = gtk_ui_manager_get_action (grid->priv->uimanager, "/ToolBar/WorkWidgetLastChunck");
	g_object_set (G_OBJECT (action), "sensitive", 
		      grid->priv->sample_size ? (grid->priv->sample_last_row < row_count ? TRUE : FALSE) : FALSE,  
		      "visible", grid->priv->mode & MG_ACTION_NAVIGATION_ARROWS ? TRUE : FALSE, NULL);
	
	if (grid->priv->sample_last_row == 0)
		str = g_strdup_printf (_("? - ? / ?"));
	else
		str = g_strdup_printf ("%d - %d / %d", 
				       grid->priv->sample_first_row + 1, grid->priv->sample_last_row, row_count);
	gtk_label_set_text (GTK_LABEL (grid->priv->nav_current_sample), str);
	g_free (str);
	gtk_widget_set_sensitive (grid->priv->nav_current_sample, TRUE);
	if (grid->priv->mode & MG_ACTION_NAVIGATION_ARROWS)
		gtk_widget_show (grid->priv->nav_current_sample);
	else
		gtk_widget_hide (grid->priv->nav_current_sample);

	gtk_ui_manager_ensure_update (grid->priv->uimanager); 
}

static void     params_set_block_changed (MgWorkGrid *grid, gboolean block);
static gboolean commit_do_run_query (MgWorkGrid *grid, MgQuery *query);
static void
modif_actions_real_do (MgWorkGrid *grid, gchar action)
{
	GtkTreeModel *model;
	GtkTreeIter iter, sel_iter;
	GtkTreeSelection *select;
	gboolean select_ok;
	GtkTreePath *path;

	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (grid->priv->treeview));
	select_ok = gtk_tree_selection_get_selected (select, &model, &sel_iter);

	switch (action) {
	case 'c': /* COMMIT changes */
		if (grid->priv->core->data_rs) {
			GtkTreeIter iter;
			GtkTreeModel *model;
			GtkTreeSelection *selection;
			
			model = gtk_tree_view_get_model (GTK_TREE_VIEW (grid->priv->treeview));

			/* FIRST row */
			if (!gtk_tree_model_get_iter_first (model, &iter))
				return;
			commit_do_work (grid, &iter);
						
			/* OTHER rows */
			while (gtk_tree_model_iter_next (model, &iter)) 
				commit_do_work (grid, &iter);

			/* reset the parameters to their correct value, or render them invalid */
			selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (grid->priv->treeview));
			if (gtk_tree_selection_get_selected (selection, NULL, &iter)) 
				mg_work_grid_set_params_from_iter (grid, &iter);
			else {
				/* render all the parameters invalid */
				GSList *list = grid->priv->core->work_context->parameters;
				while (list) {
					MgParameter *tmp;
					g_object_get (G_OBJECT (list->data), "full_bind", &tmp, NULL);
					if (! tmp) 
						mg_parameter_declare_invalid (MG_PARAMETER (list->data));
					
					list = g_slist_next (list);
				}
			}
			
			refresh_all_rows (grid, TRUE);
		}
		break;
	case 'r': /* RESET grid */
		refresh_all_rows (grid, FALSE);
		break;
	case 'i': /* INSERT */
		/* Add a row to the model with -1 as the row number, if possible */
		if (select_ok)
			gtk_list_store_insert_after (GTK_LIST_STORE (model), &iter, &sel_iter);
		else
			gtk_list_store_append (GTK_LIST_STORE (model), &iter);
		gtk_list_store_set (GTK_LIST_STORE (model), &iter, COLUMN_ROW_NUM, -1, -1);
		gtk_tree_selection_select_iter (select, &iter);
		path = gtk_tree_model_get_path (model, &iter);
		gtk_tree_view_scroll_to_cell (GTK_TREE_VIEW (grid->priv->treeview), path, NULL, FALSE, 0, 0);
		gtk_tree_path_free (path);

		/* for params which will be invalid for this row, set them to the default value */
		grid_set_user_modifs_to_default_if_notnull (grid, model, &iter, grid->priv->core);
		break;
	case 'd': /* DELETE */
		if (select_ok) {
			gint row;
			ModelUserModifiedRow *user_modifs;

			gtk_tree_model_get (model, &sel_iter, COLUMN_ROW_NUM, &row, 
					    COLUMN_USER_MODIFS_ROW, &user_modifs, -1);
			if (row >= 0) {
				if (!user_modifs) {
					user_modifs = grid_model_make_user_modifs (grid, row);
					user_modifs->to_be_deleted = TRUE;
					gtk_list_store_set (GTK_LIST_STORE (model), &sel_iter, 
							    COLUMN_USER_MODIFS_ROW, user_modifs, -1);
				}
				else {
					GtkTreePath *path = gtk_tree_model_get_path (model, &sel_iter);
					user_modifs->to_be_deleted = TRUE;
					gtk_tree_model_row_changed (model, path, &sel_iter);
					gtk_tree_path_free (path);
				}
			}
			else {
				grid_clean_data_model_row (grid, model, &sel_iter);
				gtk_list_store_remove (GTK_LIST_STORE (model), &sel_iter);
			}
		}
		break;
	case 'u': /* UNDELETE */
		if (select_ok) {
			gint row;
			ModelUserModifiedRow *user_modifs;

			gtk_tree_model_get (model, &sel_iter, COLUMN_ROW_NUM, &row, 
					    COLUMN_USER_MODIFS_ROW, &user_modifs, -1);
			if (row >= 0) {
				if (user_modifs) {
					if (user_modifs->user_values) {
						GtkTreePath *path = gtk_tree_model_get_path (model, &sel_iter);
						user_modifs->to_be_deleted = FALSE;
						gtk_tree_model_row_changed (model, path, &sel_iter);
						gtk_tree_path_free (path);
					}
					else {
						grid->priv->modified_rows = g_slist_remove (grid->priv->modified_rows, 
											    user_modifs);
						user_modifs_row_free (user_modifs);
						gtk_list_store_set (GTK_LIST_STORE (model), &sel_iter, 
								    COLUMN_USER_MODIFS_ROW, NULL, -1);
					}
				}
			}
			else
				g_assert_not_reached ();
		}
		break;
	default:
		g_assert_not_reached ();
		break;
	}
}

static void
params_set_block_changed (MgWorkGrid *grid, gboolean block)
{
	GSList *list;

	list = grid->priv->core->work_context->nodes;
	while (list) {
		MgContextNode *node;
		node = MG_CONTEXT_NODE (list->data);

		if (node->param)
			if (block)
				mg_base_block_changed (MG_BASE (node->param));
			else
				mg_base_unblock_changed (MG_BASE (node->param));
		else {
			GSList *plist = node->params;

			while (plist) {
				if (block)
					mg_base_block_changed (MG_BASE (plist->data));
				else
					mg_base_unblock_changed (MG_BASE (plist->data));
				plist = g_slist_next (plist);
			}
		}
		list = g_slist_next (list);
	}
}

static gboolean
commit_do_work (MgWorkGrid *grid, GtkTreeIter *iter)
{
	GtkTreeModel *model;
	gboolean to_be_deleted = FALSE;
	MgQuery *query = NULL;
	gint row;
	ModelUserModifiedRow *user_modifs;

	model = gtk_tree_view_get_model (GTK_TREE_VIEW (grid->priv->treeview));
	gtk_tree_model_get (model, iter, 
			    COLUMN_ROW_NUM, &row,
			    COLUMN_USER_MODIFS_ROW, &user_modifs, -1);
	if (user_modifs)
		to_be_deleted = user_modifs->to_be_deleted;

	if (to_be_deleted)
		query = grid->priv->core->query_delete;
	else {
		if (row < 0)
			query = grid->priv->core->query_insert;
		else {
			if (user_modifs)
				query = grid->priv->core->query_update;
		}
	}
	
	if (query) {
		params_set_block_changed (grid, TRUE);
		mg_work_grid_set_params_from_iter (grid, iter);
		if (commit_do_run_query (grid, query)) 
			grid_clean_data_model_row (grid, model, iter);
		params_set_block_changed (grid, FALSE);
	}

	return query ? TRUE : FALSE;
}

/*
 * Runs the provided query which is part of the priv->core's modif queries
 */
static gboolean
commit_do_run_query (MgWorkGrid *grid, MgQuery *query)
{
	return mg_util_query_execute_modif (query, grid->priv->core->work_context,
					    grid->priv->mode & MG_ACTION_ASK_CONFIRM_INSERT,
					    grid->priv->mode & MG_ACTION_ASK_CONFIRM_UPDATE,
					    grid->priv->mode & MG_ACTION_ASK_CONFIRM_DELETE,
					    GTK_WIDGET (grid), NULL, NULL);
}


/*
 * grid_model_make_user_modifs
 *
 * Creates a new ModelUserModifiedRow, and fills the pkey part with the 
 * PK fields of the corresponding @row in the grid->priv->core->data_rs if @row >= 0
 */
static ModelUserModifiedRow *
grid_model_make_user_modifs (MgWorkGrid *grid, gint row)
{
	ModelUserModifiedRow *user_modifs;

	user_modifs = g_new0 (ModelUserModifiedRow, 1);
	grid->priv->modified_rows = g_slist_append (grid->priv->modified_rows, user_modifs);
	
	if (row >= 0) {
		/* store a copy of all the pk fields' values into the ModelUserModifiedRow structure */
		GSList *pk_fields = mg_query_get_target_pkfields (grid->priv->core->query_select, 
								  grid->priv->core->modif_target);
		if (pk_fields) {
			GSList *flist = pk_fields;
			GSList *pk_values = NULL;
			while (flist) {
				const GdaValue *value;
				gint col = mg_entity_get_field_index (MG_ENTITY (grid->priv->core->query_select),
								      MG_FIELD (flist->data));
				value = mg_resultset_get_gdavalue (grid->priv->core->data_rs, row, col);
				pk_values = g_slist_append (pk_values, gda_value_copy (value));
				
				flist = g_slist_next (flist);
			}
			g_slist_free (pk_fields);
			
			user_modifs->pkey = pk_values;
		}
	}

	return user_modifs;
}

/*
 * grid_model_get_user_modifs
 * From the data model @tree_model, tries to find a ModelUserModifiedValue structure which
 * represents the data as modified by the user, for the provided @context_node.
 *
 * If @create_if_needed is TRUE, the a new ModelUserModifiedValue structure will be created
 * and returned.
 *
 * Returns: a ModelUserModifiedValue structure, or %NULL
 */
static ModelUserModifiedValue *
grid_model_get_user_modifs (MgWorkGrid *grid, GtkTreeModel *tree_model, GtkTreeIter *iter,
			    MgContextNode *context_node, gboolean create_if_needed)
{
	ModelUserModifiedRow *user_modifs;
	ModelUserModifiedValue *user_value = NULL;
	gint row;
	
	gtk_tree_model_get (tree_model, iter, COLUMN_ROW_NUM, &row, COLUMN_USER_MODIFS_ROW, &user_modifs, -1);
	if (user_modifs) {
		GSList *list;
		list = user_modifs->user_values;
		while (list && !user_value) {
			if (MODEL_USER_MODIFIED_VALUE (list->data)->context_node == context_node)
				user_value = MODEL_USER_MODIFIED_VALUE (list->data);
			list = g_slist_next (list);
		}		
	}

	if (!user_value && create_if_needed) {
		if (!user_modifs) 
			user_modifs = grid_model_make_user_modifs (grid, row);
		
		user_value = g_new0 (ModelUserModifiedValue, 1);
		user_modifs->user_values = g_slist_append (user_modifs->user_values, user_value);
		user_value->context_node = context_node;
		user_value->attributes = 0;
		user_value->value = NULL;
		gtk_list_store_set (GTK_LIST_STORE (tree_model), iter, COLUMN_USER_MODIFS_ROW, user_modifs, -1);
	}

	return user_value;
}


/*
 * grid_set_user_modifs_to_default_if_notnull
 * If core->work_context has some 'proposed' values for some parameters, then set them.
 *
 * Also, if some of the parameters in core->work_context are invalid, and if some default values are
 * available, creates ModelUserModifiedValue structures to tell that the default value is required
 * (as if it had been done by the user).
 */
static void
grid_set_user_modifs_to_default_if_notnull (MgWorkGrid *grid, GtkTreeModel *tree_model, GtkTreeIter *iter, 
					    MgWorkCore *core)
{
	GSList *list = core->params_modif_queries_value_providers;
	MgParameter *param;
	MgContextNode *node;
	const GdaValue *value;
	ModelUserModifiedValue *modval;

	while (list) {
		param = MG_PARAMETER (list->data);

		/* 'proposed' value in the context */ 
		value = mg_context_get_param_default_value (core->work_context, param);
		if (value) {
			node = mg_context_find_node_for_param (core->work_context, param);
			modval = grid_model_get_user_modifs (grid, tree_model, iter, node, TRUE);
			grid_modif_struct_set_value (grid, modval, tree_model, iter, core, value);
		}
		else {
			/* Available default value */
			value = mg_parameter_get_default_value (param);
			if (mg_parameter_get_not_null (param) && value && !gda_value_is_null (value)) {
				node = mg_context_find_node_for_param (core->work_context, param);
				g_assert (node);
				modval = grid_model_get_user_modifs (grid, tree_model, iter, node, TRUE);
				grid_modif_struct_set_status (grid, modval, tree_model, iter, core, 
							      MG_DATA_ENTRY_IS_DEFAULT);
			}
			else {
				/* If we have a MgDataCellRendererCombo then ask for the first record 
				 * of choice, if available */
				node = mg_context_find_node_for_param (core->work_context, param);
				
				if (! node->param) {
					GdaValue *first_value;
					ColumnData *column_data;

					column_data = get_column_data (grid, node);
					first_value = mg_data_cell_renderer_combo_get_first_value_available 
						(MG_DATA_CELL_RENDERER_COMBO (column_data->data_cell));
					if (first_value) {
						modval = grid_model_get_user_modifs (grid, tree_model, 
										     iter, node, TRUE);
						grid_modif_struct_set_value (grid, modval, tree_model,
									     iter, core, first_value);
						gda_value_free (first_value);
					}
				}
			}
		}
		
		list = g_slist_next (list);
	}
}

static const GdaValue *
get_value_from_recordset (GtkTreeModel *tree_model, GtkTreeIter *iter, MgWorkCore *core, MgParameter *context_param)
{
	gint col;
	MgWorkCoreNode *cnode;
	gint row;
	
	g_return_val_if_fail (context_param, NULL);

	gtk_tree_model_get (tree_model, iter, COLUMN_ROW_NUM, &row, -1);
	cnode = mg_work_core_find_core_node (core, context_param);
	g_assert (cnode);

	col = cnode->position;
	g_assert (col >= 0);
		
	return mg_resultset_get_gdavalue (core->data_rs, row, col);	
}

static void
remove_user_modif (MgWorkGrid *grid, ModelUserModifiedValue *user_value, GtkTreeModel *tree_model, GtkTreeIter *iter)
{
	ModelUserModifiedRow *user_modifs;
	
	gtk_tree_model_get (tree_model, iter, COLUMN_USER_MODIFS_ROW, &user_modifs, -1);
	g_return_if_fail (user_modifs);
	g_return_if_fail (g_slist_find (user_modifs->user_values, user_value));

	user_modifs->user_values = g_slist_remove (user_modifs->user_values, user_value);
	if (user_value->value)
		gda_value_free (user_value->value);
	g_free (user_value);

	/* if there is not anymore any user modified values, then get rid of the ModelUserModifiedRow */
	if (!user_modifs->user_values && !user_modifs->to_be_deleted) {
		grid->priv->modified_rows = g_slist_remove (grid->priv->modified_rows, user_modifs);
		user_modifs_row_free (user_modifs);
		gtk_list_store_set (GTK_LIST_STORE (tree_model), iter, COLUMN_USER_MODIFS_ROW, NULL, -1);
	}
}


/*
 * grid_modif_struct_set_value
 * Updates @user_modif with @value, and delete it if @value is the same as the one it already has.
 */
static void
grid_modif_struct_set_value (MgWorkGrid *grid, ModelUserModifiedValue *user_modif, 
			     GtkTreeModel *tree_model, GtkTreeIter *iter,
			     MgWorkCore *core, const GdaValue *value)
{
	guint attrs = 0;
	gint row;
	GtkTreePath *path;

	/* see if the value is the same as the original one, and if that is the case, 
	 * get rid of the 'user_modif' and return.
	 * row == -1 if a new row has been added (one which is not in the core->resultset) */
	gtk_tree_model_get (tree_model, iter, COLUMN_ROW_NUM, &row, -1);

	if (row > -1) {
		if (user_modif->context_node->param) { /* one single param */
			const GdaValue *value_orig;
			
			value_orig = get_value_from_recordset (tree_model, iter, core, user_modif->context_node->param);
			g_assert (value_orig);

			if (((!value || gda_value_is_null (value)) && (!value_orig || gda_value_is_null (value_orig))) ||
			    (value && value_orig && (gda_value_get_type (value) == gda_value_get_type (value_orig)) &&
			     !gda_value_compare (value, value_orig))) {
				remove_user_modif (grid, user_modif, tree_model, iter);
				return;
			}
		}
		else { /* multiple parameters */
			gboolean same_as_orig = TRUE;
			GSList *params;
			GList *user_modifs;
			
			g_assert (gda_value_isa (value, GDA_VALUE_TYPE_LIST));
			
			params = user_modif->context_node->params;
			user_modifs = (GList *) gda_value_get_list (value);
			while (params && same_as_orig) {
				const GdaValue *value = NULL;
				const GdaValue *orig_value;
				
				value = g_list_nth_data (user_modifs, 
							 GPOINTER_TO_INT (g_hash_table_lookup 
									  (user_modif->context_node->params_pos_in_query, 
									   params->data)));
				orig_value = get_value_from_recordset (tree_model, iter, core, MG_PARAMETER (params->data));
				
				if (((!value || gda_value_is_null (value)) && 
				     (!orig_value || gda_value_is_null (orig_value))) ||
				    (value && orig_value && 
				     (gda_value_get_type (value) == gda_value_get_type (orig_value)) &&
				     !gda_value_compare (value, orig_value)))
					same_as_orig = TRUE;
				else
					same_as_orig = FALSE;
				
				params = g_slist_next (params);
			}

			if (same_as_orig) {
				remove_user_modif (grid, user_modif, tree_model, iter);
				return;
			}
		}
	}

	/* at this point the value is different from the original one, or there is no original value */
	if (user_modif->value)
		gda_value_free (user_modif->value);
	
	user_modif->value = gda_value_copy (value);
	attrs = user_modif->attributes;

	/* resetting some attributes */
	attrs = attrs & ~MG_DATA_ENTRY_IS_NULL;
	attrs = attrs & ~MG_DATA_ENTRY_IS_UNCHANGED;
	attrs = attrs & ~MG_DATA_ENTRY_IS_DEFAULT;

	/* setting new attributes */
	if (!value || gda_value_is_null (value))
		attrs = attrs | MG_DATA_ENTRY_IS_NULL;

	user_modif->attributes = attrs;

	/* signal the model for data update */
	path = gtk_tree_model_get_path (tree_model, iter);
	gtk_tree_model_row_changed (tree_model, path, iter);
	gtk_tree_path_free (path);
}

/*
 * grid_clean_data_model_row
 *
 * Cleans the data allocated inside @tree_model (ModelUserModifiedValue structures), at row pointed
 * by @iter; and sets the COLUMN_ROW_NUM to -2.
 */
static void 
grid_clean_data_model_row (MgWorkGrid *grid, GtkTreeModel *tree_model, GtkTreeIter *iter)
{
	ModelUserModifiedRow *user_modifs;

	gtk_tree_model_get (tree_model, iter, COLUMN_USER_MODIFS_ROW, &user_modifs, -1);
	/* REM: user_modifs can be NULL if the user has not changed anything, and no safe default was set */
	if (!user_modifs)
		return;

	grid->priv->modified_rows = g_slist_remove (grid->priv->modified_rows, user_modifs);
	user_modifs_row_free (user_modifs);
	gtk_list_store_set (GTK_LIST_STORE (tree_model), iter, COLUMN_ROW_NUM, -2, COLUMN_USER_MODIFS_ROW, NULL, -1);
}


static void
grid_modif_struct_set_status (MgWorkGrid *grid, ModelUserModifiedValue *user_modif,
			      GtkTreeModel *tree_model, GtkTreeIter *iter,
			      MgWorkCore *core, MgDataEntryAttribute requested_status)
{
	g_return_if_fail (user_modif);
	guint attrs;
	GtkTreePath *path;
	gint row;

	switch (requested_status) {
	case MG_DATA_ENTRY_IS_NULL:
		/* see if  the original value is NULL
		 * to get rid of the 'user_modif' and return. */
		gtk_tree_model_get (tree_model, iter, COLUMN_ROW_NUM, &row, -1);

		if (row > -1) {
			if (user_modif->context_node->param) { /* one single param */
				const GdaValue *value_orig;
				
				value_orig = get_value_from_recordset (tree_model, iter, core, user_modif->context_node->param);
				g_assert (value_orig);
				
				if (!value_orig || gda_value_is_null (value_orig)) {
					remove_user_modif (grid, user_modif, tree_model, iter);
					return;
				}
			}
			else { /* multiple parameters */
				GSList *params;
				gboolean allnull = TRUE;
				const GdaValue *orig_value;
				
				params = user_modif->context_node->params;
				while (params && allnull) {
					orig_value = get_value_from_recordset (tree_model, iter, core, 
									       MG_PARAMETER (params->data));
					
					if (orig_value && !gda_value_is_null (orig_value))
						allnull = FALSE;
					
					params = g_slist_next (params);
				}
				
				if (allnull) {
					remove_user_modif (grid, user_modif, tree_model, iter);
					return;
				}
			}
		}

		if (user_modif->value) {
			gda_value_free (user_modif->value);
			user_modif->value = NULL;
		}
		user_modif->value = gda_value_new_null ();
		user_modif->attributes = user_modif->attributes | MG_DATA_ENTRY_IS_NULL;

		/* signal the model for data update */
		path = gtk_tree_model_get_path (tree_model, iter);
		gtk_tree_model_row_changed (tree_model, path, iter);
		gtk_tree_path_free (path);
		break;
	case MG_DATA_ENTRY_IS_DEFAULT:
		attrs = user_modif->attributes;
		if (user_modif->context_node->param) { /* single parameter */
			const GdaValue *value = mg_parameter_get_default_value (user_modif->context_node->param);
			if (user_modif->value) {
				gda_value_free (user_modif->value);
				user_modif->value = NULL;
			}

			if (value && 
			    (gda_value_get_type (value) == mg_server_data_type_get_gda_type (mg_parameter_get_data_type 
			     (user_modif->context_node->param)))) {
				user_modif->value = gda_value_copy (value);
			}
			if (! user_modif->value)
				user_modif->value = gda_value_new_null ();

			attrs = attrs | MG_DATA_ENTRY_IS_DEFAULT;
		}
		else { /* several parameters constrained by a query */
			TO_IMPLEMENT;
			attrs = attrs | MG_DATA_ENTRY_IS_DEFAULT;
		}
		user_modif->attributes = attrs;

		/* signal the model for data update */
		path = gtk_tree_model_get_path (tree_model, iter);
		gtk_tree_model_row_changed (tree_model, path, iter);
		gtk_tree_path_free (path);
		break;
	case MG_DATA_ENTRY_IS_UNCHANGED:
		remove_user_modif (grid, user_modif, tree_model, iter);
		break;
	default:
		g_assert_not_reached ();
		break;
	}
}






/*
 * MgWorkWidget interface implementation
 */
static void
mg_work_grid_run (MgWorkWidget *iface, guint mode)
{
	MgWorkGrid *grid;

	g_return_if_fail (iface && IS_MG_WORK_GRID (iface));
	grid = MG_WORK_GRID (iface);
	g_return_if_fail (grid->priv);
	g_return_if_fail (grid->priv->core->query_select);

	/* REM : we don't check for missing parameters: the user must do it himself
	 * using the mg_work_widget_get_exec_context() function and mg_context_is_valid() */

	/*
	 * Signals connecting
	 */
	if (grid->priv->core->args_context)
		g_signal_connect (G_OBJECT (grid->priv->core->args_context), "changed",
				  G_CALLBACK (arg_param_changed_cb), grid);
	grid->priv->has_run = TRUE;

	/*
	 * Actual start
	 */
	if (mode)
		grid->priv->mode = mode;
	refresh_all_rows (grid, FALSE);
}

static void
mg_work_grid_set_mode (MgWorkWidget *iface, guint mode)
{
	MgWorkGrid *grid;

	g_return_if_fail (iface && IS_MG_WORK_GRID (iface));
	grid = MG_WORK_GRID (iface);
	g_return_if_fail (grid->priv);

	grid->priv->mode = mode;

	/* updating the various possible actions */
	modif_buttons_update (grid);
}

static void
mg_work_grid_set_entry_editable (MgWorkWidget *iface, MgQfield *field, gboolean editable)
{
	MgWorkGrid *grid;
	MgParameter *param;
	ColumnData *column_data;
	MgContextNode *node;

	g_return_if_fail (iface && IS_MG_WORK_GRID (iface));
	grid = MG_WORK_GRID (iface);
	g_return_if_fail (grid->priv);
	
	param = mg_work_grid_get_param_for_field (iface, field, NULL, FALSE);
	node = mg_context_find_node_for_param (grid->priv->core->work_context, param);
	g_return_if_fail (node);
	column_data = get_column_data (grid, node);
	g_return_if_fail (column_data);

	if (editable)
		column_data->data_locked = grid->priv->core->modif_target ? FALSE : TRUE;
	else
		column_data->data_locked = TRUE;

	g_object_set (G_OBJECT (column_data->data_cell), "editable", !column_data->data_locked, NULL);
}


static void
mg_work_grid_show_entry_actions (MgWorkWidget *iface, MgQfield *field, gboolean show_actions)
{
	MgParameter *param;
	MgContextNode *cnode;
	ColumnData *cdata;
	MgWorkGrid *grid;

	g_return_if_fail (iface && IS_MG_WORK_GRID (iface));
	grid = MG_WORK_GRID (iface);
	g_return_if_fail (grid->priv);

	if (!field)
		/* TODO: if @field is NULL, then apply to all the fields */
		TO_IMPLEMENT;
	
	param = mg_work_grid_get_param_for_field (iface, field, NULL, FALSE);
	g_return_if_fail (param);
	
	cnode = mg_context_find_node_for_param (grid->priv->core->work_context, param);
	g_return_if_fail (cnode);

	cdata = get_column_data (grid, cnode);
	g_return_if_fail (cdata);

	if (show_actions != cdata->info_shown) {
		cdata->info_shown = show_actions;
		g_object_set (G_OBJECT (cdata->info_cell), "visible", cdata->info_shown, NULL);
	}
}

static void
mg_work_grid_show_global_actions (MgWorkWidget *iface, gboolean show_actions)
{
	MgWorkGrid *grid;

	g_return_if_fail (iface && IS_MG_WORK_GRID (iface));
	grid = MG_WORK_GRID (iface);
	g_return_if_fail (grid->priv);

	if (show_actions)
		gtk_widget_show (grid->priv->modif_all);
	else
		gtk_widget_hide (grid->priv->modif_all);
}

static MgParameter *
mg_work_grid_get_param_for_field (MgWorkWidget *iface, MgQfield *field, const gchar *field_name, gboolean in_exec_context)
{
	MgWorkGrid *grid;
	MgParameter *param = NULL;

	g_return_val_if_fail (iface && IS_MG_WORK_GRID (iface), NULL);
	grid = MG_WORK_GRID (iface);
	g_return_val_if_fail (grid->priv, NULL);
	g_return_val_if_fail (field || (field_name && *field_name), NULL);

	if (field) {
		/* use the 'field' argument */
		g_return_val_if_fail (field && IS_MG_QFIELD (field), NULL);
		param = mg_work_core_find_param (grid->priv->core, field, in_exec_context);
	}
	else {
		/* use the 'field_name' argument */
		MgField *f = mg_entity_get_field_by_name (MG_ENTITY (grid->priv->core->query_select), field_name);
		if (f)
			param = mg_work_core_find_param (grid->priv->core, MG_QFIELD (f), in_exec_context);	
	}

	return param;
}

static gboolean
mg_work_grid_has_been_changed (MgWorkWidget *iface)
{
	MgWorkGrid *grid;

	g_return_val_if_fail (iface && IS_MG_WORK_GRID (iface), FALSE);
	grid = MG_WORK_GRID (iface);
	g_return_val_if_fail (grid->priv, FALSE);

	return grid->priv->modified_rows ? TRUE : FALSE;
}

static MgContext *
mg_work_grid_get_exec_context (MgWorkWidget *iface)
{
	MgWorkGrid *grid;

	g_return_val_if_fail (iface && IS_MG_WORK_GRID (iface), NULL);
	grid = MG_WORK_GRID (iface);
	g_return_val_if_fail (grid->priv, NULL);
	
	return grid->priv->core->args_context;
}

static GtkActionGroup *
mg_work_grid_get_actions_group (MgWorkWidget *iface)
{
	MgWorkGrid *grid;
	
	g_return_val_if_fail (iface && IS_MG_WORK_GRID (iface), NULL);
	grid = MG_WORK_GRID (iface);
	g_return_val_if_fail (grid->priv, NULL);

	return grid->priv->actions_group;
}

/**
 * mg_work_grid_set_sample_size
 * @grid:
 * @sample_size:
 *
 */
void
mg_work_grid_set_sample_size (MgWorkGrid *grid, gint sample_size)
{
	g_return_if_fail (grid && IS_MG_WORK_GRID (grid));
	g_return_if_fail (grid->priv);

	if (sample_size <0)
		grid->priv->sample_size = 0;
	else
		grid->priv->sample_size = sample_size;

	update_simple_grid (grid, FALSE);
}

/**
 * mg_work_grid_set_sample_start
 * @grid:
 * @sample_start:
 *
 */
void
mg_work_grid_set_sample_start (MgWorkGrid *grid, gint sample_start)
{
	g_return_if_fail (grid && IS_MG_WORK_GRID (grid));
	g_return_if_fail (grid->priv);
	
	if (sample_start <0)
		grid->priv->sample_first_row = 0;
	else
		grid->priv->sample_first_row = sample_start;

	update_simple_grid (grid, FALSE);
}
