#include <stdlib.h>
#include <string.h>

#include <stdio.h>

#ifdef HAVE_DMALLOC
#include <dmalloc.h>
#endif

#include "buffer.h"

/**
 * init the buffer 
 * 
 */

buffer* buffer_init() {
	buffer *b;
	
	b = malloc(sizeof(buffer));
	
	b->ptr = 0;
	b->size = 0;
	b->used = 0;
	
	return b;
}

/**
 * free the buffer 
 * 
 */

void buffer_free(buffer *b) {
	if (!b) return;
	
	if (b->size) {
		free(b->ptr);
		b->size = 0;
		b->used = 0;
	}
	free(b);
}

void buffer_reset(buffer *b) {
	if (!b) return;
	
	b->used = 0;
}


/**
 * 
 * allocate (if neccessary) enough space for 'size' bytes and 
 * set the 'used' coutner to 0
 * 
 */

int buffer_prepare_copy(buffer *b, size_t size) {
	if (!b) return -1;
	
	if (0 == b->size) {
		b->size = size;
		b->ptr = malloc(b->size);
	} else if (size > b->size) {
		free(b->ptr);
		
		b->size = size;
		b->ptr = malloc(b->size);
	}
	b->used = 0;
	return 0;
}

/**
 * 
 * increase the internal buffer (if neccessary) to append another 'size' byte
 * ->used isn't changed
 * 
 */

int buffer_prepare_append(buffer *b, size_t size) {
	if (!b) return -1;
	
	if (0 == b->size) {
		b->size = size;
		b->ptr = malloc(b->size);
		b->used = 0;
	} else if (b->used + size > b->size) {
		b->size += size;
		b->ptr = realloc(b->ptr, b->size);
	}
	return 0;
}

int buffer_copy_string(buffer *b, const char *s) {
	size_t s_len;
	
	if (!s || !b) return -1;
	
	s_len = strlen(s);
	if (buffer_prepare_copy(b, s_len + 1)) return -1;
	
	memcpy(b->ptr, s, s_len + 1);
	b->used = s_len + 1;
	
	return 0;
}

int buffer_copy_string_len(buffer *b, const char *s, size_t s_len) {
	if (!s || !b) return -1;
	
	if (s_len == 0) return 0;
	
	if (buffer_prepare_copy(b, s_len + 1)) return -1;
	
	memcpy(b->ptr, s, s_len);
	b->ptr[s_len] = '\0';
	b->used = s_len + 1;
	
	return 0;
}

int buffer_copy_string_buffer(buffer *b, const buffer *src) {
	if (src->used == 0) {
		b->used = 0;
		return 0;
	}
	return buffer_copy_string_len(b, src->ptr, src->used - 1);
}

int buffer_append_string(buffer *b, const char *s) {
	size_t s_len;
	
	if (!s || !b) return -1;
	
	/* the buffer is empty, fallback to copy */
	if (b->used == 0) {
		return buffer_copy_string(b, s);
	}
	
	if (b->ptr[b->used - 1] != '\0') {
		SEGFAULT();
	}
	
	s_len = strlen(s);
	if (buffer_prepare_append(b, s_len)) return -1;
	
	memcpy(b->ptr + b->used - 1, s, s_len + 1);
	b->used += s_len;
	
	return 0;
}

int buffer_append_string_rfill(buffer *b, const char *s, size_t maxlen) {
	size_t s_len;
	ssize_t fill_len;
	
	if (!s || !b) return -1;
	
	/* the buffer is empty, fallback to copy */
	if (b->used == 0) {
		return buffer_copy_string(b, s);
	}
	
	if (b->ptr[b->used - 1] != '\0') {
		/* seg-fault */
		SEGFAULT();
	}
	
	s_len = strlen(s);
	if (buffer_prepare_append(b, MAX(s_len, maxlen + 1))) return -1;
	
	fill_len = maxlen - s_len;
	
	if (fill_len > 0) {
		memcpy(b->ptr + b->used - 1, s, s_len);
		memset(b->ptr + b->used + s_len - 1, ' ', fill_len);
		*(b->ptr + b->used + s_len + fill_len - 1) = '\0';
	} else {
		memcpy(b->ptr + b->used - 1, s, s_len + 1);
	}
	
	b->used += MAX(s_len, maxlen);
	
	return 0;
}

/**
 * append a string to the end of the buffer
 * 
 * the resulting buffer is terminated with a '\0' 
 * s is treated as a un-terminated string (a \0 is handled a normal character)
 * 
 * @param b a buffer
 * @param s the string
 * @param s_len size of the string (without the terminating \0)
 */

int buffer_append_string_len(buffer *b, const char *s, size_t s_len) {
	if (!s || !b) return -1;
	
	if (s_len == 0) return 0;
	
	/* the buffer is empty, fallback to copy */
	if (b->used == 0) {
		return buffer_copy_string_len(b, s, s_len);
	}
	
	if (b->ptr[b->used - 1] != '\0') {
		SEGFAULT();
	}
	
	if (buffer_prepare_append(b, s_len)) return -1;
	
	memcpy(b->ptr + b->used - 1, s, s_len);
	b->ptr[b->used + s_len - 1] = '\0';
	b->used += s_len;
	
	return 0;
}

int buffer_append_string_buffer(buffer *b, const buffer *src) {
	if (src->used == 0) return 0;
	
	return buffer_append_string_len(b, src->ptr, src->used - 1);
}

int buffer_append_memory(buffer *b, const char *s, size_t s_len) {
	if (!s || !b) return -1;
	
	if (s_len == 0) return 0;
	
	if (buffer_prepare_append(b, s_len)) return -1;
	
	memcpy(b->ptr + b->used, s, s_len);
	b->used += s_len;
	
	return 0;
}

int buffer_copy_memory(buffer *b, const char *s, size_t s_len) {
	if (!s || !b) return -1;
	
	b->used = 0;
	
	return buffer_append_memory(b, s, s_len);
}



int ltostr(char *s, long l) {
	int i, sign = 0;
	
	if (l < 0) {
		sign = 1;
		l = -l;
	}
	
	for (i = 0; l > 9; l /= 10, i++) {
		s[i] = '0' + (l % 10);
	}
	
	s[i] = '0' + l;
	if (sign) {
		s[++i] = '-';
	}
	s[i+1] = '\0';
	
	/* swap bytes again :) */
	if (i > 0) {
		int li = i;
		for (; i > li / 2; i--) {
			char c;
			
			c = s[i];
			s[i] = s[li - i];
			s[li - i] = c;
		}
	}
	
	return 0;
}

int buffer_copy_long(buffer *b, long l) {
	int i, sign = 0;
	char *s;
	
	if (!b) return -1;
	
	b->used = 0;
	
	if (buffer_prepare_append(b, 32)) return -1;
	
	s = b->ptr + b->used;
	
	if (l < 0) {
		sign = 1;
		l = -l;
	}
	
	for (i = 0; l > 9; l /= 10, i++) {
		s[i] = '0' + (l % 10);
	}
	
	s[i] = '0' + l;
	if (sign) {
		s[++i] = '-';
	}
	s[i+1] = '\0';
	b->used = i + 2;
	
	/* swap bytes again :) */
	if (i > 0) {
		int li = i;
		for (; i > li / 2; i--) {
			char c;
			
			c = s[i];
			s[i] = s[li - i];
			s[li - i] = c;
		}
	}
	
	return 0;
}


int buffer_append_long(buffer *b, long l) {
	int i, sign = 0;
	char *s;
	
	if (!b) return -1;
	
	/* the buffer is empty, fallback to copy */
	if (b->used == 0) {
		SEGFAULT();
	}
	
	if (b->ptr[b->used - 1] != '\0') {
		SEGFAULT();
	}
	
	if (buffer_prepare_append(b, 32)) return -1;
	
	s = b->ptr + b->used - 1;
	
	if (l < 0) {
		sign = 1;
		l = -l;
	}
	
	for (i = 0; l > 9; l /= 10, i++) {
		s[i] = '0' + (l % 10);
	}
	
	s[i] = '0' + l;
	if (sign) {
		s[++i] = '-';
	}
	s[i+1] = '\0';
	b->used += i + 1;
	
	/* swap bytes again :) */
	if (i > 0) {
		int li = i;
		for (; i > li / 2; i--) {
			char c;
			
			c = s[i];
			s[i] = s[li - i];
			s[li - i] = c;
		}
	}
	
	return 0;
}


int ultostr(char *s, unsigned long l) {
	int i;
	
	for (i = 0; l > 9; l /= 10, i++) {
		s[i] = '0' + (l % 10);
	}
	
	s[i] = '0' + l;
	s[i+1] = '\0';
	
	/* swap bytes again :) */
	if (i > 0) {
		int li = i;
		for (; i > li / 2; i--) {
			char c;
			
			c = s[i];
			s[i] = s[li - i];
			s[li - i] = c;
		}
	}
	
	return 0;
}

int ultohex(char *s, unsigned long l) {
	int i;
	
	for (i = 0; l > 15; l /= 16, i++) {
		s[i] = (l % 16) < 10 ? '0' + (l % 16) : 'a' + (l % 16) - 10;
	}
	
	s[i] = l < 10 ? '0' + l : 'a' + l - 10;
	s[i+1] = '\0';
	
	/* swap bytes again :) */
	if (i > 0) {
		int li = i;
		for (; i > li / 2; i--) {
			char c;
			
			c = s[i];
			s[i] = s[li - i];
			s[li - i] = c;
		}
	}
	
	return 0;
}

char int2hex(char i) {
	if (i >= 0 && i < 10) return i + '0';
	if (i >= 10 && i < 16) return i + 'a' - 10;
	
	return '_';
}

char hex2int(unsigned char c) {
	return (c >= '0' && c <= '9') ? c-'0' : (c|32)-'a'+10;
}


/**
 * init the buffer 
 * 
 */

buffer_array* buffer_array_init() {
	buffer_array *b;
	
	b = malloc(sizeof(buffer_array));
	
	b->ptr = 0;
	b->size = 0;
	b->used = 0;
	
	return b;
}

/**
 * free the buffer_array 
 * 
 */

void buffer_array_free(buffer_array *b) {
	size_t i;
	if (!b) return;
	
	for (i = 0; i < b->size; i++) {
		if (b->ptr[i]) buffer_free(b->ptr[i]);
	}
	free(b->ptr);
	free(b);
}

buffer *buffer_array_append_get_buffer(buffer_array *b) {
	size_t i;
	if (b->size == 0) {
		b->size = 16;
		b->ptr = malloc(sizeof(buffer *) * b->size);
		for (i = 0; i < b->size; i++) {
			b->ptr[i] = NULL;
		}
	} else if (b->size == b->used) {
		b->size += 16;
		b->ptr = realloc(b->ptr, sizeof(buffer *) * b->size);
		for (i = b->used; i < b->size; i++) {
			b->ptr[i] = NULL;
		}
	}
	
	if (b->ptr[b->used] == NULL) {
		b->ptr[b->used] = buffer_init();
	}
	
	b->ptr[b->used]->used = 0;
	
	return b->ptr[b->used++];
}


char * buffer_search_string_len(buffer *b, char *needle, size_t len) {
	size_t i;
	if (len == 0) return NULL;
	if (needle == NULL) return NULL;
	
	if (b->used < len) return NULL;
	
	for(i = 0; i < b->used - len; i++) {
		if (0 == memcmp(b->ptr + i, needle, len)) {
			return b->ptr + i;
		}
	}
	
	return NULL;
}

buffer *buffer_init_string(const char *str) {
	buffer *b = buffer_init();
	
	buffer_copy_string(b, str);
	
	return b;
}

int buffer_is_empty(buffer *b) {
	return (b->used == 0);
}

int buffer_is_equal(buffer *a, buffer *b) {
	size_t i;
	
	if (a->used != b->used) return 0;
	if (a->used == 0) return 1;

#if 1
	for (i = a->used - 2; i < a->used && i % 4; i -= 1) {
		if (a->ptr[i] != b->ptr[i]) return 0;
	}
	
	for (i -= 4; i < a->used; i -= 4) {
		if (*((unsigned long *)(a->ptr + i)) != 
		    *((unsigned long *)(b->ptr + i))) return 0;
	}
#else
	if (0 != strcmp(a->ptr, b->ptr)) return 0;
#endif

	return 1;
}

int buffer_append_string_hex(buffer *b, char *in, size_t in_len) {
	size_t i;
	unsigned char j;
	
	/* BO protection */
	if (in_len * 2 < in_len) return -1;
	
	if (b->used > 0) {
		if (b->ptr[b->used-1] == '\0') b->used--;
	}
	
	buffer_prepare_append(b, in_len * 2 + 1);
	
	for (i = 0; i < in_len; i++) {
		j = (in[i] >> 4) & 0xf;
		b->ptr[b->used++] = j + ((j <= 9) ? '0' : 'a' - 10);
		
		j = in[i] & 0xf;
		b->ptr[b->used++] = j + ((j <= 9) ? '0' : 'a' - 10);
	}
	b->ptr[b->used++] = '\0';
	
	return 0;
}

int buffer_append_string_url_encoded(buffer *b, const char *s) {
	unsigned char *ds, *d;
	size_t s_len;
	size_t d_len;
	
	if (!s || !b) return -1;
	
	if (b->ptr[b->used - 1] != '\0') {
		SEGFAULT();
	}
	
	s_len = strlen(s);
	
	/* count to-be-encoded-characters */
	for (ds = (unsigned char *)s, d_len = 0; *ds; ds++) {
		if (*ds < 32 || *ds > 127) {
			d_len += 3;
		} else {
			switch (*ds) {
			case '$':
			case '&':
			case '+':
			case ',':
			case '/':
			case ':':
			case ';':
			case '=':
			case '?':
			case '@':
			case ' ':
			case '#':
			case '%':
			case '<':
			case '>':
			case '"':
			case '\'':
				d_len += 3;
				break;
			default:
				d_len ++;
				break;
			}
		}
	}
	
	if (buffer_prepare_append(b, d_len)) return -1;
	
	for (ds = (unsigned char *)s, d = b->ptr + b->used - 1, d_len = 0; *ds; ds++) {
		if (*ds < 32 || *ds > 127) {
			d[d_len++] = '%';
			d[d_len++] = int2hex(*ds >> 4);
			d[d_len++] = int2hex(*ds & 0xf);
		} else {
			switch (*ds) {
			case '$':
			case '&':
			case '+':
			case ',':
			case '/':
			case ':':
			case ';':
			case '=':
			case '?':
			case '@':
			case ' ':
			case '#':
			case '%':
			case '<':
			case '>':
			case '"':
			case '\'':
				d[d_len++] = '%';
				d[d_len++] = int2hex(*ds >> 4);
				d[d_len++] = int2hex(*ds & 0xf);
				break;
			default:
				d[d_len++] = *ds;
				break;
			}
		}
	}
	
	b->ptr[b->used + d_len - 1] = '\0';
	
	b->used += d_len;
	
	return 0;
}

int buffer_append_string_html_encoded(buffer *b, const char *s) {
	unsigned char *ds, *d;
	size_t s_len;
	size_t d_len;
	
	if (!s || !b) return -1;
	
	if (b->ptr[b->used - 1] != '\0') {
		SEGFAULT();
	}
	
	s_len = strlen(s);
	
	/* count to-be-encoded-characters */
	for (ds = (unsigned char *)s, d_len = 0; *ds; ds++) {
		switch (*ds) {
		case '>':
		case '<':
			d_len += 4;
			break;
		case '&':
			d_len += 5;
			break;
		default:
			d_len++;
			break;
		}
	}
	
	if (buffer_prepare_append(b, d_len)) return -1;
	
	for (ds = (unsigned char *)s, d = b->ptr + b->used - 1, d_len = 0; *ds; ds++) {
		switch (*ds) {
		case '>':
			d[d_len++] = '&';
			d[d_len++] = 'g';
			d[d_len++] = 't';
			d[d_len++] = ';';
			
			break;
		case '<':
			d[d_len++] = '&';
			d[d_len++] = 'l';
			d[d_len++] = 't';
			d[d_len++] = ';';
			
			break;
		case '&':
			d[d_len++] = '&';
			d[d_len++] = 'a';
			d[d_len++] = 'm';
			d[d_len++] = 'p';
			d[d_len++] = ';';
			
			break;
				
		default:
			d[d_len++] = *ds;
			break;
		}
	}
	
	b->ptr[b->used + d_len - 1] = '\0';
	b->used += d_len;
	
	return 0;
}
