/* presburger_transduction.c - DFA package example application */

/*
 * MONA
 * Copyright (C) 1997-2002 BRICS.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the  Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 * USA.
 */

/* This example illustrates how to use transduction and  
 * iteration. It constructs an automaton for 'p_index = value'
 * using Presburger encoding by starting with an automaton 
 * for the relation 'p_i = 1' and iteratively adding 1 
 * to the constant using projection, product and index-replacing.
 * (This is not supposed to be an efficient way of 
 * constructing these automata - the purpuse is to illustrate 
 * the iterative approach.)
 */ 

#define INDEX 7
#define VALUE 13

#include <stdio.h>
#include <stdlib.h>
#include "dfa.h"
#include "mem.h"

int main()
{
  DFA *a1, *a2, *a3, *a4, *a5, *a6, *a7, *a8, *a9;
  int i;
  
  /* initialize the BDD package */
  bdd_init();
    
  /* import the 'presburger_plus_012.dfa' automaton 
     generated by running 'mona presburger.mona' */
  {
    char **vars; /* 0 terminated array of variable names */
    int *orders; /* array of variable orders (0/1/2) */
    a1 = dfaImport("presburger_plus_012.dfa", &vars, &orders);
    if (!a1) {
      printf("error: unable to read 'presburger_plus_012.dfa' "
	     "(run 'mona -n presburger.mona')\n");
      exit(-1);
    }
    /* 'vars' now contains {"p","q","r"}, i.e. the variables in their 
       original index ordering, 'orders' contains {2,2,2} */

    for (i = 0; vars[i]; i++)
      mem_free(vars[i]);
    mem_free(vars);
    mem_free(orders);
  }

  /* make automaton for 'p_1 = 1' */
  a2 = dfaPresbConst(1, 1);

  /* make the intersection product of 'a1' and 'a2' 
     and project(+right-quotient) 'p_1' away -
     this corresponds to making an automaton for the  
     following formula: 'ex p_1: p_0 + p_1 = p_2 & p_1 = 1' */  
  a3 = dfaProduct(a1, a2, dfaAND);
  a4 = dfaMinimize(a3);
  dfaRightQuotient(a4, 1);
  a5 = dfaProject(a4, 1);
  a6 = dfaMinimize(a5);
  /* a6 now represents the formula 'p_0 + 1 = p_2' */

  /* clean up the temporary automata */
  dfaFree(a1);
  dfaFree(a3);
  dfaFree(a4);
  dfaFree(a5);

  /* for the base case of the iteration, we need an automaton
     for 'p_0 = 1' - reuse a2 by replacing the index */
  {
    int map[2];
    map[1] = 0; /* means: replace 1 with 0 in all BDD nodes */
    dfaReplaceIndices(a2, map);
  }

  /* the main part: iterate the transduction */
  for (i = 1; i < VALUE; i++) {
    
    /* make an automaton for 'ex p_0: p_0 = i & p_0 + 1 = p_2' */
    a7 = dfaProduct(a6, a2, dfaAND);
    a8 = dfaMinimize(a7);
    dfaRightQuotient(a8, 0);
    a9 = dfaProject(a8, 0);

    /* make automaton for 'p_0 = i+1' by a replace-indices */
    {
      int map[3];
      map[2] = 0;
      dfaReplaceIndices(a9, map);
    }

    /* clean up */
    dfaFree(a7);
    dfaFree(a8);
    dfaFree(a2);

    a2 = a9;
    /* now a2 represents 'p_0 = i+1' */
  }
  
  /* clean up */
  dfaFree(a6);

  /* set the index */
  {
    int map[1];
    map[0] = INDEX;
    dfaReplaceIndices(a2, map);
  }

  /* output the resulting automaton in Graphviz format */
  {
    unsigned indices[1]; /* array of indices in alphabet */
    indices[0] = INDEX;
    dfaPrintGraphviz(a2, 1, indices);
  }

  dfaFree(a2);
  return 0;
}
