/*

    mpv_curses.c

    Curses Interface (Linux/Unix)

    mp - Programmer Text Editor

    Copyright (C) 1991-2001 Angel Ortega <angel@triptico.com>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

    http://www.triptico.com

*/

#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <termios.h>
#include <fcntl.h>
#include <curses.h>
#include <signal.h>
#include <ctype.h>
#include <sys/stat.h>

#if !defined(WITHOUT_GLOB)
#include <glob.h>
#endif

#include "mp_core.h"
#include "mp_video.h"
#include "mp_synhi.h"
#include "mp_iface.h"

#include "mp_lang.h"

/*******************
	Data
********************/

#define START_LINE 1
#define END_LINE 1

/* signal grabbing */
static sigset_t Sigset;

/* standard attributes */
static int _attrs[MP_COLOR_NUM+1];

/* other attributes */
static int _title_attr;
static int _menu_sel_attr;
static int _menu_act_attr;
static int _menu_frame_attr;
static int _menu_frame2_attr;
static int _scrollbar_attr;
static int _scrollbar_thumb_attr;
static int _tab_sel_attr;

/** menu **/

/* menu item */

struct mpv_menu_item
{
	char label[32]; 		/* label */
	int k;				/* MPK_ key */
	int hk; 			/* hotkey (not used) */
	int * toggle;			/* toggle mark */
	struct mpv_menu_item * next;
	struct mpv_menu_item * prev;
};

/* menu bar */

struct mpv_menu_bar
{
	char label[15]; 		/* label */
	struct mpv_menu_item * first;	/* first item */
	struct mpv_menu_item * last;	/* last item */
	int items;			/* item number */
	int xpos;			/* horizontal position */
	struct mpv_menu_bar * next;
	struct mpv_menu_bar * prev;
};

/* pointers to first and last menu bars */
struct mpv_menu_bar * _mpv_menu=NULL;
struct mpv_menu_bar * _mpv_menu_last=NULL;

/* menu bar horizontal position */
static int _mpv_menu_xpos=2;

#ifndef NCURSES_VERSION
#define NO_TRANSPARENCY
#endif


/* history for readlines */

#ifndef NUM_READLINE_HISTORIES
#define NUM_READLINE_HISTORIES 8
#endif

struct readhist
{
	int idx;
	char * lines[NUM_READLINE_HISTORIES];

} readline_history[MPR_LAST+1];

#ifndef DEFAULT_INK_COLOR
#define DEFAULT_INK_COLOR COLOR_WHITE
#endif

#ifndef DEFAULT_PAPER_COLOR
#define DEFAULT_PAPER_COLOR COLOR_BLACK
#endif

/* hint text to the left of first line */
char * _mpv_hint_text=NULL;


#ifdef POOR_MAN_BOXES
#define MPBOX_HLINE '-'
#define MPBOX_VLINE '|'
#define MPBOX_TL_CORNER '+'
#define MPBOX_TR_CORNER '+'
#define MPBOX_BL_CORNER '+'
#define MPBOX_BR_CORNER '+'
#define MPBOX_THUMB ' '
#else
#define MPBOX_HLINE ACS_HLINE
#define MPBOX_VLINE ACS_VLINE
#define MPBOX_TL_CORNER ACS_ULCORNER
#define MPBOX_TR_CORNER ACS_URCORNER
#define MPBOX_BL_CORNER ACS_LLCORNER
#define MPBOX_BR_CORNER ACS_LRCORNER
#define MPBOX_THUMB ACS_CKBOARD
#endif

/*****************
       Code
******************/

static void _mpv_attrset(int attr)
{
#ifndef ABSOLUTELY_NO_COLORS
	attrset(attr);
#endif
}


static void _mpv_wattrset(WINDOW * w, int attr)
{
#ifndef ABSOLUTELY_NO_COLORS
	wattrset(w,attr);
#endif
}


/**
 * _sigwinch - SIGWINCH capture function.
 * @s: signal
 *
 * The capture function for the SIGWINCH signal, raised
 * whenever a window resize has been done (from i.e. xterm).
 */
static void _sigwinch(int s)
{
	/* restart curses */
	mpv_shutdown();
	mpv_startup();

	/* invalidate main window */
	clearok(stdscr,1);
	refresh();

	/* recalc dimensions */
	_mpv_x_size=COLS - 1;
	_mpv_y_size=LINES - START_LINE - END_LINE;

	/* redraw everything */
	mpv_title(NULL);
	mpi_draw_all(_mp_active);

	/* reattach */
	signal(SIGWINCH,_sigwinch);
}


/**
 * mpv_strcasecmp - Case ignoring string compare
 * @s1: first string
 * @s2: second string
 *
 * Case ignoring string compare. System dependent
 * (strcasecmp in Unix, stricmp in Win32)
 */
int mpv_strcasecmp(char * s1, char * s2)
{
	return(strcasecmp(s1,s2));
}


/**
 * mpv_goto - Positions the cursor to start drawing.
 * @x: the x position
 * @y: the y position
 *
 * Positions the cursor to start drawing the document window.
 */
void mpv_goto(int x, int y)
{
	move(START_LINE+y,x);
}


/**
 * mpv_char - Draws a char with color in the document window.
 * @c: the char
 * @color: the color (one of MP_COLOR_ constants)
 *
 * Draws a char with color in current cursor position of the
 * document window.
 */
void mpv_char(int c, int color)
{
	_mpv_attrset(_attrs[color]);
	addch((unsigned char) c);
}


/**
 * mpv_str - Draws a string with color in the document window.
 * @str: the string
 * @color: the color (one of MP_COLOR_ constants)
 *
 * Draws a string, calling mpv_char() for each of their chars.
 */
void mpv_str(char * str, int color)
{
	_mpv_attrset(_attrs[color]);
	addstr(str);
}


/**
 * mpv_cursor - Positions the hardware cursor.
 * @x: the real x position
 * @y: the real y position
 *
 * Sets the hardware cursor to x, y position.
 */
void mpv_cursor(int x, int y)
{
	static int ox=0;
	static int oy=0;

	if(x==-1) x=ox;
	if(y==-1) y=oy;

	move(START_LINE+y,x);

	ox=x;
	oy=y;
}


/**
 * mpv_refresh - Refresh the screen.
 *
 * Orders the underlying system to redraw the screen.
 */
void mpv_refresh(void)
{
	refresh();
}


/**
 * _mpv_title_status - Draws the title or status line.
 * @y: the y position
 * @str1: first string
 * @str2: second (optional) string
 *
 * Draws the title or status line with str1 and, if defined,
 * concatenates str2. The y variable indicates the vertical
 * position where the line will be drawn. It is filled with
 * spaces to the right margin (internal).
 */
static void _mpv_title_status(int y, char * str1, char * str2)
{
	char * ptr;
	int n;

	move(y,0);
	_mpv_attrset(_title_attr);

	if(str1!=NULL)
	{
		for(n=0;*str1 && n < _mpv_x_size;n++,str1++)
			addch(*str1);
	}

	if(str2!=NULL)
	{
		for(ptr=" - ";*ptr && n < _mpv_x_size;n++,ptr++)
			addch(*ptr);
		for(;*str2 && n < _mpv_x_size;n++,str2++)
			addch(*str2);
	}

	if(_mpv_hint_text==NULL) _mpv_hint_text=L(MSG_DEFAULTHINT);

	if(y!=0)
	{
		for(;n < _mpv_x_size - strlen(_mpv_hint_text) + 1;n++)
			addch(' ');
		addstr(_mpv_hint_text);
	}
	else
	{
		for(;n<_mpv_x_size+1;n++)
			addch(' ');
	}

	mpv_cursor(-1,-1);
}


/**
 * mpv_title - Sets the string to be drawn as title
 * @str: the string
 *
 * Sets the string to be drawn as title of the document window.
 */
void mpv_title(char * str)
{
	mp_txt * t;
	char files[1024];
	int n,m;
	char * ptr;

	if(str==NULL && _mp_active!=NULL)
	{
		str=files;
		if(_mp_active->mod)
			strcpy(files,"*");
		else
			files[0]='\0';

		strncat(files, _mp_active->name, sizeof(files)-1);

		t=_mp_active->next;
		n=strlen(files);

		m=sizeof(files)-1;
		if(_mpv_x_size < m) m=_mpv_x_size;

		while(n < sizeof(files)-1)
		{
			if(t==NULL) t=_mp_txts;
			if(t==_mp_active) break;

			for(ptr=" | ";*ptr && n < m;n++,ptr++)
				files[n]=*ptr;

			if(n < m && t->mod)
				files[n++]='*';

			for(ptr=t->name;*ptr && n < m;n++,ptr++)
				files[n]=*ptr;

			t=t->next;
		}

		files[n]='\0';
	}

	_mpv_title_status(0," mp " VERSION, str);
}


/**
 * mpv_status_line - Sets the string to be drawn as status line
 * @str: the string
 *
 * Sets the string to be drawn as the status line.
 */
void mpv_status_line(char * str)
{
	_mpv_title_status(LINES - 1,str,NULL);
}


/**
 * mpv_zoom - Zooms the document window.
 * @inc: the increment (+1 or -1)
 *
 * Increases / decreases font size of the document window,
 * if applicable.
 */
int mpv_zoom(int inc)
{
	return(0);
}


#define ctrl(k) ((k)&31)

/**
 * _mpv_key - Processes a key.
 * @k: the key
 *
 * Processes a raw (curses) key and converts it to a MPK_ constant
 * or normal ISO-8859-1 character.
 */
static int _mpv_key(int k)
{
	/* special case: some terminals return the backspace
	   key as a 0x7f (del or rubout) */
	if(k==0x7f) k='\b';

	/* test if in special shift mode */
	if(_mpi_shift)
	{
		/* if 0..9, convert to function */
		if(k>='0' && k<='9')
		{
			if(!(k-='0')) k+=10;
			k=KEY_F(k);
		}
		else
		/* HOME and END does not work in some xterms */
		if(k==KEY_LEFT)
			k=ctrl('J');
		else
		if(k==KEY_RIGHT)
			k=ctrl('K');
		else
		if(k==KEY_DOWN)
			k=KEY_NPAGE;
		else
		if(k==KEY_UP)
			k=KEY_PPAGE;
		else
		if(k==KEY_END)
			k=ctrl('E');
		else
		if(k==KEY_HOME)
			k=ctrl('W');
		else
		if(k>=32 && k<=255)
			k=ctrl(k);

		_mpi_shift=0;
	}

	/* if it is a printable character, we're done */
	if(k>=32 && k<=255)
		return(k);

	/* special keys */
	switch(k)
	{
	case KEY_UP:	k=MPK_UP; break;
	case KEY_DOWN:	k=MPK_DOWN; break;
	case KEY_LEFT:	k=MPK_LEFT; break;
	case KEY_RIGHT: k=MPK_RIGHT; break;
	case KEY_NPAGE: k=MPK_PGDN; break;
	case KEY_PPAGE: k=MPK_PGUP; break;
	case KEY_HOME:	k=MPK_BOL; break;
	case KEY_END:	k=MPK_EOL; break;
	case ctrl('W'): k=MPK_BOF; break;
	case ctrl('E'): k=MPK_EOF; break;
	case ctrl('J'): k=MPK_WORDLEFT; break;
	case ctrl('K'): k=MPK_WORDRIGHT; break;

	case ctrl('B'): k=MPK_SEEK; break;
	case ctrl('L'): k=MPK_SEEKNEXT; break;
	case ctrl('R'): k=MPK_REPLACE; break;
	case ctrl('G'): k=MPK_GOTO; break;

	case KEY_IC:	k=MPK_INSMODE; break;
	case KEY_DC:	k=MPK_DELCHAR; break;
	case '\b':
	case KEY_BACKSPACE:
			k=MPK_BACKSPC; break;
	case ctrl('Y'): k=MPK_DELLINE; break;

	case ctrl('C'):
	case ctrl('D'): k=MPK_COPY; break;
	case ctrl('Q'):
	case ctrl('T'): k=MPK_CUT; break;
	case ctrl('P'):
	case ctrl('V'): k=MPK_PASTE; break;
	case ctrl('U'): k=MPK_TEMPLATE; break;
	case ctrl('F'): k=MPK_TAG; break;
	case ctrl('S'): k=MPK_COMPLETION; break;

	case KEY_F(1):	k=MPK_HELP; break;
	case KEY_F(2):	k=MPK_SAVE; break;
	case KEY_F(3):	k=MPK_OPEN; break;
	case KEY_F(4):	k=MPK_CLOSE; break;
	case KEY_F(5):	k=MPK_NEW; break;
	case KEY_F(6):	k=MPK_NEXT; break;
	case KEY_F(7):	k=MPK_PLAYMACRO; break;
	case KEY_F(8):	k=MPK_UNMARK; break;
	case KEY_F(9):	k=MPK_MARK; break;
	case KEY_F(10): k=MPK_RECORDMACRO; break;
	case ctrl('O'): k=MPK_OPEN2; break;

	case '\e':
	case ctrl('N'): k=MPK_SHIFT; break;
	case ctrl(' '):
	case ctrl('A'): k=MPK_MENU; break;

	case ctrl('X'): k=MPK_EXIT; break;

#ifdef NCURSES_MOUSE_VERSION

	case KEY_MOUSE:
	{
		MEVENT m;

		getmouse(&m);
		_mpi_instant_x=m.x;
		_mpi_instant_y=m.y-1;

		if(m.bstate & BUTTON1_PRESSED)
			k=MPK_UNMARK;
		else
			k=MPK_MARK;

		break;
	}
#endif

	}

	return(k);
}


/**
 * mpv_key - Reads a key from terminal and processes it.
 *
 * Reads a key from terminal and processes it via _mpv_key().
 */
static int mpv_key(void)
{
	return(_mpv_key(getch()));
}


/**
 * mpv_add_menu - Creates a new menu bar.
 * @label: the label
 *
 * Creates a new menu bar.
 */
void mpv_add_menu(char * label)
{
	struct mpv_menu_bar * m;
	int n;

	m=(struct mpv_menu_bar *) malloc(sizeof(struct mpv_menu_bar));

	if(m==NULL) return;

	for(n=0;*label;label++)
	{
		if(*label!='&')
			m->label[n++]=*label;
	}
	m->label[n]='\0';

	m->first=m->last=NULL;
	m->xpos=_mpv_menu_xpos;

	m->next=NULL;

	if(_mpv_menu==NULL)
	{
		_mpv_menu=m;
		m->prev=NULL;
	}
	else
	{
		_mpv_menu_last->next=m;
		m->prev=_mpv_menu_last;
	}

	/* this is the last, by now */
	_mpv_menu_last=m;

	_mpv_menu_xpos+=(strlen(m->label) + 3);
}


/**
 * mpv_add_menu_item - Adds a menu item to current menu bar.
 * @label: the label
 * @key: the associated MPK_ key.
 * @toggle: a pointer to int with the toggle variable status.
 *
 * Adds a menu item to current menu bar. If toggle is defined,
 * it is a pointer to an integer to toggle if the item is
 * activated.
 */
void mpv_add_menu_item(char * label, int key, int * toggle)
{
	struct mpv_menu_item * i;
	int n,m,l;

	i=(struct mpv_menu_item *) malloc(sizeof(struct mpv_menu_item));

	if(i==NULL) return;

	i->k=key;
	i->toggle=toggle;
	i->next=NULL;

	/* calculate tabulator size */
	l=(sizeof(i->label)-strlen(label))-1;

	for(n=0;label[n]!='\t' && label[n] && n<sizeof(i->label);n++)
		i->label[n]=label[n];

	for(m=n;l>0;l--,m++)
		i->label[m]=' ';

	if(label[n]=='\t') n++;
		else m--;
	for(;label[n];n++,m++)
		i->label[m]=label[n];
	i->label[m]='\0';

	if(_mpv_menu_last->first==NULL)
	{
		_mpv_menu_last->first=i;
		i->prev=NULL;
	}
	else
	{
		_mpv_menu_last->last->next=i;
		i->prev=_mpv_menu_last->last;
	}

	_mpv_menu_last->last=i;

	_mpv_menu_last->items++;
}


static void _mpv_box(WINDOW * w, int tx, int ty)
{
	int x,y;

	wmove(w,0,0);
	waddch(w,MPBOX_TL_CORNER);
	for(x=1;x < tx-1;x++)
		waddch(w,MPBOX_HLINE);
	waddch(w,MPBOX_TR_CORNER);

	for(y=1;y < ty-1;y++)
	{
		_mpv_wattrset(w,_menu_frame_attr);
		wmove(w,y,0);
		waddch(w,MPBOX_VLINE);
		_mpv_wattrset(w,_menu_frame2_attr);
		wmove(w,y,tx-1);
		waddch(w,MPBOX_VLINE);
	}

	waddch(w,MPBOX_BL_CORNER);
	for(x=1;x < tx-1;x++)
		waddch(w,MPBOX_HLINE);
	waddch(w,MPBOX_BR_CORNER);
}


/**
 * mpv_menu - Manages the menu.
 *
 * Manages the menu (drawing it, if applicable).
 * Returns the key associated to the activated menu item,
 * or 0 if none was. 
 */
int mpv_menu(void)
{
	int k=0,ok;
	char tmp[1024];
	struct mpv_menu_bar * m;
	struct mpv_menu_item * i;
	struct mpv_menu_item * i2;
	WINDOW * w;
	int x,y,c,cc,sy;

	tmp[0]='\0';
	for(m=_mpv_menu;m!=NULL;m=m->next)
	{
		strcat(tmp,"   ");
		strcat(tmp,m->label);
	}

	_mpv_hint_text=L(MSG_CANCELHINT);
	_mpv_title_status(0,tmp,NULL);
	m=_mpv_menu;

	ok=0;
	while(!ok)
	{
		i=m->first;

		w=newwin(m->items+2, sizeof(i->label), 1, m->xpos);
		_mpv_wattrset(w,_menu_frame_attr);
/*		  box(w,0,0); */
		_mpv_box(w,sizeof(i->label),m->items+2);
		wrefresh(w);

		while(!ok)
		{
			/* draw items */
			for(sy=y=1,i2=m->first;i2!=NULL;y++,i2=i2->next)
			{
				if(i2==i)
				{
					c=_menu_sel_attr;
					sy=y;
				}
				else
					c=_title_attr;

				/* if '-', then item is a separator */
				if(i2->label[0]=='-')
				{
					wmove(w,y,1);
					_mpv_wattrset(w,_menu_frame_attr);

					/* waddch(w,ACS_LTEE); */
					for(x=0;x<sizeof(i2->label)-2;x++)
						waddch(w,MPBOX_HLINE);
					/* waddch(w,ACS_RTEE); */
				}
				else
				{
					wmove(w,y,1);
					_mpv_wattrset(w,c);

					for(x=0;x<sizeof(i->label) && (cc=i2->label[x]);x++)
					{
						if(cc=='&')
						{
							/* _mpv_wattrset(w,_menu_act_attr); */
							cc=i2->label[++x];
						}
						else
							_mpv_wattrset(w,c);

						waddch(w,(unsigned char)cc);
					}

					/* draws toggle mark */
					if(i2->toggle!=NULL && *i2->toggle)
						waddch(w,'*');
					else
						waddch(w,' ');
				}
			}

			wmove(w,sy,1);
			wrefresh(w);
			k=getch();

			/* possible keys */
			if(k==0x1b || k==ctrl('A') || k==ctrl('N') || k==ctrl(' '))
			{
				k=0;
				ok=1;
			}
			else
			if(k==KEY_RIGHT)
			{
				if((m=m->next)==NULL)
					m=_mpv_menu;
				break;
			}
			else
			if(k==KEY_LEFT)
			{
				if((m=m->prev)==NULL)
					m=_mpv_menu_last;
				break;
			}
			else
			if(k==KEY_DOWN)
			{
				if((i=i->next)==NULL)
					i=m->first;

				/* this assumes a separator can't be
				   the last item of a menu, so don't do it */
				if(i->label[0]=='-')
					i=i->next;
			}
			else
			if(k==KEY_UP)
			{
				if((i=i->prev)==NULL)
					i=m->last;

				/* this assumes a separator can't be
				   the last item of a menu, so don't do it */
				if(i->label[0]=='-')
					i=i->prev;
			}
			else
			if(k=='\r')
			{
				k=i->k;
				ok=1;
			}
			else
			{
				k=_mpv_key(k);
				ok=1;
			}
		}

		delwin(w);
		touchwin(stdscr);
		refresh();
	}

	_mpv_hint_text=NULL;

	return(k);
}


/**
 * _mpv_prompt - Asks yes or no.
 * @prompt: the prompt
 * @prompt2: an optional second prompt.
 *
 * Asks yes or no.
 * Returns 1 if 'yes' choosen.
 */
static int _mpv_prompt(char * prompt, char * prompt2)
{
	int c;
	char * yes;
	char * no;

	move(LINES - 1,0);
	_mpv_attrset(_title_attr);

	addstr(prompt);
	if(prompt2)
		addstr(prompt2);

	yes=L(MSG_YES);
	no=L(MSG_NO);

	for(;;)
	{
		c=toupper(getch());

		if(c==*yes || c==*no || c=='\r')
		{
			mpv_status_line(NULL);
			break;
		}
	}

	if(c==*no) return(0);

	return(1);
}


/**
 * mpv_confirm - Asks for confirmation.
 * @prompt: the question to ask
 *
 * Asks for confirmation.
 * Returns 1 if choosen 'yes'.
 */
int mpv_confirm(char * prompt)
{
	return(_mpv_prompt(prompt,L(MSG_YESNO)));
}


/**
 * mpv_alert - Alerts the user.
 * @msg: the alert message
 * @msg2: a possible second message.
 *
 * Alerts the user by showing the message and
 * asking for validation.
 */
void mpv_alert(char * msg, char * msg2)
{
	char tmp[4096];

	if(msg2==NULL)
		strncpy(tmp,msg,sizeof(tmp));
	else
		sprintf(tmp,msg,msg2);

	_mpv_prompt(tmp,L(MSG_ENTER));
}


/**
 * mpv_list - Manages a selection list
 * @title: the title or caption of the list
 * @txt: the text containing the list to show
 *
 * Shows a unique element selection list to the user.
 * The list must be previously built into txt.
 * Returns the selected element (the number of the
 * line where the element is) or -1 on cancellation.
 */
int mpv_list(char * title, mp_txt * txt)
{
	int c;

	_mpv_hint_text=L(MSG_CANCELHINT);
	mpv_title(title);

	mp_move_bof(txt);
	_mpi_list_selection=-1;

	if(txt->lasty==0)
	{
		/* no lines or just one line: exit */
		_mpv_hint_text=NULL;
		return(0);
	}

	txt->type=MP_TYPE_LIST;
	txt->mod=0;

	mpi_draw_all(txt);

	for(;;)
	{
		c=mpv_key();
		if(mpi_action(txt, c))
			mpi_draw_all(txt);

		if(c==MPK_SHIFT || c==MPK_CLOSE || _mpi_list_selection!=-1)
			break;
	}

	_mpv_hint_text=NULL;

	return(_mpi_list_selection);
}


/**
 * open_file_list - Shows a list of files to open
 * @rcpt: buffer to receive the selected file
 *
 * Asks for a file scheme and shows a list of files
 * matching it.
 * Returns the pushed key to exit ('\r' or MPK_ESCAPE).
 * If no or just one file is matched, or if glob()
 * is unsupported, '\r' is also returned.
 * The selected file will be written into the rcpt
 * buffer.
 * This is an ugly hack: it will be rewritten
 * sooner or later.
 */
static int open_file_list(char * rcpt)
{
#if !defined(WITHOUT_GLOB)
	mp_txt * txt;
	int n,m,c;
	glob_t globbuf;
	struct stat s;
	char * ptr;
	int l;

	txt=mp_create_sys_txt("<file list>");

	globbuf.gl_offs=1;
	if(glob(rcpt,GLOB_MARK,NULL,&globbuf))
		return('\r');

	for(n=0;globbuf.gl_pathv[n]!=NULL;n++)
	{
		ptr=globbuf.gl_pathv[n];

		if(stat(ptr,&s)==-1) continue;
		if(s.st_mode & S_IFDIR) continue;

		m=strlen(ptr);

		mp_put_str(txt,ptr,1);
		mp_put_char(txt,':',1);

		while(m++ < 40)
			mp_put_char(txt,' ',1);

		{
			char tmp[128];

			sprintf(tmp,"%12d bytes",(int)s.st_size);
			mp_put_str(txt,tmp,1);
		}

		mp_insert_line(txt);
	}

	globfree(&globbuf);

	/* deletes last insert line */
	mp_move_left(txt);
	mp_delete_char(txt);

	if((l=mpv_list("file list", txt))!=-1)
	{
		mp_move_xy(txt,0,l);
		mp_get_str(txt,rcpt,1024,':');
		c='\r';
	}
	else
		c=MPK_ESCAPE;

	mp_delete_sys_txt(txt);
	return(c);
#else /* WITHOUT_GLOB */

	return('\r');

#endif /* WITHOUT_GLOB */
}


/**
 * mpv_readline - Ask for user input.
 * @type: the type of input asked (one of MPR_ constants)
 * @prompt: the prompt
 * @def: the default value
 *
 * Asks for user input. The expected data type is
 * described in type.
 * Returns a pointer to a static buffer with the
 * data entered by the user, or NULL if user cancelled.
 */
char * mpv_readline(int type, char * prompt, char * def)
{
	static char tmp[1024];
	int c,cursor,rhidx;
	struct readhist * rh;
	int n;

	move(LINES - 1,0);
	_mpv_attrset(_title_attr);

	/* clean line */
	for(n=0;n < COLS;n++) addch(' ');
	move(LINES - 1,0);

	if(def==NULL)
		tmp[0]='\0';
	else
		strncpy(tmp,def,sizeof(tmp));

	cursor=strlen(tmp);

	addstr(prompt);
	addstr(tmp);

	rh=&readline_history[type];
	rhidx=rh->idx;

	for(;;)
	{
		c=mpv_key();

		if(c=='\r' || c==MPK_ESCAPE)
			break;

		if(c==MPK_BACKSPC && cursor>0)
		{
			addstr("\b \b");
			tmp[--cursor]='\0';
			refresh();
		}
		else
		if(c==MPK_CLEARLINE)
		{
			while(cursor--) { addstr("\b \b"); }
			tmp[cursor=0]='\0';
			refresh();
		}
		else
		if(c==MPK_UP)
		{
			if(--rhidx < 0)
				rhidx=NUM_READLINE_HISTORIES - 1;

			if(readline_history[type].lines[rhidx]!=NULL)
				strncpy(tmp,readline_history[type].lines[rhidx],sizeof(tmp));
			else
				tmp[0]='\0';

			while(cursor--) { addstr("\b \b"); }
			addstr(tmp);
			cursor=strlen(tmp);
			refresh();
		}
		else
		if(c==MPK_DOWN)
		{
			if(++rhidx == NUM_READLINE_HISTORIES)
				rhidx=0;

			if(readline_history[type].lines[rhidx]!=NULL)
				strncpy(tmp,readline_history[type].lines[rhidx],sizeof(tmp));
			else
				tmp[0]='\0';

			while(cursor--) { addstr("\b \b"); }
			addstr(tmp);
			cursor=strlen(tmp);
			refresh();
		}
		else
		if(c>=32 && c<=255 && cursor < 56)
		{
			addch(c);
			tmp[cursor++]=c;
			tmp[cursor]='\0';
			refresh();
		}
	}

	if(c==MPK_ESCAPE)
		return(NULL);

	if(type==MPR_OPEN)
	{
		if(tmp[0]=='\0')
			strncpy(tmp,"*",sizeof(tmp));
		else
		{
			struct stat s;

			if(!stat(tmp,&s))
			{
				/* add a globber if it's a directory */
				if(s.st_mode & S_IFDIR)
					strcat(tmp,"/*");
			}
		}

		if(strchr(tmp,'*')!=NULL || strchr(tmp,'?')!=NULL)
			c=open_file_list(tmp);
	}

	mpv_status_line(NULL);

	if(c==MPK_ESCAPE)
		return(NULL);

	/* store line in history */
	if(rh->lines[rh->idx]!=NULL)
		free(rh->lines[rh->idx]);

	rh->lines[rh->idx]=(char *) malloc(strlen(tmp)+1);
	strncpy(rh->lines[rh->idx],tmp,strlen(tmp)+1);

	if(++rh->idx == NUM_READLINE_HISTORIES)
		rh->idx=0;

	return(tmp);
}


/**
 * mpv_help - Shows the available help for a term
 * @term: the term
 * @synhi: the syntax highlighter
 *
 * Shows the available help for the term. The argument
 * filetype is a string describing the file type,
 * taken from the syntax hilighter (this allows to
 * retrieve the help from different sources for C
 * or Perl code, for example).
 * Returns 0 on error or if no help is available
 * for this term.
 */
int mpv_help(char * term, int synhi)
{
	char tmp[1024];
	FILE * f;
	mp_txt * txt;
	char ** ptr;

	sprintf(tmp,L(MSG_HELP),term);

	if(synhi==0 || (txt=mp_create_txt(tmp))==NULL)
		return(0);

	if((ptr=_mps_synhi[synhi - 1].helpers)!=NULL)
	{
		for(;*ptr!=NULL;ptr++)
		{
			snprintf(tmp,sizeof(tmp),*ptr,term);

			if((f=popen(tmp,"r"))!=NULL)
			{
				mp_load_open_file(txt,f);

				if(!pclose(f))
					break;
			}
		}
	}

	if(ptr==NULL || *ptr==NULL)
	{
		mp_delete_txt(txt);
		mpv_alert(L(MSG_NOHELPFOR),term);
		return(0);
	}

	mps_auto_synhi(txt);
	txt->type=MP_TYPE_READ_ONLY;
	txt->mod=0;

	return(1);
}


/**
 * mpv_about - Shows the 'About Minimum Profit...' information.
 *
 * Shows a text or dialog box showing the information
 * about the program, version and such.
 */
void mpv_about(void)
{
	mp_txt * txt;

	txt=mp_create_txt(L(MSG_ABOUT));

	mp_put_str(txt,MP_LICENSE,1);
	mp_move_bof(txt);

	txt->type=MP_TYPE_READ_ONLY;
	txt->mod=0;
}


/**
 * mpv_notify - The notify function.
 * @str: the str
 *
 * The notify function for mp_set_notify().
 */
static void mpv_notify(char * str)
{
	mpv_status_line(str);
	mpv_refresh();
}


#define R3(a,b,c) (((a)*(b))/(c))

/**
 * mpv_scrollbar - Draws / updates the scrollbar.
 * @pos: current position
 * @size: vertical size
 * @max: maximum value
 *
 * Draws / updates the scrollbar. pos is current position
 * (the line where the cursor is), size is the number of
 * lines of the document window and max is the maximum
 * value (the total number of lines in current text).
 */
void mpv_scrollbar(int pos, int size, int max)
{
	int n;
	int l,u,a;

	if(max<size)
	{
		l=0;
		u=_mpv_y_size;
		a=1;
	}
	else
	{
		l=R3(pos,size,max);
		u=R3(pos+size,size,max);
		a=0;
	}

	for(n=0;n<_mpv_y_size;n++)
	{
		move(START_LINE+n,_mpv_x_size);

		if(n>=l && n<=u && !a)
		{
			_mpv_attrset(_scrollbar_thumb_attr);
			addch(MPBOX_THUMB);
		}
		else
		{
			_mpv_attrset(_scrollbar_attr);
			addch(MPBOX_VLINE);
		}
	}
}


/**
 * mpv_filetabs - Draws the tab set containing the file names
 *
 * Draws the names of the opened files in the form of a tab set.
 */
void mpv_filetabs(void)
{
	/* this function is only a dummy, as the file
	   names are shown in the title bar */
}

#ifdef IEXTEN
#define TTY_MODE ISIG|ICANON|ECHO|IEXTEN
#else				/* not IEXTEN */
#define TTY_MODE ISIG|ICANON|ECHO
#endif				/* not IEXTEN */

void mpv_xonxoff(int state)
{
	static struct termios _ioval;
	static int orig_state=-1;

	/* Get current settings */
	tcgetattr(0, &_ioval);

	/* Store XON/XOFF state */
	if ( orig_state == -1 ) {
		orig_state = (_ioval.c_iflag & (IXON | IXOFF));
	}

	if ( state ) {
	
		_ioval.c_lflag &= ~TTY_MODE;
		_ioval.c_iflag &= ~(IXON | IXOFF);

		tcsetattr(0, TCSANOW, &_ioval);
		return;

	} else {
		/* Paranoid? */
		if (orig_state == -1) {
			return;
		}

		_ioval.c_lflag |= TTY_MODE;
		_ioval.c_iflag |= orig_state;

		tcsetattr(0, TCSANOW, &_ioval);
		return;
	}
}



/**
 * mpv_startup - Starts up the system dependent driver.
 *
 * Starts up the system dependent driver.
 * Returns 1 if succesful.
 */
int mpv_startup(void)
{
	int dic,dpc;
	
	mpv_xonxoff(1);

	initscr();
	start_color();

#ifndef NO_TRANSPARENCY
	use_default_colors();
#else
	mpi_transp_mode=0;
#endif

	dic=DEFAULT_INK_COLOR;
	dpc=DEFAULT_PAPER_COLOR;

	if(mpi_transp_mode) dic=dpc=-1;

#ifdef NCURSES_MOUSE_VERSION
	if(mpi_mouse)
		mousemask(BUTTON1_PRESSED|BUTTON2_PRESSED, NULL);
#endif

	keypad(stdscr, TRUE);
	nonl();
	cbreak();
	noecho();

	if(mpi_monochrome)
	{
		init_pair(MP_COLOR_NORMAL,dic,dpc);
		init_pair(MP_COLOR_SELECTED,dic,dpc);
		init_pair(MP_COLOR_COMMENT,dic,dpc);
		init_pair(MP_COLOR_STRING,dic,dpc);
		init_pair(MP_COLOR_TOKEN,dic,dpc);
		init_pair(MP_COLOR_VAR,dic,dpc);
		init_pair(MP_COLOR_CURSOR,dic,dpc);
		init_pair(MP_COLOR_CAPS,dic,dpc);
		init_pair(MP_COLOR_LOCAL,dic,dpc);

		init_pair(10,dic,dpc);
		init_pair(11,dic,dpc);
		init_pair(12,dic,dpc);
		init_pair(13,dic,dpc);
	}
	else
	{
		init_pair(MP_COLOR_NORMAL,dic,dpc);
		init_pair(MP_COLOR_SELECTED,COLOR_RED,COLOR_WHITE);
		init_pair(MP_COLOR_COMMENT,COLOR_GREEN,dpc);
		init_pair(MP_COLOR_STRING,COLOR_BLUE,dpc);
		init_pair(MP_COLOR_TOKEN,COLOR_GREEN,dpc);
		init_pair(MP_COLOR_VAR,COLOR_RED,dpc);
		init_pair(MP_COLOR_CURSOR,dic,dpc);
		init_pair(MP_COLOR_CAPS,COLOR_YELLOW,dpc);
		init_pair(MP_COLOR_LOCAL,COLOR_CYAN,dpc);

		init_pair(10,mpi_alt_color ? COLOR_RED:COLOR_BLUE,COLOR_WHITE);
		init_pair(11,mpi_alt_color ? COLOR_RED:COLOR_BLUE,
			     mpi_alt_color ? COLOR_YELLOW:COLOR_BLUE);
		init_pair(12, COLOR_WHITE,
			     mpi_alt_color ? COLOR_BLUE:COLOR_BLACK);
		init_pair(13,mpi_alt_color ? COLOR_RED:COLOR_BLUE,
			     mpi_alt_color ? COLOR_BLACK:COLOR_BLACK);
	}

	_attrs[MP_COLOR_NORMAL]=COLOR_PAIR(MP_COLOR_NORMAL);
	_attrs[MP_COLOR_SELECTED]=COLOR_PAIR(MP_COLOR_SELECTED)|A_REVERSE;
	_attrs[MP_COLOR_COMMENT]=COLOR_PAIR(MP_COLOR_COMMENT);
	_attrs[MP_COLOR_STRING]=COLOR_PAIR(MP_COLOR_STRING)|A_BOLD;
	_attrs[MP_COLOR_TOKEN]=COLOR_PAIR(MP_COLOR_TOKEN)|A_BOLD;
	_attrs[MP_COLOR_VAR]=COLOR_PAIR(MP_COLOR_VAR);
	_attrs[MP_COLOR_CURSOR]=COLOR_PAIR(MP_COLOR_CURSOR)|A_REVERSE;
	_attrs[MP_COLOR_CAPS]=COLOR_PAIR(MP_COLOR_CAPS)|A_BOLD;
	_attrs[MP_COLOR_LOCAL]=COLOR_PAIR(MP_COLOR_LOCAL)|A_UNDERLINE;

	_title_attr=COLOR_PAIR(10)|A_BOLD|A_REVERSE;
	_menu_frame_attr=COLOR_PAIR(11)|A_BOLD|A_REVERSE;
	_menu_frame2_attr=COLOR_PAIR(13)|A_BOLD|A_REVERSE;
	_menu_sel_attr=COLOR_PAIR(12)|A_BOLD;
	_menu_act_attr=_menu_frame_attr;
	_scrollbar_attr=COLOR_PAIR(MP_COLOR_NORMAL);
	_scrollbar_thumb_attr=_title_attr;
	_tab_sel_attr=COLOR_PAIR(10)|A_BOLD;

	_mpv_x_size=COLS - 1;
	_mpv_y_size=LINES - START_LINE - END_LINE;


	signal(SIGWINCH,_sigwinch);

	mp_set_notify(mpv_notify);
	mpv_status_line(NULL);

	fclose(stderr);

	/* ignore unwanted signals */
	sigemptyset (&Sigset);
	sigaddset (&Sigset, SIGTERM);
	sigaddset (&Sigset, SIGHUP);
	sigaddset (&Sigset, SIGTSTP);
	sigaddset (&Sigset, SIGINT);
	sigaddset (&Sigset, SIGIOT);
	sigaddset (&Sigset, SIGQUIT);
	sigprocmask (SIG_BLOCK, &Sigset, 0);

	/* init readlines */
	memset(readline_history,'\0',sizeof(readline_history));

	return(1);
}


/**
 * mpv_shutdown - Shuts down the system dependent driver.
 *
 * Shuts down the system dependent driver.
 */
void mpv_shutdown(void)
{
	sigprocmask (SIG_UNBLOCK, &Sigset, 0);

	endwin();
	
	mpv_xonxoff(0);
}


/**
 * usage - Prints the usage help.
 *
 * Prints the usage help to standard output.
 */
void usage(void)
{
	printf("mp " VERSION " - Programmer Text Editor\n");
	printf("Copyright (C) 1991-2002 Angel Ortega <angel@triptico.com>\n");
	printf("%s\n", __DATE__ " " __TIME__);
	printf("This software is covered by the GPL license. NO WARRANTY.\n\n");

	printf("%s", L(MSG_USAGE_TEXT));
	printf("%s\n",mps_enumerate_modes());
}


int main(int argc, char * argv[])
{
	int c, r;

	r=mpi_args_1(argc, argv);

	if(r==-1) { usage(); exit(0); }
	if(r==-2) { printf(VERSION "\n"); exit(0); }

	mp_startup();
	mps_startup();
	mpi_startup();
	mpv_startup();

	r=mpi_args_2(argc, argv);

	if(r==-1)
	{
		mpv_shutdown();
		printf("%s\n",L(MSG_BADMODE));
		printf("%s\n",mps_enumerate_modes());
		exit(1);
	}

	/* force the reading of the tags file, if one exists */
	mpi_find_tag(NULL,0);

	/* create empty text if no file is open */
	if(_mp_active==NULL)
	{
		mp_create_txt(L(MSG_UNNAMED));
		mps_auto_synhi(_mp_active);
	}

	mpv_title(NULL);
	mpi_draw_all(_mp_active);

	/* bark if bad tags */
	if(r==-2)
	{
		mpv_alert(L(MSG_TAGNOTFOUND),"");
		mpi_draw_all(_mp_active);
	}

	/* build _mpi_template_file */
	{
		char * home=getenv("HOME");

		if(home!=NULL)
			sprintf(_mpi_template_file,"%s/.mp_templates",home);
		else
			strncpy(_mpi_template_file,".mp_templates",sizeof(_mpi_template_file));
	}

	/* main loop */

	while(! _mpi_exit_requested)
	{
		c=mpv_key();

		if(mpi_action(_mp_active,c))
			mpi_draw_all(_mp_active);
	}

	/* close everything */
	mpi_shutdown();

	mpv_title(NULL);
	mpv_refresh();

	mpv_shutdown();
	mp_shutdown();

	return(0);
}
