/***************************************************************************
 Mutella - A commandline/HTTP client for the Gnutella filesharing network.

 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2
 of the License, or (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 preferences.cpp  -  Application preferences and associated classes.
 
 the original version of this file was taken from Gnucleus (http://gnucleus.sourceforge.net)                             
    begin                : Tue May 29 2001
    copyright            : (C) 2001 by
    email                : maksik@gmx.co.uk
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "mutella.h"

#include "basicstruct.h"
#include "property.h"
#include "preferences.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

MGnuPreferences::MGnuPreferences(MController* pController) : m_ipnetsetBlackList(&m_mutex), m_ipnetsetWhiteList(&m_mutex), m_setWordFilter(&m_mutex)
{
	m_pController = pController;
	
	m_pContainer = new MPropertyContainer;
	ASSERT(m_pContainer);
	MProperty* pP;

	// Local
	m_pContainer->AddSection("LocalHost");

	pP = m_pContainer->AddProperty("LocalIP",         &m_ipLocalHost,      MakeIP(127,0,0,1));
	pP->SetPropertyDescription( "IP address of the local host",
		"The value is automatically defined on the client start and modified later "
		"once a connection to the Gnutella network is established (this is required "
		"for determination of the correct IP for box with multiple interfaces");
	pP = m_pContainer->AddProperty("ForceIP",         &m_ipForcedHost,     MakeIP(0,0,0,0));
	pP->SetPropertyDescription( "Your IP address reported to the Gnutella-net",
		"Set to your firewall IP if there is a static mapping for your Mutella");
	pP = m_pContainer->AddProperty("DynDnsFirewall",  m_szDynDnsFirewall,  MAX_URL, "");
	pP->SetPropertyDescription( "Alternative to ForceIP when IP of the firewall is not static",
		"Set to the symbolic name of your firewall. The 'ForceIP' variable will be "
		"automatically updated each 10 minutes");
	pP = m_pContainer->AddProperty("AcceptRemoteIpHeader", &m_bAcceptRemoteIpHeader, true);
	pP->SetPropertyDescription( "Whether the 'LocalIP' should be updated from RemoteIP replies",
		"Must work correctly in most cases. Only effective when ForceIP is set to "
		"0.0.0.0 and DynDnsFirewall is empty. Setting this option to 'true' enables "
		"to detect firewall IP automatically in a most robust way");
	pP = m_pContainer->AddProperty("LocalPort",       &m_dwLocalPort,      6346);
	pP->SetPropertyDescription( "Local listening port",
		"Modifying the value during client operation have no effect. Edit "
		"~/.mutella/mutellarc file when client is stopped, or set LocalPort in Mutella "
		"and exit it. Upon the next client start Mutella will listen on the new port");
	pP = m_pContainer->AddProperty("ForcePort",       &m_dwForcedPort,     0);
	pP->SetPropertyDescription("Listening port reported to the Gnutella-net",
		"Normally combines with ForceIP. Should be set to the listening port on the "
		"firewall if there is a static mapping for your Mutella");
	pP = m_pContainer->AddProperty("SpeedStatic",     &m_dwSpeedStat,      0);
	pP->SetPropertyDescription( "The speed of your connection in Kbps",
		"Set this property if you want to override the auto-detected setting, "
		"othervise use 0 or negstive value");
	pP = m_pContainer->AddProperty("SpeedDynamic",    &m_dwSpeedDyn,       33);
	pP->SetPropertyDescription( "The detected speed of your connection in Kbps",
		"Set this property if you want to override the auto-detected setting");
	pP = m_pContainer->AddProperty("SaveStatePeriod", &m_nSaveStatePeriod, 10); // auto save state period in minutes
	pP->SetPropertyDescription("Period for auto-saving the configuration files",
		"Set to 0 or negative value if you want to disable the feature");
	
	// Local Firewall
	pP = m_pContainer->AddProperty("Firewall",         &m_bBehindFirewall,   true); // this flag will be automatically reset on incoming connection
	pP->SetPropertyDescription("Auto-detected firewall flag",
		"It is 'true' if it is impossible to connect to your host or 'false' otherwise. "
		"In most cases auto-detection works. If you want to manually remove search "
		"results from private subnets set 'ReachableForPush' setting to 'false'");
	pP = m_pContainer->AddProperty("ReachableForPush", &m_bReachableForPush, true);
	pP->SetPropertyDescription("Specify whether your host is reachable for push transfers",
		"If this is set to 'false', results from private subnets will be filtered out. "
		"If set to 'true', your host will attempt to request pushes to hosts which are on "
		"private subnets if your 'Firewall' flag is not 'true'");

	// Connect
	m_pContainer->AddSection("Connections");

	pP = m_pContainer->AddProperty("QuietMode",           &m_bQuietMode,   false);
	pP->SetPropertyDescription("Do not allow Mutella making outgoing connections",
		"This mode is useful when it is advisable to reduce amount of the outgoing "
		"connections, for example when running Mutella from the office computer. Has "
		"effect on Gnutella connections and push-uploads");
	pP = m_pContainer->AddProperty("MinConnections",      &m_nMinConnects, 4);
	pP->SetPropertyDescription("Minimum number of connections (not effective in LEAF mode)",
		"When number of connections drops below this value Mutella starts to make "
		"outgoing connections. This is not true when 'QuietMode' is set to 'true'");
	pP = m_pContainer->AddProperty("MaxConnections",      &m_nMaxConnects, 10);
	pP->SetPropertyDescription("Maximum number of the Gnutella-net connections, excluding leaves",
		"Does not make sense in LEAF mode");
	pP = m_pContainer->AddProperty("MaxIncomingConns",    &m_nMaxIncomingConns, -1);
	pP->SetPropertyDescription("Maximum number of the incoming Gnutella-net connections, excluding leaves",
		"Does not make any sense in LEAF mode");
	pP = m_pContainer->AddProperty("MaxConnPerSubnetA",   &m_nMaxConnPerSubnetA, -1);
	pP->SetPropertyDescription("Maximum number of the simultaneous connections per A-class subnet",
		"This property is used to improve coverage. Normally you don't have to adjust it");
	pP = m_pContainer->AddProperty("MaxConnPerSubnetB",   &m_nMaxConnPerSubnetB, 4);
	pP->SetPropertyDescription("Maximum number of the simultaneous connections per B-class subnet",
		"This property is used to improve coverage. Normally you don't have to adjust it");
	pP = m_pContainer->AddProperty("MaxConnPerSubnetC",   &m_nMaxConnPerSubnetC, 2);
	pP->SetPropertyDescription("Maximum number of the simultaneous connections per C-class subnet",
		"This property is used to improve coverage. Normally you don't have to adjust it");
	pP = m_pContainer->AddProperty("LeafModeConnects",    &m_nLeafModeConnects, 5);
	pP->SetPropertyDescription("Number of the Gnutella-net connections when in LEAF mode",
		"The number of Ultrapeers your leaf needs to connect to");
	pP = m_pContainer->AddProperty("EnableUltrapeerMode", &m_bEnableSuperNodeMode, true);
	pP->SetPropertyDescription("Enable participation in Ultrapeer election",
		"When set to 'true' enables Mutella to become ulrapeer. In order "
		"to do so the client's uptime should be substantial and the connection"
		"bandwidth should be hight enough (>32K/s)");
	pP = m_pContainer->AddProperty("ForceUltrapeerMode",  &m_bForceSuperNodeMode, false);
	pP->SetPropertyDescription("Enforce Ultrapeer mode",
		"When set to 'true' forces Mutella to become ulrapeer, regardless of "
		"the election process or local bandwidth limits and so on. Only use "
		"this option if you are certain you know what you are doing");
	pP = m_pContainer->AddProperty("MaxLeaves",           &m_nMaxLeaves, 32);
	pP->SetPropertyDescription("Maximum mumber of leaves allowed to connect to our ULTRAPEER",
		"Set it in accordance with your bandwidth");

	// Initial Gnu Web Caches
	m_pContainer->AddSection("GWebCaches");
	
	pP = m_pContainer->AddProperty("GWebCache1", m_szGWebCache1, MAX_URL, "http://www.gnucleus.net/gcache/gcache.php");
	pP->SetPropertyDescription("GWebCache server #1 (host cache) URL",
		"GWebCacheX is used to initiate the connection to the Gnutella-net "
		"Alternatively it will be used when Mutella looses the connection to the "
		"net for whatever reason and host cache is empty. Accepts standard "
		"HTTP URL format");
	pP = m_pContainer->AddProperty("GWebCache2", m_szGWebCache2, MAX_URL, "http://cache.mynapster.com/index.php");
	pP->SetPropertyDescription("GWebCache server #2 (host cache) URL",
		"GWebCacheX is used to initiate the connection to the Gnutella-net "
		"Alternatively it will be used when Mutella looses the connection to the "
		"net for whatever reason and host cache is empty. Accepts standard "
		"HTTP URL format");
	pP = m_pContainer->AddProperty("GWebCache3", m_szGWebCache3, MAX_URL, "http://gwebcache.bearshare.net/gcache.php");
	pP->SetPropertyDescription("GWebCache server #3 (host cache) URL",
		"GWebCacheX is used to initiate the connection to the Gnutella-net "
		"Alternatively it will be used when Mutella looses the connection to the "
		"net for whatever reason and host cache is empty. Accepts standard "
		"HTTP URL format");
	pP = m_pContainer->AddProperty("GWebCacheFollowRedirect", &m_bGWebCacheFollowRedirect, false);
	pP->SetPropertyDescription("Whether to follow redirects in GWebCache replies",
		"Defines whether mutella should follow redirects in GWebCache replies."
		"Initial specifications have said 'yes, follow', but now some people are"
		"concerned about possible DND attacs.");

	// Timeouts
	m_pContainer->AddSection("Timeouts");
	
	pP = m_pContainer->AddProperty("ConnectTimeout",  &m_dwConnectTimeout,  30);
	pP->SetPropertyDescription("Timeout for establishing Gnutella connections",
		"The default value works fine in most cases");
	pP = m_pContainer->AddProperty("PushTimeout",     &m_dwPushTimeout,     60);
	pP->SetPropertyDescription("Timeout for establishing PUSH-mode download",
		"The default value works fine in most cases");
	pP = m_pContainer->AddProperty("SpeedTimeout",    &m_dwSpeedTimeout,    60);
	pP->SetPropertyDescription("Defines how long download is allowed be under min-speed",
		"The default value works fine in most cases");
	pP = m_pContainer->AddProperty("TransferTimeout", &m_dwTransferTimeout, 60);
	pP->SetPropertyDescription("Timeout for the upload and download data transfer",
		"The default value works fine in most cases");
	
	// paths
	m_pContainer->AddSection("Paths");
	
	pP = m_pContainer->AddProperty("DownloadPath", m_szDownloadPath, MAX_PATH, GetDefaultDir().c_str());
	pP->SetPropertyDescription("Path to the download directory",
		"Defines the location on your file system, where the downloaded files are to be stored");
	pP = m_pContainer->AddProperty("SharePath", m_szSharePath, MAX_PATH, "");
	pP->SetPropertyDescription("Path to the shared directory",
		"Path to the shared directory. Changing it during Mutella session has no "
		"immediate effect. To actualise new shared path you need to issue `scan' "
		"command. To share files located on different file systems use symbolic links");

	// Search
	m_pContainer->AddSection("SearchOptions");
	
	pP = m_pContainer->AddProperty("StrictSearch",      &m_bStrictSearch,        false);
	pP->SetPropertyDescription("Whether not to run 'passive' searches",
		"When set to 'true' only the search results that are originating from "
		"our requests are checked. Otherwise, all query-reply traffick is checked. "
		"Only set it to rtue if you want to save your CPU cycles at expence of "
		"reduced search efficiency");
	pP = m_pContainer->AddProperty("GroupStrict",       &m_bStrictSha1Grouping,  false);
	pP->SetPropertyDescription("Enforces strict SHA1 grouping",
		"When set to 'true' only results with exactly the same SHA1 are grouped");
	pP = m_pContainer->AddProperty("MinFileSize",       &m_nMinFileKSize,        1024); //1M
	pP->SetPropertyDescription("Minimum file size (in K) for search results",
		"Required for protection against millions of partial files people share as well as other spam");
	pP = m_pContainer->AddProperty("MaxFileSize",       &m_nMaxFileKSize,        2097152); //2G (spam protection)
	pP->SetPropertyDescription("Maximum file size (in K) for search results",
		"Required for spam-protection. Default is very reasonable");
	pP = m_pContainer->AddProperty("RequeryPeriod",     &m_nResubmitSearches,    30); // resubmit all the searches in 30 minutes
	pP->SetPropertyDescription("Pereod for query resubmition, in minutes",
		"Default of 30 minutes is prescribed by teh GDF. Note, that shortening this period "
		"would nor improve search efficiency much because major Gnutella servents whould "
		"drop repeated queries");
	pP = m_pContainer->AddProperty("MaxSearches",       &m_nMaxSearches,         64); // performance limitations
	pP->SetPropertyDescription("Maximum number of the simultaneous searches",
		"It's only there for a fool protection, which basically stops users from creating "
		"infinite number of searches. Set in accordance with your hosts performance");
	pP = m_pContainer->AddProperty("MaxResults",        &m_nMaxPerSearchResults, 256);// performance limitations
	pP->SetPropertyDescription("Maximum number of results per search",
		"High number of search results consumes more CPU and memory. Keep this in mind.");
	pP = m_pContainer->AddProperty("SaveSearches",      &m_bSaveSearches,        true);// save/load searches between sessions
	pP->SetPropertyDescription("Whether searches need to be persistent",
		"The default is very reasonable");

	// Search Screen
	m_pContainer->AddSection("SearchFilters");
	
	pP = m_pContainer->AddProperty("BlackListActive",   &m_bBlackListActive,  true);
	pP->SetPropertyDescription("Activates and deactivates the 'BlackList' feature",
		"Enables to quickly de/re-activate the 'BlackList' feature");
	pP = m_pContainer->AddSet("BlackList",  &m_ipnetsetBlackList); // put spammers in there
	pP->SetPropertyDescription("Put spammers or hammering hosts here",
		"Hosts/nets listed in here are denied connections and uploads. Search results from "
		"those hosts are ignored");
	pP = m_pContainer->AddProperty("WordFilterActive",  &m_bWordFilterActive, true);
	pP->SetPropertyDescription("Activates and deactivates the 'WordFilter' feature",
		"Enables to quickly de/re-activate the 'WordFilter' feature");
	pP = m_pContainer->AddSet("WordFilter", &m_setWordFilter);  // list bad words you dont whant to see in your results
	pP->SetPropertyDescription("Ignore results containing certain words",
		"Search results containing the words listed in here are dropped. Changing the filter "
		"does not affect the results already collected");

	// Share
	m_pContainer->AddSection("ShareOptions");
	
	pP = m_pContainer->AddProperty("ShareFilter",   m_szShareFilter,   MAX_PATH, "");
	pP->SetPropertyDescription("File name filter used when scanning the shared directory",
		"Space separated list of file patterns like *.avi *.mp? *.mpeg and so on. "
		"Empty string means no filterding is done. Changing it during Mutella "
		"session has no immediate effect in the same way as for 'SharePath'. To "
		"actualise new filter in is necessary to issue `scan' command.");
	pP = m_pContainer->AddProperty("ReplyFilePath", &m_bReplyFilePath, true); // include relative path to the share names
	pP->SetPropertyDescription("Add file path to search replies",
		"Defines whether to add a file path to the search replies we send out. The path added is "
		"relative to the shared directory");
	pP = m_pContainer->AddProperty("MaxReplies",    &m_nMaxReplies,    64);
	pP->SetPropertyDescription("Maximum number of the replies for an incoming search request",
		"Default is OK for most cases");
	pP = m_pContainer->AddProperty("ReplyIfAvail",  &m_bSendOnlyAvail, true); // makes sense if sharing only big files
	pP->SetPropertyDescription("Send search-replies only if upload slots are available",
		"Reduces the load of the host, especially when popular content is shared. Only set "
		"to 'false' if you are certain that you understand the consequencies (or you have a T3 line)");
	pP = m_pContainer->AddProperty("ShareDownloadDir", &m_bShareDownloadDir, true);
	pP->SetPropertyDescription("Auto-shares the downloaded files",
		"defines whether the frehly-downloaded files are shared automatically");
	pP = m_pContainer->AddProperty("ShareDotFiles", &m_bShareDotFiles, false);
	pP->SetPropertyDescription("Enables to share 'hidden' files",
		"You most probably won't need that");

	// Transfer
	m_pContainer->AddSection("TransferOptions");
	
	pP = m_pContainer->AddProperty("MaxDownloads",        &m_nMaxDownloads,        20);
	pP->SetPropertyDescription("Maximum number of the simulteneous downloads",
		"Only limited by your host's performance. In any case, default is big enough");
	pP = m_pContainer->AddProperty("MaxUploads",          &m_nMaxUploads,          10);
	pP->SetPropertyDescription("Maximum number of the simultaneous uploads",
		"Maximum number of the upload 'slots' on your system. Set it in accordance "
		"with your connection speed, host performance and the amount of good will");
	pP = m_pContainer->AddProperty("RetryDelay",          &m_nRetryWait,           10); // delay between download retries
	pP->SetPropertyDescription("Download retry delay",
		"Retry delay for broken or busy downloads. Do not set it to too small wallue, "
		"people will put you into their blacklists");
	pP = m_pContainer->AddProperty("MaxPerHostDownloads", &m_nMaxPerHostDownloads, 2);
	pP->SetPropertyDescription("Maximum number of parallel downloads per host",
		"Maximum number of simultaneous downloads from the same host");
	pP = m_pContainer->AddProperty("MaxPerHostUploads",   &m_nMaxPerHostUploads,   2);
	pP->SetPropertyDescription("Maximum number of parallel uploads per host",
		"Maximum number of simultaneous uploads to the same host");
	pP = m_pContainer->AddProperty("MaxPerFileUploads",   &m_nMaxPerFileUploads,   3); // to enable sharing both rare and popular content
	pP->SetPropertyDescription("Maximum number of parallel uploads per file",
		"Maximum number of simultaneous uploads of the same file. Thus option allows "
		"so save some uploads slots for rare stuff in case you also share something "
		"very popular");
	pP = m_pContainer->AddProperty("MinDownloadSpeed",  &m_dMinDownSpeed,      -1); // drop download if it is slower than that
	pP->SetPropertyDescription("Minimum download speed in bytes per second",
		"Minimum download speed defines the threshold, at which download connection "
		"is dropped and another connection is attemped. Mutella only drops connections "
		"if it has alternative download hosts");
	pP = m_pContainer->AddProperty("MinUploadSpeed",    &m_dMinUpSpeed,        -1); // same about uploads
	pP->SetPropertyDescription("Minimum upload speed in bytes per second",
		"Minimum upload speed defines the threshold, at which upload connection is "
		"considered to be 'too ineffective' and dropped");
	pP = m_pContainer->AddSet("WhiteList",  &m_ipnetsetWhiteList); // put your friends in there so that they get uploads
	pP->SetPropertyDescription("List of IPs or nets, for which uploads are always given",
		"List of IPs or nets, for which uploads are always given, regardless of the other "
		"upload limits set. Put your friends in here to guarantee effective file transfer "
		"to their hosts");

	// Bandwidth
	m_pContainer->AddSection("Bandwidth");
	
	pP = m_pContainer->AddProperty("BandwidthTotal",    &m_dBandwidthTotal,    -1); // not really functional
	pP->SetPropertyDescription("Bandwidth allocated for all non-ours traffic",
		"Limits the amount of bandwidth (in Kbytes per second) given for both uploads "
		"and Gnutella-net. Works well in combination with BandwidthConnects, when the "
		"later is less or equal to 50% of the BandwidthTotal. Actual upload traffic is "
		"limited by the (BandwidthTotal - actual_connection_traffic)");
	pP = m_pContainer->AddProperty("BandwidthUpload", &m_dBandwidthTransfer, 10); // 10 K/sec
	pP->SetPropertyDescription("Bandwidth allocated for uploads",
		"Limits the amount of bandwidth (in Kbytes per second) allowed for uploads. "
		"When BandwidthTotal is set actual upload traffic is limited by the minimum "
		"between BandwidthUpload and (BandwidthTotal - actual_connection_traffic)");
	pP = m_pContainer->AddProperty("BandwidthConnects", &m_dBandwidthConnects, -1);
	pP->SetPropertyDescription("Bandwidth allocated for Guntella-net connections",
		"The amount of network bandwidth (in Kbytes per second) allowed to spend for "
		"Gnutella-net related traffic. Because of the nature of this traffic Mutella "
		"is unable to follow this limitation exactly. The value given here will only "
		"approximately limit the connection bandwidth");
	pP = m_pContainer->AddProperty("BandwidthDownload", &m_dBandwidthDownl,    -1); // unlimited
	pP->SetPropertyDescription("Bandwidth allocated for downloads",
		"Limits the amount of bandwidth (in Kbytes per second) allowed for downloads. "
		"This option enables one to preserve a bit of bandwidth for e.g. internet browsing, "
		"even while mutella is busy downloading things");
	
	m_pContainer->SetCurrentSection(NULL);
}

MGnuPreferences::~MGnuPreferences()
{
	delete m_pContainer;
}

CString MGnuPreferences::GetDefaultDir()
{
	return "~/mutella";
}

